package com.ejianc.business.assist.material.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.assist.material.bean.MaterialCheckDetailEntity;
import com.ejianc.business.assist.material.bean.MaterialCheckEntity;
import com.ejianc.business.assist.material.bean.MaterialContractEntity;
import com.ejianc.business.assist.material.bean.MaterialSettleEntity;
import com.ejianc.business.assist.material.enums.BillTypeEnum;
import com.ejianc.business.assist.material.mapper.MaterialSettleMapper;
import com.ejianc.business.assist.material.service.IMaterialCheckDetailService;
import com.ejianc.business.assist.material.service.IMaterialCheckService;
import com.ejianc.business.assist.material.service.IMaterialContractService;
import com.ejianc.business.assist.material.service.IMaterialSettleService;
import com.ejianc.business.assist.material.vo.MaterialCheckDetailVO;
import com.ejianc.business.assist.material.vo.MaterialSettleVO;
import com.ejianc.business.assist.material.vo.record.MaterialSettleRecordVO;
import com.ejianc.business.assist.rmat.enums.SettleEnum;
import com.ejianc.business.assist.rmat.utils.DateUtil;
import com.ejianc.business.assist.rmat.vo.report.RmatConSettleReportVO;
import com.ejianc.business.contractbase.pool.enums.ContractTypeEnum;
import com.ejianc.business.contractbase.pool.enums.SettleSourceTypeEnum;
import com.ejianc.business.contractbase.pool.settlepool.api.ISettlePoolApi;
import com.ejianc.business.contractbase.pool.settlepool.vo.SettlePoolVO;
import com.ejianc.business.contractpub.util.BeanConvertorUtil;
import com.ejianc.business.pro.rmat.utils.PushSupUtil;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseEntity;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 辅料中心采购结算单
 * 
 * @author generator
 * 
 */
@Service("materialSettleService")
public class MaterialSettleServiceImpl extends BaseServiceImpl<MaterialSettleMapper, MaterialSettleEntity> implements IMaterialSettleService{


    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String BILL_CODE = "ASSISTMATERIAL_CONTRACT_SETTLE";//此处需要根据实际修改
    private static final String OPERATE = "SETTLE_BILL_SYNC";
    private static final String BILL_TYPE = BillTypeEnum.辅料中心采购合同结算单.getCode();
    private static final String BILL_NAME = BillTypeEnum.辅料中心采购合同结算单.getName();

    private static final String PUSH_BILL_SERVER_URL = "/ejc-supbusiness-web/openapi/assistmaterial/settle/billSync";
    private static final String DEL_SUP_BILL_SERVER_URL = "/ejc-supbusiness-web/openapi/assistmaterial/settle/billDel";


    @Autowired
    private IBillCodeApi billCodeApi;
    
    @Autowired
    private IMaterialContractService contractService;

    @Autowired
    private ISettlePoolApi settlePoolApi;

    @Autowired
    private IMaterialCheckService checkService;

    @Autowired
    private IMaterialCheckDetailService checkDetailService;

    @Autowired
    private PushSupUtil pushSupUtil;

    @Autowired
    private MaterialSettleMapper materialSettleMapper;

    @Override
    public MaterialSettleVO saveOrUpdate(MaterialSettleVO saveOrUpdateVO) {
        MaterialSettleEntity entity = BeanMapper.map(saveOrUpdateVO, MaterialSettleEntity.class);
        if(entity.getId() == null || entity.getId() == 0){
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(),saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }

            // 签字状态
            entity.setSupplierSignStatus(0);
        }

        // 校验合同
        validateContract(entity.getContractId(), entity.getId());
        // 校验结算日期
        MaterialSettleVO lastVo = queryLastSettleTaxMny(entity.getContractId(), entity.getId());
        if (lastVo.getId() != null) {
            int compareDate = DateUtil.compareDate(lastVo.getSettleDate(), entity.getSettleDate());
            if (compareDate > 0) {
                throw new BusinessException("本次结算日期" + DateUtil.formatDate(entity.getSettleDate()) + "不能小于上次结算日期" + DateUtil.formatDate(lastVo.getSettleDate()));
            }
        }

        this.saveOrUpdate(entity, false);

        return BeanMapper.map(selectById(entity.getId()), MaterialSettleVO.class);
    }

    @Override
    public String validateContract(Long contractId, Long billId) {
        // 同一个合同只能存在一个自由态或审批中的单据
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.NOT_IN, "1,3"));
        if(billId != null){
            queryParam.getParams().put("id", new Parameter(QueryParam.NE, billId));
        }
        List<MaterialSettleEntity> list = super.queryList(queryParam, false);
        if (ListUtil.isNotEmpty(list)) {
            throw new BusinessException("当前合同存在非审批通过态的结算单，不允许新增!");
        }
        return "校验通过！";
    }

    @Override
    public MaterialSettleVO queryLastSettleTaxMny(Long contractId, Long billId) {
        LambdaQueryWrapper<MaterialSettleEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(MaterialSettleEntity::getContractId, contractId);
        wrapper.in(MaterialSettleEntity::getBillState, Arrays.asList(1, 3));
        if(billId != null){
            wrapper.ne(MaterialSettleEntity::getId, billId);
        }
        wrapper.orderByDesc(MaterialSettleEntity::getCreateTime);

        List<MaterialSettleEntity> list = this.list(wrapper);

        if (ListUtil.isEmpty(list)) {
            return new MaterialSettleVO();
        }else {
            return BeanMapper.map(list.get(0), MaterialSettleVO.class);
        }

    }

    @Override
    public List<MaterialCheckDetailVO> queryCheckData(Long contractId, String lastSettleDate, String settleDate) {
        LambdaQueryWrapper<MaterialCheckEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(MaterialCheckEntity::getContractId, contractId);
        wrapper.eq(MaterialCheckEntity::getSettleFlag, 0);
        wrapper.in(MaterialCheckEntity::getBillState, Arrays.asList(1, 3));
        wrapper.orderByDesc(MaterialCheckEntity::getCreateTime);
        if (StringUtils.isBlank(lastSettleDate)) {
            wrapper.le(MaterialCheckEntity::getCheckDate, settleDate);
        }else {
            wrapper.between(MaterialCheckEntity::getCheckDate, lastSettleDate, settleDate);
        }

        List<MaterialCheckEntity> list = checkService.list(wrapper);
        if (ListUtil.isEmpty(list)) {
            return new ArrayList<>();
        }

        List<Long> idsList = list.stream().map(BaseEntity::getId).collect(Collectors.toList());
        LambdaQueryWrapper<MaterialCheckDetailEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.in(MaterialCheckDetailEntity::getCheckId, idsList);
        List<MaterialCheckDetailEntity> detailList = checkDetailService.list(queryWrapper);
        if (ListUtil.isEmpty(detailList)) {
            return new ArrayList<>();
        }

        return BeanMapper.mapList(detailList, MaterialCheckDetailVO.class);

    }

    @Override
    public MaterialSettleRecordVO querySettleRecord(Long id) {
        MaterialContractEntity entity = contractService.selectById(id);

        MaterialSettleRecordVO vo = new MaterialSettleRecordVO();
        vo.setId(id);
        vo.setMainContractId(id);
        vo.setContractMny(entity.getContractMny());
        vo.setContractTaxMny(entity.getContractTaxMny());
        vo.setPerformanceStatus(entity.getPerformanceStatus());
        vo.setChangeStatus(entity.getChangeStatus());
        vo.setSignatureStatus(entity.getSignatureStatus());

        // 查询变更记录
        LambdaQueryWrapper<MaterialSettleEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(MaterialSettleEntity::getContractId, id);
        wrapper.orderByDesc(MaterialSettleEntity::getCreateTime);
        List<MaterialSettleEntity> settleList = super.list(wrapper);

        if (ListUtil.isNotEmpty(settleList)) {
            vo.setTotalSettleTaxMny(settleList.get(0).getTotalSettleTaxMny());
            vo.setTotalSettleMny(settleList.get(0).getTotalSettleMny());
            vo.setSettleRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalSettleTaxMny(), vo.getTotalSettleMny()), new BigDecimal("100")));
        }

        vo.setDetailList(BeanMapper.mapList(settleList, MaterialSettleVO.class));

        return vo;
    }

    @Override
    public Boolean pushSettleToPool(MaterialSettleVO vo) {
        SettlePoolVO poolVO = new SettlePoolVO();
        try {
            logger.info("结算单对象 -> 结算池对象自动转换开始");
            // 对象自动转换
            BeanConvertorUtil.convert(vo, poolVO);

            // 个别字段需要手动封装
            poolVO.setSourceId(vo.getId());
            poolVO.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            poolVO.setSourceType(SettleSourceTypeEnum.辅料中心周转材采购结算.getCode());
            poolVO.setSourceTypeName(SettleSourceTypeEnum.辅料中心周转材采购结算.getName());
            poolVO.setSettleProperty(Integer.valueOf(SettleEnum.支出.getCode()));
            poolVO.setSettlePropertyName(SettleEnum.支出.getName());
            poolVO.setUltimateFlag(Integer.valueOf(vo.getSettleType()));
            
            // 查询合同
            MaterialContractEntity contractEntity = contractService.selectById(vo.getContractId());
            poolVO.setSupplementFlag(contractEntity.getSupplementFlag());
            poolVO.setMaiContractId(contractEntity.getId());
            poolVO.setMaiContractCode(contractEntity.getBillCode());
            poolVO.setMaiContractName(contractEntity.getContractName());
            poolVO.setSignDate(contractEntity.getSignDate());
            poolVO.setContractType(ContractTypeEnum.辅料中心周转材采购合同.getTypeCode());
            poolVO.setContractTypeName(ContractTypeEnum.辅料中心周转材采购合同.getTypeName());
            poolVO.setContractFlag(1);// 合同标识
            
            poolVO.setCreateTime(vo.getCreateTime());
            poolVO.setCreateUserCode(vo.getCreateUserCode());
            poolVO.setUpdateTime(vo.getUpdateTime());
            poolVO.setUpdateUserCode(vo.getUpdateUserCode());


            CommonResponse<SettlePoolVO> res = settlePoolApi.saveOrUpdateSettle(poolVO);
            if(res.isSuccess()) {
                logger.info("结算单推送至结算池成功！结算单id-{}", vo.getId());
                return true;
            } else {
                logger.error("结算单推送合同池失败！结算单id-{}，{}", vo.getId(), res.getMsg());
            }
        } catch (Exception e) {
            logger.error("结算单推送合同池失败！结算单id-{}", vo.getId(), e);
        }
        return false;
    }

    @Override
    public Boolean delSettleFromPool(Long id) {
        SettlePoolVO poolVO = new SettlePoolVO();
        poolVO.setSourceId(id);

        CommonResponse<SettlePoolVO> res = settlePoolApi.deleteSettle(poolVO);
        if (res.isSuccess()) {
            logger.info("将结算单从结算池中删除成功！结算单id-{}", id);
            return true;
        }

        logger.error("将结算单从结算池中删除失败！结算单id-{}，{}", id, res.getMsg());
        return false;
    }

    /**
     * 供方签字确认信息回写
     * @param request
     * @return
     */
    @Override
    public String updateBillSupSignSyncInfo(HttpServletRequest request) {
        String billId = request.getParameter("billId");
        MaterialSettleEntity entity = super.selectById(billId);
        String msg = pushSupUtil.updateBillSupSignSyncInfo(request, (JSONObject) JSONObject.toJSON(entity),
                MaterialSettleEntity.class, OPERATE, BILL_TYPE, BILL_NAME);
        logger.info("返回消息msg" + msg);
        return msg;
    }

    /**
     * 单据推送到供方协同服务
     * @param entity
     * @return
     */
    @Override
    public Boolean pushBillToSupCenter(MaterialSettleEntity entity) {
        Boolean syncFlag = pushSupUtil.pushBillToSupCenter((JSONObject)JSONObject.toJSON(entity), OPERATE,
                BILL_TYPE, BILL_NAME, PUSH_BILL_SERVER_URL);
        return syncFlag;
    }

    /**
     * 将推送至供方的单据作废
     * @param entity
     * @return
     */
    @Override
    public Boolean delPushBill(MaterialSettleEntity entity) {
        Boolean delSuc = pushSupUtil.delPushBill((JSONObject) JSONObject.toJSON(entity), OPERATE,
                BILL_TYPE, BILL_NAME, DEL_SUP_BILL_SERVER_URL);
        return delSuc;

    }

    @Override
    public Map<String, Object> count(Map<String, Object> paramMap) {
        Map<String, Object> resp = new HashMap<>();
        Map<String, Object> rs = materialSettleMapper.countSettleInfo(paramMap);
        if(null == rs) {
            rs = new HashMap<>();
        }

        resp.put("total", null == rs.get("total") ? 0 : Integer.valueOf(rs.get("total").toString()));
        resp.put("curTotalSettleMny", null == rs.get("curTotalSettleMny") ? BigDecimal.ZERO : new BigDecimal(rs.get("curTotalSettleMny").toString()));
        resp.put("curTotalSettleTaxMny", null == rs.get("curTotalSettleTaxMny") ? BigDecimal.ZERO : new BigDecimal(rs.get("curTotalSettleTaxMny").toString()));
        resp.put("totalSettleMnyThisYear", null == rs.get("totalSettleMnyThisYear") ? BigDecimal.ZERO : new BigDecimal(rs.get("totalSettleMnyThisYear").toString()));
        resp.put("totalSettleTaxMnyThisYear", null == rs.get("totalSettleTaxMnyThisYear") ? BigDecimal.ZERO : new BigDecimal(rs.get("totalSettleTaxMnyThisYear").toString()));
        resp.put("totalStartSettleMny", null == rs.get("totalStartSettleMny") ? BigDecimal.ZERO : new BigDecimal(rs.get("totalStartSettleMny").toString()));
        resp.put("totalStartSettleTaxMny", null == rs.get("totalStartSettleTaxMny") ? BigDecimal.ZERO : new BigDecimal(rs.get("totalStartSettleTaxMny").toString()));

        return resp;
    }

    @Override
    public List<RmatConSettleReportVO> pageList(Map<String, Object> paramMap) {
        return materialSettleMapper.pageList(paramMap);
    }

}
