package com.ejianc.foundation.cust.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.ejianc.foundation.cust.bean.BusinessColumnEntity;
import com.ejianc.foundation.cust.bean.BusinessTableEntity;
import com.ejianc.foundation.cust.mapper.BusinessTableMapper;

import cn.hutool.core.collection.CollectionUtil;

/**
 * <pre>
 * 针对一张表的操作者，系统表的操作有以下几种
 * 1 针对表本身的：建表，删表
 * 2 针对字段：增字段，改字段，删字段
 * 3 针对数据：增，删，查，改
 * </pre>
 */
public class TableOperator {
	protected Logger logger = LoggerFactory.getLogger(this.getClass());
	/**
	 * 表对象
	 */
	protected BusinessTableEntity table;
	private BusinessTableMapper businessTableMapper;

	/**
	 * @param table
	 * @param jdbcTemplate
	 */
	public TableOperator(BusinessTableMapper businessTableMapper, BusinessTableEntity table) {
		this.businessTableMapper = businessTableMapper;
		this.table = table;
	}
	
	private String columnToSql(BusinessColumnEntity column) {
		StringBuilder sb = new StringBuilder();
		sb.append("" + column.getName() + "");
		if (ColumnType.CLOB.equalsWithKey(column.getType())) {
			sb.append(" text");
		} else if (ColumnType.DATE.equalsWithKey(column.getType())) {
			sb.append(" datetime");
		} else if (ColumnType.NUMBER.equalsWithKey(column.getType())) {
			sb.append(" decimal(" + column.getColLength() + "," + column.getColDecimal() + ")");
		} else if (ColumnType.VARCHAR.equalsWithKey(column.getType())) {
			sb.append(" varchar(" + column.getColLength() + ")");
		}

		if (1 == column.getRequired() || 1 ==column.getPrimary()) {
			sb.append(" NOT NULL");
		} else {
			sb.append(" NULL");
		}
		sb.append(" COMMENT '" + column.getComment() + "'");
		return sb.toString();
	}

	/**
	 * <pre>
	 * 创建表
	 * </pre>
	 */
	public void createTable() {
		if (isTableCreated()) {
			logger.debug("表[" + table.getName() + "(" + table.getComment() + ")]已存在数据库中，无需再次生成");
			return;
		}
		// 建表语句
		StringBuilder sql = new StringBuilder();
		sql.append("CREATE TABLE " + table.getName() + " (" + "\n");
		for (BusinessColumnEntity column : table.getColumns()) {
			sql.append(columnToSql(column) + ",\n");
		}
		sql.append("PRIMARY KEY (id)" + "\n)");
		//默认设置字符集
		sql.append(" ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE utf8_bin ");
		if (StringUtils.isNotBlank(table.getComment())) {
			sql.append(" COMMENT='" + table.getComment() + "'");
		}
		// 建表结束
		sql.append(";");
		businessTableMapper.createBusiTable(sql.toString());
	}

	/**
	 * <pre>
	 * 删除表
	 * </pre>
	 */
	public void dropTable() {
		if (!isTableCreated()) {
			return;
		}
		String sql = "drop table " + table.getName() + "";
		businessTableMapper.dropBusiTable(sql);
	}

	/**
	 * <pre>
	 * 表是否被生成过
	 * 或者说，表是否已存在数据库
	 * </pre>
	 *
	 * @return
	 */
	public boolean isTableCreated() {
		String sql = "select count(1) from information_schema.TABLES t where table_name = "+table.getName();
		List<String> resultList = businessTableMapper.queryTableExist(sql);
		return (resultList != null && resultList.size() > 0) ? true : false;
	}

	/**
	 * <pre>
	 * 增加字段
	 * </pre>
	 *
	 * @param column
	 *            字段
	 */
	public void addColumn(BusinessColumnEntity column) {
		StringBuilder sql = new StringBuilder();
		sql.append("ALTER TABLE " + table.getName() + "");
		sql.append(" ADD COLUMN " + columnToSql(column) + ";");
		businessTableMapper.executeSql(sql.toString());
	}

	/**
	 * <pre>
	 * 更新字段
	 * </pre>
	 *
	 * @param column
	 *            字段
	 */
	public void updateColumn(BusinessColumnEntity column) {
		StringBuilder sql = new StringBuilder();
		sql.append("ALTER TABLE " + table.getName() + "");
		sql.append(" MODIFY COLUMN " + columnToSql(column) + ";");
		businessTableMapper.executeSql(sql.toString());
	}

	/**
	 * <pre>
	 * 删除字段
	 * </pre>
	 *
	 * @param column
	 */
	public void dropColumn(String columnName) {
		StringBuilder sql = new StringBuilder();
		sql.append("ALTER TABLE " + table.getName() + "");
		sql.append(" DROP COLUMN " + columnName + ";");
		businessTableMapper.executeSql(sql.toString());
	}
	
	/**
	 * 获取DB信息
	 * 
	 * @return
	 */
	public BusinessTableEntity getDbTable(Long tableId) {
		return businessTableMapper.selectById(tableId);
	}

	/**
	 * <pre>
	 * 插入数据
	 * </pre>
	 *
	 * @param data
	 *            数据
	 */
	public void insertData(Map<String, Object> data) {
		StringBuilder sql = new StringBuilder();
		sql.append("INSERT INTO " + table.getName());

		StringBuilder columnNameSql = new StringBuilder();// 字段名字sql
		StringBuilder paramNameSql = new StringBuilder();// 参数sql
		for (Entry<String, Object> entry : data.entrySet()) {
			if (columnNameSql.length() > 0) {
				columnNameSql.append(",");
				paramNameSql.append(",");
			}
			columnNameSql.append(entry.getKey());
			paramNameSql.append("'"+entry.getValue()+"'");
		}
		sql.append("(" + columnNameSql + ") VALUES(" + paramNameSql + ")");
		businessTableMapper.executeSql(sql.toString());
	}

	/**
	 * <pre>
	 * 根据主键删除数据
	 * </pre>
	 *
	 * @param id
	 *            主键值
	 */
	public void deleteData(Object id) {
		String sql = "DELETE FROM " + table.getName() + " where id = "+id;
		businessTableMapper.executeSql(sql);
	}

	/**
	 * <pre>
	 * 根据参数删除数据
	 * </pre>
	 *
	 * @param param
	 *            参数
	 */
	public void deleteData(Map<String, Object> param) {
		if (param.isEmpty()) {
			throw new RuntimeException("操作删除表[" + table.getComment() + "(" + table.getName() + ")]时，条件参数为空(会导致全表数据清空)");
		}

		StringBuilder sql = new StringBuilder();
		sql.append("DELETE FROM " + table.getName() + " where ");
		for (Entry<String, Object> entry : param.entrySet()) {
			sql.append(entry.getKey() + " = '"+entry.getValue()+"' ");
		}
		businessTableMapper.executeSql(sql.toString());
	}

	/**
	 * <pre>
	 * 更新数据
	 * 获取数据中的主键来进行更新
	 * </pre>
	 *
	 * @param data
	 *            数据
	 */
	public void updateData(Map<String, Object> data) {
		Object id = data.get("id");
		if (id == null) {
			throw new RuntimeException("操作更新表[" + table.getComment() + "(" + table.getName() + ")]时，参数中有没主键[(id)]");
		}

		StringBuilder sql = new StringBuilder();
		sql.append("UPDATE " + table.getName() + " SET ");
		for (Entry<String, Object> entry : data.entrySet()) {
			// 主键跳过
			if (entry.getKey().equals("id")) {
				continue;
			}
			sql.append(entry.getKey() + " = '"+entry.getValue()+"'");
		}
		// 主键设置为过滤条件
		sql.append(" WHERE id = "+id);
		businessTableMapper.executeSql(sql.toString());
	}

	/**
	 * <pre>
	 * 根据主键获取唯一数据
	 * </pre>
	 *
	 * @param columnName
	 *            要查的字段
	 * @param id
	 *            主键值
	 * @return
	 */
	public Map<String, Object> selectData(List<String> columnName, Object id) {
		Map<String, Object> param = new HashMap<>();
		param.put("id", id);
		List<Map<String, Object>> list = selectData(columnName, param);
		if (!list.isEmpty()) {
			return list.get(0);
		}
		return null;
	}

	/**
	 * <pre>
	 * 根据主键获取唯一数据
	 * </pre>
	 *
	 * @param id
	 *            主键值
	 * @return
	 */
	public Map<String, Object> selectData(Object id) {
		Map<String, Object> param = new HashMap<>();
		param.put("id", id);
		List<Map<String, Object>> list = selectData(param);
		if (!list.isEmpty()) {
			return list.get(0);
		}
		return null;
	}

	/**
	 * <pre>
	 * </pre>
	 *
	 * @param param
	 *            参数
	 * @return
	 */
	public List<Map<String, Object>> selectData(Map<String, Object> param) {
		return selectData(null, param);
	}

	/**
	 * <pre>
	 * </pre>
	 *
	 * @param columnName
	 *            要查的字段
	 * @param param
	 *            参数列表
	 * @return
	 */
	public List<Map<String, Object>> selectData(List<String> columnName, Map<String, Object> param) {
		StringBuilder sql = new StringBuilder();
		if (CollectionUtil.isEmpty(columnName)) {
			sql.append("SELECT * FROM " + table.getName());
		} else {
			sql.append("SELECT");
			for (String cn : columnName) {
				if (!sql.toString().endsWith("SELECT")) {
					sql.append(",");
				}
				sql.append(" " + cn);
			}
			sql.append(" FROM " + table.getName());
		}

		sql.append(" WHERE ");

		for (Entry<String, Object> entry : param.entrySet()) {
			sql.append(entry.getKey() + " = '"+entry.getValue()+"'");
		}
		return businessTableMapper.executeSelectSql(sql.toString());
	}

	/**
	 * <pre>
	 * 同步table和数据库中的字段信息
	 * 目前只处理字段的增删
	 * </pre>
	 */
	public void syncColumn() {
		// 未生成表，不处理
		if (!isTableCreated()) {
			return;
		}
		Set<String> dbColumnNames = new HashSet<>();// 数据库中存在的字段名
		BusinessTableEntity dbTable = getDbTable(table.getId());
		for (BusinessColumnEntity c : dbTable.getColumns()) {
			dbColumnNames.add(c.getName());
		}

		for (String columnName : dbColumnNames) {
			if (this.table.getColumn(columnName) == null) {// 数据库表内有，但是结构没有，删除
				dropColumn(columnName);
			}
		}

		for (BusinessColumnEntity column : table.getColumns()) {
			boolean exits = false;
			for (String columnName : dbColumnNames) {
				if (columnName.equalsIgnoreCase(column.getName())) {
					exits = true;
					break;
				}
			}
			if (!exits) {// 结构有，数据库表内没有，增加
				addColumn(column);
			} else if (!dbTable.getColumn(column.getName()).equals(column)) {
				updateColumn(column);// 更新一遍结构
			}
		}
	}

}
