package com.ejianc.business.contractbase.report.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.contractbase.home.consts.ProjectSurveyEnum;
import com.ejianc.business.contractbase.pool.settlepool.bean.SettlePoolEntity;
import com.ejianc.business.contractbase.pool.settlepool.service.ISettlePoolService;
import com.ejianc.business.contractbase.report.consts.DateInConsts;
import com.ejianc.business.contractbase.report.vo.SupplierSettleVO;
import com.ejianc.business.contractbase.utils.EJCDateUtil;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

import static com.ejianc.framework.skeleton.template.BaseServiceImpl.changeToQueryWrapper;


/**
 * @author songlx
 * @description: 供应商报表-履约分析
 * @date: 2022/5/26
 */
@RestController
@RequestMapping("/supplierReport")
public class SupplierReportController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi iOrgApi;


    @Autowired
    private ISettlePoolService service;

    /**
     * @description: 履约分析柱状图
     * @author songlx
     * @date: 2022/5/26
     */
    @RequestMapping(value = "/settleList", method = RequestMethod.GET)
    public CommonResponse<List<SupplierSettleVO>> settleList(@RequestParam(value = "supplierId", required = false) Long supplierId, @RequestParam(defaultValue = "near6Mon") String dateIn, @RequestParam(value = "orgId", required = false) Long orgId) {

        QueryWrapper wrapper = new QueryWrapper();
        wrapper.eq("tenant_id", InvocationInfoProxy.getTenantid());
        wrapper.eq(supplierId != null, "partyb_id", supplierId);

        Long _orgId = orgId == null ? InvocationInfoProxy.getOrgId() : orgId;

        //若当前上下文为项目部，则根据项目部Id来进行查询
        if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
            wrapper.eq("org_id", _orgId);
        } else {
            CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(_orgId);
            if (!orgResp.isSuccess()) {
                logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
                return CommonResponse.error("查询失败，获取组织信息失败！");
            }
            List<Long> ids = orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
            wrapper.in("parent_org_id", ids);
        }

        String startMonth = null;
        String endMonth = null;
        if (DateInConsts.TIME_RANGE_THIS_YEAR.equals(dateIn)) {
            startMonth = EJCDateUtil.getYear() + "-01";
            endMonth = EJCDateUtil.getYear() + "-12";
        } else if (DateInConsts.TIME_RANGE_3_MONTH.equals(dateIn)) {
            startMonth = EJCDateUtil.getMonth(-2);
            endMonth = EJCDateUtil.getMonth(0);
        } else if (DateInConsts.TIME_RANGE_6_MONTH.equals(dateIn)) {
            startMonth = EJCDateUtil.getMonth(-5);
            endMonth = EJCDateUtil.getMonth(0);
        } else {
            startMonth = EJCDateUtil.getLastYear() + "-01";
            endMonth = EJCDateUtil.getLastYear() + "-12";
        }

      /*
        SELECT
        DATE_FORMAT( settle_date, '%Y-%m' ) ext1,
                SUM(
                        IF
                                ( source_type = 'labor_final' OR source_type = 'pro_final', total_process_tax_difference, cur_tax_mny )) cur_tax_mny
                FROM
        ejc_conbase_settle_pool
        */

        //分包结算金额 = 月度结算的本期结算金额合计（cur_tax_mny） + 完工结算的完工单过程结算差额（total_process_tax_difference）
        //labor_month-劳务分包月度结算,labor_node-劳务分包节点结算,labor_final-劳务分包最终结算,pro_month-专业分包月度结算,c-专业分包节点结算,pro_final-专业分包最终结算

        wrapper.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        wrapper.select("DATE_FORMAT(settle_date,'%Y-%m') ext1", "SUM( IF( source_type = 'labor_final' OR source_type = 'pro_final', total_process_tax_difference, cur_tax_mny ) ) cur_tax_mny");
        wrapper.ge("DATE_FORMAT(settle_date,'%Y-%m')", startMonth);
        wrapper.le("DATE_FORMAT(settle_date,'%Y-%m')", endMonth);
        wrapper.notIn("source_type", Arrays.asList("labor_node", "pro_node"));
        wrapper.groupBy("DATE_FORMAT(settle_date,'%Y-%m')");
        List<SettlePoolEntity> list = service.list(wrapper);

        List<String> monthData = EJCDateUtil.getMonthBetween(startMonth, endMonth);

        Map<String, BigDecimal> dataMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(list)) {
            dataMap = list.stream().collect(Collectors.toMap(SettlePoolEntity::getExt1, SettlePoolEntity::getCurTaxMny));
        }
        ArrayList<SupplierSettleVO> dataList = new ArrayList<>();
        for (String mon : monthData) {
            SupplierSettleVO supplierSettleVO = new SupplierSettleVO();
            supplierSettleVO.setSettleDate(mon);
            supplierSettleVO.setSumTaxMny(ComputeUtil.scaleTwo(ComputeUtil.nullToZero(dataMap.get(mon))));
            dataList.add(supplierSettleVO);
        }

        return CommonResponse.success("查询履约分析数据成功！", dataList);
    }


}
