package com.ejianc.business.contractbase.pool.contractpool.controller.api;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.ejianc.business.contractbase.pool.contractpool.bean.ContractPoolEntity;
import com.ejianc.business.contractbase.pool.contractpool.bean.ContractSubPaymentNodeEntity;
import com.ejianc.business.contractbase.pool.contractpool.dto.ContractDTO;
import com.ejianc.business.contractbase.pool.contractpool.dto.CostDesktopDTO;
import com.ejianc.business.contractbase.pool.contractpool.mapper.ContractPoolMapper;
import com.ejianc.business.contractbase.pool.contractpool.service.IContractPoolService;
import com.ejianc.business.contractbase.pool.contractpool.service.IContractSubPaymentNodeService;
import com.ejianc.business.contractbase.pool.contractpool.vo.*;
import com.ejianc.business.contractbase.pool.enums.ContractPerformanceStateEnum;
import com.ejianc.business.contractbase.pool.enums.ContractPropertyEnum;
import com.ejianc.business.contractbase.pool.enums.ContractTypeEnum;
import com.ejianc.business.contractbase.pool.settlepool.bean.SettlePoolEntity;
import com.ejianc.business.contractbase.pool.settlepool.service.ISettlePoolService;
import com.ejianc.business.contractbase.vo.ContractMnyAndNumRatioVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IProjectPoolApi;
import com.ejianc.foundation.share.vo.ProjectVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.CollectionUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Collectors;

import static com.ejianc.framework.skeleton.template.BaseServiceImpl.changeToQueryWrapper;

/**
 * 合同池管理API
 *
 * @author CJ
 * @Description: 合同池管理API
 * @date 2022/2/26 14:55
 */
@RestController
@RequestMapping
public class ContractPoolApiController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IContractPoolService contractPoolService;
    
    
    @Autowired
    private IContractSubPaymentNodeService nodeService;
    @Autowired
    private ISettlePoolService settleService;
    @Autowired
    private IOrgApi iOrgApi;

    @Autowired
    private IBillTypeApi billTypeApi;

    @Autowired
    private IProjectPoolApi projectPoolApi;

    // 单据编码规则
    private static final String BILL_CODE = "CONTRACT_POOL";

    // 更新合同履约状态
    @PostMapping(value = "/api/contractpool/updatePerformanceStatusByContractId")
    public CommonResponse<ContractPoolVO> updatePerformanceStatusByContractId(@RequestBody ContractPoolVO contractPoolVO) {
        logger.info("接收合同同步数据: {}", JSONObject.toJSONString(contractPoolVO));
        ContractPoolVO resp = contractPoolService.updatePerformanceStatusByContractId(contractPoolVO);
        return CommonResponse.success("更新合同池中合同的履约状态成功", resp);
    }

    /**
     * 更新合同履约状态---分布式事务版
     *
     * @param contractPoolVO
     * @return
     */
    @PostMapping(value = "/api/contractpool/updatePerformanceStatusById")
    public CommonResponse<ContractPoolVO> updatePerformanceStatusById(@RequestBody ContractPoolVO contractPoolVO) {
        return updatePerformanceStatusByContractId(contractPoolVO);
    }

    /**
     * 新增或更新合同信息
     *
     * @param contract
     * @return
     */
    @PostMapping(value = "/api/contractpool/saveOrUpdateContract")
    public CommonResponse<ContractPoolVO> saveOrUpdateContract(@RequestBody ContractPoolVO contract) {
        logger.info("接收合同同步数据: {}", JSONObject.toJSONString(contract));
        ContractPoolVO resp = contractPoolService.saveOrUpdateContract(contract);

        return CommonResponse.success(resp);
    }

    /**
     * 合同删除
     *
     * @param contract
     * @return
     */
    @PostMapping(value = "/api/contractpool/deleteContract")
    public CommonResponse<String> deleteContract(@RequestBody ContractPoolVO contract) {
        logger.info("接收合同-{}删除请求", JSONObject.toJSONString(contract));
        //合同是否被其他单据引用

        ContractPoolEntity contractEntity = null;
        if(null != contract.getSourceId()) {
            contractEntity = contractPoolService.selectById(contract.getSourceId());
        }
        if(null == contractEntity && null != contract.getId()) {
            contractEntity = contractPoolService.selectById(contract.getId());
        }

        if(null == contractEntity) {
            logger.info("合同-【{}】删除操作结束，在合同池中无匹配的数据！", JSONObject.toJSONString(contract));
            return CommonResponse.success("删除成功！");
        }

        CommonResponse<String> res = billTypeApi.checkQuote("EJCBT202204000016", contractEntity.getId());
        logger.info("平台返回查询被引用情况" + res.isSuccess() + "----" + res.getMsg());
        if(!res.isSuccess()) {
            return CommonResponse.error("当前合同已被下游业务引用，不能删除！");
        }
        //此处逻辑删除，合同推送时物理删除
        contractPoolService.removeById(contractEntity.getId(), false);
        return CommonResponse.success("删除成功！");
    }


    /**
     * @param contractId 合同id
     * @return 合同子级list
     */
    @ResponseBody
    @RequestMapping(value = "/api/contractpool/queryPayment",method = RequestMethod.GET)
    public CommonResponse<List<ContractSubPaymentNodeVO>> queryPayment(
            @RequestParam(value = "contractId") String contractId){
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("pid",new Parameter(QueryParam.EQ,contractId));
        List<ContractSubPaymentNodeEntity> contractSubPaymentNodeEntities = nodeService.queryList(queryParam);
        List<ContractSubPaymentNodeVO> contractSubPaymentNodeVOS = BeanMapper.mapList(contractSubPaymentNodeEntities, ContractSubPaymentNodeVO.class);
        return CommonResponse.success("查询数据成功！",contractSubPaymentNodeVOS);
    }

    /**
     * 通过合同名称查询合同详情
     * @param contractCode
     * @return
     */
    @ResponseBody
    @RequestMapping(value = "/api/contractpool/queryContractByCode",method = RequestMethod.GET)
    public CommonResponse<ContractPoolVO> queryContractByCode( @RequestParam(value = "contractCode") String contractCode){
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractCode", new Parameter(QueryParam.EQ, contractCode));
        List<ContractPoolEntity> contractPoolVOList = contractPoolService.queryList(queryParam);
        ContractPoolVO contractPoolVO = new ContractPoolVO();
        if (CollectionUtils.isNotEmpty(contractPoolVOList) && contractPoolVOList.get(0) != null){
            contractPoolVO = BeanMapper.map(contractPoolVOList.get(0), ContractPoolVO.class);
        }
        return CommonResponse.success("查询数据成功！", contractPoolVO);

    }

    /**
     * 根据条件查询合同池列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "/api/contractpool/queryList")
    public CommonResponse<List<ContractPoolVO>> queryList(@RequestBody QueryParam queryParam) {
        List<ContractPoolVO> resp = new ArrayList<>();
        List<ContractPoolEntity> entityList = contractPoolService.queryList(queryParam);
        if(CollectionUtils.isNotEmpty(entityList)) {
            resp = BeanMapper.mapList(entityList, ContractPoolVO.class);
        }
        return CommonResponse.success(resp);
    }
    /**
     * 根据条件查询供应商id
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "/api/contractpool/querySupplierList")
    public CommonResponse<List<Long>> querySupplierList(@RequestBody QueryParam queryParam){
        List<ContractPoolVO> resp = new ArrayList<>();
        List<ContractPoolEntity> entityList = contractPoolService.queryList(queryParam);
        if(CollectionUtils.isNotEmpty(entityList)) {
            List<Long> collect = entityList.stream().map(ContractPoolEntity::getPartybId).distinct().collect(Collectors.toList());
            return CommonResponse.success("查询成功！", collect);
        }
        return CommonResponse.error("查询失败！！");
    }
    /**
     * 根据条件查询合同池的合同金额
     * @param queryParam
     * @return
     *
     */
    @PostMapping(value = "/api/contractpool/queryContractMny")
    public CommonResponse<BigDecimal> queryContractMny(@RequestBody QueryParam queryParam){
        //modify by CJ 2022-12-27 针对后端一直报错处理
        Parameter projectId = queryParam.getParams().get("projectId");
        if(null != projectId && "in".equals(projectId.getType().toLowerCase())) {
            if(CollectionUtils.isEmpty(JSONObject.parseArray(JSONObject.toJSONString(projectId.getValue())))) {
                queryParam.getParams().remove("projectId");
            }
        }
        Parameter project_id = queryParam.getParams().get("project_id");
        if(null != project_id && "in".equals(project_id.getType().toLowerCase())) {
            if(CollectionUtils.isEmpty(JSONObject.parseArray(JSONObject.toJSONString(project_id.getValue())))) {
                queryParam.getParams().remove("project_id");
            }
        }

        QueryWrapper<ContractPoolEntity> wrapper = changeToQueryWrapper(queryParam);
        wrapper.select("IFNULL(SUM(contract_tax_mny),0) as contractTaxMny");
        ContractPoolEntity entity = contractPoolService.getOne(wrapper);
        QueryWrapper<ContractPoolEntity> objectQueryWrapper = changeToQueryWrapper(queryParam);
        objectQueryWrapper.select("IFNULL(SUM(total_reply_tax_mny),0) as totalTeplyTaxMny");
        ContractPoolEntity one = contractPoolService.getOne(objectQueryWrapper);
        BigDecimal contractTaxMny = ComputeUtil.safeAdd(entity.getContractTaxMny(), one.getTotalTeplyTaxMny());
        return CommonResponse.success("查询数据成功！",contractTaxMny);
    }


    @PostMapping(value = "/api/contractpool/querySumContractMny")
    public CommonResponse<BigDecimal> querySumContractMny(@RequestBody QueryParam queryParam){
        QueryWrapper<ContractPoolEntity> wrapper = changeToQueryWrapper(queryParam);
        wrapper.select("IFNULL(SUM(contract_tax_mny),0) as contractTaxMny");
        ContractPoolEntity entity = contractPoolService.getOne(wrapper);
        BigDecimal contractTaxMny =entity.getContractTaxMny();
        return CommonResponse.success("查询数据成功！",contractTaxMny);
    }
    /**
     * 根据供应商 id 和开始时间啊 结束时间 查询合同列表
     * @param
     * @return
     */
    @PostMapping(value = "/api/contractpool/getContractList")
    public CommonResponse<JSONArray> getContractList(@RequestBody SupplyVO vo){
        logger.info("接收到请求参数:"+JSONObject.toJSONString(vo));
        List<ContractDTO> contractList = contractPoolService.getContractList(vo);
        logger.info("查询结果:"+JSONObject.toJSONString(contractList));
        return CommonResponse.success("查询成功！！！！",JSONObject.parseArray(JSONObject.toJSONString(contractList)));
    }

    /**
     * 根据供应商 id 和开始时间啊 结束时间 查询合同列表
     * @param
     * @return
     */
    @PostMapping(value = "/api/contractpool/getContractListByHeadUser")
    public CommonResponse<JSONArray> getContractListByHeadUser(@RequestBody HeadUserVO vo){
        logger.info("接收到请求参数:"+JSONObject.toJSONString(vo));
        List<ContractDTO> contractList = contractPoolService.getContractListByHeadUser(vo);
        logger.info("查询结果:"+JSONObject.toJSONString(contractList));
        return CommonResponse.success("查询成功！！！！",JSONObject.parseArray(JSONObject.toJSONString(contractList)));
    }


    /**
     * 根据供应商Id查询供应商历史业绩列表
     *
     * @param supplierId
     * @return
     */
    @GetMapping(value = "/api/contractpool/supHisPerformanceList")
    public CommonResponse<List<SupHisPerformanceVO>> supHisPerformanceList(@RequestParam Long supplierId) {

        List<SupHisPerformanceVO> resp = new ArrayList<>();
        QueryWrapper<ContractPoolEntity> query = new QueryWrapper<>();
        query.select("if(source_type = 'laborSub', supplier_project_manager_name, '') as headUserName," +
                " project_name as engineeringName, project_address as engineeringAddress, " +
                " partya_name as cooperationOrg, source_type_name as cooperationContent, contract_tax_mny as contractMny, contract_name as contractName, category_name as contractCategoryName, " +
                " supplier_project_manager_name as supplierManagerName, supplier_project_manager_link as supplierManagerLink, source_type as contractCategoryProperty," +
                " sign_date as startDay, finish_settle_date as endDay, employee_name as contacts, employee_phone as contactsPhone, partyb_id as supplierId");
        query.eq("partyb_id", supplierId);

        List<Map<String, Object>> mapList = contractPoolService.listMaps(query);
        if(CollectionUtils.isNotEmpty(mapList)) {
            resp = BeanMapper.mapList(mapList, SupHisPerformanceVO.class);
        }

        return CommonResponse.success(resp);
    }
    /**
     * @param contractId 合同id
     */
    @ResponseBody
    @RequestMapping(value = "/api/contractpool/queryById",method = RequestMethod.GET)
    public CommonResponse<ContractPoolVO> queryById(@RequestParam(value = "contractId") Long contractId){
        ContractPoolEntity contractPoolEntity = contractPoolService.selectById(contractId);
        if (contractPoolEntity==null){
            return CommonResponse.error("查询失败！根据id未查询到合同");
        }
        ContractPoolVO vo = BeanMapper.map(contractPoolEntity, ContractPoolVO.class);
        return CommonResponse.success("查询数据成功！",vo);
    }

    @ResponseBody
    @RequestMapping(value = "/api/contractpool/queryContractHonourMny",method = RequestMethod.POST)
    public CommonResponse<BigDecimal> queryContractHonourMny(@RequestBody ContractPoolVO vo){
        BigDecimal bigDecimal = contractPoolService.queryContractHonourMny(vo);
        return CommonResponse.success("查询数据成功！",bigDecimal);
    }

    // 根据项目id查询当前项目下的 最新一条施工合同的合同金额 - 专业合同的合同金额的累计
    @GetMapping(value = "/api/contractpool/queryContractionSubtractProTaxMny")
    public CommonResponse<BigDecimal> queryContractionSubtractProTaxMny(@RequestParam Long projectId) {
        return CommonResponse.success("查询数据成功！", contractPoolService.queryContractionSubtractProTaxMny(projectId));
    }
    /**
     * 根据供应商id和orgIds 查询 供应商的合同金额 和结算金额
     */
    @ResponseBody
    @RequestMapping(value = "/api/contractpool/queryContractSettleTaxMny",method = RequestMethod.POST)
    public CommonResponse<List<SupplierInfoVO>> queryContractSettleTaxMny(@RequestBody SupplierInfoVO vo){
        List<SupplierInfoVO> vos= new ArrayList<>();
        QueryWrapper<ContractPoolEntity> wrapper = new QueryWrapper<>();
        wrapper.select("IFNULL(SUM(contract_tax_mny),0) as contractTaxMny,partyb_id");
        wrapper.in("partyb_id",vo.getSupplierIds());
        wrapper.in("org_id",vo.getOrgIds());
        wrapper.groupBy("partyb_id");
        List<ContractPoolEntity> list = contractPoolService.list(wrapper);
        QueryWrapper<SettlePoolEntity> settleWrapper = new QueryWrapper();
        wrapper.select("IFNULL(SUM(cur_tax_mny),0) as curTaxMny,partyb_id");
        settleWrapper.in("partyb_id",vo.getSupplierIds());
        settleWrapper.in("org_id",vo.getOrgIds());
        settleWrapper.groupBy("partyb_id");
        List<SettlePoolEntity> settleList = settleService.list(settleWrapper);
        Map<Long, SettlePoolEntity> settlePoolMap = settleList.stream().collect(Collectors.toMap(k -> k.getPartybId(), (k) -> k));
        if (CollectionUtils.isNotEmpty(list)){
            for (ContractPoolEntity entity : list) {
                SupplierInfoVO supplierInfoVO = new SupplierInfoVO();
                supplierInfoVO.setContractTaxMny(entity.getContractTaxMny());
                if (settlePoolMap.containsKey(entity.getPartybId())){
                    supplierInfoVO.setSettleTaxMny(settlePoolMap.get(entity.getPartybId()).getCurTaxMny());
                }
                supplierInfoVO.setSupplierId(entity.getPartybId());
                vos.add(supplierInfoVO);
            }
        }

        return CommonResponse.success("查询成功",vos);
    }
    /**
     * 根据条件查询合同池的合同金额
     * @param  sourceType,String dateIn
     * @return
     *
     */
    @GetMapping(value = "/api/contractpool/queryThisYearContractMny")
    public CommonResponse<BigDecimal> queryThisYearContractMny(@RequestParam(value = "sourceType",required = false) String sourceType,@RequestParam(value = "dateIn",required = false)String dateIn, @RequestParam(value = "orgId",required = false) Long orgId ){
        QueryWrapper<ContractPoolEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.select("IFNULL(SUM(total_reply_tax_mny),0) as totalTeplyTaxMny");
        queryWrapper.eq(StringUtils.isNotEmpty(dateIn),"DATE_FORMAT(sign_date,\"%Y\")", dateIn);
        queryWrapper.eq(StringUtils.isNotEmpty(sourceType),"source_type",sourceType);

        QueryWrapper<ContractPoolEntity> wrapper = new QueryWrapper<>();
        wrapper.select("IFNULL(SUM(contract_tax_mny),0) as contractTaxMny");
        wrapper.eq(StringUtils.isNotEmpty(dateIn),"DATE_FORMAT(sign_date,\"%Y\")", dateIn);
        wrapper.eq(StringUtils.isNotEmpty(sourceType),"source_type",sourceType);
        Long _orgId = orgId == null ? InvocationInfoProxy.getOrgId() : orgId;
        /** 数据隔离 本下 没有组织orgId的删除下面代码-------------开始 */
        List<OrgVO> orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentId(_orgId), true, "查询失败，获取当前本下组织信息失败。");
        //普通组织 id
        List<Long> commonOrgIds = new ArrayList<>();
        //项目部 id
        List<Long> departmentIds = new ArrayList<>();
        orgVOList.stream().forEach(org -> {
            if (5 == org.getOrgType()) {
                //项目部
                departmentIds.add(org.getId());
            } else {
                //普通组织
                commonOrgIds.add(org.getId());
            }
        });
        /** 要求主表有orgId字段，保存单据所属组织 */
        if (commonOrgIds.size() != 0) {
            wrapper.in("parent_org_id", commonOrgIds);
            queryWrapper.in("parent_org_id", commonOrgIds);
        } else if (departmentIds.size() != 0) {
            wrapper.in("org_id", departmentIds);
            queryWrapper.in("org_id", departmentIds);
        }
        ContractPoolEntity entity = contractPoolService.getOne(wrapper);
        ContractPoolEntity one = contractPoolService.getOne(queryWrapper);
        BigDecimal contractTaxMny = ComputeUtil.safeAdd(entity.getContractTaxMny(), one.getTotalTeplyTaxMny());
        return CommonResponse.success("查询数据成功！",contractTaxMny);
    }

    /**
     * 累加回写【合同池】对应合同的“累计实付/收金额”、“累计实付/收金额(无税)”
     */
    @GetMapping(value = "/api/contractpool/updateActualPayRecMny")
    public CommonResponse<String> updateActualPayRecMny(@RequestParam(value = "contractId") Long contractId,@RequestParam(value = "taxMny") BigDecimal taxMny,@RequestParam(value = "mny") BigDecimal mny,@RequestParam(value = "flag") boolean flag){
        LambdaUpdateWrapper<ContractPoolEntity> query = new LambdaUpdateWrapper<>();
        query.eq(ContractPoolEntity::getContractId,contractId);
        if (flag){
            query.setSql("total_actual_pay_rec_tax_mny = IFNULL(total_actual_pay_rec_tax_mny,0) + IFNULL("+taxMny+",0),total_actual_pay_rec_mny = IFNULL(total_actual_pay_rec_mny,0) + IFNULL("+mny+",0)");
        }else {
            query.setSql("total_actual_pay_rec_tax_mny = IFNULL(total_actual_pay_rec_tax_mny,0) - IFNULL("+taxMny+",0),total_actual_pay_rec_mny = IFNULL(total_actual_pay_rec_mny,0) - IFNULL("+mny+",0)");
        }
        contractPoolService.update(query);
        return CommonResponse.success("回写成功");
    }

    /**
     * 获取RPC数据
     * resp 返回值
     * isMustSuc 是否必须成功
     * errMsg 失败提示
     */
    private Object getRespData(CommonResponse<?> resp, boolean isMustSuc, String errMsg) {
        if (isMustSuc && !resp.isSuccess()) {
            throw new BusinessException(org.apache.commons.lang3.StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
        }
        return resp.getData();
    }

    /**
     * 在建项目金额区位排名前topNum名
     *
     * @param orgId
     * @param topNum
     * @return
     */
    @GetMapping(value = "/api/contractpool/getProjectAreaContractMny")
    public CommonResponse<List<Map<String, Object>>> getProjectAreaContractMny(@RequestParam Long orgId,
                                                                      @RequestParam Integer topNum,
                                                                      @RequestParam(value = "contractTypes",required = false) String contractTypes,
                                                                      @RequestParam(required = false) String dateIn) {

        CommonResponse<List<ProjectVO>> prjResp = projectPoolApi.queryProjectsByOrgId(orgId);
        if(!prjResp.isSuccess()) {
            logger.error("根据组织Id-[{}]获取在建项目列表信息失败，原因：{}", orgId, prjResp.getMsg());
            return CommonResponse.error("获取项目信息失败！");
        }
        List<ProjectVO> projectList = prjResp.getData();

        if(CollectionUtils.isEmpty(projectList)) {
            logger.info("在建项目金额区位排名 获取当前组织-【{}】下在建项目列表为空！", orgId);
            return CommonResponse.success(new ArrayList<>());
        }

        List<JSONObject> projectProvinceMap = new ArrayList<>();
        JSONObject tmp = null;
        for(ProjectVO project : projectList) {
            tmp = new JSONObject();
            if(StringUtils.isBlank(project.getArea())) {
                tmp.put("provinceName", "未知");
            } else {
                tmp.put("provinceName", project.getAreaName().split("/")[0]);
            }
            tmp.put("projectId", project.getId());
            projectProvinceMap.add(tmp);
        }

        Map<String, Object> param = new HashMap<>();
        param.put("projectProvinceMap", projectProvinceMap);
        param.put("topNum", topNum);

        if(StringUtils.isNotBlank(contractTypes)) {
            param.put("contractTypes", Arrays.asList(contractTypes.split(",")));
        } else {
            param.put("contractTypes", Arrays.asList(new String[]{ContractTypeEnum.施工合同.getTypeCode()}));
        }

        if(StringUtils.isNotBlank(dateIn)) {
            param.put("signDateYear", dateIn);
        }

        List<Map<String, Object>> projectContractMny = contractPoolService.getProjectAreaContractMny(param);

        return CommonResponse.success(projectContractMny);
    }

    /**
     * 承揽子分单位排名
     *   根据组织过滤（本级+下级），查询【合同池】中“合同类别”为“施工合同”的“合同总金额”
     *   维度：按照【组织管理】中“组织类型”为“分（子）公司”、“经理部”的“组织名称”维度统计【合同池】“合同总金额”，按照金额降序显示前 topNum 名组织的金额，时间是指定年度范围内的合同
     *
     * @param orgId
     * @param topNum
     * @param contractTypes
     * @return
     */
    @GetMapping(value = "/api/contractpool/getMolecularUnitRange")
    public CommonResponse<List<Map<String, Object>>> getMolecularUnitRange(@RequestParam Long orgId,
                                                                           @RequestParam Integer topNum,
                                                                           @RequestParam(value = "contractTypes",required = false) String contractTypes) {

        CommonResponse<List<OrgVO>> orgResp = iOrgApi.findOrgByTypesAndOrgId(InvocationInfoProxy.getTenantid(), Arrays.asList(new Integer[]{2,3}), null, orgId);
        if(!orgResp.isSuccess()) {
            logger.error("根据参数tenantId-{}, orgTypes-{}, orgId-{}查询本下组织列表失败，{}",
                    InvocationInfoProxy.getTenantid(), JSONObject.toJSONString(Arrays.asList(new Integer[]{2,3})), orgId, orgResp.getMsg());
            return CommonResponse.error("获取组织信息失败！");
        }
        List<OrgVO> orgList = orgResp.getData();
        if(CollectionUtils.isEmpty(orgList)) {
            logger.info("根据参数tenantId-{}, orgTypes-{}, orgId-{}查询本下组织列表为空！",
                    InvocationInfoProxy.getTenantid(), JSONObject.toJSONString(Arrays.asList(new Integer[]{2,3})), orgId);
            return CommonResponse.success(new ArrayList<>());
        }

        List<Long> orgIds = orgList.stream().map(OrgVO::getId).collect(Collectors.toList());

        Map<String, Object> param = new HashMap<>();
        param.put("parentOrgIds", orgIds);

        if(StringUtils.isNotBlank(contractTypes)) {
            param.put("contractTypes", Arrays.asList(contractTypes.split(",")));
        } else {
            param.put("contractTypes", Arrays.asList(new String[]{ContractTypeEnum.施工合同.getTypeCode()}));
        }
        param.put("topNum", topNum);
        List<Map<String, Object>> data = contractPoolService.getMolecularUnitRange(param);
        return CommonResponse.success(data);
    }



    /**
     * 年度支出合同签订占比（金额、数量）
     *
     * @param orgId
     * @param year 年度
     * @return
     */
    @GetMapping(value = "/api/contractpool/getPayContractRatio")
    public CommonResponse<List<ContractMnyAndNumRatioVO>> getPayContractRatio(@RequestParam Long orgId,
                                                                              @RequestParam(value = "year",required = false) String year){

        List<ContractMnyAndNumRatioVO> returnList = new CopyOnWriteArrayList<>();
        QueryWrapper wrapper = new QueryWrapper();
        wrapper.eq("tenant_id", InvocationInfoProxy.getTenantid());

//        Long _orgId = orgId == null ? InvocationInfoProxy.getOrgId() : orgId;
//
//        //若当前上下文为项目部，则根据项目部Id来进行查询
//        if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
//            wrapper.eq("org_id", _orgId);
//        } else {
//            CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(_orgId);
//            if (!orgResp.isSuccess()) {
//                logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
//                return CommonResponse.error("查询失败，获取组织信息失败！");
//            }
//            List<Long> ids = orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
//            wrapper.in("parent_org_id", ids);
//        }

        if (orgId == null) {
            orgId = InvocationInfoProxy.getOrgId();
        }
        List<Long> orgIds = iOrgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        wrapper.in("org_id", orgIds);

        wrapper.eq("contract_property", ContractPropertyEnum.支出合同.getPropertyCode());
        wrapper.select("source_type", "source_type_name", "SUM(contract_tax_mny) as signTaxMny", "count(*) as signNum");
//        wrapper.eq("performance_status", ContractPerformanceStateEnum.履约中.getStateCode());
        wrapper.eq("DATE_FORMAT(sign_date,'%Y')", year);
        wrapper.eq("dr", 0);
        wrapper.groupBy("source_type");
        List<ContractPoolEntity> list = contractPoolService.list(wrapper);
        if (CollectionUtils.isNotEmpty(list)){
            for (ContractPoolEntity contractPoolEntity : list) {
                ContractMnyAndNumRatioVO vo = new ContractMnyAndNumRatioVO();
                vo.setContractType(contractPoolEntity.getSourceType());
                vo.setContractTypeName(contractPoolEntity.getSourceTypeName());
                vo.setSignTaxMny(contractPoolEntity.getSignTaxMny());
                vo.setSignNum(contractPoolEntity.getSignNum());
                returnList.add(vo);
            }
        }
        List<String> contractTypes = new ArrayList<>();
        contractTypes.add(ContractTypeEnum.物资采购合同.getTypeCode());
        contractTypes.add(ContractTypeEnum.混凝土合同.getTypeCode());
        contractTypes.add(ContractTypeEnum.周转材合同.getTypeCode());
        contractTypes.add(ContractTypeEnum.设备采购.getTypeCode());
        contractTypes.add(ContractTypeEnum.设备租赁.getTypeCode());
        contractTypes.add(ContractTypeEnum.安拆合同.getTypeCode());
        contractTypes.add(ContractTypeEnum.劳务分包合同.getTypeCode());
        contractTypes.add(ContractTypeEnum.专业分包合同.getTypeCode());
        contractTypes.add(ContractTypeEnum.其他支出.getTypeCode());
        returnList.sort(Comparator.comparingInt(item -> contractTypes.indexOf(item.getContractType())));
        return CommonResponse.success("查询数据成功！", returnList);
    }


    // 根据项目id查询当前项目下的 施工合同总金额
    @GetMapping(value = "/api/contractpool/queryContractionTaxMny")
    public CommonResponse<BigDecimal> queryContractionTaxMny(@RequestParam Long projectId) {
        LambdaQueryWrapper<ContractPoolEntity> incomeQueryWrapper = new LambdaQueryWrapper();
        incomeQueryWrapper.eq(ContractPoolEntity::getProjectId,projectId);
        incomeQueryWrapper.eq(ContractPoolEntity::getTenantId,InvocationInfoProxy.getTenantid());
        incomeQueryWrapper.eq(ContractPoolEntity::getSourceType, ContractTypeEnum.施工合同.getTypeCode());
        incomeQueryWrapper.eq(ContractPoolEntity::getDr,0);
        List<ContractPoolEntity> listContractPool = contractPoolService.list(incomeQueryWrapper);
        if (CollectionUtils.isEmpty(listContractPool)){
            return CommonResponse.success("查询数据成功！", BigDecimal.ZERO);
        }
        List<ContractPoolEntity> zsht = listContractPool.stream().filter(x->x.getIsEstimation()==null || x.getIsEstimation()==0).collect(Collectors.toList());
        List<ContractPoolEntity> zght = listContractPool.stream().filter(x->null != x.getIsEstimation() && 1==x.getIsEstimation()).collect(Collectors.toList());
        //施工合同总金额（取值与数据模型一致）
        BigDecimal sumIncomeContractTaxMny = CollectionUtil.isNotEmpty(zsht) ?
                zsht.stream().map(item -> ComputeUtil.safeAdd(item.getNotIncludeProvisionalMny(), item.getTotalCostAdjustTaxMny())).filter(Objects::nonNull).reduce(BigDecimal::add).orElse(BigDecimal.ZERO) :
                zght.stream().map(item -> ComputeUtil.safeAdd(item.getNotIncludeProvisionalMny(), item.getTotalCostAdjustTaxMny())).filter(Objects::nonNull).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);

        logger.info("施工合同总金额：{}", sumIncomeContractTaxMny.toString());
        return CommonResponse.success("查询数据成功！", sumIncomeContractTaxMny);
    }


    // 查询本年已签合同的供应商个数
    @GetMapping(value = "/api/contractpool/queryContractionSupplierNum")
    public CommonResponse<Integer> queryContractionSupplierNum() {

        Integer integer = contractPoolService.queryContractionSupplierNum(new QueryParam());

        logger.info("本年已签合同的供应商个数：{}", integer);
        return CommonResponse.success("查询数据成功！", integer);
    }


    @Autowired
    private ContractPoolMapper mapper;


    // 查询本年已签合同的供应商个数明细
    @GetMapping(value = "/api/contractpool/queryContractionSupplierNumDesktop")
    public CommonResponse<List<CostDesktopDTO>> queryContractionSupplierNumDesktop() {

        //Integer integer = contractPoolService.queryContractionSupplierNum(new QueryParam());
        List<CostDesktopDTO> costDesktopDTOS = mapper.queryContractionSupplierNumDasktop();
        //logger.info("本年已签合同的供应商个数：{}", integer);
        return CommonResponse.success("查询数据成功！", costDesktopDTOS);
    }

}
