package com.ejianc.business.contractbase.util;

import com.ejianc.framework.core.exception.BusinessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.Period;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * 日期处理工具类
 */
public class DateUtils {
    private static Logger logger = LoggerFactory.getLogger(DateUtils.class);
    private DateUtils() {
    }

    /** 日期格式化：yyyy-MM-dd */
    public static final String YYYY_MM_DD = "yyyy-MM-dd";

    /**
     * 计算2个日期之间相差的  以年、月、日为单位，各自计算结果是多少
     * 比如：2011-02-02 到  2017-03-03
     * 计算结果为 1,1  一个月零一天
     *
     * @param fromDate 开始日期
     * @param toDate   结束日期
     * @return 计算结果
     */
    public static int[] dayCompare(Date fromDate, Date toDate) {
        Instant instant = fromDate.toInstant();
        Instant instant1 = toDate.toInstant();
        ZoneId zoneId = ZoneId.systemDefault();
        LocalDate one = instant.atZone(zoneId).toLocalDate();
        LocalDate two = instant1.atZone(zoneId).toLocalDate();
        int year = Period.between(one, two).getYears();
        int month = Period.between(one, two).getMonths();
        int day = Period.between(one, two).getDays();
        month = year * 12 + month;
        return new int[]{month, day};
    }

    /**
     * 计算两个日期之间相差的天数
     *
     * @param smdate 较小的时间
     * @param bdate  较大的时间
     * @return 相差天数
     */
    public static Integer daysBetween(Date smdate, Date bdate) {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(YYYY_MM_DD);
            smdate = sdf.parse(sdf.format(smdate));
            bdate = sdf.parse(sdf.format(bdate));
            Calendar cal = Calendar.getInstance();
            cal.setTime(smdate);
            long time1 = cal.getTimeInMillis();
            cal.setTime(bdate);
            long time2 = cal.getTimeInMillis();
            long between_days = (time2 - time1) / (1000 * 3600 * 24);
            return Integer.parseInt(String.valueOf(between_days));
        }
        catch (Exception e) {
            logger.error("计算日期失败，失败原因：{}", e.getMessage(), e);
            throw new BusinessException("计算日期失败，失败原因：" + e.getMessage());
        }
    }

    /**
     * 计算月份，加几月或者减几月  整数  往后推,负数往前移动
     *
     * @param date  需要计算的日期
     * @param month 计算月数
     * @return 计算结果
     */
    public static Date calculationMonth(Date date, Integer month) {
        Calendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, month); //把日期往后增加一天,整数  往后推,负数往前移动
        return calendar.getTime(); //这个时间就是日期往后推一天的结果
    }

    /**
     * 计算日期，加几天或者减几天  整数  往后推,负数往前移动
     *
     * @param date 需要计算的日期
     * @param days 计算天数
     * @return 计算结果
     */
    public static Date calculationDay(Date date, Integer days) {
        Calendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, days); //把日期往后增加一天,整数  往后推,负数往前移动
        return calendar.getTime(); //这个时间就是日期往后推一天的结果
    }

    /**
     * 创建格式为YYYY-MM-DD的日期
     *
     * @param date 日期
     * @return 创建结果
     */
    public static Date createDate(String date) {
        return createDate(date, YYYY_MM_DD);
    }

    /**
     * 创建日期
     *
     * @param date    需要创建日期的对象
     * @param pattern 日期格式
     * @return 创建结果
     */
    public static Date createDate(String date, String pattern) {
        try {
            return new SimpleDateFormat(pattern).parse(date);
        }
        catch (Exception e) {
            return null;
        }
    }

    /**
     * 格式化日期
     *
     * @param date    需要格式化的日期
     * @param pattern 格式
     * @return 格式化结果
     */
    public static String transformationDate(Date date, String pattern) {
        return new SimpleDateFormat(pattern).format(date);
    }

    /**
     * date2String  默认yyyy-MM-dd
     *
     * @param date 日期
     * @return 转换结果
     */
    public static String transformationDefaultDate(Date date) {
        return transformationDate(date, YYYY_MM_DD);
    }

    /**
     * 格式化日期
     *
     * @param date    需要格式化的日期
     * @param pattern 格式
     * @return 格式化结果
     */
    public static Date formatDate(Date date, String pattern) {
        return createDate(transformationDate(date, pattern), pattern);
    }
}
