package com.ejianc.business.utils;

import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.beans.PropertyDescriptor;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.*;


/**
 * @description: 比较对象值变化
 * 结合注解@DifferenceComparison和@FieldDifferenceComparison使用
 * @DifferenceComparison 需要对比的实体类加此注解，说明见注解类注释
 * @FieldDifferenceComparison 需要对比的字段需要加此注解，说明见注解类注释
 *
 * @author songlx
 * @date: 2024/4/26
 */
public class CompareDifferenceUtil {

    private static final Logger log = LoggerFactory.getLogger(CompareDifferenceUtil.class);

    private static final String ADD = "add";
    private static final String UPDATE = "update";

    private static final Map<String, Set<Field>> fieldCache = new HashMap<>();


    /**
     * 获取两个单个对象的值变化列表
     * @param oldObj 旧对象
     * @param newObj 新对象
     * @return List<FieldDifference> 差异字段集合
     */
    public static List<FieldDifference> compareObj(Object oldObj, Object newObj) {
        List<FieldDifference> result = new ArrayList<>();

        if (oldObj == null) {
            throw new RuntimeException("CompareDifferenceUtil.compareObj oldObj is null");
        }
        if (null == newObj) {
            throw new RuntimeException("CompareDifferenceUtil.compareObj newObj is null");
        }
        // 同一个对象就没必要对比了
        if (oldObj == newObj) {
            return result;
        }
        Class<?> newObjClass = newObj.getClass();
        DifferenceComparison differenceComparison = newObjClass.getAnnotation(DifferenceComparison.class);

        Set<Field> fields = getField(newObjClass);

        if (CollectionUtils.isNotEmpty(fields)) {
            String fieldName = null;
            for (Field field : fields) {
                if (!field.isAnnotationPresent(FieldDifferenceComparison.class)) {
                    continue;
                }
                FieldDifferenceComparison fieldDifferenceComparison = field.getAnnotation(FieldDifferenceComparison.class);
                fieldName = field.getName();
                Class<?> type = field.getType();
                String FieldType = type.getSimpleName();
                Object oldValue = getFieldValueByName(fieldName, oldObj);

                Object newValue = getFieldValueByName(fieldName, newObj);

                if (null == oldValue) {
                    if (null != newValue) {
                        result.add(FieldDifference.getInstance(fieldName, fieldDifferenceComparison.name(), FieldType, null, newValue));
                    }
                } else {
                    if (StringUtils.isNotEmpty(differenceComparison.beforeKeyPrefix())) {
                        String beforeValKey = differenceComparison.beforeKeyPrefix().concat(StringUtils.capitalize(fieldName));
                        setFieldValueByName(beforeValKey, oldValue, newObj);
                    }

                    if (null != newValue) {
                        if (newValue instanceof BigDecimal) {
                            BigDecimal newValue1 = ComputeUtil.toBigDecimal(newValue);
                            BigDecimal oldValue1 = ComputeUtil.toBigDecimal(oldValue);
                            if (!ComputeUtil.equals(newValue1, oldValue1)) {
                                result.add(FieldDifference.getInstance(fieldName, fieldDifferenceComparison.name(), FieldType, oldValue, newValue));
                            }
                        } else if (!oldValue.equals(newValue)) {
                            result.add(FieldDifference.getInstance(fieldName, fieldDifferenceComparison.name(), FieldType, oldValue, newValue));
                        }
                    } else {
                        result.add(FieldDifference.getInstance(fieldName, fieldDifferenceComparison.name(), FieldType, oldValue, null));
                    }
                }
            }
        }

        if (CollectionUtils.isNotEmpty(result)) {
            if (StringUtils.isNotEmpty(differenceComparison.changeFlag())) {
                setFieldValueByName(differenceComparison.changeFlag(), UPDATE, newObj);
            }
            if (StringUtils.isNotEmpty(differenceComparison.differnceList())) {
                setFieldValueByName(differenceComparison.differnceList(), result, newObj);
            }
        }
        return result;
    }


    /**
     * @description: 比较两个集合对象的值变化
     * 适用于子表
     * @param oldList 旧集合
     * @param newList 新集合
     * @author songlx
     * @date: 2024/4/28
     */
    public static void compareList(List oldList, List newList) {
        if (CollectionUtils.isNotEmpty(newList)) {
            DifferenceComparison differenceComparisonNew = newList.get(0).getClass().getAnnotation(DifferenceComparison.class);
            if (StringUtils.isEmpty(differenceComparisonNew.changeFlag())) {
                throw new BusinessException("CompareDifferenceUtil.compareList error: newList changeFlag is null");
            }
            String changeFlagName = differenceComparisonNew.changeFlag();

            if (StringUtils.isEmpty(differenceComparisonNew.compareKey())) {
                throw new BusinessException("CompareDifferenceUtil.compareList error: newList compareKey is null");
            }
            String compareKeyNew = differenceComparisonNew.compareKey();

            if (CollectionUtils.isNotEmpty(oldList)) {
                DifferenceComparison differenceComparisonOld = oldList.get(0).getClass().getAnnotation(DifferenceComparison.class);
                if (StringUtils.isEmpty(differenceComparisonOld.compareKey())) {
                    throw new BusinessException("CompareDifferenceUtil.compareList error: oldList compareKey is null");
                }
                String compareKeyOld = differenceComparisonOld.compareKey();

                HashMap<Object, Object> oldMap = new HashMap<>();
                for (Object t : oldList) {
                    Object fieldValue = getFieldValueByName(compareKeyOld, t);
                    oldMap.put(fieldValue, t);
                }
                for (Object newItem : newList) {
                    Object _fieldValue = getFieldValueByName(compareKeyNew, newItem);
                    Object oldItem = oldMap.get(_fieldValue);
                    if (oldItem == null) {
                        setFieldValueByName(changeFlagName, ADD, newItem);
                    } else {
                        compareObj(oldItem, newItem);
                    }
                }
            } else {
                // 如果老列表为空，则新列表全部为新增
                for (Object newItem : newList) {
                    setFieldValueByName(changeFlagName, ADD, newItem);
                }
            }
        }
    }


    /**
     * 获取类的需要对比的字段（加注解@FieldDifferenceComparison）
     */
    @SuppressWarnings("unchecked")
    private static Set<Field> getField(Class<?> type) {
        String className = type.getName();
        if (CollectionUtils.isEmpty(fieldCache.get(className))) {
            synchronized (className) {
                if (CollectionUtils.isEmpty(fieldCache.get(className))) {
                    Set<Field> fields = new HashSet<>();
                    Field[] declaredFields = type.getDeclaredFields();
                    for (Field declaredField : declaredFields) {
                        if (declaredField.isAnnotationPresent(FieldDifferenceComparison.class)) {
                            fields.add(declaredField);
                        }
                    }
                    fieldCache.put(className, fields);
                }
            }
        }
        return fieldCache.get(className);
    }


    private static Object getFieldValueByName(String fieldName, Object o) {
        try {
            PropertyDescriptor namePd = new PropertyDescriptor(fieldName, o.getClass());
            Method method = namePd.getReadMethod();
            Object value = method.invoke(o, new Object[]{});
            return value;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    private static boolean setFieldValueByName(String fieldName, Object val, Object o) {
        try {
            PropertyDescriptor namePd = new PropertyDescriptor(fieldName, o.getClass());
            Method method = namePd.getWriteMethod();
            method.invoke(o, val);
            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }


    /**
     * @description: 字段差异封装VO
     */
    public static class FieldDifference implements Serializable {

        private String column;
        private String columnName;
        private String columnType;
        private Object oldValue;
        private Object newValue;

        public static FieldDifference getInstance(String column, String columnName, String columnType, Object oldValue, Object newValue) {
            return new FieldDifference(column, columnName, columnType, oldValue, newValue);
        }

        public FieldDifference(String column, String columnName, String columnType, Object oldValue, Object newValue) {
            this.column = column;
            this.columnName = columnName;
            this.columnType = columnType;
            this.oldValue = oldValue;
            this.newValue = newValue;
        }

        public String getColumn() {
            return column;
        }

        public void setColumn(String column) {
            this.column = column;
        }

        public String getColumnName() {
            return columnName;
        }

        public void setColumnName(String columnName) {
            this.columnName = columnName;
        }

        public String getColumnType() {
            return columnType;
        }

        public void setColumnType(String columnType) {
            this.columnType = columnType;
        }

        public Object getOldValue() {
            return oldValue;
        }

        public void setOldValue(Object oldValue) {
            this.oldValue = oldValue;
        }

        public Object getNewValue() {
            return newValue;
        }

        public void setNewValue(Object newValue) {
            this.newValue = newValue;
        }
    }


}