package com.ejianc.business.dxcheck.service.impl;

import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.dxcheck.dao.RecordDao;
import com.ejianc.business.dxcheck.dao.RecordSubDao;
import com.ejianc.business.dxcheck.entity.RecordEntity;
import com.ejianc.business.dxcheck.entity.RecordSubEntity;
import com.ejianc.business.dxcheck.model.res.RecordRes;
import com.ejianc.business.dxcheck.model.vo.RecordAddEditVo;
import com.ejianc.business.dxcheck.model.vo.RecordBatchAddVo;
import com.ejianc.business.dxcheck.model.vo.RecordDelVo;
import com.ejianc.business.dxcheck.model.vo.RecordScoreVo;
import com.ejianc.business.dxcheck.service.RecordServer;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.math.NumberUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Date;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/5/29 上午9:46
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class RecordServerImpl implements RecordServer {
    private final RecordDao recordDao;
    private final RecordSubDao recordSubDao;
    private final SessionManager sessionManager;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public RecordRes add(RecordAddEditVo vo) {
        UserContext userContext = sessionManager.getUserContext();
        /**
         * 主信息
         */
        RecordEntity recordEntity = new RecordEntity();
        BeanUtils.copyProperties(vo, recordEntity);
        recordEntity.setState(NumberUtils.INTEGER_ONE);
        recordEntity.setOrgId(userContext.getOrgId());
        recordEntity.setOrgCode(userContext.getOrgCode());
        recordEntity.setOrgName(userContext.getOrgName());
        recordEntity.setParentOrgId(userContext.getDeptId());
        recordEntity.setParentOrgCode(userContext.getDeptCode());
        recordEntity.setParentOrgName(userContext.getDeptName());
        recordDao.save(recordEntity);
        /**
         * 指标信息
         */
        List<RecordSubEntity> saveRecordSubEntityList = Lists.newArrayList();
        List<RecordAddEditVo.AddEditRecordSubInfo> recordSubEntities = vo.getRecordSubEntities();
        recordSubEntities.forEach(e -> {
            BeanUtils.copyProperties(recordEntity, e);

            RecordSubEntity recordSubEntity = new RecordSubEntity();
            BeanUtils.copyProperties(e, recordSubEntity);
            recordSubEntity.setId(null);
            recordSubEntity.setRecordId(recordEntity.getId());
            recordSubEntity.setState(Objects.nonNull(recordSubEntity.getSelfScore()) ? 1 : 0);
            saveRecordSubEntityList.add(recordSubEntity);
        });
        recordSubDao.saveBatch(saveRecordSubEntityList);

        return buildRes(recordEntity, saveRecordSubEntityList);
    }

    @Override
    public List<RecordRes> batchAdd(RecordBatchAddVo vo) {
        UserContext userContext = sessionManager.getUserContext();
        String batchAssessmentCode = RandomUtil.randomString(20);
        List<RecordBatchAddVo.BathAddRecordInfo> recordInfos = vo.getRecordInfos();

        List<RecordRes> resList = Lists.newArrayList();
        List<RecordEntity> recordEntityList = Lists.newArrayList();
        List<RecordSubEntity> saveRecordSubEntityList = Lists.newArrayList();
        for (RecordBatchAddVo.BathAddRecordInfo recordInfo : recordInfos) {
            /**
             * 主信息
             */
            RecordEntity recordEntity = new RecordEntity();
            BeanUtils.copyProperties(recordInfo, recordEntity);
            recordEntity.setState(NumberUtils.INTEGER_ONE);
            recordEntity.setOrgId(userContext.getOrgId());
            recordEntity.setOrgCode(userContext.getOrgCode());
            recordEntity.setOrgName(userContext.getOrgName());
            recordEntity.setParentOrgId(userContext.getDeptId());
            recordEntity.setParentOrgCode(userContext.getDeptCode());
            recordEntity.setParentOrgName(userContext.getDeptName());
            recordEntity.setBatchAssessmentCode(batchAssessmentCode);
            recordDao.save(recordEntity);
            recordEntityList.add(recordEntity);
            /**
             * 指标信息
             */
            List<RecordBatchAddVo.BathAddRecordSubInfo> recordSubEntities = vo.getRecordSubEntities();
            recordSubEntities.forEach(e -> {
                BeanUtils.copyProperties(recordEntity, e);

                RecordSubEntity recordSubEntity = new RecordSubEntity();
                BeanUtils.copyProperties(e, recordSubEntity);
                recordSubEntity.setId(null);
                recordSubEntity.setRecordId(recordEntity.getId());
                recordSubEntity.setState(Objects.nonNull(recordSubEntity.getSelfScore()) ? 1 : 0);
                saveRecordSubEntityList.add(recordSubEntity);
            });

        }
        recordSubDao.saveBatch(saveRecordSubEntityList);

        recordEntityList.forEach(recordEntity -> {
            resList.add(buildRes(recordEntity, saveRecordSubEntityList));
        });
        return resList;
    }

    @Override
    public void del(RecordDelVo vo) {
        List<Long> ids = vo.getIds();
        List<Long> subIdList = vo.getSubIdList();
        /**
         * 删除主信息
         */
        if (!ids.isEmpty()) {
            try {
                recordDao.removeByIds(ids);
                recordSubDao.lambdaUpdate().in(RecordSubEntity::getRecordId, ids).remove();
                return;
            } catch (Exception e) {
                throw new BusinessException("列表删除失败，原因:" + e.getMessage());
            }
        }
        /**
         * 删除指标
         */
        if (!subIdList.isEmpty()) {
            try {
                recordSubDao.removeByIds(subIdList);
                return;
            } catch (Exception e) {
                throw new BusinessException("明细删除失败，原因:" + e.getMessage());
            }
        }
        throw new BusinessException("列表删除失败，请选择删除的条目");
    }

    @Override
    public RecordRes detail(Long id) {
        RecordEntity recordEntity = recordDao.lambdaQuery().eq(BaseEntity::getId, id).one();
        if (Objects.isNull(recordEntity)) {
            throw new BusinessException("查询失败,数据不存在");
        }
        List<RecordSubEntity> recordSubEntityList = recordSubDao.lambdaQuery().eq(RecordSubEntity::getRecordId, id).list();

        return buildRes(recordEntity, recordSubEntityList);
    }

    @Override
    public IPage<RecordEntity> list(QueryParam param) {
        //模糊
        param.getFuzzyFields().addAll(Arrays.asList());
        //排序
        param.getOrderMap().put("createTime", QueryParam.DESC);

        return recordDao.queryPage(param, false);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public RecordRes edit(RecordAddEditVo vo) {
        /**
         * 修改主信息
         */
        RecordEntity recordEntity = recordDao.getById(vo.getId());
        BeanUtils.copyProperties(vo, recordEntity);
        recordEntity.setModificationTime(new Date(System.currentTimeMillis()));
        recordDao.updateById(recordEntity);

        List<RecordAddEditVo.AddEditRecordSubInfo> recordSubEntities = vo.getRecordSubEntities();
        if (recordSubEntities.isEmpty()) {
            throw new BusinessException("至少需要一条指标");
        }

        /**
         * 修改指标
         */
        List<RecordSubEntity> saveRecordSubEntityList = Lists.newArrayList();
        List<RecordSubEntity> editRecordSubEntityList = Lists.newArrayList();
        recordSubEntities.forEach(e -> {
            Long recordSubId = e.getId();
            if (Objects.isNull(recordSubId)) {
                /**
                 * 修改-新增新指标
                 */
                BeanUtils.copyProperties(recordEntity, e);
                e.setRecordId(recordEntity.getId());

                RecordSubEntity recordSubEntity = new RecordSubEntity();
                BeanUtils.copyProperties(e, recordSubEntity);
                recordSubEntity.setModifierJobNum(vo.getModifierJobNum());
                recordSubEntity.setState(Objects.nonNull(recordSubEntity.getSelfScore()) ? 1 : 0);
                saveRecordSubEntityList.add(recordSubEntity);
            } else {
                /**
                 * 修改-修改原指标
                 */
                RecordSubEntity recordSubEntity = recordSubDao.getById(recordSubId);
                recordSubEntity.setAssessmentUnitCode(vo.getAssessmentUnitCode());
                recordSubEntity.setAssessmentUnitName(vo.getAssessmentUnitName());
                recordSubEntity.setAssessmentDepartmentCode(vo.getAssessmentDepartmentCode());
                recordSubEntity.setAssessmentDepartmentName(vo.getAssessmentDepartmentName());
                recordSubEntity.setAssessmentEventType(vo.getAssessmentEventType());
                recordSubEntity.setAssessmentContent(vo.getAssessmentContent());
                recordSubEntity.setAssessorJobNum(vo.getAssessorJobNum());
                recordSubEntity.setAssessorName(vo.getAssessorName());
                recordSubEntity.setModifierJobNum(vo.getModifierJobNum());

                recordSubEntity.setScoreOfYear(e.getScoreOfYear());
                recordSubEntity.setScoreOfSeason(e.getScoreOfSeason());
                recordSubEntity.setAssessmentSubject(e.getAssessmentSubject());
                recordSubEntity.setState(Objects.nonNull(e.getSelfScore()) ? 1 : 0);
                recordSubEntity.setSelfScore(e.getSelfScore());
                recordSubEntity.setSelfScoreReason(e.getSelfScoreReason());
                recordSubEntity.setSupportMaterialSelf(e.getSupportMaterialSelf());
                editRecordSubEntityList.add(recordSubEntity);
            }
        });
        recordSubDao.saveBatch(saveRecordSubEntityList);
        recordSubDao.updateBatchById(editRecordSubEntityList);

        List<RecordSubEntity> recordSubEntityList = Lists.newArrayList();
        recordSubEntityList.addAll(saveRecordSubEntityList);
        recordSubEntityList.addAll(editRecordSubEntityList);

        /**
         * 修改-删除指标
         */
        List<Long> removeRecordSubIdList = vo.getRemoveRecordSubIdList();
        if (Objects.nonNull(removeRecordSubIdList) && !removeRecordSubIdList.isEmpty()) {
            recordSubDao.lambdaUpdate().in(BaseEntity::getId, removeRecordSubIdList)
                    .set(RecordSubEntity::getModifierJobNum, vo.getModifierJobNum())
                    .update();
            recordSubDao.removeByIds(removeRecordSubIdList);
        }

        RecordRes recordRes = buildRes(recordEntity, recordSubEntityList);
        recordRes.setRemoveRecordSubIdList(removeRecordSubIdList);
        return recordRes;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void score(RecordScoreVo vo) {
        List<RecordScoreVo.ScoreVo> scoreList = vo.getScoreList();
        try {
            for (RecordScoreVo.ScoreVo scoreVo : scoreList) {
                Long recordSubId = scoreVo.getRecordSubId();
                /**
                 * 建议评分（专业组建议）
                 */
                Integer suggestScore = scoreVo.getSuggestScore();
                if (Objects.nonNull(suggestScore)) {
                    recordSubDao.lambdaUpdate().eq(BaseEntity::getId, recordSubId)
                            .set(RecordSubEntity::getState, 2)
                            .set(RecordSubEntity::getSuggestScore, suggestScore)
                            .set(RecordSubEntity::getSuggestReason, scoreVo.getSuggestReason())
                            .set(RecordSubEntity::getSupportMaterialMajor, scoreVo.getSupportMaterialMajor())
                            .update();
                }
                /**
                 * 初评分（初评组初审）
                 */
                Integer preliminaryScore = scoreVo.getPreliminaryScore();
                if (Objects.nonNull(preliminaryScore)) {
                    recordSubDao.lambdaUpdate().eq(BaseEntity::getId, recordSubId)
                            .set(RecordSubEntity::getState, 3)
                            .set(RecordSubEntity::getPreliminaryScore, preliminaryScore)
                            .set(RecordSubEntity::getPreliminaryReason, scoreVo.getPreliminaryReason())
                            .set(RecordSubEntity::getSupportMaterialPrelim, scoreVo.getSupportMaterialPrelim())
                            .update();
                }
                /**
                 * 终评分（考评组长终评）
                 */
                Integer finalScore = scoreVo.getFinalScore();
                if (Objects.nonNull(finalScore)) {
                    recordSubDao.lambdaUpdate().eq(BaseEntity::getId, recordSubId)
                            .set(RecordSubEntity::getState, 4)
                            .set(RecordSubEntity::getFinalScore, finalScore)
                            .set(RecordSubEntity::getFinalReason, scoreVo.getFinalReason())
                            .update();
                }
            }
        } catch (Exception e) {
            throw new BusinessException("评分失败，原因：:" + e.getMessage());
        } finally {
            Long recordId = vo.getRecordId();
            /**
             * 评分后更新主表状态信息  以最小为准
             */
            Integer minState = recordSubDao.lambdaQuery().eq(RecordSubEntity::getRecordId, recordId).list()
                    .stream().map(RecordSubEntity::getState).min(Integer::compareTo).orElse(0);

            recordDao.lambdaUpdate().eq(BaseEntity::getId, recordId).set(RecordEntity::getState, minState).update();
        }

    }

    /**
     * 构建返回
     *
     * @param recordEntity            主信息
     * @param saveRecordSubEntityList 指标信息
     * @return RecordRes 返回
     * @see RecordRes
     */
    private RecordRes buildRes(RecordEntity recordEntity, List<RecordSubEntity> saveRecordSubEntityList) {
        RecordRes recordRes = new RecordRes();
        BeanUtils.copyProperties(recordEntity, recordRes);
        List<RecordRes.RecordSubInfoRes> subEntityList = saveRecordSubEntityList.stream().map(recordSubEntity -> {
            RecordRes.RecordSubInfoRes resRecordSubEntity = new RecordRes.RecordSubInfoRes();

            BeanUtils.copyProperties(recordSubEntity, resRecordSubEntity);
            return resRecordSubEntity;
        }).collect(Collectors.toList());
        recordRes.setRecordSubEntitiyList(subEntityList);
        return recordRes;
    }
}
