package com.ejianc.business.dxcheck.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.ReUtil;
import cn.hutool.core.util.StrUtil;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.dxcheck.dao.NormDao;
import com.ejianc.business.dxcheck.entity.NormEntity;
import com.ejianc.business.dxcheck.model.vo.NormAddVo;
import com.ejianc.business.dxcheck.model.vo.NormEditVo;
import com.ejianc.business.dxcheck.service.NormServer;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/5/28 下午2:18
 * @Description: 考核指标
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class NormServerImpl implements NormServer {
    private final NormDao dao;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public NormEntity add(NormAddVo vo) {
        try {
            String firstClassify = vo.getFirstClassify();
            String secondClassify = vo.getSecondClassify();

            NormEntity normEntity = NormEntity.builder().build();
            BeanUtils.copyProperties(vo, normEntity);

            String serialNum = dealSerialNum(firstClassify, secondClassify);
            normEntity.setSerialNum(serialNum);
            normEntity.setNormVersion("V1");
            dao.save(normEntity);
            return normEntity;
        } catch (BeansException e) {
            throw new BusinessException("新增失败，原因：" + e.getMessage());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void del(List<Long> ids) {
        try {
            dao.removeByIds(ids);
        } catch (BeansException e) {
            throw new BusinessException("删除失败，原因：" + e.getMessage());
        }
    }

    @Override
    public NormEntity detail(Long id) {
        NormEntity one = dao.lambdaQuery().eq(BaseEntity::getId, id).one();
        if (Objects.isNull(one)) {
            throw new BusinessException("查询失败,数据不存在");
        }
        return one;
    }

    @Override
    public Map<String, Map<String, List<NormEntity>>> list() {
        List<NormEntity> list = dao.lambdaQuery().orderByAsc(NormEntity::getSerialNum).list();
        return list.stream().collect(Collectors.groupingBy(e->e.getSerialNum().split("\\.")[0],Collectors.groupingBy(e->e.getSerialNum().split("\\.")[1])));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public NormEntity edit(NormEditVo vo) {
        Long oldNormId = vo.getId();
        try {
            NormEntity byId = dao.getById(oldNormId);
            if (byId == null) {
                throw new BusinessException("修改失败，修改数据不存在");
            }
            //新增修改后的指标
            NormEntity saveEntity = NormEntity.builder().build();
            BeanUtils.copyProperties(byId, saveEntity);
            saveEntity.setId(null);
            saveEntity.setCreateTime(new DateTime());
            //版本处理
            int newNormVersion = ReUtil.getFirstNumber(byId.getNormVersion()) + 1;
            saveEntity.setNormVersion("V" + newNormVersion);
            saveEntity.setNormContent(vo.getNormContent());
            saveEntity.setPlusLimit(vo.getPlusLimit());
            saveEntity.setReduceLimit(vo.getReduceLimit());
            saveEntity.setHeadquarterDepartment(vo.getHeadquarterDepartment());
            saveEntity.setSubsidiaryCompany(vo.getSubsidiaryCompany());
            saveEntity.setStraightProject(vo.getStraightProject());
            saveEntity.setModifierJobNum(vo.getModifierJobNum());
            saveEntity.setModifierName(vo.getModifierName());
            saveEntity.setModifyTime(new Date(System.currentTimeMillis()));
            saveEntity.setOldNorm(oldNormId);
            dao.save(saveEntity);
            //删除原先指标
            byId.setModifierJobNum(vo.getModifierJobNum());
            byId.setModifierName(vo.getModifierName());
            byId.setModifyTime(new Date(System.currentTimeMillis()));
            dao.updateById(byId);
            dao.removeById(byId.getId());
            return saveEntity;
        } catch (BeansException e) {
            throw new BusinessException("修改失败，原因：" + e.getMessage());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<String> move(Long id, String move) {
        List<NormEntity> updateEntityList = Lists.newArrayList();
        NormEntity moveEntity = dao.getById(id);
        if (Objects.isNull(moveEntity)) {
            throw new BusinessException("移动失败，数据不存在");
        }
        //序号
        String serialNum = moveEntity.getSerialNum();
        try {
            if ("up".equals(move)) {
                String thirdNumber = StrUtil.subAfter(serialNum, ".", true);
                if ("1".equals(thirdNumber)) {
                    throw new BusinessException("上移失败，序号位于第一位");
                }
                //获取上层信息
                String[] parts = serialNum.split("\\.");
                int superThirdNumber = Integer.parseInt(parts[parts.length - 1]) - 1;
                String superVersion = parts[0] + "." + parts[1] + "." + superThirdNumber;
                NormEntity superEntity = dao.lambdaQuery().eq(NormEntity::getSerialNum, superVersion).one();

                superEntity.setSerialNum(moveEntity.getSerialNum());
                moveEntity.setSerialNum(superVersion);
                updateEntityList.add(superEntity);
                updateEntityList.add(moveEntity);
                dao.updateBatchById(updateEntityList);
                return CommonResponse.success("上移成功");
            }
            if ("down".equals(move)) {
                //获取下层信息
                String[] parts = serialNum.split("\\.");
                int lowerThirdNumber = Integer.parseInt(parts[parts.length - 1]) + 1;
                String lowerVersion = parts[0] + "." + parts[1] + "." + lowerThirdNumber;
                NormEntity lowerEntity = dao.lambdaQuery().eq(NormEntity::getSerialNum, lowerVersion).one();
                if (Objects.isNull(lowerEntity)) {
                    throw new BusinessException("下移失败，序号位无法下移");
                }
                moveEntity.setSerialNum(lowerEntity.getSerialNum());
                lowerEntity.setSerialNum(serialNum);
                updateEntityList.add(lowerEntity);
                updateEntityList.add(moveEntity);
                dao.updateBatchById(updateEntityList);
                return CommonResponse.success("下移成功");
            }
        } catch (BusinessException e) {
            throw new BusinessException("操作失败，原因：" + e.getMessage());
        }
        return CommonResponse.error("操作不存在");
    }

    /**
     * 处理序号
     *
     * @param firstClassify  一级分类
     * @param secondClassify 二级分类
     * @return 序号
     */
    private String dealSerialNum(String firstClassify, String secondClassify) {
        String firstSerialNum;
        String secondSerialNum;
        String thirdSerialNum;
        List<NormEntity> list = dao.list();
        Map<String, Map<String, List<NormEntity>>> firstAndSecondMap = list.stream().collect(Collectors.groupingBy(NormEntity::getFirstClassify,
                Collectors.groupingBy(NormEntity::getSecondClassify)));
        if (firstAndSecondMap.containsKey(firstClassify)) {
            firstSerialNum = firstAndSecondMap.get(firstClassify).values().stream().findFirst().get().get(0).getSerialNum().split("\\.")[0];
            Map<String, List<NormEntity>> secondMap = firstAndSecondMap.get(firstClassify);
            if (secondMap.containsKey(secondClassify)) {
                secondSerialNum = secondMap.get(secondClassify).get(0).getSerialNum().split("\\.")[1];
                //获取新增的一级二级的信息
                List<NormEntity> firstSecondList = dao.getFirstSecondListOrderBySerialNum(firstClassify, secondClassify);
                String lastSerialNum = firstSecondList.get(firstSecondList.size() - 1).getSerialNum();
                thirdSerialNum = String.valueOf(NumberUtil.parseInt(StrUtil.subAfter(lastSerialNum, ".", true)) + 1);
            } else {
                secondSerialNum = String.valueOf(secondMap.size() + 1);
                thirdSerialNum = "1";
            }
        } else {
            firstSerialNum = String.valueOf(firstAndSecondMap.size() + 1);
            secondSerialNum = "1";
            thirdSerialNum = "1";
        }
        return StrUtil.join(".", firstSerialNum, secondSerialNum, thirdSerialNum);
    }
}
