package com.ejianc.business.dxcheck.service.impl;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.NumberUtil;
import com.beust.jcommander.internal.Lists;
import com.ejianc.business.dxcheck.dao.RecordSubDao;
import com.ejianc.business.dxcheck.entity.RecordSubEntity;
import com.ejianc.business.dxcheck.model.res.AssessmentRankRes;
import com.ejianc.business.dxcheck.model.res.RecordStatisticsRes;
import com.ejianc.business.dxcheck.service.StatisticsServer;
import com.ejianc.framework.skeleton.template.BaseEntity;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author: LCL
 * @Date: 2024/6/4 下午3:36
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class StatisticsServerImpl implements StatisticsServer {
    private final RecordSubDao recordSubDao;
    /**
     * 本季开始时间
     */
    private static final DateTime QUARTER_START_TIME = DateUtil.beginOfQuarter(new Date());
    /**
     * 本季结束时间
     */
    private static final DateTime QUARTER_END_TIME = DateUtil.endOfQuarter(new Date());
    /**
     * 本年开始时间
     */
    private static final DateTime YEAR_START_TIME = DateUtil.beginOfYear(new Date());
    /**
     * 本年结束时间
     */
    private static final DateTime YEAR_END_TIME = DateUtil.endOfYear(new Date());

    @Override
    public AssessmentRankRes assessmentRank() {
        /**
         * 当前季度
         */
        List<RecordSubEntity> nowQuarterList = recordSubDao.lambdaQuery()
                .eq(BaseEntity::getDr, 0)
                .isNotNull(RecordSubEntity::getFinalScore)
                .between(BaseEntity::getCreateTime, QUARTER_START_TIME, QUARTER_END_TIME)
                .list();
        /**
         * 当年
         */
        List<RecordSubEntity> nowYearList = recordSubDao.lambdaQuery()
                .eq(BaseEntity::getDr, 0)
                .isNotNull(RecordSubEntity::getFinalScore)
                .between(BaseEntity::getCreateTime, YEAR_START_TIME, YEAR_END_TIME)
                .list();
        /**
         * 本部 org_type=4
         */
        List<RecordSubEntity> bbYearList = nowYearList.stream().filter(e -> e.getOrgType() == 4).collect(Collectors.toList());
        List<RecordSubEntity> bbQuarterList = nowQuarterList.stream().filter(e -> e.getOrgType() == 4).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> bbInfoList = disposeScores(bbQuarterList, bbYearList);
        /**
         * 分子公司 org_type=2
         */
        List<RecordSubEntity> fzgsYearList = nowYearList.stream().filter(e -> e.getOrgType() == 2).collect(Collectors.toList());
        List<RecordSubEntity> fzgsQuarterList = nowQuarterList.stream().filter(e -> e.getOrgType() == 2).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> fzgsInfoList = disposeScores(fzgsQuarterList, fzgsYearList);
        /**
         * 项目 org_type=3、5
         */
        List<RecordSubEntity> xmYearList = nowYearList.stream().filter(e -> e.getOrgType() == 3 || e.getOrgType() == 5).collect(Collectors.toList());
        List<RecordSubEntity> xmQuarterList = nowQuarterList.stream().filter(e -> e.getOrgType() == 3 || e.getOrgType() == 5).collect(Collectors.toList());
        List<AssessmentRankRes.AssessmentRankInfo> xmInfoList = disposeScores(xmQuarterList, xmYearList);

        return AssessmentRankRes.builder()
                .bb(bbInfoList)
                .fzgs(fzgsInfoList)
                .xm(xmInfoList)
                .build();
    }

    @Override
    public RecordStatisticsRes record(String departmentCode) {
        List<RecordSubEntity> recordSubEntityList = recordSubDao.lambdaQuery()
                .eq(RecordSubEntity::getAssessmentDepartmentCode, departmentCode)
                .isNotNull(RecordSubEntity::getFinalScore)
                .list();
        if (recordSubEntityList.isEmpty()) {
            return RecordStatisticsRes.builder().build();
        }
        ////本季得分
        int quarterScore = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(QUARTER_START_TIME) && e.getCreateTime().before(QUARTER_END_TIME))
                .collect(Collectors.toList())
                .stream()
                .mapToInt(RecordSubEntity::getFinalScore)
                .sum();
        //本年度加减分
        int yearScore = recordSubEntityList.stream()
                .filter(e -> e.getCreateTime().after(YEAR_START_TIME) && e.getCreateTime().before(YEAR_END_TIME))
                .collect(Collectors.toList())
                .stream()
                .mapToInt(RecordSubEntity::getFinalScore)
                .sum();
        //年度总分
        int totalAnnualScore = 100 + yearScore;
        //该部门所有考核记录情况
        Map<Long, List<RecordSubEntity>> recordSubMap = recordSubEntityList.stream()
                .collect(Collectors.groupingBy(RecordSubEntity::getNormId));
        /**
         * 板块平均得分(本部/分子公司/项目)
         */
        //获取板块信息
        Integer orgType = recordSubEntityList.get(0).getOrgType();
        List<RecordSubEntity> orgTypeRecordSubList = recordSubDao.lambdaQuery()
                .eq(RecordSubEntity::getOrgType, orgType)
                .isNotNull(RecordSubEntity::getFinalScore)
                .list();
        //板块总分
        int plateScoreSum = orgTypeRecordSubList.stream().mapToInt(RecordSubEntity::getFinalScore).sum();
        //板块数量
        Map<String, List<RecordSubEntity>> collect = orgTypeRecordSubList.stream()
                .collect(Collectors.groupingBy(RecordSubEntity::getAssessmentDepartmentCode));
        int plateSize = collect.size();
        //板块平均得分
        BigDecimal plateAverageScore = BigDecimal.valueOf(NumberUtil.div(plateScoreSum, plateSize));
        //排名
        int rank = orgTypeRecordSubList.stream()
                //汇总
                .collect(Collectors.groupingBy(RecordSubEntity::getAssessmentDepartmentCode
                        , Collectors.summingInt(RecordSubEntity::getFinalScore)))
                .entrySet()
                .stream()
                //排序
                .sorted(Map.Entry.<String, Integer>comparingByValue().reversed())
                .map(Map.Entry::getKey)
                .collect(Collectors.toList())
                //获取当前部门排名
                .indexOf(departmentCode) + 1;
        return RecordStatisticsRes.builder()
                .quarterScore(quarterScore)
                .yearScore(yearScore)
                .totalAnnualScore(totalAnnualScore)
                .plateAverageScore(plateAverageScore)
                .rank(rank)
                .total(plateSize)
                .recordSubMap(recordSubMap)
                .build();
    }

    /**
     * 分数处理 年度总分 = 初始分 + 当年分数(可能为负数)
     *
     * @param quarterList 当前季度  只是用来显示，不参与分数计算
     * @param yearList    当年  纬度最大，以这个为基础计算总分 用来参与分数计算  加上年度总分初始赋分
     * @return NucleusRankRes.NucleusRankInfo
     */
    private List<AssessmentRankRes.AssessmentRankInfo> disposeScores(List<RecordSubEntity> quarterList, List<RecordSubEntity> yearList) {
        List<AssessmentRankRes.AssessmentRankInfo> infoList = Lists.newArrayList();
        Map<String, Map<String, List<RecordSubEntity>>> unitDepartmentYearMap = yearList.stream().collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode,
                Collectors.groupingBy(RecordSubEntity::getAssessmentDepartmentCode)));

        Map<String, Map<String, List<RecordSubEntity>>> unitDepartmentQuarterMap = quarterList.stream().collect(Collectors.groupingBy(RecordSubEntity::getAssessmentUnitCode,
                Collectors.groupingBy(RecordSubEntity::getAssessmentDepartmentCode)));
        for (Map.Entry<String, Map<String, List<RecordSubEntity>>> stringMapEntry : unitDepartmentYearMap.entrySet()) {
            String unitCode = stringMapEntry.getKey();
            Map<String, List<RecordSubEntity>> value = stringMapEntry.getValue();
            for (Map.Entry<String, List<RecordSubEntity>> stringListEntry : value.entrySet()) {
                //年度总分初始分
                int totalAnnualScore = 100;
                String departmentCode = stringListEntry.getKey();
                List<RecordSubEntity> recordSubList = stringListEntry.getValue();
                //该 单位——部门 年度分
                int yearScore = recordSubList.stream().mapToInt(RecordSubEntity::getFinalScore).sum();
                //该部门年度总分
                totalAnnualScore += yearScore;
                //当前季度分
                int quarterScore = 0;
                Map<String, List<RecordSubEntity>> stringListMap = unitDepartmentQuarterMap.get(unitCode);
                if (Objects.nonNull(stringListMap)) {
                    List<RecordSubEntity> recordSubEntityList = stringListMap.get(departmentCode);
                    if (recordSubEntityList != null) {
                        quarterScore = recordSubEntityList.stream().mapToInt(RecordSubEntity::getFinalScore).reduce(Integer::sum).orElse(0);
                    }
                }
                AssessmentRankRes.AssessmentRankInfo infoRes = AssessmentRankRes.AssessmentRankInfo.builder()
                        .unitCode(unitCode)
                        .unitName(recordSubList.get(0).getAssessmentUnitName())
                        .departmentCode(departmentCode)
                        .departmentName(recordSubList.get(0).getAssessmentDepartmentName())
                        .quarterScore(quarterScore)
                        .yearScore(yearScore)
                        .totalAnnualScore(totalAnnualScore)
                        .build();
                infoList.add(infoRes);
            }
        }
        infoList.sort(Comparator.comparing(AssessmentRankRes.AssessmentRankInfo::getTotalAnnualScore).reversed());
        return infoList;
    }
}
