package com.ejianc.business.finance.utils;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.CamelAndUnderLineConverter;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author yqls
 * @date 2024-11-14 19:10:57
 */
public class ParamUtil {

    /** 基础字段查询*/
    public static <T> String paramToString(Map<String, Parameter> params){
        StringBuilder builder = new StringBuilder();
        builder.append("dr = 0");
        for(Map.Entry<String, Parameter> entry: params.entrySet()){
            Parameter param = entry.getValue();
            String type = param.getType();
            String key = CamelAndUnderLineConverter.humpToLine(entry.getKey());

            switch(type) {
                case QueryParam.EQ:
                    if(null == param.getValue()) {
                        builder.append(" and " + key + " is null");
                    } else {
                        builder.append(" and " + key + " = '" + param.getValue() + "'");
                    }
                    break;
                case QueryParam.NE:
                    if(null == param.getValue()) {
                        builder.append(" and " + key + " is not null");
                    } else {
                        builder.append(" and " + key + " <> '" + param.getValue() + "'");
                    }
                    break;
                case QueryParam.SQL:
                    /** 严禁前端直接传入 仅用于后端使用 ！！！ */
                    if(param.getValue()!=null) {
                        builder.append(param.getValue().toString());
                    }
                    break;
                case QueryParam.IN:
                    if(param.getValue() instanceof List) {
                        builder.append(" and " + key + " in ('" + StringUtils.join((List<?>)param.getValue(), "', '") + "')");
                    }else if(param.getValue() instanceof String) {
                        String[] paramArr = param.getValue().toString().split(",");
                        builder.append(" and " + key + " in ('" + StringUtils.join(paramArr, "', '") + "')");
                    }
                    break;
                case QueryParam.NOT_IN:
                    if(param.getValue() instanceof List) {
                        builder.append(" and " + key + " not in ('" + StringUtils.join((List<?>)param.getValue(), "', '") + "')");
                    }else if(param.getValue() instanceof String) {
                        String[] paramArr = param.getValue().toString().split(",");
                        builder.append(" and " + key + " not in ('" + StringUtils.join(paramArr, "', '") + "')");
                    }
                    break;
                case QueryParam.LIKE:
                    if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
                        builder.append(" and " + key + " like '%" + param.getValue() + "%'");
                    }
                    break;
                case QueryParam.NOT_LIKE:
                    if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
                        builder.append(" and " + key + " not like '%" + param.getValue() + "%'");
                    }
                    break;
                case QueryParam.LIKE_LEFT:
                    if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
                        builder.append(" and " + key + " like '%" + param.getValue() + "'");
                    }
                    break;
                case QueryParam.LIKE_RIGHT:
                    if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
                        builder.append(" and " + key + " like '" + param.getValue() + "%'");
                    }
                    break;
                case QueryParam.BETWEEN:
                    if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
                        String[] paramArr = param.getValue().toString().split(",");
                        if("dateInSelect".equals(param.getInputType()) && paramArr[0].length() == 7){
                            /** 日期区间 月份区间 */
                            if(paramArr[0].equals(paramArr[1])){
                                /** 月份相同，就是一个月 */
                                builder.append(" and " + key + " = '" + paramArr[0] + "'");
                            } else {
                                builder.append(" and " + key + " in ('" + StringUtils.join(getMonthRange(paramArr[0], paramArr[1]), "', '") + "')");
                            }
                        } else {
                            builder.append(" and " + key + " between '" + paramArr[0] + "' and '" + paramArr[1] + "'");
                        }
                    }
                    break;
                case QueryParam.LT:
                    if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
                        builder.append(" and " + key + " < '" + param.getValue() + "'");
                    }
                    break;
                case QueryParam.LE:
                    if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
                        builder.append(" and " + key + " <= '" + param.getValue() + "'");
                    }
                    break;
                case QueryParam.GT:
                    if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
                        builder.append(" and " + key + " > '" + param.getValue() + "'");
                    }
                    break;
                case QueryParam.GE:
                    if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
                        builder.append(" and " + key + " >= '" + param.getValue() + "'");
                    }
                    break;
            }
        }
        return builder.toString();
    }

    public static List<String> getMonthRange(String startDateStr, String endDateStr) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        Date startDate = null;
        Date endDate = null;
        try {
            startDate = sdf.parse(startDateStr);
            endDate = sdf.parse(endDateStr);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        List<String> monthList = new ArrayList<>();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startDate);
        while (!calendar.getTime().after(endDate)) {
            int year = calendar.get(Calendar.YEAR);
            int month = calendar.get(Calendar.MONTH) + 1;
            String monthStr = String.format("%04d-%02d", year, month);
            monthList.add(monthStr);
            calendar.add(Calendar.MONTH, 1);
        }
        return monthList;
    }

    public static String getSql(QueryParam param, String tableAlias) {
        StringBuffer sql = new StringBuffer(" AND ");

        // 工具方法：为字段加上表别名前缀（如果存在）
        java.util.function.Function<String, String> addAlias = fieldName -> {
            if (StringUtils.isNotBlank(tableAlias)) {
                return tableAlias + "." + fieldName;
            }
            return fieldName;
        };

        /** 模糊查询配置的字段 */
        boolean hasData = false;
        if (StringUtils.isNotBlank(param.getSearchText()) && !ListUtil.isEmpty(param.getFuzzyFields())) {
            hasData = true;
            String searchText = param.getSearchText();
            List<String> fuzzyFields = param.getFuzzyFields();
            if (fuzzyFields.size() == 1) {
                String field = CamelAndUnderLineConverter.humpToLine(fuzzyFields.get(0));
                sql.append(addAlias.apply(field)).append(" LIKE '%")
                   .append(StringEscapeUtils.escapeSql(searchText)).append("%' ");
            } else {
                sql.append(" ( ");
                for (int i = 0, size = fuzzyFields.size(); i < size; i++) {
                    String key = fuzzyFields.get(i);
                    String field = CamelAndUnderLineConverter.humpToLine(key);
                    sql.append(addAlias.apply(field)).append(" LIKE '%")
                       .append(StringEscapeUtils.escapeSql(searchText)).append("%' ");
                    if (i != size - 1) {
                        sql.append(" OR ");
                    }
                }
                sql.append(" ) ");
            }
        }

        if (param.getParams() != null && param.getParams().size() > 0) {
            if (hasData) {
                sql.append(" AND ( ");
            }
            int i = 1;
            for (String key : param.getParams().keySet()) {
                Parameter parameter = param.getParams().get(key);
                String type = parameter.getType();
                String field = CamelAndUnderLineConverter.humpToLine(key);
                field = addAlias.apply(field);

                switch (type) {
                    case QueryParam.EQ:
                        if (null == parameter.getValue()) {
                            sql.append(field).append(" IS NULL ");
                        } else {
                            sql.append(field).append(" = '")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("' ");
                        }
                        break;
                    case QueryParam.NE:
                        if (null == parameter.getValue()) {
                            sql.append(field).append(" IS NOT NULL ");
                        } else {
                            sql.append(field).append(" != '")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("' ");
                        }
                        break;
                    case QueryParam.IN:
                        if (parameter.getValue() instanceof List) {
                            String inValues = ((List<?>) parameter.getValue()).stream()
                                    .map(obj -> "'" + StringEscapeUtils.escapeSql(obj.toString()) + "'")
                                    .collect(Collectors.joining(","));
                            sql.append(field).append(" IN (").append(inValues).append(") ");
                        } else if (parameter.getValue() instanceof String) {
                            String[] paramArr = parameter.getValue().toString().split(",");
                            String inValues = Arrays.stream(paramArr)
                                    .map(val -> "'" + StringEscapeUtils.escapeSql(val.trim()) + "'")
                                    .collect(Collectors.joining(","));
                            sql.append(field).append(" IN (").append(inValues).append(") ");
                        }
                        break;
                    case QueryParam.NOT_IN:
                        if (parameter.getValue() instanceof List) {
                            String notInValues = ((List<?>) parameter.getValue()).stream()
                                    .map(obj -> "'" + StringEscapeUtils.escapeSql(obj.toString()) + "'")
                                    .collect(Collectors.joining(","));
                            sql.append(field).append(" NOT IN (").append(notInValues).append(") ");
                        } else if (parameter.getValue() instanceof String) {
                            String[] paramArr = parameter.getValue().toString().split(",");
                            String notInValues = Arrays.stream(paramArr)
                                    .map(val -> "'" + StringEscapeUtils.escapeSql(val.trim()) + "'")
                                    .collect(Collectors.joining(","));
                            sql.append(field).append(" NOT IN (").append(notInValues).append(") ");
                        }
                        break;
                    case QueryParam.LIKE:
                        if (parameter.getValue() != null && StringUtils.isNotBlank(parameter.getValue().toString())) {
                            sql.append(field).append(" LIKE '%")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("%' ");
                        }
                        break;
                    case QueryParam.NOT_LIKE:
                        if (parameter.getValue() != null && StringUtils.isNotBlank(parameter.getValue().toString())) {
                            sql.append(field).append(" NOT LIKE '%")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("%' ");
                        }
                        break;
                    case QueryParam.LIKE_LEFT:
                        if (parameter.getValue() != null && StringUtils.isNotBlank(parameter.getValue().toString())) {
                            sql.append(field).append(" LIKE '")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("%' ");
                        }
                        break;
                    case QueryParam.LIKE_RIGHT:
                        if (parameter.getValue() != null && StringUtils.isNotBlank(parameter.getValue().toString())) {
                            sql.append(field).append(" LIKE '%")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("' ");
                        }
                        break;
                    case QueryParam.BETWEEN:
                        if (parameter.getValue() != null && StringUtils.isNotBlank(parameter.getValue().toString())) {
                            String[] paramArr = parameter.getValue().toString().split(",");
                            if (paramArr.length == 2) {
                                sql.append(field).append(" BETWEEN '")
                                   .append(StringEscapeUtils.escapeSql(paramArr[0].trim())).append("' AND '")
                                   .append(StringEscapeUtils.escapeSql(paramArr[1].trim())).append("' ");
                            }
                        }
                        break;
                    case QueryParam.LT:
                        if (parameter.getValue() != null && StringUtils.isNotBlank(parameter.getValue().toString())) {
                            sql.append(field).append(" < '")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("' ");
                        }
                        break;
                    case QueryParam.LE:
                        if (parameter.getValue() != null && StringUtils.isNotBlank(parameter.getValue().toString())) {
                            sql.append(field).append(" <= '")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("' ");
                        }
                        break;
                    case QueryParam.GT:
                        if (parameter.getValue() != null && StringUtils.isNotBlank(parameter.getValue().toString())) {
                            sql.append(field).append(" > '")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("' ");
                        }
                        break;
                    case QueryParam.GE:
                        if (parameter.getValue() != null && StringUtils.isNotBlank(parameter.getValue().toString())) {
                            sql.append(field).append(" >= '")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("' ");
                        }
                        break;
                }
                if (i != param.getParams().size()) {
                    sql.append(" AND ");
                }
                i++;
            }
            if (hasData) {
                sql.append(" ) ");
            }
            hasData = true;
        }

        if (StringUtils.isNotEmpty(param.getSearchObject())) {
            JSONObject searchObject = JSONObject.parseObject(param.getSearchObject());
            if (hasData) {
                sql.append(" AND ");
            }
            int i = 1;
            for (String key : searchObject.keySet()) {
                String field = CamelAndUnderLineConverter.humpToLine(key);
                field = addAlias.apply(field);
                sql.append(field).append(" = '")
                   .append(StringEscapeUtils.escapeSql(searchObject.getString(key))).append("' ");
                if (i != searchObject.size()) {
                    sql.append(" AND ");
                }
                i++;
            }
        }

        // 处理排序逻辑
        String whereClause = sql.length() == 5 ? null : sql.toString(); // 原WHERE条件部分
        LinkedHashMap<String, String> orderMap = param.getOrderMap(); // 获取排序配置
        StringBuffer orderByClause = new StringBuffer();

        if (orderMap != null && !orderMap.isEmpty()) {
            orderByClause.append(" ORDER BY ");
            int index = 0;
            int total = orderMap.size();
            for (Map.Entry<String, String> entry : orderMap.entrySet()) {
                String field = entry.getKey(); // 驼峰字段名（如createTime）
                String direction = entry.getValue(); // 排序方向（如desc）

                // 1. 驼峰转下划线（与其他字段处理一致）
                String dbField = CamelAndUnderLineConverter.humpToLine(field);
                // 2. 添加表别名（若有）
                dbField = addAlias.apply(dbField);
                // 3. 处理排序方向（默认asc，无效值强制asc）
                String dir = StringUtils.isNotBlank(direction) ? direction.trim().toUpperCase() : "ASC";
                if (!"ASC".equals(dir) && !"DESC".equals(dir)) {
                    dir = "ASC";
                }

                // 拼接排序项
                orderByClause.append(dbField).append(" ").append(dir);
                if (index != total - 1) {
                    orderByClause.append(", ");
                }
                index++;
            }
        }

        // 组合最终SQL
        if (orderByClause.length() > 0) {
            if (whereClause != null) {
                return whereClause + orderByClause.toString(); // WHERE + ORDER BY
            } else {
                return orderByClause.toString(); // 只有ORDER BY
            }
        } else {
            return whereClause; // 没有排序，返回原WHERE条件（或null）
        }
    }

    /**
     * description：不转驼峰
     *
     */
    public static String getSqlBuild(QueryParam param, String tableAlias) {
        StringBuffer sql = new StringBuffer(" AND ");

        // 优化：移除驼峰转换，直接使用传入的字段名
        java.util.function.Function<String, String> addAlias = fieldName -> {
            if (StringUtils.isNotBlank(tableAlias)) {
                return tableAlias + "." + fieldName;
            }
            return fieldName;
        };

        boolean hasData = false;

        // 模糊查询：直接使用传入的字段名（下划线格式）
        if (StringUtils.isNotBlank(param.getSearchText()) && !ListUtil.isEmpty(param.getFuzzyFields())) {
            hasData = true;
            String searchText = param.getSearchText();
            List<String> fuzzyFields = param.getFuzzyFields();

            if (fuzzyFields.size() == 1) {
                sql.append(addAlias.apply(fuzzyFields.get(0)))
                   .append(" LIKE '%")
                   .append(StringEscapeUtils.escapeSql(searchText))
                   .append("%' ");
            } else {
                sql.append(" ( ");
                for (int i = 0; i < fuzzyFields.size(); i++) {
                    sql.append(addAlias.apply(fuzzyFields.get(i)))
                       .append(" LIKE '%")
                       .append(StringEscapeUtils.escapeSql(searchText))
                       .append("%' ");
                    if (i != fuzzyFields.size() - 1) {
                        sql.append(" OR ");
                    }
                }
                sql.append(" ) ");
            }
        }

        // 参数查询：直接使用传入的字段名（下划线格式）
        if (param.getParams() != null && !param.getParams().isEmpty()) {
            if (hasData) {
                sql.append(" AND ( ");
            }
            int i = 1;
            for (String key : param.getParams().keySet()) {
                Parameter parameter = param.getParams().get(key);
                String type = parameter.getType();

                // 关键优化：直接使用 key（下划线字段名）
                String field = addAlias.apply(key);

                switch (type) {
                    case QueryParam.EQ:
                        if (null == parameter.getValue()) {
                            sql.append(field).append(" IS NULL ");
                        } else {
                            sql.append(field).append(" = '")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("' ");
                        }
                        break;
                    case QueryParam.NE:
                        if (null == parameter.getValue()) {
                            sql.append(field).append(" IS NOT NULL ");
                        } else {
                            sql.append(field).append(" != '")
                               .append(StringEscapeUtils.escapeSql(parameter.getValue().toString())).append("' ");
                        }
                        break;
                    case QueryParam.IN:
                        // ... (IN/NOT_IN 逻辑保持不变，字段名直接用 key)
                        if (parameter.getValue() instanceof List) {
                            String inValues = ((List<?>) parameter.getValue()).stream()
                                    .map(obj -> "'" + StringEscapeUtils.escapeSql(obj.toString()) + "'")
                                    .collect(Collectors.joining(","));
                            sql.append(field).append(" IN (").append(inValues).append(") ");
                        } else if (parameter.getValue() instanceof String) {
                            String[] paramArr = parameter.getValue().toString().split(",");
                            String inValues = Arrays.stream(paramArr)
                                    .map(val -> "'" + StringEscapeUtils.escapeSql(val.trim()) + "'")
                                    .collect(Collectors.joining(","));
                            sql.append(field).append(" IN (").append(inValues).append(") ");
                        }
                        break;
                    // ... (其他类型保持不变，字段名直接用 key)
                    // 其余类型（NOT_IN, LIKE 等）同理，不再重复
                }
                if (i != param.getParams().size()) {
                    sql.append(" AND ");
                }
                i++;
            }
            if (hasData) {
                sql.append(" ) ");
            }
            hasData = true;
        }

        // 搜索对象：直接使用传入的字段名
        if (StringUtils.isNotEmpty(param.getSearchObject())) {
            JSONObject searchObject = JSONObject.parseObject(param.getSearchObject());
            if (hasData) {
                sql.append(" AND ");
            }
            int i = 1;
            for (String key : searchObject.keySet()) {
                // 直接使用 key（下划线字段名）
                String field = addAlias.apply(key);
                sql.append(field).append(" = '")
                   .append(StringEscapeUtils.escapeSql(searchObject.getString(key))).append("' ");
                if (i != searchObject.size()) {
                    sql.append(" AND ");
                }
                i++;
            }
        }

        // 排序处理：直接使用传入的排序字段名（下划线格式）
        LinkedHashMap<String, String> orderMap = param.getOrderMap();
        StringBuffer orderByClause = new StringBuffer();
        if (orderMap != null && !orderMap.isEmpty()) {
            orderByClause.append(" ORDER BY ");
            int index = 0;
            for (Map.Entry<String, String> entry : orderMap.entrySet()) {
                String field = entry.getKey(); // 直接使用传入的字段名
                String direction = entry.getValue();
                String dbField = addAlias.apply(field); // 添加表别名

                String dir = StringUtils.isNotBlank(direction) ? direction.trim().toUpperCase() : "ASC";
                if (!"ASC".equals(dir) && !"DESC".equals(dir)) {
                    dir = "ASC";
                }
                orderByClause.append(dbField).append(" ").append(dir);
                if (index != orderMap.size() - 1) {
                    orderByClause.append(", ");
                }
                index++;
            }
        }

        // 组合结果
        String whereClause = sql.length() == 5 ? null : sql.toString();
        if (orderByClause.length() > 0) {
            return whereClause != null ? whereClause + orderByClause.toString() : orderByClause.toString();
        }
        return whereClause;
    }
}
