package com.ejianc.business.finance.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.finance.bean.PayContractEntity;
import com.ejianc.business.finance.bean.PayForegiftEntity;
import com.ejianc.business.finance.bean.PayForegiftPlanEntity;
import com.ejianc.business.finance.bean.PayForegiftRecordEntity;
import com.ejianc.business.finance.mapper.PayForegiftMapper;
import com.ejianc.business.finance.service.IPayForegiftPlanService;
import com.ejianc.business.finance.service.IPayForegiftRecordService;
import com.ejianc.business.finance.service.IPayForegiftService;
import com.ejianc.business.finance.util.BillTypeCodeEnum;
import com.ejianc.business.finance.util.ValidateUtil;
import com.ejianc.business.finance.vo.PayForegiftPlanVO;
import com.ejianc.business.finance.vo.PayForegiftRecordVO;
import com.ejianc.business.finance.vo.PayForegiftVO;
import com.ejianc.business.finance.vo.TotalColumnVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 *  服务实现类
 * </p>
 *
 * @author yqls
 * @since 2020-06-04
 */
@Service
public class PayForegiftServiceImpl extends BaseServiceImpl<PayForegiftMapper, PayForegiftEntity> implements IPayForegiftService {

    private static final String PAY_FOREGIFT_BILL_CODE = "PAY_FOREGIFT";

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IPayForegiftPlanService planService;

    @Autowired
    private IPayForegiftRecordService recordService;

    @Autowired
    private IOrgApi orgApi;

    @Override
    public PayForegiftVO insertOrUpdate(PayForegiftVO vo) {
        PayForegiftEntity entity = BeanMapper.map(vo, PayForegiftEntity.class);
        if(!vo.getManageFlag()){
            // 校验同合同、同组织下只能有一张未生效单据
            this.validateBeforeSave(entity);
        }
        // 保存时校验合同version是否一致
        String billTypeCode = BillTypeCodeEnum.getContractBillTypeCode(vo.getContractType());
        if(!ValidateUtil.validateUpstreamVersion(String.valueOf(vo.getContractId()), billTypeCode, vo.getContractVersion())){
            throw new BusinessException("该合同已被更新，请刷新后重做！");
        }
        // 自动生成编码
        this.autoSetBillCode(entity);
        // 保存主表
        super.saveOrUpdate(entity);
        Long id = entity.getId();
        // 保存预计退还计划
        List<PayForegiftPlanVO> planVOList = this.updatePlanVOList(vo, id);
        // 保存退还记录
        List<PayForegiftRecordVO> recordVOList = this.updateRecordVOList(vo, id);
        // 返回VO
        PayForegiftVO backVO = BeanMapper.map(super.getById(id), PayForegiftVO.class);
        backVO.setPlanVOList(planVOList);// 计划列表
        backVO.setRecordVOList(recordVOList);// 记录列表
        return backVO;
    }

    /**
     * 预计退还计划
     * @param foregiftVO
     * @param payapplyId
     * @return
     */
    private List<PayForegiftPlanVO> updatePlanVOList(PayForegiftVO foregiftVO, Long payapplyId) {
        // 新增
        List<PayForegiftPlanVO> planVOList = foregiftVO.getPlanVOList();
        if(planVOList != null && !planVOList.isEmpty()){
            List<PayForegiftPlanEntity> planEntityList = BeanMapper.mapList(planVOList, PayForegiftPlanEntity.class);
            // 保存预计退还时间
            this.savePlanTime(payapplyId, planEntityList);
            planService.saveOrUpdateBatch(planEntityList, planEntityList.size(), false);
            planVOList = BeanMapper.mapList(planEntityList, PayForegiftPlanVO.class);
        }
        // 删除
        List<Long> ids = planVOList.stream().map(PayForegiftPlanVO::getId).collect(Collectors.toList());
        QueryWrapper wrapper = new QueryWrapper<PayForegiftPlanEntity>();
        wrapper.eq("payapply_id", payapplyId);
        wrapper.notIn(!ids.isEmpty(),"id", ids);
        planService.remove(wrapper, false);
        return planVOList;
    }

    /**
     * 保存预计退还时间
     * @param payapplyId
     * @param planEntityList
     */
    private void savePlanTime(Long payapplyId, List<PayForegiftPlanEntity> planEntityList) {
        Date planTime = null;
        Date newTime = new Date();
        for(PayForegiftPlanEntity planEntity : planEntityList){
            planEntity.setPayapplyId(payapplyId);
            if(planEntity.getPlanTime().after(newTime)){
                if(planTime == null){
                    planTime = planEntity.getPlanTime();
                } else {
                    planTime = planTime.before(planEntity.getPlanTime()) ? planTime : planEntity.getPlanTime();
                }
            }
        }
        if(planTime != null){
            PayForegiftEntity forgiftEntity = baseMapper.selectById(payapplyId);
            forgiftEntity.setPlanTime(planTime);
            super.saveOrUpdate(forgiftEntity);
        }
    }

    /**
     * 预计退还计划
     * @param foregiftVO
     * @param payapplyId
     * @return
     */
    private List<PayForegiftRecordVO> updateRecordVOList(PayForegiftVO foregiftVO, Long payapplyId) {
        // 新增
        List<PayForegiftRecordVO> recordVOList = foregiftVO.getRecordVOList();
        if(recordVOList != null && !recordVOList.isEmpty()){
            List<PayForegiftRecordEntity> recordEntityList = BeanMapper.mapList(recordVOList, PayForegiftRecordEntity.class);
            for(PayForegiftRecordEntity recordEntity : recordEntityList){
                recordEntity.setPayapplyId(payapplyId);
            }
            recordService.saveOrUpdateBatch(recordEntityList, recordEntityList.size(), false);
            recordVOList = BeanMapper.mapList(recordEntityList, PayForegiftRecordVO.class);
        }
        // 删除
        if(recordVOList != null){
            List<Long> ids = recordVOList.stream().map(PayForegiftRecordVO::getId).collect(Collectors.toList());
            QueryWrapper wrapper = new QueryWrapper<PayForegiftRecordEntity>();
            wrapper.eq("payapply_id", payapplyId);
            wrapper.notIn(!ids.isEmpty(),"id", ids);
            recordService.remove(wrapper, false);
        }
        return recordVOList;
    }

    @Override
    public PayForegiftVO queryDetail(Long id) {
        // 查询主表
        PayForegiftEntity entity = baseMapper.selectById(id);
        PayForegiftVO vo = BeanMapper.map(entity, PayForegiftVO.class);

        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("payapplyId", new Parameter(QueryParam.EQ, vo.getId()));
        queryParam.getOrderMap().put("createTime", "desc");
        // 查询预计退还计划
        List<PayForegiftPlanEntity> payForegiftPlanEntityList = planService.queryList(queryParam, false);
        vo.setPlanVOList(BeanMapper.mapList(payForegiftPlanEntityList, PayForegiftPlanVO.class));
        // 查询退还记录
        List<PayForegiftRecordEntity> payForegiftRecordEntityList = recordService.queryList(queryParam, false);
        vo.setRecordVOList(BeanMapper.mapList(payForegiftRecordEntityList, PayForegiftRecordVO.class));
        return vo;
    }

    @Override
    public List<PayForegiftVO> queryExportList(QueryParam param) {
        param.setPageIndex(0);
        param.setPageSize(-1);
        List<PayForegiftVO> resVos = (List<PayForegiftVO>) queryPageJson(param, false).get("records");
        if(!resVos.isEmpty()){
            for(int i = 0 ; i< resVos.size(); i++){
                PayForegiftVO vo = resVos.get(i);
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            };
        }
        return resVos;
    }

    @Override
    public String delete(List<Long> ids) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("id", new Parameter(QueryParam.IN, ids));
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        List<PayForegiftEntity> entityList = super.queryList(queryParam, false);
        if(CollectionUtils.isNotEmpty(entityList)) {
            super.removeByIds(ids, false);
            // 删除计划
            planService.remove(new QueryWrapper<PayForegiftPlanEntity>().in("payapply_id", ids), false);
            // 删除记录
            recordService.remove(new QueryWrapper<PayForegiftRecordEntity>().in("payapply_id", ids), false);
        }
        return "删除成功！";
    }

    @Override
    public JSONObject queryPageJson(QueryParam param, boolean isEs) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("orgName");
        fuzzyFields.add("receiveUnitName");
        fuzzyFields.add("applyUserName");
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 组织本下
        param.getParams().put("org_id",new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        // 处理是否逾期与全部返还条件
        this.dealQueryParaam(param);
        IPage<PayForegiftEntity> pageData = super.queryPage(param,false);
        List<PayForegiftVO> voList = BeanMapper.mapList(pageData.getRecords(), PayForegiftVO.class);
        // 最近一次未到期预计退还时间
        this.dealPlanTime(voList);
        com.alibaba.fastjson.JSONObject page = new com.alibaba.fastjson.JSONObject();
        page.put("records", voList);
        page.put("total", pageData.getTotal());
        page.put("current", pageData.getCurrent());
        page.put("size", pageData.getSize());
        page.put("pages", pageData.getPages());
        return page;
    }

    @Override
    public TotalColumnVO getTotalColumnInfo(QueryParam param, boolean isEs) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("orgName");
        fuzzyFields.add("receiveUnitName");
        fuzzyFields.add("applyUserName");
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 组织本下
        param.getParams().put("org_id",new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        // 处理是否逾期与全部返还条件
        this.dealQueryParaam(param);
        List<PayForegiftEntity> list = this.queryList(param);
        List<PayForegiftVO> voList = BeanMapper.mapList(list, PayForegiftVO.class);
        // 最近一次未到期预计退还时间
        this.dealPlanTime(voList);

        // 过滤为空的数据并求和
        BigDecimal sumApplyMny = voList.stream().filter(entity -> entity.getApplyMny() != null)
                .map(PayForegiftVO::getApplyMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal sumPayMny = voList.stream().filter(entity -> entity.getPayMny() != null)
                .map(PayForegiftVO::getPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);

        TotalColumnVO vo = new TotalColumnVO();
        vo.setSumApplyMny(sumApplyMny);
        vo.setSumPayMny(sumPayMny);

        return vo;
    }

    @Override
    public TotalColumnVO getApproveTotalColumnInfo(QueryParam param, boolean isEs) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("orgName");
        fuzzyFields.add("receiveUnitName");
        fuzzyFields.add("applyUserName");
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 组织本下
        param.getParams().put("org_id",new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        // 处理是否逾期与全部返还条件
        this.dealQueryParaam(param);
        List<PayForegiftEntity> list = this.queryList(param);
        List<PayForegiftVO> voList = BeanMapper.mapList(list, PayForegiftVO.class);
        // 最近一次未到期预计退还时间
        this.dealPlanTime(voList);

        // 过滤为空的数据并求和
        BigDecimal sumApplyMny = voList.stream().filter(entity -> entity.getApplyMny() != null)
                .map(PayForegiftVO::getApplyMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal sumPayMny = voList.stream().filter(entity -> entity.getPayMny() != null)
                .map(PayForegiftVO::getPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);

        TotalColumnVO vo = new TotalColumnVO();
        vo.setSumApplyMny(sumApplyMny);
        vo.setSumPayMny(sumPayMny);

        return vo;

    }

    /**
     * 最近一次未到期预计退还时间
     * @param voList
     */
    private void dealPlanTime(List<PayForegiftVO> voList) {
        for (PayForegiftVO vo : voList){
            // 查询预计退还计划
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("payapplyId", new Parameter(QueryParam.EQ, vo.getId()));
            queryParam.getParams().put("planTime", new Parameter(QueryParam.GE, new Date()));
            queryParam.getOrderMap().put("planTime", QueryParam.ASC);
            List<PayForegiftPlanEntity> entityList = planService.queryList(queryParam, false);
            Date planTime = null;
            for(PayForegiftPlanEntity entity : entityList){
                if(planTime == null){
                    planTime = entity.getPlanTime();
                } else {
                    planTime = planTime.before(entity.getPlanTime()) ? planTime : entity.getPlanTime();
                }
            }
            vo.setPlanTime(planTime);
        }
    }

    /**
     * 处理是否逾期与全部返还条件
     * @param param
     */
    private void dealQueryParaam(QueryParam param) {
        // 是否逾期
        Map<String, Parameter> parameterMap = param.getParams();
        if(parameterMap.containsKey("overdueFlag")){
            Parameter parameter = parameterMap.get("overdueFlag");
            if(!"1,2".equals(parameter.getValue()) && !"2,1".equals(parameter.getValue())){
                Integer flag = Integer.parseInt(String.valueOf(parameter.getValue()));
                List<Long> ids = baseMapper.queryIdsByFlag(flag);
                if(!ids.isEmpty()){
                    param.getParams().put("id",new Parameter(QueryParam.IN, ids));
                } else {
                    param.getParams().put("1",new Parameter(QueryParam.EQ, 0));
                }
            }
            parameterMap.remove("overdueFlag");
        }
        // 是否全部返还
        if(parameterMap.containsKey("returnAllFlag")){
            Parameter parameter = parameterMap.get("returnAllFlag");
            if(!"1,2".equals(parameter.getValue()) && !"2,1".equals(parameter.getValue())){
                LambdaQueryWrapper<PayForegiftEntity> lambda = Wrappers.lambdaQuery();
                Integer status = Integer.parseInt(String.valueOf(parameter.getValue()));
                lambda.last(status == 1,"and return_mny >= apply_mny");
                lambda.last(status == 2,"and return_mny < apply_mny");
                List<PayForegiftEntity> dataList = baseMapper.selectList(lambda);
                List<Long> ids = dataList.stream().map(PayForegiftEntity::getId).collect(Collectors.toList());
                if(!ids.isEmpty()){
                    param.getParams().put("id",new Parameter(QueryParam.IN, ids));
                } else {
                    param.getParams().put("1",new Parameter(QueryParam.EQ, 0));
                }
            }
            parameterMap.remove("returnAllFlag");
        }
    }

    /**
     * 校验同合同、同组织下只能有一张未生效单据
     * @param entity
     */
    private void validateBeforeSave(PayForegiftEntity entity) {
        LambdaQueryWrapper<PayForegiftEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(PayForegiftEntity::getContractId, entity.getContractId());
        lambda.eq(PayForegiftEntity::getOrgId, entity.getOrgId());
        lambda.eq(PayForegiftEntity::getTenantId, InvocationInfoProxy.getTenantid());
        lambda.ne(entity.getId() != null && entity.getId() > 0, PayForegiftEntity::getId, entity.getId());
        lambda.notIn(PayForegiftEntity::getBillState, Arrays.asList(
                BillStateEnum.COMMITED_STATE.getBillStateCode(),
                BillStateEnum.PASSED_STATE.getBillStateCode()));
        List<PayForegiftEntity> entityList = super.list(lambda);
        if(entityList != null && entityList.size() > 0) {
            throw new BusinessException("该合同在同一组织下存在未生效单据,不允许新增!");
        }
    }

    /**
     * 自动生成编码
     * @param entity
     */
    private void autoSetBillCode(PayForegiftEntity entity) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        if(StringUtils.isEmpty(entity.getBillCode())){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(PAY_FOREGIFT_BILL_CODE, tenantId);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        //修改  校验合同编号是否重复
        LambdaQueryWrapper<PayForegiftEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(PayForegiftEntity::getBillCode, entity.getBillCode());
        lambda.eq(PayForegiftEntity::getTenantId, tenantId);
        lambda.ne(entity.getId() != null && entity.getId() > 0, PayForegiftEntity::getId, entity.getId());
        List<PayForegiftEntity> entityList = super.list(lambda);
        if(entityList != null && entityList.size() > 0) {
            throw new BusinessException("存在相同编码，不允许保存!");
        }
    }

}
