package com.ejianc.business.finance.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.cost.api.ICostDetailApi;
import com.ejianc.business.cost.vo.CostDetailVO;
import com.ejianc.business.finance.bean.*;
import com.ejianc.business.finance.mapper.PaySporadicMapper;
import com.ejianc.business.finance.service.IPayInvoiceService;
import com.ejianc.business.finance.service.IPaySporadicService;
import com.ejianc.business.finance.service.IPaySporadicStockService;
import com.ejianc.business.finance.util.BillTypeCodeEnum;
import com.ejianc.business.finance.util.ValidateUtil;
import com.ejianc.business.finance.vo.*;
import com.ejianc.business.material.api.IInstoreApi;
import com.ejianc.business.material.vo.InstoreVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 *  服务实现类
 * </p>
 *
 * @author yqls
 * @since 2020-06-04
 */
@Service
public class PaySporadicServiceImpl extends BaseServiceImpl<PaySporadicMapper, PaySporadicEntity> implements IPaySporadicService {

    private static final String PAY_SPORADIC_BILL_CODE = "PAY_SPORADIC";

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IInstoreApi instoreApi;

    @Autowired
    private IPayInvoiceService invoiceService;

    @Autowired
    private IPaySporadicStockService stockService;

    @Autowired
    private IOrgApi orgApi;
    @Autowired
    private ICostDetailApi costDetailApi;
    @Autowired
    private SessionManager sessionManager;

    @Override
    public PaySporadicVO insertOrUpdate(PaySporadicVO vo) {
        PaySporadicEntity entity = BeanMapper.map(vo, PaySporadicEntity.class);
        if(!vo.getManageFlag()){
            // 保存前校验
            this.validateBeforeSave(entity);
        }
        // 自动生成编码
        this.autoSetBillCode(entity);
        //判断是否有入库单
        List<PaySporadicStockVO> stockVO = vo.getStockVOList();
        if (ListUtil.isNotEmpty(stockVO)){
            entity.setInputFlag("1");
        }else {
            entity.setInputFlag("0");
        }
        //设置默认值
        entity.setProportionFlag("0");
        entity.setRelationFlag("0");
        // 设置默认是否需要发票
        if (StringUtils.isEmpty(entity.getInvoiceFlag())) {
            entity.setInvoiceFlag("3");
        }
        // 保存主表
        super.saveOrUpdate(entity);
        //
        if (2 == vo.getPayStatus()){
            PaySporadicEntity paySporadicEntity = super.selectById(entity.getId());
//            costPush(paySporadicEntity);
        }
        Long id = entity.getId();
        // 更新发票
        List<PayInvoiceVO> invoiceVOList = this.updateInvoiceVOS(vo, id);
        // 更新入库
        List<PaySporadicStockVO> stockVOList = this.updateStockVOS(vo, id);
        // 返回VO
        PaySporadicVO backVO = BeanMapper.map(super.getById(id), PaySporadicVO.class);
        backVO.setInvoiceVOList(invoiceVOList);// 发票列表
        backVO.setStockVOList(stockVOList);// 入库列表
        return backVO;
    }

    /**
     * 更新发票
     * @param vo
     * @param id
     * @return
     */
    private List<PayInvoiceVO> updateInvoiceVOS(PaySporadicVO vo, Long id) {
        // 新增
        List<PayInvoiceVO> voList = vo.getInvoiceVOList();
        if(voList != null && !voList.isEmpty()){
            for(PayInvoiceVO invoiceVO : voList){
                // 保存时校验version是否一致
                if(!ValidateUtil.validateUpstreamVersion(String.valueOf(invoiceVO.getInvoiceId()),
                        BillTypeCodeEnum.税务收票.getBillTypeCode(), invoiceVO.getInvoiceVersion())){
                    throw new BusinessException("发票已被更新，请刷新后重做！");
                }
                invoiceVO.setPayapplyId(id);
            }
            List<PayInvoiceEntity> entityList = BeanMapper.mapList(voList, PayInvoiceEntity.class);
            invoiceService.saveOrUpdateBatch(entityList, entityList.size(), false);
            voList = BeanMapper.mapList(entityList, PayInvoiceVO.class);
        }
        // 删除
        List<Long> ids = voList.stream().map(PayInvoiceVO::getId).collect(Collectors.toList());
        QueryWrapper wrapper = new QueryWrapper<PayInvoiceEntity>();
        wrapper.eq("payapply_id", id);
        wrapper.notIn(!ids.isEmpty(),"id", ids);
        invoiceService.remove(wrapper, false);
        return voList;
    }

    /**
     * 更新入库
     * @param vo
     * @param id
     * @return
     */
    private List<PaySporadicStockVO> updateStockVOS(PaySporadicVO vo, Long id) {
        // 新增
        List<PaySporadicStockVO> voList = vo.getStockVOList();
        List<InstoreVO> instoreVOList = new ArrayList<>();
        List<Long> stockIds = new ArrayList<>();
        if(voList != null && !voList.isEmpty()){
            for(PaySporadicStockVO stockVO : voList){
                // 保存时校验version是否一致
                if(!ValidateUtil.validateUpstreamVersion(String.valueOf(stockVO.getStockId()),
                        BillTypeCodeEnum.物资入库.getBillTypeCode(), stockVO.getStockVersion())){
                    throw new BusinessException("入库单已被更新，请刷新后重做！");
                }
                stockVO.setPayapplyId(id);
                InstoreVO instoreVO = new InstoreVO();
                instoreVO.setId(stockVO.getStockId());
                instoreVO.setPaymentFlag(1);
                stockIds.add(stockVO.getStockId());
                instoreVOList.add(instoreVO);
            }
            List<PaySporadicStockEntity> entityList = BeanMapper.mapList(voList, PaySporadicStockEntity.class);
            stockService.saveOrUpdateBatch(entityList, entityList.size(), false);
            voList = BeanMapper.mapList(entityList, PaySporadicStockVO.class);
        }
        // 删除
        List<Long> voIds = voList.stream().map(PaySporadicStockVO::getId).collect(Collectors.toList());
        List<PaySporadicStockEntity> dataList = stockService.list(new QueryWrapper<PaySporadicStockEntity>().eq("payapply_id", id));
        List<Long> dataIds = dataList.stream().map(PaySporadicStockEntity::getId).collect(Collectors.toList());
        dataIds.removeAll(voIds);
        List<Long> delIds = dataIds;
        if(!delIds.isEmpty()){
            stockService.remove(new QueryWrapper<PaySporadicStockEntity>().in("id", delIds), false);
        }

        // 回写入库单是否付款
        for(PaySporadicStockEntity data : dataList){
            if(delIds.contains(data.getId()) && !stockIds.contains(data.getStockId())){
                InstoreVO instoreVO = new InstoreVO();
                instoreVO.setId(data.getStockId());
                instoreVO.setPaymentFlag(0);
                instoreVOList.add(instoreVO);
            }
        }
        if(!instoreVOList.isEmpty()){
            instoreApi.updatePayment(instoreVOList);
        }
        return voList;
    }

    @Override
    public PaySporadicVO queryDetail(Long id) {
        // 查询主表
        PaySporadicEntity entity = baseMapper.selectById(id);
        PaySporadicVO vo = BeanMapper.map(entity, PaySporadicVO.class);
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("payapplyId", new Parameter(QueryParam.EQ, vo.getId()));
        queryParam.getOrderMap().put("createTime", "desc");
        // 查询关联发票
        List<PayInvoiceEntity> invoiceEntityList = invoiceService.queryList(queryParam, false);
        vo.setInvoiceVOList(BeanMapper.mapList(invoiceEntityList, PayInvoiceVO.class));
        // 查询关联发票
        List<PaySporadicStockEntity> stockEntityList = stockService.queryList(queryParam, false);
        vo.setStockVOList(BeanMapper.mapList(stockEntityList, PaySporadicStockVO.class));
        return vo;
    }

    @Override
    public List<PaySporadicVO> queryExportList(QueryParam param) {
        param.setPageIndex(0);
        param.setPageSize(-1);
        List<PaySporadicVO> resVos = (List<PaySporadicVO>) queryPageJson(param, false).get("records");
        if(!resVos.isEmpty()){
            for(int i = 0 ; i< resVos.size(); i++){
                PaySporadicVO vo = resVos.get(i);
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            };
        }
        return resVos;
    }

    @Override
    public String delete(List<Long> ids) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("id", new Parameter(QueryParam.IN, ids));
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        List<PaySporadicEntity> entityList = super.queryList(queryParam, false);
        if(CollectionUtils.isNotEmpty(entityList)) {
            super.removeByIds(ids, false);
            QueryWrapper wrapper = new QueryWrapper<PaySporadicStockEntity>().in("payapply_id", ids);
            // 删除发票
            invoiceService.remove(wrapper, false);
            // 删除入库
            List<PaySporadicStockEntity> stockList = stockService.list(wrapper);
            stockService.remove(wrapper, false);
            // 回写入库单是否付款
            List<InstoreVO> instoreVOList = new ArrayList<>();
            for(PaySporadicStockEntity data : stockList){
                InstoreVO instoreVO = new InstoreVO();
                instoreVO.setId(data.getStockId());
                instoreVO.setPaymentFlag(0);
                instoreVOList.add(instoreVO);
            }
            if(!instoreVOList.isEmpty()){
                instoreApi.updatePayment(instoreVOList);
            }
        }
        return "删除成功！";
    }

    @Override
    public JSONObject queryPageJson(QueryParam param, boolean isEs) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("orgName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("applyUserName");
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 组织本下
        param.getParams().put("org_id",new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

        Parameter parameterFlag = param.getParams().get("relationInvoiceFlag");
        param.getParams().remove("relationInvoiceFlag");
        String flag = parameterFlag == null ? null : parameterFlag.getValue().toString();

        Page<PaySporadicVO> pages = new Page<>(param.getPageIndex(), param.getPageSize());
        QueryWrapper wrapper = changeToQueryWrapper(param);
        List<PaySporadicVO> list = baseMapper.queryPageList(pages, wrapper, flag);

        com.alibaba.fastjson.JSONObject page = new com.alibaba.fastjson.JSONObject();
        page.put("records", list);
        page.put("total", pages.getTotal());
        page.put("current", pages.getCurrent());
        page.put("size", pages.getSize());
        page.put("pages", pages.getPages());
        return page;
    }

    @Override
    public Map<String, Object> countAmt(Long tenantId, List<Long> projectIds, List<Long> orgIds) {
        QueryWrapper<PaySporadicEntity> queryWrapper = new QueryWrapper<>();
        if (CollectionUtils.isNotEmpty(orgIds)) {
            queryWrapper.in("org_id", orgIds);
        }

        queryWrapper.select("ifnull(sum(pay_mny),0) as amt");

        queryWrapper.eq("tenant_id", tenantId);
        queryWrapper.eq("dr", BaseVO.DR_UNDELETE);
//        付款类型：1-组织付款 2-项目付款
//        queryWrapper.eq("pay_type", 2);
//        支付状态：1-未支付 2-已支付
        queryWrapper.eq("pay_status", 2);

//        if(CollectionUtils.isNotEmpty(projectIds)) {
//            queryWrapper.in("project_id", projectIds);
//        }
//        已生效的单据
        queryWrapper.in("bill_state",
                Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()}));

        return super.getMap(queryWrapper);
    }

    @Override
    public CommonResponse<PaySporadicVO> pushCost(PaySporadicVO paySporadicVO) {
        PaySporadicEntity paySporadicEntity = baseMapper.selectById(paySporadicVO.getId());
        paySporadicEntity.setSubjectId(paySporadicVO.getSubjectId());
        paySporadicEntity.setSubjectName(paySporadicVO.getSubjectName());
        paySporadicEntity.setOrgSubjectId(paySporadicVO.getOrgSubjectId());
        paySporadicEntity.setAccountingId(paySporadicVO.getAccountingId());
        paySporadicEntity.setAccountingName(paySporadicVO.getAccountingName());
        paySporadicEntity.setOrgAccountingId(paySporadicVO.getOrgAccountingId());
        super.saveOrUpdate(paySporadicEntity,false);
        //推送数据
        if (paySporadicEntity.getBillState().equals(BillStateEnum.COMMITED_STATE.getBillStateCode()) || paySporadicEntity.getBillState().equals(BillStateEnum.PASSED_STATE.getBillStateCode())) {
            costPush(paySporadicEntity);
        }
        return CommonResponse.success(BeanMapper.map(paySporadicEntity, PaySporadicVO.class));
    }
    @Override
    public void costPush(PaySporadicEntity paySporadicEntity) {
        if (!"1".equals(paySporadicEntity.getDependOnProject())){
            return;
        }
        //1.判断是否关联
        Long subjectId = paySporadicEntity.getSubjectId();
        Long accountingId = paySporadicEntity.getAccountingId();
        boolean newRelationFlag = true;
        if(null == subjectId && null == accountingId){
            newRelationFlag = false;
        }
        //更新是否关联
        LambdaUpdateWrapper<PaySporadicEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.in(PaySporadicEntity::getId, paySporadicEntity.getId());
        updateWrapper.set(PaySporadicEntity::getRelationFlag, newRelationFlag ? "1":"0");//(1:是，0：否)
        super.update(updateWrapper);

        //判断之前的单据是否关联
        String oldRelationFlag = paySporadicEntity.getRelationFlag();
        //之前已关联
        if (oldRelationFlag.equals("1")){
            if (newRelationFlag){
                saveCost(paySporadicEntity);
            }
            if (!newRelationFlag){
                //删除成本中心之前的数据
                costDetailApi.deleteSubject(paySporadicEntity.getId());
            }
        }
        //之前未关联
        if (oldRelationFlag.equals("0")){
            if (newRelationFlag){
                //税率
                saveCost(paySporadicEntity);
            }
        }


    }
    private void saveCost(PaySporadicEntity paySporadicEntity) {
        //明细
        List<CostDetailVO> costDetailVOList = new ArrayList<>();
        if (paySporadicEntity.getSubjectId() != null && paySporadicEntity.getAccountingId() != null) {
            CostDetailVO costDetailVO = new CostDetailVO();
            costDetailVO.setSubjectId(paySporadicEntity.getSubjectId());
            costDetailVO.setSourceId(paySporadicEntity.getId());
            costDetailVO.setSourceDetailId(paySporadicEntity.getId());
            costDetailVO.setHappenTaxMny(paySporadicEntity.getPayMny());
            costDetailVO.setHappenMny(paySporadicEntity.getPayMny());
            costDetailVO.setHappenDate(paySporadicEntity.getApplyTime());
            costDetailVO.setCreateUserName(sessionManager.getUserContext().getUserName());
            costDetailVO.setSourceType(PAY_SPORADIC_BILL_CODE);
            costDetailVO.setSourceTabType(PAY_SPORADIC_BILL_CODE);
            costDetailVO.setProjectId(paySporadicEntity.getProjectId());
            // 核算对象
            costDetailVO.setAccountingId(paySporadicEntity.getAccountingId());
            costDetailVO.setOrgSubjectId(paySporadicEntity.getOrgSubjectId());
            costDetailVO.setOrgAccountingId(paySporadicEntity.getOrgAccountingId());
            costDetailVOList.add(costDetailVO);
        }
        //成本中心
        if (ListUtil.isNotEmpty(costDetailVOList)){
            CommonResponse<String> stringCommonResponse = costDetailApi.saveSubject(costDetailVOList);
            if (stringCommonResponse.isSuccess()){
            }else {
                throw new BusinessException(stringCommonResponse.getMsg());
            }
        }
    }
    /**
     * 校验同合同、同组织下只能有一张未生效单据
     * @param entity
     */
    private void validateBeforeSave(PaySporadicEntity entity) {
        LambdaQueryWrapper<PaySporadicEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(PaySporadicEntity::getSupplierId, entity.getSupplierId());
        lambda.eq(PaySporadicEntity::getOrgId, entity.getOrgId());
        lambda.eq(PaySporadicEntity::getTenantId, InvocationInfoProxy.getTenantid());
        lambda.ne(entity.getId() != null && entity.getId() > 0, PaySporadicEntity::getId, entity.getId());
        lambda.notIn(PaySporadicEntity::getBillState, Arrays.asList(
                BillStateEnum.COMMITED_STATE.getBillStateCode(),
                BillStateEnum.PASSED_STATE.getBillStateCode()));
        List<PaySporadicEntity> entityList = super.list(lambda);
        if(entityList != null && entityList.size() > 0) {
            throw new BusinessException("该供应商在当前组织下存在未审批通过的申请单["
                    + entityList.get(0).getBillCode() + "]，无法发起申请!");
        }
    }

    /**
     * 自动生成编码
     * @param entity
     */
    private void autoSetBillCode(PaySporadicEntity entity) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        if(StringUtils.isEmpty(entity.getBillCode())){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(PAY_SPORADIC_BILL_CODE, tenantId);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        //修改  校验合同编号是否重复
        LambdaQueryWrapper<PaySporadicEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(PaySporadicEntity::getBillCode, entity.getBillCode());
        lambda.eq(PaySporadicEntity::getTenantId, tenantId);
        lambda.ne(entity.getId() != null && entity.getId() > 0, PaySporadicEntity::getId, entity.getId());
        List<PaySporadicEntity> entityList = super.list(lambda);
        if(entityList != null && entityList.size() > 0) {
            throw new BusinessException("存在相同编码，不允许保存!");
        }
    }
    //成本改造
    @Override
    public void pullCost(Long id) {
        ////更新关联状态为未关联
        LambdaUpdateWrapper<PaySporadicEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.in(PaySporadicEntity::getId, id);
        updateWrapper.set(PaySporadicEntity::getRelationFlag, "0");//(1:是，0：否)
        super.update(updateWrapper);
        //删除成本中心数据
        costDetailApi.deleteSubject(id);
    }

    @Override
    public TotalColumnVO getTotalColumnInfo(QueryParam param, boolean isEs) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("orgName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("applyUserName");
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 组织本下
        param.getParams().put("org_id",new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

        Parameter parameterFlag = param.getParams().get("relationInvoiceFlag");
        param.getParams().remove("relationInvoiceFlag");
        String flag = parameterFlag == null ? null : parameterFlag.getValue().toString();

        QueryWrapper wrapper = changeToQueryWrapper(param);
        List<PaySporadicVO> list = baseMapper.queryPageList(null, wrapper, flag);

        // 过滤为空的数据并求和
        BigDecimal sumApplyMny = list.stream().filter(entity -> entity.getApplyMny() != null)
                .map(PaySporadicVO::getApplyMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal sumPayMny = list.stream().filter(entity -> entity.getPayMny() != null)
                .map(PaySporadicVO::getPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);

        TotalColumnVO vo = new TotalColumnVO();
        vo.setSumApplyMny(sumApplyMny);
        vo.setSumPayMny(sumPayMny);

        return vo;

    }

    @Override
    public TotalColumnVO getApproveTotalColumnInfo(QueryParam param, boolean isEs) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("orgName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("applyUserName");
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 组织本下
        param.getParams().put("org_id",new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

        Parameter parameterFlag = param.getParams().get("relationInvoiceFlag");
        param.getParams().remove("relationInvoiceFlag");
        String flag = parameterFlag == null ? null : parameterFlag.getValue().toString();

        QueryWrapper wrapper = changeToQueryWrapper(param);
        List<PaySporadicVO> list = baseMapper.queryPageList(null, wrapper, flag);

        // 过滤为空的数据并求和
        BigDecimal sumApplyMny = list.stream().filter(entity -> entity.getApplyMny() != null)
                .map(PaySporadicVO::getApplyMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal sumPayMny = list.stream().filter(entity -> entity.getPayMny() != null)
                .map(PaySporadicVO::getPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);

        TotalColumnVO vo = new TotalColumnVO();
        vo.setSumApplyMny(sumApplyMny);
        vo.setSumPayMny(sumPayMny);

        return vo;
    }
}
