package com.ejianc.business.finance.controller.api;

import com.ejianc.business.equipment.vo.PurchaseContractVO;
import com.ejianc.business.equipment.vo.RentContractVO;
import com.ejianc.business.finance.service.IPayContractService;
import com.ejianc.business.finance.vo.BlocKanBanVo;
import com.ejianc.business.finance.vo.PayContractVO;
import com.ejianc.business.material.vo.MaterialContractVO;
import com.ejianc.business.sub.api.ISubBlocKanBanApi;
import com.ejianc.business.sub.vo.ContractVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.vo.SupplierVO;
import com.ejianc.foundation.support.api.ISupplierApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @ClassName: BlocKanBanApi
 * @Description:
 * @Author: 曹鹏辉
 * @Date: 2021/9/7 9:52
 */
@RestController
@RequestMapping("/api/blocKanBan/")
public class FinanceBlocKanBanApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private ISubBlocKanBanApi subBlocKanBanApi;

    @Autowired
    private IPayContractService payContractService;

    @Autowired
    private ISupplierApi supplierApi;

    /**
     * 集团看板分包供应商top5
     * @param orgId
     * @param mnyFlag 0-支付金额(默认)，1-合同总金额，2-欠付金额
     * @param yearFlag 0-今年(默认)，1-全年
     * @return
     */
    @GetMapping(value = "querySupplierPayMny")
    @ResponseBody
    public CommonResponse<List<BlocKanBanVo>> querySupplierPayMny(@RequestParam(value = "orgId", required = false) Long orgId
            , @RequestParam(value = "mnyFlag") Integer mnyFlag, @RequestParam(value = "yearFlag") Integer yearFlag) {

        if (orgId == null) {
            Long tenantid = InvocationInfoProxy.getTenantid();
            CommonResponse<OrgVO> orgByTenantId = orgApi.findOrgByTenantId(tenantid);
            orgId = orgByTenantId.getData().getId();
        }

        // 本下组织
        List<Long> orgIds = orgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList());

        // 分包合同总金额
        CommonResponse<Map<Long, ContractVO>> mapCommonResponse = subBlocKanBanApi.queryExternalSupplierAndMny(null, yearFlag);
        if (!mapCommonResponse.isSuccess()) {
            return CommonResponse.error("查询分包数据失败");
        }
        Map<Long, ContractVO> map = mapCommonResponse.getData();
        List<ContractVO> contractVOList = new ArrayList<>(map.values());
        List<Long> supplierIds = contractVOList.stream().map(ContractVO::getSupplierId).collect(Collectors.toList());

        // 查询外部供应商对应的支付金额
        List<PayContractVO> payContractVOList = payContractService.querySupplierPayMny(orgIds, supplierIds, yearFlag);
        Map<Long, PayContractVO> payContractVOMap = payContractVOList.stream().collect(Collectors.toMap(PayContractVO::getReceiveUnitId, Function.identity()));

        Set<Long> set = new HashSet<>(map.keySet());
        set.addAll(payContractVOMap.keySet());

        List<BlocKanBanVo> voList = new ArrayList<>();
        for (Long supplierId : set) {
            BlocKanBanVo vo = new BlocKanBanVo();
            vo.setReceiveUnitId(supplierId);
            if (map.containsKey(supplierId)) {
                vo.setReceiveUnitName(map.get(supplierId).getSupplierName());
                BigDecimal contractTaxMny = map.get(supplierId).getContractTaxMny();
                contractTaxMny = contractTaxMny == null ? BigDecimal.ZERO : contractTaxMny;
                vo.setTotalContractTaxMny(contractTaxMny);
            }else {
                vo.setTotalContractTaxMny(BigDecimal.ZERO);
            }

            if (payContractVOMap.containsKey(supplierId)) {
                vo.setReceiveUnitName(payContractVOMap.get(supplierId).getReceiveUnitName());
                BigDecimal payMny = payContractVOMap.get(supplierId).getPayMny();
                payMny = payMny == null ? BigDecimal.ZERO : payMny;
                vo.setTotalPayMny(payMny);
            }else {
                vo.setTotalPayMny(BigDecimal.ZERO);
            }

            vo.setOwedMny(ComputeUtil.safeSub(vo.getTotalContractTaxMny(), vo.getTotalPayMny()));
            if (vo.getOwedMny().compareTo(new BigDecimal("0E-8")) == 0) {
                vo.setOwedMny(BigDecimal.ZERO);
            }

            voList.add(vo);
        }

        if (mnyFlag == 0) {
            // 默认 支付金额 降序
            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalPayMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getTotalPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalPayMny(), sunMny), new BigDecimal("100")));
            }
        }else if (mnyFlag == 1) {
            // 合同总金额 降序
            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalContractTaxMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getTotalContractTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalContractTaxMny(), sunMny), new BigDecimal("100")));
            }
        }else {
            // 欠付金额 降序
            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getOwedMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getOwedMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getOwedMny(), sunMny), new BigDecimal("100")));
            }
        }

        return CommonResponse.success("查询分包供应商top5成功", voList);
    }


    /**
     * 项目看板分包供应商
     * @param projectId
     * @param mnyFlag 0-支付金额(默认)，1-合同总金额，2-欠付金额
     * @param yearFlag 0-今年(默认)，1-全年
     * @return
     */
    @GetMapping(value = "querySubSupplierByProject")
    @ResponseBody
    public CommonResponse<List<BlocKanBanVo>> querySubSupplierByProject(@RequestParam(value = "projectId", required = false) Long projectId
            , @RequestParam(value = "mnyFlag") Integer mnyFlag, @RequestParam(value = "yearFlag") Integer yearFlag) {

        if (projectId == null) {
            return CommonResponse.error("项目为空！");
        }
        
        // 查询分包合同金额
        List<ContractVO> contractList = payContractService.querySubContractMny(projectId, yearFlag);

        // 查询供应商
        List<Long> supplierIdList = contractList.stream().map(ContractVO::getSupplierId).collect(Collectors.toList());
        List<SupplierVO> supplierList = new ArrayList<>();
        if (ListUtil.isNotEmpty(supplierIdList)) {
            CommonResponse<List<SupplierVO>> response = supplierApi.querySupplierByIds(supplierIdList);
            if (!response.isSuccess()) {
                return CommonResponse.error("查询供应商信息失败！");
            }
            supplierList = response.getData();
        }
       
        // 过滤出来 外部供应商
        List<Long> ids = supplierList.stream().filter(vo -> vo.getInsideOrgId() == null).map(SupplierVO::getId).collect(Collectors.toList());

        Map<Long, ContractVO> contractMap = contractList.stream().filter(vo -> ids.contains(vo.getSupplierId())).collect(Collectors.toMap(ContractVO::getSupplierId, Function.identity()));

        // 分包合同外部供应商id集合
        List<Long> supplierIdsList = new ArrayList<>(contractMap.values()).stream().map(ContractVO::getSupplierId).collect(Collectors.toList());

        // 查询外部供应商对应的支付金额
        List<PayContractVO> payContractList = payContractService.querySupplierPayMnyByProject(projectId, supplierIdsList, yearFlag);
        Map<Long, PayContractVO> payContractMap = payContractList.stream().collect(Collectors.toMap(PayContractVO::getReceiveUnitId, Function.identity()));

        Set<Long> set = new HashSet<>(contractMap.keySet());
        set.addAll(payContractMap.keySet());

        List<BlocKanBanVo> voList = new ArrayList<>();
        for (Long supplierId : set) {
            BlocKanBanVo vo = new BlocKanBanVo();
            vo.setReceiveUnitId(supplierId);
            if (contractMap.containsKey(supplierId)) {
                vo.setReceiveUnitName(contractMap.get(supplierId).getSupplierName());
                BigDecimal contractTaxMny = contractMap.get(supplierId).getContractTaxMny();
                contractTaxMny = contractTaxMny == null ? BigDecimal.ZERO : contractTaxMny;
                vo.setTotalContractTaxMny(contractTaxMny);
            }else {
                vo.setTotalContractTaxMny(BigDecimal.ZERO);
            }

            if (payContractMap.containsKey(supplierId)) {
                vo.setReceiveUnitName(payContractMap.get(supplierId).getReceiveUnitName());
                BigDecimal payMny = payContractMap.get(supplierId).getPayMny();
                payMny = payMny == null ? BigDecimal.ZERO : payMny;
                vo.setTotalPayMny(payMny);
            }else {
                vo.setTotalPayMny(BigDecimal.ZERO);
            }

            vo.setOwedMny(ComputeUtil.safeSub(vo.getTotalContractTaxMny(), vo.getTotalPayMny()));
            if (vo.getOwedMny().compareTo(new BigDecimal("0E-8")) == 0) {
                vo.setOwedMny(BigDecimal.ZERO);
            }

            voList.add(vo);
        }

        if (mnyFlag == 0) {
            // 默认 支付金额 降序
//            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalPayMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getTotalPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalPayMny(), sunMny), new BigDecimal("100")));
            }
        }else if (mnyFlag == 1) {
            // 合同总金额 降序
//            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalContractTaxMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getTotalContractTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalContractTaxMny(), sunMny), new BigDecimal("100")));
            }
        }else {
            // 欠付金额 降序
//            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getOwedMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getOwedMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getOwedMny(), sunMny), new BigDecimal("100")));
            }
        }

        return CommonResponse.success("查询分包供应商top5成功", voList);
    }


    /**
     * 项目看板材料设备供应商
     * @param projectId
     * @param mnyFlag 0-支付金额(默认)，1-合同总金额，2-欠付金额
     * @param yearFlag 0-今年(默认)，1-全年
     * @return
     */
    @GetMapping(value = "queryMESupplierByProject")
    @ResponseBody
    public CommonResponse<List<BlocKanBanVo>> queryMESupplierByProject(@RequestParam(value = "projectId", required = false) Long projectId
            , @RequestParam(value = "mnyFlag") Integer mnyFlag, @RequestParam(value = "yearFlag") Integer yearFlag) {

        if (projectId == null) {
            return CommonResponse.error("项目为空！");
        }

        // 查询物资合同金额
        List<MaterialContractVO> materialContractList = payContractService.queryMaterialContractMny(projectId, yearFlag);

        // 查询设备采购合同金额
        List<PurchaseContractVO> purchaseContractList = payContractService.queryPurchaseContractMny(projectId, yearFlag);
        
        // 查询设备租赁合同金额
        List<RentContractVO> rentContractList = payContractService.queryRentContractMny(projectId, yearFlag);

        
        // 查询供应商
        List<Long> supplierIdList = materialContractList.stream().map(MaterialContractVO::getSupplierId).collect(Collectors.toList());
        if (ListUtil.isNotEmpty(materialContractList)) {
            supplierIdList.addAll(materialContractList.stream().map(MaterialContractVO::getSupplierId).collect(Collectors.toList()));
        }
        if (ListUtil.isNotEmpty(purchaseContractList)) {
            supplierIdList.addAll(purchaseContractList.stream().map(PurchaseContractVO::getSupplierId).collect(Collectors.toList()));
        }
        if (ListUtil.isNotEmpty(rentContractList)) {
            supplierIdList.addAll(rentContractList.stream().map(RentContractVO::getSupplierId).collect(Collectors.toList()));
        }
        
        List<SupplierVO> supplierList = new ArrayList<>();
        if (ListUtil.isNotEmpty(supplierIdList)) {
            CommonResponse<List<SupplierVO>> response = supplierApi.querySupplierByIds(supplierIdList);
            if (!response.isSuccess()) {
                return CommonResponse.error("查询供应商信息失败！");
            }
            supplierList = response.getData();
        }

        // 过滤出来 外部供应商
        List<Long> ids = supplierList.stream().filter(vo -> vo.getInsideOrgId() == null).map(SupplierVO::getId).collect(Collectors.toList());

        Map<Long, MaterialContractVO> materialMap = materialContractList.stream().filter(vo -> ids.contains(vo.getSupplierId())).collect(Collectors.toMap(MaterialContractVO::getSupplierId, Function.identity()));
        Map<Long, PurchaseContractVO> purchaseMap = purchaseContractList.stream().filter(vo -> ids.contains(vo.getSupplierId())).collect(Collectors.toMap(PurchaseContractVO::getSupplierId, Function.identity()));
        Map<Long, RentContractVO> rentMap = rentContractList.stream().filter(vo -> ids.contains(vo.getSupplierId())).collect(Collectors.toMap(RentContractVO::getSupplierId, Function.identity()));

        // 分包合同外部供应商id集合
        List<Long> supplierIdsList = new ArrayList<>();
        if (!materialMap.isEmpty()) {
            supplierIdsList.addAll(materialMap.keySet());
        }
        if (!purchaseMap.isEmpty()) {
            supplierIdsList.addAll(purchaseMap.keySet());
        }
        if (!rentMap.isEmpty()) {
            supplierIdsList.addAll(rentMap.keySet());
        }
        
        // 查询外部供应商对应的支付金额
        List<PayContractVO> payContractList = payContractService.querySupplierPayMnyByProject(projectId, supplierIdsList, yearFlag);
        Map<Long, PayContractVO> payContractMap = payContractList.stream().collect(Collectors.toMap(PayContractVO::getReceiveUnitId, Function.identity()));

        Set<Long> set = new HashSet<>(materialMap.keySet());
        set.addAll(purchaseMap.keySet());
        set.addAll(rentMap.keySet());
        set.addAll(payContractMap.keySet());

        List<BlocKanBanVo> voList = new ArrayList<>();
        for (Long supplierId : set) {
            BlocKanBanVo vo = new BlocKanBanVo();
            vo.setReceiveUnitId(supplierId);
            if (materialMap.containsKey(supplierId)) {
                vo.setReceiveUnitName(materialMap.get(supplierId).getSupplierName());
                BigDecimal contractTaxMny = materialMap.get(supplierId).getAmountWithTax();
                contractTaxMny = contractTaxMny == null ? BigDecimal.ZERO : contractTaxMny;
                vo.setTotalContractTaxMny(contractTaxMny);
            }else if (purchaseMap.containsKey(supplierId)) {
                vo.setReceiveUnitName(purchaseMap.get(supplierId).getSupplierName());
                BigDecimal contractTaxMny = purchaseMap.get(supplierId).getContractTaxMny();
                contractTaxMny = contractTaxMny == null ? BigDecimal.ZERO : contractTaxMny;
                vo.setTotalContractTaxMny(contractTaxMny);
            }if (rentMap.containsKey(supplierId)) {
                vo.setReceiveUnitName(rentMap.get(supplierId).getSupplierName());
                BigDecimal contractTaxMny = rentMap.get(supplierId).getContractTaxMny();
                contractTaxMny = contractTaxMny == null ? BigDecimal.ZERO : contractTaxMny;
                vo.setTotalContractTaxMny(contractTaxMny);
            }else {
                vo.setTotalContractTaxMny(BigDecimal.ZERO);
            }

            if (payContractMap.containsKey(supplierId)) {
                vo.setReceiveUnitName(payContractMap.get(supplierId).getReceiveUnitName());
                BigDecimal payMny = payContractMap.get(supplierId).getPayMny();
                payMny = payMny == null ? BigDecimal.ZERO : payMny;
                vo.setTotalPayMny(payMny);
            }else {
                vo.setTotalPayMny(BigDecimal.ZERO);
            }

            vo.setOwedMny(ComputeUtil.safeSub(vo.getTotalContractTaxMny(), vo.getTotalPayMny()));
//            if (vo.getOwedMny().compareTo(new BigDecimal("0E-8")) == 0) {
//                vo.setOwedMny(BigDecimal.ZERO);
//            }
            if (vo.getOwedMny().equals(new BigDecimal("0E-8")) || vo.getOwedMny().equals(new BigDecimal("0E-12"))) {
                vo.setOwedMny(BigDecimal.ZERO);
            }
            
            
            voList.add(vo);
        }

        if (mnyFlag == 0) {
            // 默认 支付金额 降序
//            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalPayMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getTotalPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalPayMny(), sunMny), new BigDecimal("100")));
            }
        }else if (mnyFlag == 1) {
            // 合同总金额 降序
//            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalContractTaxMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getTotalContractTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalContractTaxMny(), sunMny), new BigDecimal("100")));
            }
        }else {
            // 欠付金额 降序
//            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getOwedMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getOwedMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getOwedMny(), sunMny), new BigDecimal("100")));
            }
        }

        return CommonResponse.success("查询材料设备供应商成功", voList);
    }
    
    
    

}
