package com.ejianc.business.finance.controller.api;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.finance.bean.*;
import com.ejianc.business.finance.enums.ConstantTypeEnum;
import com.ejianc.business.finance.service.*;
import com.ejianc.business.finance.util.CountCallable;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.vo.ExpenditureVO;
import com.ejianc.business.finance.vo.IncomeVO;
import com.ejianc.business.finance.vo.ProjectLedgerVO;
import com.ejianc.business.tax.api.IInvoiceApi;
import com.ejianc.foundation.support.api.IDefdocApi;
import com.ejianc.foundation.support.vo.DefdocDetailVO;
import com.ejianc.foundation.tenant.api.ITenantApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.Utils;
import com.ejianc.framework.skeleton.template.IBaseService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.RequestContextHolder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

/**
 * @author yqls
 * @Description: 项目看板接口
 * @date 2021/03/31 17:46
 */
@RestController
@RequestMapping(value = "/api/project/")
public class ProjectBoardApi {

    private static Logger logger = LoggerFactory.getLogger(ProjectBoardApi.class);

    private static Calendar calendar = Calendar.getInstance();

    @Value("${common.env.base-host}")
    private String BASE_HOST;

    @Autowired
    private ITenantApi tenantApi;

    @Autowired
    private IDefdocApi defdocApi;

    @Autowired
    private IInvoiceApi invoiceApi;

    @Autowired
    private IPayContractService contractService;

    @Autowired
    private IPaySporadicService sporadicService;

    @Autowired
    private IPayReimburseService reimburseService;

    @Autowired
    private IPayForegiftService foregiftService;

    @Autowired
    private IPayForegiftRecordService foregiftRecordService;

    @Autowired
    private ILoadApplyService loadApplyService;

    @Autowired
    private ILoadReimburseService loadReimburseService;

    @Autowired
    private ILoadBackService loadBackService;

    @Autowired
    private IBondDownService bondDownService;

    @Autowired
    private IBondUpService bondUpService;

    @Autowired
    private IReceiveService receiveService;

    /**
     * 根据项目id，查询项目收付款以及合同金额
     * @param projectId
     * @return
     */
    @GetMapping("queryPayAndReceiveByProjectId")
    CommonResponse<JSONObject> queryPayAndReceiveByProjectId(@RequestParam(value = "projectId") Long projectId, HttpServletRequest request){
        long startTime = System.currentTimeMillis();    //获取开始时间
        String authority = request.getHeader("authority");

        ExecutorService threadPool = Executors.newFixedThreadPool(2);
        /** 收款登记 */
        QueryParam queryParam = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        Future<JSONObject> future1 = this.excute(threadPool, queryParam, receiveService);
        /** 收入合同 */
        Future<JSONObject> future2 = this.excute(threadPool, authority, "ejc-income-web/contract/pageList", queryParam);
        BigDecimal payMoney = this.getPayMny(projectId);// 付款金额
        BigDecimal receiveMoney = null;// 收款登记金额
        BigDecimal contractMoney = null;// 收入合同金额
        try {
            receiveMoney = getSumMny(future1.get(), "receiveMny");
            contractMoney = getSumMny(future2.get(), "contractTaxMny");
        } catch (Exception e) {
            logger.error("查询项目收付款以及合同金额数据异常", e);
        } finally {
            threadPool.shutdown();
        }

        JSONObject resp = new JSONObject();

        resp.put("receiveMoney", this.roundTwoPre(receiveMoney));
        resp.put("payMoney", this.roundTwoPre(payMoney));
        resp.put("contractMoney", this.roundTwoPre(contractMoney));

        long endTime = System.currentTimeMillis();    //获取结束时间
        logger.info("queryPayAndReceiveByProjectId ---- 运行时间：" + (endTime - startTime) + "ms");//输出程序运行时间

        return CommonResponse.success(resp);
    }

    /**
     * 根据项目id，查询项目资金数据
     * @param projectId
     * @return
     */
    @GetMapping("queryReceiveInfoByProjectId")
    CommonResponse<JSONObject> queryReceiveInfoByProjectId(@RequestParam(value = "projectId") Long projectId, HttpServletRequest request){
        long startTime = System.currentTimeMillis();    //获取开始时间
        String authority = request.getHeader("authority");

        // 1.最新收款
        JSONObject latestCollection = this.getLatestCollection(projectId);

        // 2.收支
        JSONObject paymentBalance = this.getPaymentBalance(projectId, authority);

        // 3.月度收支
        JSONObject monthlyIncomeAndExpenditure = this.getMonthlyIncomeAndExpenditure(projectId);

        // 4.押金、借款、保证金
        JSONObject depositLoanMargin = this.getDepositLoanMargin(projectId);

        // 5.欠开发票top5
        JSONObject underInvoice = this.getUnderInvoice(projectId);

        JSONObject resp = new JSONObject();
        resp.put("latestCollection", latestCollection);
        resp.put("paymentBalance", paymentBalance);
        resp.put("monthlyIncomeAndExpenditure", monthlyIncomeAndExpenditure);
        resp.put("depositLoanMargin", depositLoanMargin);
        resp.put("underInvoice", underInvoice);

        long endTime = System.currentTimeMillis();    //获取结束时间
        logger.info("queryReceiveInfoByProjectId ---- 运行时间：" + (endTime - startTime) + "ms");//输出程序运行时间

        return CommonResponse.success(resp);
    }

    /**
     * 根据项目id，查询项目成本数据
     * @param projectId
     * @return
     */
    @GetMapping(value = "queryCostByProjectId")
    public CommonResponse<JSONObject> queryCostByProjectId(@RequestParam(value = "projectId") Long projectId, HttpServletRequest request) {
        long startTime = System.currentTimeMillis();    //获取开始时间
        String authority = request.getHeader("authority");

        ExecutorService threadPool = Executors.newFixedThreadPool(10);
        /** 项目总预算 */
        QueryParam queryParam = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        Future<JSONObject> future1 = this.excute(threadPool, authority, "ejc-cost-web/budgetProject/queryList", queryParam);
        /** 对甲结算 */
        Future<JSONObject> future2 = this.excute(threadPool, authority, "ejc-income-web/quote/pageList", queryParam);
        /** 分包过程结算 */
        Future<JSONObject> future3 = this.excute(threadPool, authority, "ejc-sub-web/settle/queryList", queryParam);
        /** 分包完工结算 */
        Future<JSONObject> future4 = this.excute(threadPool, authority, "ejc-sub-web/finish/queryList", queryParam);
        /** 物资采购结算 */
        Future<JSONObject> future5 = this.excute(threadPool, authority, "ejc-material-web/purchaseSettlement/queryList", queryParam);
        /** 设备采购结算 */
        Future<JSONObject> future6 = this.excute(threadPool, authority, "ejc-equipment-web/purchaseSettlement/pageList", queryParam);
        /** 设备租赁结算 */
        Future<JSONObject> future7 = this.excute(threadPool, authority, "ejc-equipment-web/rentSettlement/pageList", queryParam);
        /** 周转材结算 */
        Future<JSONObject> future12 = this.excute(threadPool, authority, "ejc-rmat-web/rentSettlement/pageList", queryParam);
        /** 其他合同结算 */
        Future<JSONObject> future13 = this.excute(threadPool, authority, "ejc-other-web/otherSettle/pageList", queryParam);
        /** 付款申请 */
        QueryParam queryParam2 = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        queryParam2.getParams().put("dependOnProject",new Parameter(QueryParam.EQ,"1"));//属于项目
        queryParam2.getParams().put("payStatus",new Parameter(QueryParam.EQ,"2"));//支付状态：2-已支付
//        Future<JSONObject> future8 = this.excute(threadPool, queryParam, contractService);
        /** 零星材料付款申请 */
        Future<JSONObject> future9 = this.excute(threadPool, queryParam2, sporadicService);
        /** 报销 */
        Future<JSONObject> future10 = this.excute(threadPool, queryParam2, reimburseService);
        /** 借款报销 */
        QueryParam queryParam3 = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        queryParam3.getParams().put("dependOnProject",new Parameter(QueryParam.EQ,"1"));//属于项目
        Future<JSONObject> future11 = this.excute(threadPool, queryParam3, loadReimburseService);

        BigDecimal generalBudget = null;// 项目总预算金额
        BigDecimal settlementToParta = null;// 对甲结算金额
        BigDecimal subcontractingCost = null;// 分包成本
        BigDecimal materialCost = null;// 材料成本
        BigDecimal mechanicsCost = null;// 机械成本
        BigDecimal indirectExpense = null;// 间接费
        BigDecimal otherCost = null;// 其他支出成本
        Map<String, Map<String, BigDecimal>> monthMap = new LinkedHashMap<>();// 月度成本
        try {
            generalBudget = this.getSumMny(future1.get(), "budgetMny");// 项目总预算金额
            settlementToParta = this.getSumMny(future2.get(), "quoteTaxMny");// 对甲结算金额
            BigDecimal subSettleMny = this.getSumMny(future3.get(), "settleTaxMny");// 分包过程结算金额
            BigDecimal subFinishMny = this.getSumMny(future4.get(), "shouldPayTaxMny");// 分包完工结算差额
            BigDecimal materialMny = this.getSumMny(future5.get(), "currentSettlementAmountTax");// 物资采购结算金额
            BigDecimal equipmentMny = this.getSumMny(future6.get(), "settlementTaxMny");// 设备采购结算金额
            BigDecimal rentMny = this.getSumMny(future7.get(), "settlementTaxMny");// 设备租赁结算金额
            BigDecimal sporadicMny = this.getSumMny(future9.get(), "payMny");// 零星材料付款申请金额
            BigDecimal reimburseMny = this.getSumMny(future10.get(), "payMny");// 报销金额
            BigDecimal loadReimburseMny = this.getSumMny(future11.get(), "reimburseMny");// 借款报销金额
            BigDecimal rmatMny = this.getSumMny(future12.get(), "settlementTaxMny");// 周转材结算金额
            BigDecimal otherMny = this.getSumMny(future13.get(), "settleTaxMny");// 其他支出合同结算金额

            subcontractingCost = MathUtil.safeAdd(subSettleMny, subFinishMny);// 分包成本
            materialCost = MathUtil.safeAdd(MathUtil.safeAdd(materialMny, sporadicMny), rmatMny);// 材料成本
            mechanicsCost = MathUtil.safeAdd(equipmentMny, rentMny);// 机械成本
            indirectExpense = MathUtil.safeAdd(reimburseMny, loadReimburseMny);// 间接费
            otherCost = otherMny; // 其他支出成本

            // 月度成本
            this.calculateCallableMny(monthMap, future3.get(), "settleDate", "shouldPayTaxMny");
            this.calculateCallableMny(monthMap, future4.get(), "settleDate", "settleTaxMny");
            this.calculateCallableMny(monthMap, future5.get(), "settlementDate", "currentSettlementAmountTax");
            this.calculateCallableMny(monthMap, future6.get(), "settlementDate", "settlementTaxMny");
            this.calculateCallableMny(monthMap, future7.get(), "settlementDate", "settlementTaxMny");
            this.calculateCallableMny(monthMap, future9.get(), "confirmTime", "payMny");
            this.calculateCallableMny(monthMap, future10.get(), "confirmTime", "payMny");
            this.calculateCallableMny(monthMap, future11.get(), "applyTime", "reimburseMny");
            this.calculateCallableMny(monthMap, future12.get(), "settlementDate", "settlementTaxMny");
            this.calculateCallableMny(monthMap, future13.get(), "signDate", "settleTaxMny");
        } catch (Exception e) {
            logger.error("查询异常", e);
        } finally {
            threadPool.shutdown();
        }

        //总成本
        BigDecimal totalCost = MathUtil.safeAdd(MathUtil.safeAdd(MathUtil.safeAdd(MathUtil.safeAdd(
                subcontractingCost, materialCost), mechanicsCost), indirectExpense), otherCost);

        JSONObject data = new JSONObject();
        JSONObject target = new JSONObject();
        target.put("generalBudget", MathUtil.roundTwoPre(generalBudget));//项目总预算
        target.put("settlementToParta", MathUtil.roundTwoPre(settlementToParta));//对甲结算
        target.put("totalCost", MathUtil.roundTwoPre(totalCost));//总成本

        JSONObject firstProcess = new JSONObject();	//第一个进度图
        firstProcess.put("process", MathUtil.roundTwoPre(MathUtil.safeDiv(totalCost, settlementToParta)));//进度条（结果保留两位小数）
        firstProcess.put("settlementBenefit", MathUtil.roundTwoPre(MathUtil.safeSub(settlementToParta, totalCost)));//结算效益
        target.put("firstProcess", firstProcess);

        JSONObject secondProcess = new JSONObject();//第二个进度图
        secondProcess.put("process", MathUtil.roundTwoPre(MathUtil.safeDiv(totalCost, generalBudget)));//进度条（结果保留两位小数）
        secondProcess.put("managementBenefit", MathUtil.roundTwoPre(MathUtil.safeSub(generalBudget, totalCost)));//经营效益
        target.put("secondProcess", secondProcess);

        JSONObject costAnalysis = new JSONObject();//成本分析
        costAnalysis.put("subcontractingCost", MathUtil.roundTwoPre(subcontractingCost));//分包成本
        costAnalysis.put("materialCost", MathUtil.roundTwoPre(materialCost));//材料成本
        costAnalysis.put("mechanicsCost", MathUtil.roundTwoPre(mechanicsCost));//机械成本
        costAnalysis.put("indirectExpense", MathUtil.roundTwoPre(indirectExpense));//间接费
        costAnalysis.put("otherCost", MathUtil.roundTwoPre(otherCost));// 其他支出成本
        target.put("costAnalysis", costAnalysis);

        JSONObject monthlyCost = new JSONObject();//月度成本
        List<String> labelList = new LinkedList<>();
        for(String key : monthMap.keySet()){
            labelList.add(key);
        }
        Collections.sort(labelList, new Comparator<String>() {
            @Override
            public int compare(String u1, String u2) {
                int diff = Integer.valueOf(u1.substring(0, u1.length() -1)) - Integer.valueOf(u2.substring(0, u2.length() -1));
                if (diff > 0) {
                    return 1;
                }else if (diff < 0) {
                    return -1;
                }
                return 0; //相等为0
            }
        });
        List<BigDecimal> incomeList = new LinkedList<>();
        for(String key : labelList){
            Map<String, BigDecimal> map = monthMap.get(key);
            incomeList.add(map.get("income") != null ? this.roundTwoPre(map.get("income")) : new BigDecimal("0.00"));
        }
        monthlyCost.put("label", labelList);
        monthlyCost.put("income", incomeList);
        target.put("monthlyCost", monthlyCost);

        data.put("target", target);

        long endTime = System.currentTimeMillis();    //获取结束时间
        logger.info("queryCostByProjectId ---- 运行时间：" + (endTime - startTime) + "ms");//输出程序运行时间
        return CommonResponse.success(data);
    }

    /**
     * 租户下生效项目过滤
     * @param projectId
     * @return
     */
    private QueryParam getNewQueryParam(@RequestParam(value = "projectId") Long projectId) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));//租户隔离
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));//单据状态已提交和审批通过
        queryParam.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
        queryParam.setPageIndex(0);
        queryParam.setPageSize(-1);
        return queryParam;
    }

    /**
     * 根据项目id，查询项目收支统计
     * @param projectId
     * @return
     */
    @GetMapping("queryIncoemAndExpendByProjectId")
    CommonResponse<JSONObject> queryIncoemAndExpendByProjectId(@RequestParam(value = "projectId") Long projectId, HttpServletRequest request){
        long startTime = System.currentTimeMillis();    //获取开始时间
        String authority = request.getHeader("authority");

        ExecutorService threadPool = Executors.newFixedThreadPool(4);
        /** 收入合同 */
        QueryParam queryParam = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        queryParam.getOrderMap().put("createTime", "desc");
        Future<JSONObject> future1 = this.excute(threadPool, authority, "ejc-income-web/contract/pageList", queryParam);
        /** 收款登记 */
        QueryParam queryParam2 = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        Future<JSONObject> future2 = this.excute(threadPool, queryParam2, receiveService);
        /** 开票 */
        Future<JSONObject> future3 = this.excute(threadPool, authority, "ejc-tax-web/invoiceOpen/pageList", queryParam2);
        /** 收票 */
        QueryParam queryParam3 = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        queryParam3.getParams().put("dependOnProject", new Parameter(QueryParam.EQ, "1"));//属于项目
        Future<JSONObject> future4 = this.excute(threadPool, authority, "ejc-tax-web/invoiceReceive/pageList", queryParam3);

        BigDecimal payMoney = this.getPayMny(projectId);// 付款金额
        BigDecimal receiveMoney = null;// 收款登记金额
        BigDecimal incomeContractTaxMny = null;// 收入合同金额（含税）
        BigDecimal incomeContractMny = null;// 收入合同金额（不含税）
        BigDecimal incomeTaxRate = null;// 合同税率
        BigDecimal financeEngineReceiveMny = null;// 工程收款（含税）
        BigDecimal financeOtherReceiveMny = null;// 其他收款（含税）
        BigDecimal openInvoiceTaxMny = null;// 开票金额（含税）
        BigDecimal openInvoiceMny = null;// 开票金额（不含税）
        BigDecimal openTaxMny = null;// 开票税金
        BigDecimal receiveSpecialInvoiceTaxMny = null;// 收票含税金额（专票）
        BigDecimal receiveSpecialInvoiceMny = null;// 收票不含税金额（专票）
        BigDecimal receiveSpecialTaxMny = null;// 收票税金（专票）
        BigDecimal receiveGeneralInvoiceTaxMny = null;// 收票含税金额（普票）
        BigDecimal receiveGeneralInvoiceMny = null;// 收票不含税金额（普票）
        BigDecimal receiveGeneralTaxMny = null;// 收票税金（普票）
        try {
            // 收入合同金额
            incomeContractTaxMny = this.getSumMny(future1.get(), "contractTaxMny");
            incomeContractMny = this.getSumMny(future1.get(), "contractMny");
            // 取新签订收入合同的税率
            JSONArray jsonArray = future1.get().getJSONArray("records");
            if (CollectionUtils.isNotEmpty(jsonArray)) {
                JSONObject obj = jsonArray.getJSONObject(0);
                incomeTaxRate = obj.getBigDecimal("taxRate");
            }

            // 收款登记金额
            receiveMoney = this.getSumMny(future2.get(), "receiveMny");
            JSONArray receiveArray = future2.get().getJSONArray("records");
            if (CollectionUtils.isNotEmpty(receiveArray)) {
                for (int i = 0; i < receiveArray.size(); i++) {
                    JSONObject obj = receiveArray.getJSONObject(i);
                    // 工程收款
                    if (Long.valueOf("1275321308270993409").equals(obj.getLong("receiveType"))){
                        financeEngineReceiveMny = MathUtil.safeAdd(financeEngineReceiveMny, obj.getBigDecimal("receiveMny"));
                    }
                    // 其他收款
                    if (Long.valueOf("1275321354706132993").equals(obj.getLong("receiveType"))){
                        financeOtherReceiveMny = MathUtil.safeAdd(financeOtherReceiveMny, obj.getBigDecimal("receiveMny"));
                    }
                }
            }

            // 开票
            openInvoiceTaxMny = this.getSumMny(future3.get(),"invoiceTaxMny");
            openInvoiceMny = this.getSumMny(future3.get(),"invoiceMny");
            openTaxMny = this.getSumMny(future3.get(),"taxMny");

            // 收票
            JSONArray invoiceArray = future4.get().getJSONArray("records");
            if (CollectionUtils.isNotEmpty(invoiceArray)) {
                for (int i = 0; i < invoiceArray.size(); i++) {
                    JSONObject obj = invoiceArray.getJSONObject(i);
                    // 专票
                    if (Long.valueOf("1277537516768632833").equals(obj.getLong("invoiceType"))){
                        receiveSpecialInvoiceTaxMny = MathUtil.safeAdd(receiveSpecialInvoiceTaxMny, obj.getBigDecimal("invoiceTaxMny"));
                        receiveSpecialInvoiceMny = MathUtil.safeAdd(receiveSpecialInvoiceMny, obj.getBigDecimal("invoiceMny"));
                        receiveSpecialTaxMny = MathUtil.safeAdd(receiveSpecialTaxMny, obj.getBigDecimal("taxMny"));
                    }
                    // 普票
                    if (Long.valueOf("1277537563879055362").equals(obj.getLong("invoiceType"))){
                        receiveGeneralInvoiceTaxMny = MathUtil.safeAdd(receiveGeneralInvoiceTaxMny, obj.getBigDecimal("invoiceTaxMny"));
                        receiveGeneralInvoiceMny = MathUtil.safeAdd(receiveGeneralInvoiceMny, obj.getBigDecimal("invoiceMny"));
                        receiveGeneralTaxMny = MathUtil.safeAdd(receiveGeneralTaxMny, obj.getBigDecimal("taxMny"));
                    }
                }
            }
        } catch (Exception e) {
            logger.error("查询项目收付款以及合同金额数据异常", e);
        } finally {
            threadPool.shutdown();
        }

        JSONObject resp = new JSONObject();
        resp.put("receiveMoney", receiveMoney);
        resp.put("payMoney", payMoney);
        resp.put("incomeContractTaxMny", incomeContractTaxMny);
        resp.put("incomeContractMny", incomeContractMny);
        resp.put("incomeTaxRate", incomeTaxRate);
        resp.put("financeEngineReceiveMny", financeEngineReceiveMny);
        resp.put("financeOtherReceiveMny", financeOtherReceiveMny);
        resp.put("openInvoiceTaxMny", openInvoiceTaxMny);
        resp.put("openInvoiceMny", openInvoiceMny);
        resp.put("openTaxMny", openTaxMny);
        resp.put("receiveSpecialInvoiceTaxMny", receiveSpecialInvoiceTaxMny);
        resp.put("receiveSpecialInvoiceMny", receiveSpecialInvoiceMny);
        resp.put("receiveSpecialTaxMny", receiveSpecialTaxMny);
        resp.put("receiveGeneralInvoiceTaxMny", receiveGeneralInvoiceTaxMny);
        resp.put("receiveGeneralInvoiceMny", receiveGeneralInvoiceMny);
        resp.put("receiveGeneralTaxMny", receiveGeneralTaxMny);

        long endTime = System.currentTimeMillis();    //获取结束时间
        logger.info("queryIncoemAndExpendByProjectId ---- 运行时间：" + (endTime - startTime) + "ms");//输出程序运行时间

        return CommonResponse.success(resp);
    }

    /*
     *
     * @description: 获取项目收支台账VO
     * @author: 曹鹏辉
     * @date: 2021/5/26 17:40
     * @param: param
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.finance.vo.ProjectLedgerVO>
     */
    @PostMapping("getProjectLedgerVO")
    public CommonResponse<ProjectLedgerVO> getProjectLedgerVO(@RequestBody QueryParam param) throws ExecutionException, InterruptedException {
        ProjectLedgerVO projectLedgerVO = new ProjectLedgerVO();

        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));//租户隔离
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));//单据状态已提交和审批通过
        param.setPageIndex(0);
        param.setPageSize(-1);

        // 开始时间、结束时间
        Parameter startDay = param.getParams().get("startDay");
        Parameter endDay = param.getParams().get("endDay");
        param.getParams().remove("startDay");
        param.getParams().remove("endDay");

        // 获取支出明细以及累计支出
        List<ExpenditureVO> expenditureList = queryExpenditureList(param, startDay, endDay)
                .stream().filter(vo -> vo.getApplyTime() != null)
                .sorted(Comparator.comparing(ExpenditureVO::getApplyTime).reversed()).collect(Collectors.toList());
        BigDecimal sumPayMny = expenditureList.stream().filter(vo -> vo.getPayMny() != null)
                .map(ExpenditureVO::getPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);

        // 获取收入明细以及累计收款
        List<IncomeVO> incomeList = queryIncomeList(param, startDay, endDay);
        BigDecimal sumReceiveMny = incomeList.stream().filter(vo -> vo.getReceiveMny() != null)
                .map(IncomeVO::getReceiveMny).reduce(BigDecimal.ZERO, BigDecimal::add);

        // 账面资金
        BigDecimal bookFunds = sumReceiveMny.subtract(sumPayMny);

        projectLedgerVO.setProjectId(Long.getLong(param.getParams().get("projectId").getValue().toString()));
        projectLedgerVO.setSumReceiveMny(sumReceiveMny);
        projectLedgerVO.setSumPayMny(sumPayMny);
        projectLedgerVO.setBookFunds(bookFunds);
        projectLedgerVO.setExpenditureList(expenditureList);
        projectLedgerVO.setIncomeList(incomeList);

        if (CollectionUtils.isNotEmpty(expenditureList) && CollectionUtils.isNotEmpty(incomeList)) {
            projectLedgerVO.setProjectName(expenditureList.get(0).getProjectName());
        }else if (CollectionUtils.isNotEmpty(expenditureList)) {
            projectLedgerVO.setProjectName(expenditureList.get(0).getProjectName());
        }else if (CollectionUtils.isNotEmpty(incomeList)) {
            projectLedgerVO.setProjectName(incomeList.get(0).getProjectName());
        }

        return CommonResponse.success("查询成功！", projectLedgerVO);
    }

    /*
     *
     * @description: 根据projectId查询已办理的合同付款、报销、零星材料付款、已生效的借款报销
     * @author: 曹鹏辉
     * @date: 2021/5/26 17:41
     * @param: param
     * @param: startDay
     * @param: endDay
     * @return: java.util.List<com.ejianc.business.finance.vo.ExpenditureVO>
     */
    private List<ExpenditureVO> queryExpenditureList(QueryParam param, Parameter startDay, Parameter endDay) {
        ExecutorService threadPool = Executors.newFixedThreadPool(4);

        if (startDay != null && endDay != null) {
            param.getParams().put("applyTime",new Parameter(QueryParam.BETWEEN, startDay.getValue() + "," + endDay.getValue()));
        }else if (startDay != null) {
            param.getParams().put("applyTime",new Parameter(QueryParam.GE, startDay.getValue()));
        }else if (endDay != null) {
            param.getParams().put("applyTime",new Parameter(QueryParam.LE, endDay.getValue()));
        }

        QueryParam queryParam = Utils.deepCopy(param);// 深拷贝
        queryParam.getParams().put("dependOnProject",new Parameter(QueryParam.EQ,"1"));//属于项目
        queryParam.getParams().put("payStatus",new Parameter(QueryParam.EQ,"2"));//支付状态：2-已支付

        /** 付款申请 */
        Future<JSONObject> future1 = this.excute(threadPool, queryParam, contractService);
        /** 零星材料付款申请 */
        Future<JSONObject> future2 = this.excute(threadPool, queryParam, sporadicService);
        /** 报销 */
        Future<JSONObject> future3 = this.excute(threadPool, queryParam, reimburseService);
        /** 借款报销 */
        QueryParam queryParam2 = Utils.deepCopy(param);// 深拷贝
        queryParam2.getParams().put("dependOnProject",new Parameter(QueryParam.EQ,"1"));//属于项目
        Future<JSONObject> future4 = this.excute(threadPool, queryParam2, loadReimburseService);

        List<ExpenditureVO> list = new ArrayList<>();
        try {
            /** 付款申请 */
            List<ExpenditureVO> payContList = BeanMapper.mapList(future1.get().getJSONArray("records"), ExpenditureVO.class);
            if (CollectionUtils.isNotEmpty(payContList)) {
                payContList.forEach(vo -> {
                    vo.setFeeTypeName("合同付款-" + ConstantTypeEnum.getEnumByCode(vo.getFeeType()).getName());
                });
                list.addAll(payContList);
            }

            /** 零星材料付款申请 */
            JSONArray sporadicJsonArray = future2.get().getJSONArray("records");
            List<PaySporadicEntity> sporadicEntityList = JSONObject.parseArray(sporadicJsonArray.toJSONString(), PaySporadicEntity.class);
            List<ExpenditureVO> sporadicList = BeanMapper.mapList(sporadicEntityList, ExpenditureVO.class);
            // 设置支付类型
            if (CollectionUtils.isNotEmpty(sporadicList)) {
                sporadicList.forEach(vo -> {
                    vo.setFeeTypeName("零星采购");
                    sporadicEntityList.forEach(entity -> {
                        if (StringUtils.equals(vo.getBillCode(), entity.getBillCode())) {
                            vo.setAccountName(entity.getAccountName());
                        }
                    });
                });
                list.addAll(sporadicList);
            }

            /** 报销 */
            List<ExpenditureVO> reimburseList = BeanMapper.mapList(future3.get().getJSONArray("records"), ExpenditureVO.class);
            list.addAll(reimburseList);

            /** 借款报销 */
            JSONArray loadReimburseJsonArray = future4.get().getJSONArray("records");
            List<LoadReimburseEntity> loadReimburseEntityList = JSONObject.parseArray(loadReimburseJsonArray.toJSONString(), LoadReimburseEntity.class);
            List<ExpenditureVO> loadReimburseList = BeanMapper.mapList(loadReimburseEntityList, ExpenditureVO.class);
            // 设置支付类型
            if (CollectionUtils.isNotEmpty(loadReimburseList)) {
                loadReimburseList.forEach(vo -> {
                    vo.setFeeTypeName("借款报销");
                    loadReimburseEntityList.forEach(entity -> {
                        if (StringUtils.equals(vo.getBillCode(), entity.getBillCode())) {
                            vo.setPayMny(entity.getReimburseMny());
                            vo.setApplyUserName(entity.getApplyEmployeeName());
                        }
                    });
                });
                list.addAll(loadReimburseList);
            }
        } catch (Exception e) {
            logger.error("查询数据异常", e);
        } finally {
            threadPool.shutdown();
        }
        // 降序并去重
        return list.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator
                .comparing(ExpenditureVO::getBillCode))), ArrayList::new));
    }

    /*
     *
     * @description: 根据projectId查询收入明细
     * @author: 曹鹏辉
     * @date: 2021/5/26 17:42
     * @param: param
     * @param: startDay
     * @param: endDay
     * @return: java.util.List<com.ejianc.business.finance.vo.IncomeVO>
     */
    private List<IncomeVO> queryIncomeList(QueryParam param, Parameter startDay, Parameter endDay){
        param.getParams().remove("applyTime");
        param.getOrderMap().put("confirmTime", QueryParam.DESC);
        if (startDay != null && endDay != null) {
            param.getParams().put("confirmTime",new Parameter(QueryParam.BETWEEN, startDay.getValue() + "," + endDay.getValue()));
        }else if (startDay != null) {
            param.getParams().put("confirmTime",new Parameter(QueryParam.GE, startDay.getValue()));
        }else if (endDay != null) {
            param.getParams().put("confirmTime",new Parameter(QueryParam.LE, endDay.getValue()));
        }

        List<ReceiveEntity> list = receiveService.queryList(param, false);
        return BeanMapper.mapList(list, IncomeVO.class);
    }

    /*
     *
     * @description: 项目收支台账导出
     * @author: 曹鹏辉
     * @date: 2021/5/27 18:25
     * @param: param
     * @param: response
     * @return: void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) throws ExecutionException, InterruptedException {
        CommonResponse<ProjectLedgerVO> commonResponse = getProjectLedgerVO(param);
        if (commonResponse.isSuccess() && commonResponse.getData() != null) {
            Map<String, Object> beans = new HashMap<>();
            ProjectLedgerVO data = commonResponse.getData();
            List<ExpenditureVO> expenditureList = data.getExpenditureList();
            List<IncomeVO> incomeList = data.getIncomeList();

            beans.put("expend", expenditureList);
            beans.put("income", incomeList);
            ExcelExport.getInstance().export("expend-income-export.xlsx", beans, response);
        }
    }

    /**
     * 根据项目汇总已办理的合同付款、报销、零星材料付款、已生效的借款报销
     * @param projectId
     * @return
     * @throws ExecutionException
     * @throws InterruptedException
     */
    private BigDecimal getPayMny(Long projectId) {
        QueryParam queryParam = this.getNewQueryParam(projectId);// 租户下生效项目过滤

        queryParam.getParams().put("dependOnProject",new Parameter(QueryParam.EQ,"1"));//属于项目
        queryParam.getParams().put("payStatus",new Parameter(QueryParam.EQ,"2"));//支付状态：2-已支付

        ExecutorService threadPool = Executors.newFixedThreadPool(4);
        /** 付款申请 */
        Future<JSONObject> future1 = this.excute(threadPool, queryParam, contractService);
        /** 零星材料付款申请 */
        Future<JSONObject> future2 = this.excute(threadPool, queryParam, sporadicService);
        /** 报销 */
        Future<JSONObject> future3 = this.excute(threadPool, queryParam, reimburseService);
        /** 借款报销 */
//        queryParam.getParams().remove("payStatus");
        QueryParam param = Utils.deepCopy(queryParam);// 深拷贝
        param.getParams().remove("payStatus");
        Future<JSONObject> future4 = this.excute(threadPool, param, loadReimburseService);
        BigDecimal payMoney = null;// 付款金额
        try {
            JSONObject c1 = future1.get();
            BigDecimal payContMny = getSumMny(c1, "payMny");

            JSONObject c2 = future2.get();
            BigDecimal sporadicMny = getSumMny(c2, "payMny");

            JSONObject c3 = future3.get();
            BigDecimal reimburseMny = getSumMny(c3, "payMny");

            JSONObject c4 = future4.get();
            BigDecimal loadReimburseMny = getSumMny(c4, "reimburseMny");
            payMoney = MathUtil.safeAdd(MathUtil.safeAdd(MathUtil.safeAdd(payContMny, sporadicMny), reimburseMny), loadReimburseMny);
        } catch (Exception e) {
            logger.error("查询数据异常", e);
        } finally {
            threadPool.shutdown();
        }
        return payMoney;
    }

    /**
     * 收支
     * @param projectId
     * @return
     */
    private JSONObject getPaymentBalance(Long projectId, String authority) {
        ExecutorService threadPool = Executors.newFixedThreadPool(3);
        /** 收款登记 */
        QueryParam queryParam = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        Future<JSONObject> future1 = this.excute(threadPool, queryParam, receiveService);
        /** 开票 */
        Future<JSONObject> future2 = this.excute(threadPool, authority, "ejc-tax-web/invoiceOpen/pageList", queryParam);
        /** 收票 */
        QueryParam queryParam2 = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        queryParam2.getParams().put("dependOnProject", new Parameter(QueryParam.EQ, "1"));//属于项目
        Future<JSONObject> future3 = this.excute(threadPool, authority, "ejc-tax-web/invoiceReceive/pageList", queryParam2);

        BigDecimal salesInvoice = null;// 开票金额
        BigDecimal inputInvoice = null;// 收票金额
        BigDecimal receiveMoney = null;// 收款金额
        BigDecimal payMoney = this.getPayMny(projectId);// 付款金额
        BigDecimal balanceMny = null;// 收付款金额
        try {
            receiveMoney = this.getSumMny(future1.get(),"receiveMny");
            balanceMny = MathUtil.safeSub(receiveMoney, payMoney);
            salesInvoice = this.getSumMny(future2.get(),"invoiceTaxMny");
            inputInvoice = this.getSumMny(future3.get(),"invoiceTaxMny");
        } catch (Exception e) {
            logger.error("查询数据异常", e);
        } finally {
            threadPool.shutdown();
        }

        JSONObject paymentBalance = new JSONObject();
        paymentBalance.put("paymentBalance", this.roundTwoPre(balanceMny));
        paymentBalance.put("projectCollection", this.roundTwoPre(receiveMoney));
        paymentBalance.put("projectExpenditure", this.roundTwoPre(payMoney));
        paymentBalance.put("salesInvoice", this.roundTwoPre(salesInvoice));
        paymentBalance.put("inputInvoice", this.roundTwoPre(inputInvoice));
        return paymentBalance;
    }

    /**
     * 查询项目下已生效收款日期最大的收款记录信息
     * @param projectId
     * @return
     */
    private JSONObject getLatestCollection(Long projectId) {
        LambdaQueryWrapper<ReceiveEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(ReceiveEntity::getProjectId, projectId);
        lambda.in(ReceiveEntity::getBillState, 1, 3);
        lambda.orderByDesc(ReceiveEntity::getCreateTime);
        List<ReceiveEntity> list = receiveService.list(lambda);
        String collectionDate = null;
        String collectionType = null;
        BigDecimal money = null;
        if(CollectionUtils.isNotEmpty(list)){
            ReceiveEntity entity = list.get(0);
            // 收款日期
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            collectionDate = sdf.format(entity.getConfirmTime());
            // 收款类型
            CommonResponse<List<DefdocDetailVO>> defResp = defdocApi.getDefDocByDefId(330358772148207658L);
            if(!defResp.isSuccess()){
                logger.error("获取自定义档案列表失败：{}", defResp.getMsg());
            }
            for(DefdocDetailVO vo : defResp.getData()){
                if(vo.getId().equals(entity.getReceiveType())){
                    collectionType = vo.getName();
                }
            }
            money = entity.getReceiveMny();
        }

        JSONObject latestCollection = new JSONObject();
        latestCollection.put("collectionDate", collectionDate);
        latestCollection.put("collectionType", collectionType);
        latestCollection.put("money", MathUtil.roundTwoPre(money));// 收款金额
        return latestCollection;
    }

    /**
     * 月度收支
     * @param projectId
     * @return
     */
    public JSONObject getMonthlyIncomeAndExpenditure(Long projectId){
        QueryParam queryParam = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        /** 收款 */
        List<ReceiveEntity> receiveEntities =  receiveService.queryList(queryParam,false);
        /** 付款申请 */
        queryParam.getParams().put("dependOnProject",new Parameter(QueryParam.EQ,"1"));//属于项目
        queryParam.getParams().put("payStatus",new Parameter(QueryParam.EQ,"2"));//支付状态：2-已支付
        List<PayContractEntity> contractEntities = contractService.queryList(queryParam,false);
        /** 零星材料付款申请 */
        List<PaySporadicEntity> paySporadicEntities = sporadicService.queryList(queryParam,false);
        /** 报销 */
        List<PayReimburseEntity> reimburseEntities = reimburseService.queryList(queryParam,false);
        /** 借款报销 */
        queryParam.getParams().remove("payStatus");
        List<LoadReimburseEntity> loadReimburseEntities = loadReimburseService.queryList(queryParam,false);

        Map<String, Map<String, BigDecimal>> monthMap = new LinkedHashMap<>();
        if(ListUtil.isNotEmpty(receiveEntities)){
            for (ReceiveEntity entity:receiveEntities){
                this.calculateSumMny(monthMap, entity.getConfirmTime(), "income", entity.getReceiveMny());
            }
        }
        if(ListUtil.isNotEmpty(contractEntities)){
            Map<String, BigDecimal> mnyMap = new HashMap<>();
            for (PayContractEntity entity:contractEntities){
                this.calculateSumMny(monthMap, entity.getConfirmTime(), "expenditure", entity.getPayMny());
            }
        }
        if(ListUtil.isNotEmpty(paySporadicEntities)){
            for (PaySporadicEntity entity:paySporadicEntities){
                this.calculateSumMny(monthMap, entity.getConfirmTime(), "expenditure", entity.getPayMny());
            }
        }
        if(ListUtil.isNotEmpty(reimburseEntities)){
            for (PayReimburseEntity entity:reimburseEntities){
                this.calculateSumMny(monthMap, entity.getConfirmTime(), "expenditure", entity.getPayMny());
            }
        }
        if(ListUtil.isNotEmpty(loadReimburseEntities)){
            for (LoadReimburseEntity entity:loadReimburseEntities){
                this.calculateSumMny(monthMap, entity.getApplyTime(), "expenditure", entity.getReimburseMny());
            }
        }
        JSONObject back = new JSONObject();
        List<String> labelList = new LinkedList<>();
        for(String key : monthMap.keySet()){
            labelList.add(key);
        }
        Collections.sort(labelList, new Comparator<String>() {
            @Override
            public int compare(String u1, String u2) {
                int diff = Integer.valueOf(u1.substring(0, u1.length() -1)) - Integer.valueOf(u2.substring(0, u2.length() -1));
                if (diff > 0) {
                    return 1;
                }else if (diff < 0) {
                    return -1;
                }
                return 0; //相等为0
            }
        });
        List<BigDecimal> incomeList = new LinkedList<>();
        List<BigDecimal> expenditureList = new LinkedList<>();
        for(String key : labelList){
            Map<String, BigDecimal> map = monthMap.get(key);
            incomeList.add(map.get("income") != null ? this.roundTwoPre(map.get("income")) : new BigDecimal("0.00"));
            expenditureList.add(map.get("expenditure") != null ? this.roundTwoPre(map.get("expenditure")) : new BigDecimal("0.00"));
        }
        back.put("label", labelList);
        back.put("income", incomeList);
        back.put("expenditure",expenditureList);
        return back;
    }

    /**
     * 押金、借款、保证金
     * @param projectId
     * @return
     */
    public JSONObject getDepositLoanMargin(Long projectId){
        QueryParam queryParam = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        /** 押金 */
        List<PayForegiftEntity> foregiftEntities =  foregiftService.queryList(queryParam,false);
        BigDecimal sumPayMny = null;
        if(CollectionUtils.isNotEmpty(foregiftEntities)){
            for(PayForegiftEntity vo : foregiftEntities){
                sumPayMny = MathUtil.safeAdd(sumPayMny, vo.getPayMny());
            }
        }
        List<Long> foregiftIds = foregiftEntities.stream().map(PayForegiftEntity::getId).collect(Collectors.toList());
        BigDecimal sumReturnMny = null;
        if(CollectionUtils.isNotEmpty(foregiftIds)){
            List<PayForegiftRecordEntity> foregiftRecordEntities = foregiftRecordService.list(
                    new QueryWrapper<PayForegiftRecordEntity>().in("payapply_id", foregiftIds));
            if(CollectionUtils.isNotEmpty(foregiftRecordEntities)){
                for(PayForegiftRecordEntity vo : foregiftRecordEntities){
                    sumReturnMny = MathUtil.safeAdd(sumReturnMny, vo.getReturnMny());
                }
            }
        }
        // 未回收押金 = 累计支付金额 - 累计退还金额
        BigDecimal uncollectedDeposit = MathUtil.safeSub(sumPayMny, sumReturnMny);

        /** 借款 */
        List<LoadApplyEntity> loadApplyEntities =  loadApplyService.queryList(queryParam,false);
        BigDecimal sumApplyMny = null;
        if(CollectionUtils.isNotEmpty(loadApplyEntities)){
            for(LoadApplyEntity vo : loadApplyEntities){
                sumApplyMny = MathUtil.safeAdd(sumApplyMny, vo.getApplyMny());
            }
        }
        List<LoadReimburseEntity> loadReimburseEntities =  loadReimburseService.queryList(queryParam,false);
        BigDecimal sumReimburseMny = null;
        if(CollectionUtils.isNotEmpty(loadReimburseEntities)){
            for(LoadReimburseEntity vo : loadReimburseEntities){
                sumReimburseMny = MathUtil.safeAdd(sumReimburseMny, vo.getReimburseMny());
            }
        }
        List<LoadBackEntity> loadBackEntities =  loadBackService.queryList(queryParam,false);
        BigDecimal sumBackMny = null;
        if(CollectionUtils.isNotEmpty(loadBackEntities)){
            for(LoadBackEntity vo : loadBackEntities){
                sumBackMny = MathUtil.safeAdd(sumBackMny, vo.getBackMny());
            }
        }
        // 未回收借款 = 累计申请金额 - 累计报销金额 - 累计退还金额
        BigDecimal uncollectedLoan = MathUtil.safeSub(MathUtil.safeSub(sumApplyMny, sumReimburseMny), sumBackMny);

        // 对下保证金
        List<BondDownEntity> bondDownEntities = bondDownService.queryList(queryParam, false);
        BigDecimal depositReturned = null;
        if(CollectionUtils.isNotEmpty(bondDownEntities)){
            for(BondDownEntity vo : bondDownEntities){
                // 待退还保证金 = 累计(支付金额 - 退还金额)
                depositReturned = MathUtil.safeAdd(depositReturned, MathUtil.safeSub(vo.getPayMny(), vo.getBackMny()));
            }
        }
        // 对下保证金
        List<BondUpEntity> bondUpEntities = bondUpService.queryList(queryParam, false);
        BigDecimal uncollectedMargin = null;
        if(CollectionUtils.isNotEmpty(bondUpEntities)){
            for(BondUpEntity vo : bondUpEntities){
                // 未回收保证金 = 累计(支付金额 - 退还金额)
                uncollectedMargin = MathUtil.safeAdd(uncollectedMargin, MathUtil.safeSub(vo.getPayMny(), vo.getBackMny()));
            }
        }

        JSONObject back = new JSONObject();
        back.put("uncollectedDeposit", this.roundTwoPre(uncollectedDeposit));
        back.put("uncollectedLoan", this.roundTwoPre(uncollectedLoan));
        back.put("depositReturned", this.roundTwoPre(depositReturned));
        back.put("uncollectedMargin", this.roundTwoPre(uncollectedMargin));
        return back;
    }

    /**
     * 欠开发票top5
     * @param projectId
     * @return
     */
    public JSONObject getUnderInvoice(Long projectId){
        QueryParam queryParam = this.getNewQueryParam(projectId);// 租户下生效项目过滤
        queryParam.getParams().put("dependOnProject", new Parameter(QueryParam.EQ, "1"));//属于项目
        queryParam.getParams().put("contractId",new Parameter(QueryParam.NE, null));

        // 收票
        CommonResponse<JSONObject> resp = invoiceApi.getReceiveListByQueryParam(queryParam);
        JSONArray jsonArray = new JSONArray();
        if (!resp.isSuccess() || resp.getData() == null) {
            logger.error("获取列表失败：{}", resp.getMsg());
        } else {
            jsonArray = resp.getData().getJSONArray("records");
        }
        Map<Long, JSONObject> map = new HashMap<>();
        if (CollectionUtils.isNotEmpty(jsonArray)) {
            for (int i = 0; i < jsonArray.size(); i++) {
                JSONObject vo = jsonArray.getJSONObject(i);
                BigDecimal invoiceTaxMny = vo.getBigDecimal("invoiceTaxMny");
                this.calculateSumMnyBySupplierId(map, vo.getLong("supplierId"), invoiceTaxMny, vo.getString("supplierName"), false);
            }
        }

        /** 付款申请 */
        queryParam.getParams().put("payStatus",new Parameter(QueryParam.EQ,"2"));//支付状态：2-已支付
        List<PayContractEntity> contractEntities = contractService.queryList(queryParam,false);
        if(CollectionUtils.isNotEmpty(contractEntities)){
            for (PayContractEntity vo : contractEntities){
                this.calculateSumMnyBySupplierId(map, vo.getReceiveUnitId(), vo.getPayMny(), vo.getReceiveUnitName(), true);
            }
        }

        // 过滤小于等于0的开发票
        Iterator<Map.Entry<Long, JSONObject>> it = map.entrySet().iterator();
        BigDecimal accumulatedUnderInvoice = null;// 累计欠开发票
        while(it.hasNext()){
            BigDecimal amountOwed = it.next().getValue().getBigDecimal("amountOwed");
            amountOwed = amountOwed != null ? amountOwed : BigDecimal.ZERO;
            // 只有当值大于0时，才是欠开发票
            if(amountOwed.compareTo(BigDecimal.ZERO) <= 0){
                it.remove();//使用迭代器的remove()方法删除元素
                continue;
            }
            accumulatedUnderInvoice = MathUtil.safeAdd(accumulatedUnderInvoice, amountOwed);
        }

        // 处理map转换为List
        List<JSONObject> dataList = new ArrayList<>();
        for(Long key : map.keySet()){
            JSONObject value = map.get(key);
            BigDecimal amountOwed = value.getBigDecimal("amountOwed") != null ?
                    value.getBigDecimal("amountOwed") : BigDecimal.ZERO;
            // 占比 = 欠开金额/累计欠开发票
            BigDecimal proportion = MathUtil.safeDiv(amountOwed, accumulatedUnderInvoice);
            value.put("amountOwed", this.roundTwoPre(amountOwed));
            value.put("proportion", proportion == null ? new BigDecimal("0.0000") : proportion.setScale(4, BigDecimal.ROUND_HALF_EVEN) );
            dataList.add(value);
        }
        // 排序并截取前5
        Collections.sort(dataList, new Comparator<JSONObject>() {
            @Override
            public int compare(JSONObject o1, JSONObject o2) {
                BigDecimal mny1 = o1.getBigDecimal("amountOwed");
                BigDecimal mny2 = o2.getBigDecimal("amountOwed");
                return mny1.compareTo(mny2); //相等为0
            }
        });
        Collections.reverse(dataList);
        dataList = dataList.size()> 5 ? dataList.subList(0, 5) : dataList;
        JSONArray array = JSONArray.parseArray(JSON.toJSONString(dataList));

        JSONObject back = new JSONObject();
        back.put("accumulatedUnderInvoice", this.roundTwoPre(accumulatedUnderInvoice != null ? accumulatedUnderInvoice : BigDecimal.ZERO));
        back.put("list", array);
        return back;
    }

    /**
     *  汇总queryParam查询到的List<Entity>中的某一金额字段
     * @param resp
     * @param field
     * @return
     */
    private BigDecimal getSumMny(CommonResponse<JSONObject> resp, String records, String field) {
        JSONArray jsonArray = new JSONArray();
        if (!resp.isSuccess() || resp.getData() == null) {
            logger.error("获取列表失败：{}", resp.getMsg());
        } else {
            logger.info("查询列表结果：{}", resp);
            jsonArray = resp.getData().getJSONArray(records);
        }
        BigDecimal sumMny = null;
        if (CollectionUtils.isNotEmpty(jsonArray)) {
            for (int i = 0; i < jsonArray.size(); i++) {
                JSONObject obj = jsonArray.getJSONObject(i);
                sumMny = MathUtil.safeAdd(sumMny, obj.getBigDecimal(field));
            }
        }
        return sumMny;
    }

    /**
     * 汇总金额字段
     * @param json
     * @param field
     * @return
     */
    private BigDecimal getSumMny(JSONObject json, String field) {
        JSONArray jsonArray = json.getJSONArray("records");
        if(CollectionUtils.isEmpty(jsonArray)){
            return null;
        }
        BigDecimal sumMny = null;
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject obj = jsonArray.getJSONObject(i);
            sumMny = MathUtil.safeAdd(sumMny, obj.getBigDecimal(field));
        }
        return sumMny;
    }

    /**
     *  使用Callable汇总月度Map的金额字段
     * @param monthMap
     * @param json
     * @return
     */
    private void calculateCallableMny(Map<String, Map<String, BigDecimal>> monthMap,
                                      JSONObject json, String dateField, String mnyField) {
        JSONArray jsonArray = json.getJSONArray("records");
        if (CollectionUtils.isNotEmpty(jsonArray)) {
            for (int i = 0; i < jsonArray.size(); i++) {
                JSONObject obj = jsonArray.getJSONObject(i);
                this.calculateSumMny(monthMap, obj.getDate(dateField), "income",  obj.getBigDecimal(mnyField));
            }
        }
    }

    /**
     * 汇总月度Map的收付款金额
     * @param monthMap
     * @param date
     * @param key
     * @param value
     */
    private void calculateSumMny(Map<String, Map<String, BigDecimal>> monthMap, Date date, String key, BigDecimal value) {
//        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        String month = (calendar.get(Calendar.MONTH) + 1) + "月";

        Map<String, BigDecimal> mnyMap = new HashMap<>();
        BigDecimal mny = null;
        if(monthMap.containsKey(month)){
            mnyMap = monthMap.get(month);
            mny = MathUtil.safeAdd(mnyMap.get(key), value);
            mnyMap.put(key, mny);
        } else {
            mnyMap = new HashMap<>();
            mnyMap.put(key, value);
        }
        monthMap.put(month, mnyMap);
    }

    /**
     * 根据供应商主键汇总金额字段放入map
     * 欠开金额=（按供应商统计）合同付款金额-合同收票金额
     * @param map
     * @param supplierId
     * @param flag
     * @param mny
     */
    private void calculateSumMnyBySupplierId(Map<Long, JSONObject> map, Long supplierId, BigDecimal mny, String supplierName, Boolean flag) {
        JSONObject data = new JSONObject();
        BigDecimal amountOwed = null;
        if (map.containsKey(supplierId)) {
            data = map.get(supplierId);
            if(flag){
                amountOwed = MathUtil.safeAdd(data.getBigDecimal("amountOwed"), mny);
            } else {
                amountOwed = MathUtil.safeSub(data.getBigDecimal("amountOwed"), mny);
            }
        } else {
            if(flag){
                amountOwed = MathUtil.safeAdd(BigDecimal.ZERO, mny);
            } else {
                amountOwed = MathUtil.safeSub(BigDecimal.ZERO, mny);
            }
        }
        data.put("amountOwed", amountOwed);
        data.put("supplierName", supplierName);
        map.put(supplierId, data);
    }

    /**
     * 使用Callable查询数据列表
     * @param threadPool
     * @param authority
     * @param url
     * @param queryParam
     * @return
     * @throws ExecutionException
     * @throws InterruptedException
     */
    private Future<JSONObject> excute(ExecutorService threadPool, String authority, String url, QueryParam queryParam) {
        Callable<JSONObject> callable = new CountCallable(
                RequestContextHolder.getRequestAttributes(), authority, BASE_HOST + url, queryParam, null);
        Future<JSONObject> future = threadPool.submit(callable);
        return future;
    }

    /**
     *  注入service使用Callable查询数据列表
     * @param threadPool
     * @param queryParam
     * @param service
     * @return
     * @throws ExecutionException
     * @throws InterruptedException
     */
    private Future<JSONObject> excute(ExecutorService threadPool, QueryParam queryParam, IBaseService service) {
        Callable<JSONObject> callable = new CountCallable(
                RequestContextHolder.getRequestAttributes(), null, null, queryParam, service);
        Future<JSONObject> future = threadPool.submit(callable);
        return future;
    }

    public BigDecimal roundTwoPre(BigDecimal src){
        return src == null ? new BigDecimal("0.00") : src.setScale(2, BigDecimal.ROUND_HALF_EVEN) ;
    }

}
