package com.ejianc.business.finance.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.finance.bean.ReceiveEntity;
import com.ejianc.business.finance.bean.ReceiveInvoiceEntity;
import com.ejianc.business.finance.bean.ReceiveReceiptEntity;
import com.ejianc.business.finance.mapper.ReceiveMapper;
import com.ejianc.business.finance.service.IReceiveInvoiceService;
import com.ejianc.business.finance.service.IReceiveReceiptService;
import com.ejianc.business.finance.service.IReceiveService;
import com.ejianc.business.finance.util.BillTypeCodeEnum;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.util.ValidateUtil;
import com.ejianc.business.finance.utils.FeignUtil;
import com.ejianc.business.finance.utils.NCUtil;
import com.ejianc.business.finance.vo.ReceiveInvoiceVO;
import com.ejianc.business.finance.vo.ReceiveReceiptVO;
import com.ejianc.business.finance.vo.ReceiveVO;
import com.ejianc.business.finance.vo.SumReceiveVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 *  服务实现类
 * </p>
 *
 * @author yqls
 * @since 2020-06-04
 */
@Service
public class ReceiveServiceImpl extends BaseServiceImpl<ReceiveMapper, ReceiveEntity> implements IReceiveService {

    private static final String RECEIVE_BILL_CODE = "FINANCE_RECEIVE";

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IReceiveInvoiceService invoiceService;

    @Autowired
    private IReceiveReceiptService receiptService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private FeignUtil feignUtil;

    @Override
    public ReceiveVO insertOrUpdate(ReceiveVO vo) {
        ReceiveEntity entity = BeanMapper.map(vo, ReceiveEntity.class);
//        // 校验同合同、同组织下只能有一张未生效单据
//        this.validateBeforeSave(entity);
        // 保存时校验合同version是否一致
        if(!ValidateUtil.validateUpstreamVersion(String.valueOf(vo.getContractId()),
                BillTypeCodeEnum.施工合同.getBillTypeCode(), vo.getContractVersion())){
            throw new BusinessException("该合同已被更新，请刷新后重做！");
        }
        // 自动生成编码
        this.autoSetBillCode(entity);
        // 保存主表
        super.saveOrUpdate(entity);
        // 更新发票
        List<ReceiveInvoiceVO> invoiceVOList = this.updateInvoiceVOS(vo, entity.getId());

        List<ReceiveReceiptVO> receiptVOList = this.updateReceiveReceipt(vo,entity.getId());
        // 返回VO
        ReceiveVO backVO = BeanMapper.map(super.getById(entity.getId()), ReceiveVO.class);
        backVO.setInvoiceVOList(invoiceVOList);// 发票列表
        backVO.setIndexList(receiptVOList);
        return backVO;
    }

    private List<ReceiveInvoiceVO> updateInvoiceVOS(ReceiveVO receiveVO, Long id) {
        // 新增
        List<ReceiveInvoiceVO> voList = receiveVO.getInvoiceVOList();
        if(voList != null && !voList.isEmpty()){
            for(ReceiveInvoiceVO invoiceVO : voList){
                // 保存时校验version是否一致
                if(!ValidateUtil.validateUpstreamVersion(String.valueOf(invoiceVO.getInvoiceId()),
                        BillTypeCodeEnum.税务开票.getBillTypeCode(), invoiceVO.getInvoiceVersion())){
                    throw new BusinessException("发票已被更新，请刷新后重做！");
                }
                invoiceVO.setReceiveId(id);
            }
            List<ReceiveInvoiceEntity> entityList = BeanMapper.mapList(voList, ReceiveInvoiceEntity.class);
            invoiceService.saveOrUpdateBatch(entityList, entityList.size(), false);
            voList = BeanMapper.mapList(entityList, ReceiveInvoiceVO.class);
        }
        // 删除
        List<Long> ids = voList.stream().map(ReceiveInvoiceVO::getId).collect(Collectors.toList());
        QueryWrapper wrapper = new QueryWrapper<ReceiveInvoiceEntity>();
        wrapper.eq("receive_id", id);
        wrapper.notIn(!ids.isEmpty(),"id", ids);
        invoiceService.remove(wrapper, false);
        return voList;
    }

    private List<ReceiveReceiptVO> updateReceiveReceipt(ReceiveVO receiveVO, Long id) {
        // 新增

        List<ReceiveReceiptVO> voList = receiveVO.getIndexList();
        if(voList != null && !voList.isEmpty()){
            for(ReceiveReceiptVO receiptVO : voList){
                receiptVO.setReceiveId(id);
            }
            List<ReceiveReceiptEntity> entityList = BeanMapper.mapList(voList, ReceiveReceiptEntity.class);
            receiptService.saveOrUpdateBatch(entityList, entityList.size(), false);
            for(ReceiveReceiptVO receiptVO : voList){
               if(receiptVO.getRowState() != null && receiptVO.getRowState().equals("del")){
                   receiptService.removeById(receiptVO.getId());
               }
            }

            LambdaQueryWrapper<ReceiveReceiptEntity>query=new LambdaQueryWrapper<>();
            query.eq(ReceiveReceiptEntity::getReceiveId,id);
            query.eq(ReceiveReceiptEntity::getDr,0);
            List<ReceiveReceiptEntity> entities = receiptService.list(query);
            voList = BeanMapper.mapList(entities, ReceiveReceiptVO.class);
        }
        return voList;
    }
    @Override
    public ReceiveVO queryDetail(Long id) {
        // 查询主表
        ReceiveEntity entity = baseMapper.selectById(id);
        ReceiveVO vo = BeanMapper.map(entity, ReceiveVO.class);
        // 查询关联发票
        QueryParam param = new QueryParam();
        param.getParams().put("receiveId", new Parameter(QueryParam.EQ, vo.getId()));
        param.getOrderMap().put("createTime", "desc");
        List<ReceiveInvoiceEntity> invoiceEntityList = invoiceService.queryList(param, false);
        List<ReceiveReceiptEntity> receiveReceiptEntityList = receiptService.queryList(param, false);
        vo.setInvoiceVOList(BeanMapper.mapList(invoiceEntityList, ReceiveInvoiceVO.class));
        vo.setIndexList(BeanMapper.mapList(receiveReceiptEntityList, ReceiveReceiptVO.class));
        return vo;
    }

    @Override
    public List<ReceiveVO> queryExportList(QueryParam param) {
        param.setPageIndex(0);
        param.setPageSize(-1);
        List<ReceiveVO> resVos = (List<ReceiveVO>) queryPageJson(param, false).get("records");
        Map<Long, String> defdocMap = feignUtil.getDefdocMap(330358772148207658L);
        if(!resVos.isEmpty()){
            for(int i = 0 ; i< resVos.size(); i++){
                ReceiveVO vo = resVos.get(i);
                vo.setReceiveTypeName(defdocMap.get(vo.getReceiveType()));
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            };
        }
        return resVos;
    }

    @Override
    public String delete(List<Long> ids) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("id", new Parameter(QueryParam.IN, ids));
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        List<ReceiveEntity> entityList = super.queryList(queryParam, false);
        if(CollectionUtils.isNotEmpty(entityList)) {
            super.removeByIds(ids, false);
            // 删除发票
            invoiceService.remove(new QueryWrapper<ReceiveInvoiceEntity>().in("receive_id", ids), false);
            receiptService.remove(new QueryWrapper<ReceiveReceiptEntity>().in("receive_id", ids), false);
        }
        return "删除成功！";
    }

    @Override
    public JSONObject queryPageJson(QueryParam param, boolean isEs) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("orgName");
        // 组织本下
        param.getParams().put("org_id",new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getOrderMap().put("createTime", QueryParam.DESC);
        // 是否生成凭证
        NCUtil.voucherParam(param);
        IPage<ReceiveEntity> pageData = super.queryPage(param,false);
        com.alibaba.fastjson.JSONObject page = new com.alibaba.fastjson.JSONObject();
        page.put("records", BeanMapper.mapList(pageData.getRecords(), ReceiveVO.class));
        page.put("total", pageData.getTotal());
        page.put("current", pageData.getCurrent());
        page.put("size", pageData.getSize());
        page.put("pages", pageData.getPages());
        return page;
    }

    @Override
    public SumReceiveVO getSumReceiveVOList(Long contractId) {
        QueryParam param = new QueryParam();
        param.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        //已生效状态的单据
        param.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        param.getOrderMap().put("createTime", QueryParam.DESC);
        List<ReceiveEntity> entityList = super.queryList(param, false);
        // 累计收款金额
        BigDecimal sumReceiveMny = null;
        for(ReceiveEntity entity : entityList){
            sumReceiveMny = MathUtil.safeAdd(sumReceiveMny, entity.getReceiveMny());
        }
        SumReceiveVO vo = new SumReceiveVO();
        vo.setContractId(contractId);
        vo.setSumReceiveMny(sumReceiveMny);
        vo.setReceiveVOList(BeanMapper.mapList(entityList, ReceiveVO.class));
        return vo;
    }

    @Override
    public Map<String, Object> countRecAmt(Long tenantId, List<Long> projectIds, boolean isWithContract, List<Long> orgIds) {
        QueryWrapper<ReceiveEntity> queryWrapper = new QueryWrapper<>();
        if (CollectionUtils.isNotEmpty(orgIds)) {
            queryWrapper.in("org_id", orgIds);
        }
        //指定租户下的
        queryWrapper.eq("tenant_id", tenantId);
        queryWrapper.eq("dr", BaseVO.DR_UNDELETE);
        //有效的单据
        queryWrapper.in("bill_state", Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()}));
        //匹配的项目范围
//        if(CollectionUtils.isNotEmpty(projectIds)) {
//            queryWrapper.in("project_id", projectIds);
//        } else {
//            //2-项目收款
//            queryWrapper.isNotNull("project_id");
//        }
        if(isWithContract) {
            //查询有合同的收入
            queryWrapper.isNotNull("contract_id");
        }
        queryWrapper.select(" round(ifnull(sum(receive_mny),0) / 10000, 2) as amt ");
        return super.getMap(queryWrapper);
    }

    /**
     * 校验同合同、同组织下只能有一张未生效单据
     * @param entity
     */
    private void validateBeforeSave(ReceiveEntity entity) {
        LambdaQueryWrapper<ReceiveEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(ReceiveEntity::getContractId, entity.getContractId());
//        lambda.eq(ReceiveEntity::getOrgId, entity.getOrgId());
        lambda.eq(ReceiveEntity::getTenantId, InvocationInfoProxy.getTenantid());
        lambda.ne(entity.getId() != null && entity.getId() > 0, ReceiveEntity::getId, entity.getId());
        lambda.notIn(ReceiveEntity::getBillState, Arrays.asList(
                BillStateEnum.COMMITED_STATE.getBillStateCode(),
                BillStateEnum.PASSED_STATE.getBillStateCode()));
        List<ReceiveEntity> entityList = super.list(lambda);
        if(entityList != null && entityList.size() > 0) {
            throw new BusinessException("该合同下存在未生效单据,不允许新增!");
        }
    }

    /**
     * 自动生成编码
     * @param entity
     */
    private void autoSetBillCode(ReceiveEntity entity) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        if(StringUtils.isEmpty(entity.getBillCode())){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(RECEIVE_BILL_CODE, tenantId);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        //修改  校验合同编号是否重复
        LambdaQueryWrapper<ReceiveEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(ReceiveEntity::getBillCode, entity.getBillCode());
        lambda.eq(ReceiveEntity::getTenantId, tenantId);
        lambda.ne(entity.getId() != null && entity.getId() > 0, ReceiveEntity::getId, entity.getId());
        List<ReceiveEntity> entityList = super.list(lambda);
        if(entityList != null && entityList.size() > 0) {
            throw new BusinessException("存在相同编码，不允许保存!");
        }
    }

}
