package com.ejianc.business.finance.controller.api;

import com.ejianc.business.finance.service.IPayContractService;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.vo.BlocKanBanVo;
import com.ejianc.business.finance.vo.PayContractVO;
import com.ejianc.business.sub.api.ISubBlocKanBanApi;
import com.ejianc.business.sub.vo.ContractVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.ISupplierApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @ClassName: BlocKanBanApi
 * @Description:
 * @Author: 曹鹏辉
 * @Date: 2021/9/7 9:52
 */
@RestController
@RequestMapping("/api/blocKanBan/")
public class FinanceBlocKanBanApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private ISubBlocKanBanApi subBlocKanBanApi;

    @Autowired
    private IPayContractService payContractService;

    @Autowired
    private ISupplierApi supplierApi;

    /**
     * 集团看板分包供应商top5
     * @param orgId
     * @param mnyFlag 0-支付金额(默认)，1-合同总金额，2-欠付金额
     * @param yearFlag 0-今年(默认)，1-全年
     * @return
     */
    @GetMapping(value = "querySupplierPayMny")
    @ResponseBody
    public CommonResponse<List<BlocKanBanVo>> querySupplierPayMny(@RequestParam(value = "orgId", required = false) Long orgId
            , @RequestParam(value = "mnyFlag") Integer mnyFlag, @RequestParam(value = "yearFlag") Integer yearFlag) {

        if (orgId == null) {
            Long tenantid = InvocationInfoProxy.getTenantid();
            CommonResponse<OrgVO> orgByTenantId = orgApi.findOrgByTenantId(tenantid);
            orgId = orgByTenantId.getData().getId();
        }

        // 本下组织
        List<Long> orgIds = orgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList());

        // 分包合同总金额
        CommonResponse<Map<Long, ContractVO>> mapCommonResponse = subBlocKanBanApi.queryExternalSupplierAndMny(null, yearFlag);
        if (!mapCommonResponse.isSuccess()) {
            return CommonResponse.error("查询分包数据失败");
        }
        Map<Long, ContractVO> map = mapCommonResponse.getData();
        List<ContractVO> contractVOList = new ArrayList<>(map.values());
        List<Long> supplierIds = contractVOList.stream().map(ContractVO::getSupplierId).collect(Collectors.toList());

        // 查询外部供应商对应的支付金额
        List<PayContractVO> payContractVOList = payContractService.querySupplierPayMny(orgIds, supplierIds, yearFlag);
        Map<Long, PayContractVO> payContractVOMap = payContractVOList.stream().collect(Collectors.toMap(PayContractVO::getReceiveUnitId, Function.identity()));

        Set<Long> set = new HashSet<>(map.keySet());
        set.addAll(payContractVOMap.keySet());

        List<BlocKanBanVo> voList = new ArrayList<>();
        for (Long supplierId : set) {
            BlocKanBanVo vo = new BlocKanBanVo();
            vo.setReceiveUnitId(supplierId);
            if (map.containsKey(supplierId)) {
                vo.setReceiveUnitName(map.get(supplierId).getSupplierName());
                BigDecimal contractTaxMny = map.get(supplierId).getContractTaxMny();
                contractTaxMny = contractTaxMny == null ? BigDecimal.ZERO : contractTaxMny;
                vo.setTotalContractTaxMny(contractTaxMny);
            }else {
                vo.setTotalContractTaxMny(BigDecimal.ZERO);
            }

            if (payContractVOMap.containsKey(supplierId)) {
                vo.setReceiveUnitName(payContractVOMap.get(supplierId).getReceiveUnitName());
                BigDecimal payMny = payContractVOMap.get(supplierId).getPayMny();
                payMny = payMny == null ? BigDecimal.ZERO : payMny;
                vo.setTotalPayMny(payMny);
            }else {
                vo.setTotalPayMny(BigDecimal.ZERO);
            }

            vo.setOwedMny(ComputeUtil.safeSub(vo.getTotalContractTaxMny(), vo.getTotalPayMny()));
            if (vo.getOwedMny().compareTo(new BigDecimal("0E-8")) == 0) {
                vo.setOwedMny(BigDecimal.ZERO);
            }

            voList.add(vo);
        }

        if (mnyFlag == 0) {
            // 默认 支付金额 降序
            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalPayMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getTotalPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalPayMny(), sunMny), new BigDecimal("100")));
            }
        }else if (mnyFlag == 1) {
            // 合同总金额 降序
            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalContractTaxMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getTotalContractTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalContractTaxMny(), sunMny), new BigDecimal("100")));
            }
        }else {
            // 欠付金额 降序
            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getOwedMny).reversed()).limit(5).collect(Collectors.toList());
            BigDecimal sunMny = voList.stream().map(BlocKanBanVo::getOwedMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo vo : voList) {
                vo.setSupplierRate(ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getOwedMny(), sunMny), new BigDecimal("100")));
            }
        }

        return CommonResponse.success("查询分包供应商top5成功", voList);
    }


    /**
     * 项目看板分包供应商
     * @param projectId
     * @param mnyFlag 0-支付金额(默认)，1-合同总金额，2-欠付金额
     * @param yearFlag 0-今年(默认)，1-全年
     * @return
     */
    @GetMapping(value = "querySubSupplierByProject")
    @ResponseBody
    public CommonResponse<List<BlocKanBanVo>> querySubSupplierByProject(@RequestParam(value = "projectId", required = false) Long projectId
            , @RequestParam(value = "mnyFlag") Integer mnyFlag, @RequestParam(value = "yearFlag") Integer yearFlag
            , @RequestParam(defaultValue = "10") Integer topN) {

        if (projectId == null) {
            return CommonResponse.error("项目为空！");
        }

        List<BlocKanBanVo> voList = new ArrayList<>();
        if (mnyFlag == 0) {
            // 支付金额、查询分包合同供应商付款单、默认
            List<PayContractVO> payContractList = payContractService.querySupplierPayMnyByProject(projectId, yearFlag, Arrays.asList(1));
            BigDecimal totalPayMny = payContractList.stream().map(PayContractVO::getPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (PayContractVO payContractVO : payContractList) {
                BlocKanBanVo vo = new BlocKanBanVo();
                vo.setReceiveUnitId(payContractVO.getReceiveUnitId());
                vo.setReceiveUnitName(payContractVO.getReceiveUnitName());
                vo.setTotalPayMny(payContractVO.getPayMny());
                BigDecimal supplierRate = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalPayMny(), totalPayMny), new BigDecimal("100"));
                vo.setSupplierRate(MathUtil.judgeZeroError(supplierRate));

                voList.add(vo);
            }

            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalPayMny).reversed()).limit(topN).collect(Collectors.toList());
        }else if (mnyFlag == 1) {
            // 合同总金额、查询分包合同总金额
            List<ContractVO> contractList = payContractService.querySubContractMny(projectId, yearFlag);
            BigDecimal totalContractTaxMny = contractList.stream().map(ContractVO::getContractTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (ContractVO contractVO : contractList) {
                BlocKanBanVo vo = new BlocKanBanVo();
                vo.setReceiveUnitId(contractVO.getSupplierId());
                vo.setReceiveUnitName(contractVO.getSupplierName());
                vo.setTotalContractTaxMny(contractVO.getContractTaxMny());
                BigDecimal supplierRate = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalContractTaxMny(), totalContractTaxMny), new BigDecimal("100"));
                vo.setSupplierRate(MathUtil.judgeZeroError(supplierRate));

                voList.add(vo);
            }

            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalContractTaxMny).reversed()).limit(topN).collect(Collectors.toList());
        }else {
            // 欠付金额、查询分包结算-支付金额
            List<BlocKanBanVo> banVoList = payContractService.querySubOwedMny(projectId, yearFlag);
            // 查询支付金额
            List<PayContractVO> payContractList = payContractService.querySupplierPayMnyByProject(projectId, yearFlag, Arrays.asList(1));
            Map<Long, PayContractVO> map = payContractList.stream().collect(Collectors.toMap(PayContractVO::getReceiveUnitId, Function.identity()));

            for (BlocKanBanVo banVo : banVoList) {
                BigDecimal settleTaxMny = MathUtil.judgeZeroError(banVo.getSettleTaxMny());
                BigDecimal finishSettleTaxMny = MathUtil.judgeZeroError(banVo.getFinishSettleTaxMny());
                // 结算金额
                BigDecimal taxMny = finishSettleTaxMny.equals(BigDecimal.ZERO) ? settleTaxMny : finishSettleTaxMny;

                if (map.containsKey(banVo.getReceiveUnitId())) {
                    banVo.setOwedMny(ComputeUtil.safeSub(taxMny, map.get(banVo.getReceiveUnitId()).getPayMny()));
                }else {
                    banVo.setOwedMny(taxMny);
                }
            }

            // 总欠付金额
            BigDecimal totalOwedMny = banVoList.stream().map(BlocKanBanVo::getOwedMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo banVo : banVoList) {
                BigDecimal supplierRate = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(banVo.getOwedMny(), totalOwedMny), new BigDecimal("100"));
                banVo.setSupplierRate(MathUtil.judgeZeroError(supplierRate));
            }

            voList = banVoList.stream().sorted(Comparator.comparing(BlocKanBanVo::getOwedMny).reversed()).limit(topN).collect(Collectors.toList());
        }

        return CommonResponse.success("查询分包供应商成功", voList);
    }


    /**
     * 项目看板材料设备供应商
     * @param projectId
     * @param mnyFlag 0-支付金额(默认)，1-合同总金额，2-欠付金额
     * @param yearFlag 0-今年(默认)，1-全年
     * @return
     */
    @GetMapping(value = "queryMESupplierByProject")
    @ResponseBody
    public CommonResponse<List<BlocKanBanVo>> queryMESupplierByProject(@RequestParam(value = "projectId", required = false) Long projectId
            , @RequestParam(value = "mnyFlag") Integer mnyFlag, @RequestParam(value = "yearFlag") Integer yearFlag
            , @RequestParam(defaultValue = "10") Integer topN) {

        if (projectId == null) {
            return CommonResponse.error("项目为空！");
        }

        List<BlocKanBanVo> voList = new ArrayList<>();
        if (mnyFlag == 0) {
            // 支付金额、查询物资采购、设备采购、设备租赁合同供应商付款单、默认
            List<PayContractVO> payContractList = payContractService.querySupplierPayMnyByProject(projectId, yearFlag, Arrays.asList(2,3,4));
            BigDecimal totalPayMny = payContractList.stream().map(PayContractVO::getPayMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (PayContractVO payContractVO : payContractList) {
                BlocKanBanVo vo = new BlocKanBanVo();
                vo.setReceiveUnitId(payContractVO.getReceiveUnitId());
                vo.setReceiveUnitName(payContractVO.getReceiveUnitName());
                vo.setTotalPayMny(payContractVO.getPayMny());
                BigDecimal supplierRate = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(vo.getTotalPayMny(), totalPayMny), new BigDecimal("100"));
                vo.setSupplierRate(MathUtil.judgeZeroError(supplierRate));

                voList.add(vo);
            }

            voList = voList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalPayMny).reversed()).limit(topN).collect(Collectors.toList());
        }else if (mnyFlag == 1) {
            // 合同总金额、查询物资采购、设备采购、设备租赁合同总金额
            List<BlocKanBanVo> contractList = payContractService.queryMaterialContractMny(projectId, yearFlag);
            BigDecimal totalContractTaxMny = contractList.stream().map(BlocKanBanVo::getTotalContractTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo banVo : contractList) {
                BigDecimal supplierRate = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(banVo.getTotalContractTaxMny(), totalContractTaxMny), new BigDecimal("100"));
                banVo.setSupplierRate(MathUtil.judgeZeroError(supplierRate));
            }

            voList = contractList.stream().sorted(Comparator.comparing(BlocKanBanVo::getTotalContractTaxMny).reversed()).limit(topN).collect(Collectors.toList());
        }else {
            // 欠付金额、查询查询物资采购、设备采购、设备租赁合同结算-支付金额
            // 查询结算金额、支付金额
            List<BlocKanBanVo> settleList = payContractService.queryMaterialSettleMny(projectId, yearFlag);
            List<PayContractVO> payContractList = payContractService.querySupplierPayMnyByProject(projectId, yearFlag, Arrays.asList(2,3,4));
            Map<Long, PayContractVO> map = payContractList.stream().collect(Collectors.toMap(PayContractVO::getReceiveUnitId, Function.identity()));

            for (BlocKanBanVo banVo : settleList) {
                BigDecimal settleTaxMny = MathUtil.judgeZeroError(banVo.getSettleTaxMny());

                if (map.containsKey(banVo.getReceiveUnitId())) {
                    banVo.setOwedMny(ComputeUtil.safeSub(settleTaxMny, map.get(banVo.getReceiveUnitId()).getPayMny()));
                }else {
                    banVo.setOwedMny(settleTaxMny);
                }
            }

            // 总欠付金额
            BigDecimal totalOwedMny = settleList.stream().map(BlocKanBanVo::getOwedMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            for (BlocKanBanVo banVo : settleList) {
                BigDecimal supplierRate = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(banVo.getOwedMny(), totalOwedMny), new BigDecimal("100"));
                banVo.setSupplierRate(MathUtil.judgeZeroError(supplierRate));
            }

            voList = settleList.stream().sorted(Comparator.comparing(BlocKanBanVo::getOwedMny).reversed()).limit(topN).collect(Collectors.toList());
        }

        return CommonResponse.success("查询材料设备供应商成功！", voList);
    }

}
