package com.ejianc.business.finance.controller;


import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.equipment.api.IEquipmentContractApi;
import com.ejianc.business.finance.bean.PayContractEntity;
import com.ejianc.business.finance.service.IPayContractService;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.vo.*;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.ComplexParam;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 *  前端控制器
 * </p>
 *
 * @author yqls
 * @since 2020-06-04
 */
@Controller
@RequestMapping("/payContract")
public class PayContractController {

    @Autowired
    private IPayContractService contractService;

    @Autowired
    private IEquipmentContractApi equipmentContractApi;

    @Autowired
    private IOrgApi orgApi;

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 新增或者修改
     * @param payContractVO
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PayContractVO> saveOrUpdate(@RequestBody PayContractVO payContractVO) {
        PayContractVO backVO = contractService.insertOrUpdate(payContractVO);
        return CommonResponse.success("保存或修改单据成功！", backVO);
    }
    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 查询详情
     * @param id
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PayContractVO> queryDetail(Long id) {
        PayContractVO backVO = contractService.queryDetail(id);
        return CommonResponse.success("查询详情数据成功！", backVO);
    }
    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description queryPrint 查询打印详情
     * @param id
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryPrint", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryPrint(Long id) {
        JSONObject json = new JSONObject();
        json.put("PayContract", contractService.queryDetail(id));
        return CommonResponse.success("查询打印数据成功！",json);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
        param.getOrderMap().put("createTime", QueryParam.DESC);
        JSONObject page = contractService.queryPageJson(param,false);
        return CommonResponse.success("查询列表数据成功！", page);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryApproveList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryApproveList(@RequestBody QueryParam param) {
        //已生效状态的单据
        param.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        param.getOrderMap().put("approveTime", QueryParam.DESC);
        JSONObject page = contractService.queryPageJson(param,false);
        return CommonResponse.success("查询列表数据成功！", page);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description delete 批量删除单据
     * @Param [ids]
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<PayContractVO> vos) {
        List<Long> ids = vos.stream().map(PayContractVO::getId).collect(Collectors.toList());
        String msg = contractService.delete(ids);
        return CommonResponse.success("删除成功！");
    }


    /**
     * excel导出
     *
     * @param response
     * @return
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        List<PayContractVO> supplierVos = contractService.queryExportList(queryParam);
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", supplierVos);
        ExcelExport.getInstance().export("contract-payapply-export.xlsx", beans, response);
    }

    /**
     * 查询列表
     * @param pageNumber
     * @param pageSize
     * @param searchText
     * @param condition
     * @return
     */
    @RequestMapping(value = "/queryContractRefer", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryContractRefer(
            @RequestParam(defaultValue = "1") int pageNumber, @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(required=false) String searchText, @RequestParam(required=false) String condition) {
        QueryParam queryParam = new QueryParam();
        queryParam.setPageSize(pageSize);
        queryParam.setPageIndex(pageNumber);
        queryParam.setSearchText(searchText);
        // condition
        if(StringUtils.isNotBlank(condition)) {
            JSONObject obj = JSONObject.parseObject(condition);
            if(condition.contains("projectId")) {
                queryParam.getParams().put("projectId", new Parameter(QueryParam.EQ, obj.getLong("projectId")));
            }
        }
        // 组织本下
        queryParam.getParams().put("org_id",new Parameter("in",orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        CommonResponse<JSONObject> apiContract = equipmentContractApi.getContract(queryParam);
        JSONObject page = new JSONObject();
        if(apiContract.isSuccess()) {
            page = apiContract.getData();
        }else{
            throw new BusinessException("网络异常， 请稍后再试");
        }
        return CommonResponse.success("查询列表数据成功！", page);
    }

    /**
     * 查询列表
     * @param pageNumber
     * @param pageSize
     * @param searchText
     * @param condition
     * @return
     */
    @RequestMapping(value = "/querySettleRefer", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> querySettleRefer(
            @RequestParam(defaultValue = "1") int pageNumber, @RequestParam(defaultValue = "10") int pageSize,
            @RequestParam(required=false) String searchText, @RequestParam(required=false) String condition) {
        QueryParam queryParam = new QueryParam();
        queryParam.setPageSize(pageSize);
        queryParam.setPageIndex(pageNumber);
        queryParam.setSearchText(searchText);
        Integer contractType = null;
        // condition
        if(StringUtils.isNotBlank(condition)) {
            JSONObject obj = JSONObject.parseObject(condition);
            if(condition.contains("contractId")) {
                queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, obj.getLong("contractId")));
            } else {
                return CommonResponse.error("condition条件缺少contractId！");
            }
            if(condition.contains("projectId")) {
                queryParam.getParams().put("projectId", new Parameter(QueryParam.EQ, obj.getLong("projectId")));
            }
            if(condition.contains("orgId")) {
                queryParam.getParams().put("orgId", new Parameter(QueryParam.EQ, obj.getLong("orgId")));
            } else {
                return CommonResponse.error("condition条件缺少orgId！");
            }
            if(condition.contains("contractType")) {
                contractType = obj.getInteger("contractType");
            } else {
                return CommonResponse.error("condition条件缺少contractType！");
            }
        }
        JSONObject page = contractService.querySettleRefer(queryParam, contractType);
        return CommonResponse.success("查询列表数据成功！", page);
    }

    /**
     * 查询收款账户信息
     * @param receiveUnitId
     * @return
     */
    @RequestMapping(value = "/queryReceiveInfo", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PayContractVO> queryReceiveInfo(Long receiveUnitId) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("receiveUnitId", new Parameter(QueryParam.EQ, receiveUnitId));
        //已生效状态的单据
        queryParam.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        queryParam.getOrderMap().put("createTime", QueryParam.DESC);
        List<PayContractEntity> entityList = contractService.queryList(queryParam,false);
        PayContractVO backVO = !entityList.isEmpty() ? BeanMapper.map( entityList.get(0), PayContractVO.class) : new PayContractVO();
        return CommonResponse.success("查询数据成功！", backVO);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 查询预付款金额与已支付金额
     * @param contractId
     * @param orgId
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "getSumPayMny", method=RequestMethod.GET)
    @ResponseBody
    public CommonResponse<SumPayMnyVO> getSumPayMny( @RequestParam(value="contractId", required=true) Long contractId,
            @RequestParam(value="orgId", required=true) Long orgId) {
        SumPayMnyVO vo = contractService.getSumPayMny(contractId, orgId);
        return CommonResponse.success("查询成功！", vo);
    }

    /**
     * @Author yqls
     * @Date 2020/5/28
     * @Description saveOrUpdate 查询待处理列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryPendingList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryPendingList(@RequestBody QueryParam param) {
        JSONObject page = contractService.queryPendingPageJson(param, false);
        return CommonResponse.success("查询列表数据成功！", page);
    }

    /**
     * excel导出待处理
     *
     * @param response
     * @return
     */
    @RequestMapping(value = "/excelExportPending", method = RequestMethod.POST)
    @ResponseBody
    public void excelExportPending(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        List<PayApplyPubVO> records = contractService.queryExportPendingList(queryParam);
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", records);
        ExcelExport.getInstance().export("pending-payapply-export.xlsx", beans, response);
    }

    /**
     * @Author yqls
     * @Date 2020/8/04
     * @Description saveOrUpdate 批量支付
     * @param vos
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/batchConfirm", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> batchConfirm(@RequestBody List<PayApplyPubVO> vos) {
        String msg = contractService.batchConfirm(vos);
        return CommonResponse.success("批量支付成功！");
    }

    /**
     * @Author yqls
     * @Date 2020/7/7
     * @Description saveOrUpdate 支出管理金额汇总
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "getStatisticsMny", method=RequestMethod.GET)
    @ResponseBody
    public CommonResponse<SumMnyStatisticVO> getStatisticsMny() {
        SumMnyStatisticVO vo = contractService.getStatisticsMny();
        return CommonResponse.success("查询成功！", vo);
    }

    /**
     * @Author yqls
     * @Date 2020/7/7
     * @Description saveOrUpdate 查询管理费明细已收金额与已支付金额
     * @param projectId
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "getManageDetail", method=RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ManageDetailVO> getManageDetail(@RequestParam(value="projectId", required=true) Long projectId) {
        ManageDetailVO vo = contractService.getManageDetail(projectId);
        return CommonResponse.success("查询成功！", vo);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/9/23 根据所给项目ids，查询付款申请已支付金额总和
     * @Description getTotalPayApplyAmountByProjectIds
     * @Param [projectIds]
     * @Return com.ejianc.framework.core.response.CommonResponse<java.math.BigDecimal>
     */
    @RequestMapping(value = "getTotalPayApplyAmountByProjectIds", method=RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> getTotalPayApplyAmountByProjectIds(@RequestBody Long orgId) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("tenantId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));//租户隔离
//        queryParam.getParams().put("dependOnProject",new Parameter(QueryParam.EQ,"1"));//属于项目
        queryParam.getParams().put("payStatus",new Parameter(QueryParam.EQ,"2"));//支付状态：2-已支付
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));//单据状态已提交和审批通过
//        if(ListUtil.isNotEmpty(projectIds)){
//            queryParam.getParams().put("projectId",new Parameter(QueryParam.IN,projectIds));
//        }
        queryParam.getParams().put("org_id", new Parameter("in", orgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

        List<PayContractEntity> contractEntities = contractService.queryList(queryParam,false);
        BigDecimal payApplyAmount = new BigDecimal("0.00");
        if(ListUtil.isNotEmpty(contractEntities)){
            for (PayContractEntity p:contractEntities){
                if(p.getPayMny()!=null){
                    payApplyAmount = MathUtil.safeAdd(payApplyAmount, p.getPayMny());
                }
            }
        }
        JSONObject back = new JSONObject();
        back.put("payApplyAmount",payApplyAmount);
        return CommonResponse.success(back);
    }
    /**
     * @Description  月度收支统计
     */
    @RequestMapping(value = "getMonthInfo", method=RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<FinanceMonthVO>> getMonthInfo(@RequestParam(value="projectId", required=true) Long projectId
            , @RequestParam(value = "timeRange", required = false) String timeRange) {
        List<FinanceMonthVO> list = contractService.getMonthInfo(projectId, timeRange);
        return CommonResponse.success("查询成功！", list);
    }
    /**
     * @Description  支出分类汇总
     */
    @RequestMapping(value = "getCostInfo", method=RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<FinanceCostVO>> getCostInfo(@RequestParam(value="projectId", required=true) Long projectId) {
        List<FinanceCostVO> list = contractService.getCostInfo(projectId);
        return CommonResponse.success("查询成功！", list);
    }

    /**
     * @Author 曹鹏辉
     * @Date 2021/9/17
     * @Description 获取付款申请 合计栏信息(累计申请金额总计，累计支付金额总计)
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<TotalColumnVO>
     */
    @RequestMapping(value = "getTotalColumnInfo", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TotalColumnVO> getTotalColumnInfo(@RequestBody QueryParam param) {
        param.getOrderMap().put("createTime", QueryParam.DESC);
        return CommonResponse.success("查询成功！", contractService.getTotalColumnInfo(param, false));
    }


    /**
     * @Author 曹鹏辉
     * @Date 2021/9/17
     * @Description 获取支出管理 合计栏信息(累计申请金额总计，累计支付金额总计)
     * @Param
     * @Return com.ejianc.framework.core.response.CommonResponse<TotalColumnVO>
     */
    @RequestMapping(value = "getApproveTotalColumnInfo", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TotalColumnVO> getApproveTotalColumnInfo(@RequestBody QueryParam param) {
        //已生效状态的单据
        param.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        param.getOrderMap().put("approveTime", QueryParam.DESC);
        return CommonResponse.success("查询成功！", contractService.getApproveTotalColumnInfo(param, false));
    }

    /**
     * @Author 曹鹏辉
     * @Date 2021/9/17
     * @Description 获取支出管理(未处理) 合计栏信息(累计申请金额总计，累计支付金额总计)
     * @Param
     * @Return com.ejianc.framework.core.response.CommonResponse<TotalColumnVO>
     */
    @RequestMapping(value = "getPendingTotalColumnInfo", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<TotalColumnVO> getPendingTotalColumnInfo(@RequestBody QueryParam param) {
        return CommonResponse.success("查询成功！", contractService.getPendingTotalColumnInfo(param, false));
    }

    /**
     * @Description  参数校验
     */
    @RequestMapping(value = "checkParams", method=RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ParamsCheckVO> checkParams(@RequestParam(value="purchaseType") Integer purchaseType, @RequestParam(value="sumPayMny") BigDecimal sumPayMny, @RequestParam(value="payMny") BigDecimal payMny, @RequestParam(value="contractMny") BigDecimal contractMny) {
        ParamsCheckVO paramsCheckVO = contractService.checkParams(purchaseType, sumPayMny, payMny, contractMny);
        return CommonResponse.success("参数校验成功！", paramsCheckVO);
    }
    /**
     * @Description queryBankInfo 查询银行信息
     * @param id
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryBankInfo", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PayContractVO> queryBankInfo(@RequestParam(value="contractId", required=true) Long contractId) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("tenantId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));//租户隔离
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));//单据状态已提交和审批通过
        //queryParam.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getOrderMap().put("createTime",QueryParam.DESC);
        List<PayContractEntity> contractEntities = contractService.queryList(queryParam,false);
        PayContractVO backVO = BeanMapper.map(contractEntities.get(0), PayContractVO.class);
        return CommonResponse.success("查询详情数据成功！", backVO);
    }
}
