package com.ejianc.controller;

import com.ejianc.entity.UpLoad;
import com.ejianc.entity.TemplateEntity;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentRequestVO;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.service.ITemplateService;
import feign.Response;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.*;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;

@RestController
@RequestMapping(value = "/upload")
public class UploadFileController {

    private static final Logger logger = LoggerFactory.getLogger(UploadFileController.class);

    @Autowired
    private IAttachmentApi attachmentApi;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private ITemplateService templateService;


    //pageoffice调用上传
    @PostMapping(value = "/uploadFile")
    public CommonResponse<List<AttachmentVO>> uploadFile(@RequestBody UpLoad upload) {
        logger.info("开始处理文件上传，单据类型: {}, 来源ID: {}", upload.getBillType(), upload.getSourceId());

        try {
            // 参数验证
            if (!validateUploadParams(upload)) {
                return CommonResponse.error("参数验证失败");
            }

            // 构建数据URI
            String dataUri = buildDataUri(upload.getFileName(), upload.getBase64String());

            // 删除已有文件
            deleteExistingFiles(upload);

            // 上传新文件
            return uploadNewFile(upload, dataUri);

        } catch (Exception e) {
            logger.error("文件上传失败，单据类型: {}, 来源ID: {}", upload.getBillType(), upload.getSourceId(), e);
            return CommonResponse.error("文件上传失败: " + e.getMessage());
        }
    }

    /**
     * 验证上传参数
     */
    private boolean validateUploadParams(UpLoad upload) {
        if (upload == null) {
            logger.error("上传参数为空");
            return false;
        }

        if (StringUtils.isBlank(upload.getBase64String())) {
            logger.error("base64字符串为空");
            return false;
        }

        if (StringUtils.isBlank(upload.getBillType()) ||
                StringUtils.isBlank(upload.getSourceId()) ||
                StringUtils.isBlank(upload.getSourceType())) {
            logger.error("必要参数缺失: billType={}, sourceId={}, sourceType={}",
                    upload.getBillType(), upload.getSourceId(), upload.getSourceType());
            return false;
        }

        if (!isValidBase64(upload.getBase64String())) {
            logger.error("无效的base64字符串");
            return false;
        }

        return true;
    }

    /**
     * 删除已有文件
     */
    private void deleteExistingFiles(UpLoad upload) {
        try {
            Long sourceId = Long.valueOf(upload.getSourceId());
            attachmentApi.deleteFileByParam(sourceId, upload.getBillType(), upload.getSourceType());
            logger.debug("已删除已有文件，来源ID: {}", upload.getSourceId());
        } catch (NumberFormatException e) {
            logger.warn("来源ID格式错误，跳过删除操作: {}", upload.getSourceId());
        } catch (Exception e) {
            logger.warn("删除已有文件失败，继续上传: {}", e.getMessage());
        }
    }

    /**
     * 上传新文件
     */
    private CommonResponse<List<AttachmentVO>> uploadNewFile(UpLoad upload, String dataUri) {
        List<String> fileList = new ArrayList<>();
        fileList.add(dataUri);

        AttachmentRequestVO request = new AttachmentRequestVO();
        request.setBillType(upload.getBillType());
        request.setSourceId(upload.getSourceId());
        request.setSourceType(upload.getSourceType());
        request.setFileList(fileList);

        CommonResponse<List<AttachmentVO>> response = attachmentApi.uploadForBase64(request);

        if (response.isSuccess()) {
            logger.info("文件上传成功，单据类型: {}, 来源ID: {}, 文件数量: {}",
                    upload.getBillType(), upload.getSourceId(),
                    response.getData() != null ? response.getData().size() : 0);
        } else {
            logger.error("文件上传失败，单据类型: {}, 来源ID: {}, 错误信息: {}",
                    upload.getBillType(), upload.getSourceId(), response.getData());
        }

        return response;
    }

    /**
     * 验证base64字符串是否有效
     *
     * @param base64String base64字符串
     * @return 是否有效
     */
    private boolean isValidBase64(String base64String) {
        if (StringUtils.isBlank(base64String)) {
            return false;
        }

        try {
            Base64.getDecoder().decode(base64String);
            return true;
        } catch (IllegalArgumentException e) {
            logger.debug("无效的base64字符串: {}", e.getMessage());
            return false;
        }
    }

    /**
     * 构建base64数据URI
     *
     * @param fileName     文件名
     * @param base64String base64编码的文件内容
     * @return 完整的数据URI
     */
    private String buildDataUri(String fileName, String base64String) {
        String extension = getFileExtension(fileName);
        return "data:application/" + extension + ";base64," + base64String;
    }

    /**
     * 获取文件扩展名
     *
     * @param fileName 文件名
     * @return 文件扩展名
     */
    private String getFileExtension(String fileName) {
        if (StringUtils.isBlank(fileName)) {
            return "tmp";
        }

        int lastDotIndex = fileName.lastIndexOf('.');
        if (lastDotIndex > 0 && lastDotIndex < fileName.length() - 1) {
            return fileName.substring(lastDotIndex + 1);
        }

        return "docx";
    }

    @PostMapping(value = "/uploadTemplate")
    @ResponseBody
    public void uploadFileTemplate(@RequestBody UpLoad upload) throws IOException {
        logger.info("开始处理模板文件上传，文件ID: {}, 模板名称: {}", 
                   upload.getFileId(), upload.getTemplateName());
        
        try {
            // 保存模板信息
            TemplateEntity templateEntity = createTemplateEntity(upload);
            templateService.save(templateEntity);
            logger.info("模板信息保存成功: {}", templateEntity.getTemplateName());
            
            // 下载并上传文件
            uploadFileFromAttachment(upload.getFileId());
            
        } catch (Exception e) {
            logger.error("模板文件上传失败，文件ID: {}", upload.getFileId(), e);
            throw new IOException("模板文件上传失败: " + e.getMessage(), e);
        }
    }

    /**
     * 从附件服务下载并上传文件
     */
    private void uploadFileFromAttachment(String fileId) throws IOException {
        // 下载文件
        Response response = attachmentApi.downloadFileById(Long.valueOf(fileId));
        if (response == null || response.body() == null) {
            throw new IOException("无法下载文件，文件ID: " + fileId);
        }
        
        // 获取文件信息
        CommonResponse<AttachmentVO> attachmentResponse = attachmentApi.queryDetail(fileId);
        if (!attachmentResponse.isSuccess() || attachmentResponse.getData() == null) {
            throw new IOException("无法获取文件信息，文件ID: " + fileId);
        }
        
        String fileName = attachmentResponse.getData().getFileName();
        if (StringUtils.isBlank(fileName)) {
            throw new IOException("文件名为空，文件ID: " + fileId);
        }
        
        // 转换为MultipartFile并上传
        try (InputStream inputStream = response.body().asInputStream()) {
            MultipartFile multipartFile = createMultipartFile(inputStream, fileName);
            callUploadFileInterface(multipartFile);
            logger.info("文件上传成功，文件名: {}", fileName);
        }
    }

    /**
     * 将InputStream转换为MultipartFile
     * @param inputStream 输入流
     * @param fileName 文件名
     * @return MultipartFile对象
     * @throws IOException IO异常
     */
    private MultipartFile createMultipartFile(InputStream inputStream, String fileName) throws IOException {
        // 读取输入流的所有字节 - Java 8 兼容版本
        byte[] fileBytes = readAllBytes(inputStream);
        
        return new MultipartFile() {
            @Override
            public String getName() {
                return "file";
            }

            @Override
            public String getOriginalFilename() {
                return fileName;
            }

            @Override
            public String getContentType() {
                return getContentTypeByFileName(fileName);
            }

            @Override
            public boolean isEmpty() {
                return fileBytes.length == 0;
            }

            @Override
            public long getSize() {
                return fileBytes.length;
            }

            @Override
            public byte[] getBytes() throws IOException {
                return fileBytes;
            }

            @Override
            public InputStream getInputStream() throws IOException {
                return new java.io.ByteArrayInputStream(fileBytes);
            }

            @Override
            public void transferTo(java.io.File dest) throws IOException, IllegalStateException {
                Files.write(dest.toPath(), fileBytes);
            }
        };
    }

    /**
     * 创建TemplateEntity对象
     * @param upload 上传参数
     * @return TemplateEntity对象
     */
    private TemplateEntity createTemplateEntity(UpLoad upload) {
        TemplateEntity  templateEntity = new TemplateEntity();
        
        // 设置基本信息
        templateEntity.setTemplateName(upload.getTemplateName());
        templateEntity.setCategoryName(upload.getCategoryName());
        templateEntity.setBelongOrgName(upload.getBelongOrgName());
        templateEntity.setFileName(upload.getFileName());
        
        // 设置文件ID
        if (StringUtils.isNotBlank(upload.getFileId())) {
            templateEntity.setFileId(Long.valueOf(upload.getFileId()));
        }
        
        // 设置默认值
        templateEntity.setEnableStatus(1); // 默认启用
        templateEntity.setSequence(1); // 默认排序号
        
        // 根据文件名设置文件类型
        if (StringUtils.isNotBlank(upload.getFileName())) {
            String fileType = getFileTypeByFileName(upload.getFileName());
            templateEntity.setFileType(fileType);
        }
        
        logger.info("TemplateEntity创建完成: 模板名称={}, 分类名称={}, 所属组织={}, 文件名={}", 
                   templateEntity.getTemplateName(), templateEntity.getCategoryName(), 
                   templateEntity.getBelongOrgName(), templateEntity.getFileName());
        
        return templateEntity;
    }

    /**
     * 根据文件名获取文件类型
     * @param fileName 文件名
     * @return 文件类型
     */
    private String getFileTypeByFileName(String fileName) {
        if (StringUtils.isBlank(fileName)) {
            return "w"; // 默认word类型
        }

        String extension = getFileExtension(fileName).toLowerCase();
        switch (extension) {
            case "pdf":
                return "f"; // pdf
            case "doc":
            case "docx":
                return "w"; // word
            case "xls":
            case "xlsx":
                return "s"; // excel
            case "ppt":
            case "pptx":
                return "p"; // powerpoint
            default:
                return "w"; // 默认word类型
        }
    }

    /**
     * Java 8 兼容的读取所有字节方法
     * @param inputStream 输入流
     * @return 字节数组
     * @throws IOException IO异常
     */
    private byte[] readAllBytes(InputStream inputStream) throws IOException {
        java.io.ByteArrayOutputStream buffer = new java.io.ByteArrayOutputStream();
        int nRead;
        byte[] data = new byte[1024];
        while ((nRead = inputStream.read(data, 0, data.length)) != -1) {
            buffer.write(data, 0, nRead);
        }
        buffer.flush();
        return buffer.toByteArray();
    }

    /**
     * 根据文件名获取Content-Type
     * @param fileName 文件名
     * @return Content-Type
     */
    private String getContentTypeByFileName(String fileName) {
        if (StringUtils.isBlank(fileName)) {
            return "application/octet-stream";
        }

        String extension = getFileExtension(fileName).toLowerCase();
        switch (extension) {
            case "pdf":
                return "application/pdf";
            case "doc":
                return "application/msword";
            case "docx":
                return "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
            case "xls":
                return "application/vnd.ms-excel";
            case "xlsx":
                return "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";
            case "txt":
                return "text/plain";
            case "jpg":
            case "jpeg":
                return "image/jpeg";
            case "png":
                return "image/png";
            case "gif":
                return "image/gif";
            default:
                return "application/octet-stream";
        }
    }

    /**
     * 调用upload/uploadFile接口
     * @param multipartFile 文件对象
     */
    private void callUploadFileInterface(MultipartFile multipartFile) {
        try {
            // 设置请求头
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.MULTIPART_FORM_DATA);

            // 创建文件资源
            ByteArrayResource fileResource = new ByteArrayResource(multipartFile.getBytes()) {
                @Override
                public String getFilename() {
                    return multipartFile.getOriginalFilename();
                }
            };

            // 构建请求体
            MultiValueMap<String, Object> body = new LinkedMultiValueMap<>();
            body.add("file", fileResource);

            // 创建HTTP实体
            HttpEntity<MultiValueMap<String, Object>> requestEntity = new HttpEntity<>(body, headers);

            // 调用接口 - 这里需要根据实际情况修改URL
            // 如果目标接口在同一个应用中，可以使用相对路径
            String uploadUrl = "http://10.136.208.154:8080/upload/uploadFile";
            
            // 如果目标接口在其他服务中，需要使用完整的URL
            //String uploadUrl = "http://localhost:8081/upload/uploadFile";
            
            ResponseEntity<String> response = restTemplate.exchange(
                uploadUrl,
                HttpMethod.POST,
                requestEntity,
                String.class
            );

            // 处理响应
            if (response.getStatusCode() == HttpStatus.OK) {
                logger.info("文件上传成功，响应: {}", response.getBody());
            } else {
                logger.error("文件上传失败，状态码: {}, 响应: {}", response.getStatusCode(), response.getBody());
            }

        } catch (Exception e) {
            logger.error("调用上传接口时发生异常", e);
        }
    }

}

