package com.ejianc.controller;

import com.ejianc.entity.UpLoad;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.service.IUploadFileService;
import com.ejianc.util.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.util.List;

/**
 * 文件上传控制器
 * @author: 系统生成
 * @version: 1
 */
@RestController
@RequestMapping(value = "/upload")
public class UploadFileController {

    private static final Logger logger = LoggerFactory.getLogger(UploadFileController.class);

    @Autowired
    private IUploadFileService uploadFileService;

    /**
     * PageOffice调用上传
     * @param upload 上传参数
     * @return 上传结果
     */
    @PostMapping(value = "/uploadFile")
    public CommonResponse<List<AttachmentVO>> uploadFile(@RequestBody UpLoad upload) {
        logger.info("接收到文件上传请求，单据类型: {}, 来源ID: {}", upload.getBillType(), upload.getSourceId());
        
        // 参数校验
        CommonResponse<List<AttachmentVO>> validationResult = validateUploadFileParams(upload);
        if (!validationResult.isSuccess()) {
            return validationResult;
        }
        
        return uploadFileService.uploadFile(upload);
    }

    /**
     * 上传模板文件
     * @param upload 上传参数
     * @throws IOException IO异常
     */
    @PostMapping(value = "/uploadTemplate")
    @ResponseBody
    public CommonResponse<Void> uploadFileTemplate(@RequestBody UpLoad upload) throws IOException {
        logger.info("接收到模板文件上传请求，文件ID: {}, 模板名称: {}", 
                   upload.getFileId(), upload.getTemplateName());
        
        // 参数校验
        CommonResponse<Void> validationResult = validateTemplateParams(upload);
        if (!validationResult.isSuccess()) {
            return validationResult;
        }
        
        uploadFileService.uploadTemplate(upload);
        return CommonResponse.success("模板文件上传成功");
    }

    /**
     * 删除模板文件
     * @param upload 删除参数
     * @throws IOException IO异常
     */
    @PostMapping(value = "/deleteTemplate")
    @ResponseBody
    public CommonResponse<Void> deleteTemplate(@RequestBody UpLoad upload) throws IOException {
        logger.info("接收到模板删除请求，文件ID: {}, 模板名称: {}",
                upload.getFileId(), upload.getFileName());
        
        // 参数校验
        CommonResponse<Void> validationResult = validateDeleteTemplateParams(upload);
        if (!validationResult.isSuccess()) {
            return validationResult;
        }
        
        uploadFileService.deleteTemplate(upload);
        return CommonResponse.success("模板文件删除成功");
    }

    /**
     * 校验文件上传参数
     * @param upload 上传参数
     * @return 校验结果
     */
    private CommonResponse<List<AttachmentVO>> validateUploadFileParams(UpLoad upload) {
        // 校验基础参数
        if (upload == null) {
            logger.error("上传参数为空");
            return CommonResponse.error("上传参数不能为空");
        }

        // 校验base64字符串
        if (StringUtils.isBlank(upload.getBase64String())) {
            logger.error("base64字符串为空");
            return CommonResponse.error("base64字符串不能为空");
        }

        // 校验必要参数
        if (StringUtils.isBlank(upload.getBillType()) ||
                StringUtils.isBlank(upload.getSourceId()) ||
                StringUtils.isBlank(upload.getSourceType())) {
            logger.error("必要参数缺失: billType={}, sourceId={}, sourceType={}",
                    upload.getBillType(), upload.getSourceId(), upload.getSourceType());
            return CommonResponse.error("必要参数缺失: 单据类型、来源ID、来源类型不能为空");
        }

        // 校验base64字符串有效性
        if (!FileUtils.isValidBase64(upload.getBase64String())) {
            logger.error("无效的base64字符串");
            return CommonResponse.error("无效的base64字符串格式");
        }

        logger.info("文件上传参数校验通过");
        return CommonResponse.success("参数校验通过", null);
    }

    /**
     * 校验模板上传参数
     * @param upload 上传参数
     * @return 校验结果
     */
    private CommonResponse<Void> validateTemplateParams(UpLoad upload) {
        // 校验基础参数
        if (upload == null) {
            logger.warn("模板上传参数为空");
            return CommonResponse.error("模板上传参数不能为空");
        }

        // 校验文件ID
        if (StringUtils.isBlank(upload.getFileId())) {
            logger.warn("模板文件ID为空");
            return CommonResponse.error("模板文件ID不能为空");
        }

        // 校验文件ID格式
        try {
            Long.valueOf(upload.getFileId());
        } catch (NumberFormatException e) {
            logger.warn("无效的模板文件ID格式: {}", upload.getFileId());
            return CommonResponse.error("无效的模板文件ID格式");
        }

        // 校验模板名称
        if (StringUtils.isBlank(upload.getTemplateName())) {
            logger.warn("模板名称为空");
            return CommonResponse.error("模板名称不能为空");
        }

        // 校验分类名称
        if (StringUtils.isBlank(upload.getCategoryName())) {
            logger.warn("分类名称为空");
            return CommonResponse.error("分类名称不能为空");
        }

        // 校验所属组织名称
        if (StringUtils.isBlank(upload.getBelongOrgName())) {
            logger.warn("所属组织名称为空");
            return CommonResponse.error("所属组织名称不能为空");
        }

        logger.info("模板上传参数校验通过");
        return CommonResponse.success("参数校验通过");
    }

    /**
     * 校验模板删除参数
     * @param upload 删除参数
     * @return 校验结果
     */
    private CommonResponse<Void> validateDeleteTemplateParams(UpLoad upload) {
        // 校验基础参数
        if (upload == null) {
            logger.warn("模板删除参数为空");
            return CommonResponse.error("模板删除参数不能为空");
        }

        // 校验文件ID
        if (StringUtils.isBlank(upload.getFileId())) {
            logger.warn("删除模板文件ID为空");
            return CommonResponse.error("删除模板文件ID不能为空");
        }

        // 校验文件ID格式
        try {
            Long.valueOf(upload.getFileId());
        } catch (NumberFormatException e) {
            logger.warn("无效的删除模板文件ID格式: {}", upload.getFileId());
            return CommonResponse.error("无效的删除模板文件ID格式");
        }

        // 校验模板名称
        if (StringUtils.isBlank(upload.getFileName())) {
            logger.warn("删除模板名称为空");
            return CommonResponse.error("删除模板名称不能为空");
        }

        logger.info("模板删除参数校验通过");
        return CommonResponse.success("参数校验通过");
    }
}

