package com.ejianc.foundation.cache;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.permission.bean.ButtonEntity;
import com.ejianc.foundation.permission.bean.RolePostRelationEntity;
import com.ejianc.foundation.permission.bean.RoleUserRelationEntity;
import com.ejianc.foundation.permission.service.IButtonService;
import com.ejianc.foundation.permission.vo.ButtonVO;
import com.ejianc.foundation.permission.vo.RolePostRelationVO;
import com.ejianc.framework.core.kit.collection.ListUtil;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.ejianc.foundation.permission.vo.AppVO;
import com.ejianc.foundation.permission.vo.RoleUserRelationVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.cache.serializer.Serializer;
import com.ejianc.framework.cache.serializer.impl.DefaultJDKSerializer;

import redis.clients.jedis.Jedis;

@Component
public class PermissionCacheManage {

	public static final String DEFAULT_CHARSET = "UTF-8";
	//当前用户的角色KEY
	private final static String CURRENTUSER_ROLE_KEY = "currentuser_role_key:";
	//当前角色的应用KEY
	private final static String CURRENTROLE_APP_KEY = "currentrole_app_key:";
	//查看当前角色的用户KEY
	private final static String CURRENTROLE_USER_KEY = "currentrole_user_key:";
	//当前角色的应用按钮KEY
	private final static String CURRENTROLE_APP_BUTTON_KEY = "currentrole_app_button_key:";
	//当前角色的岗位字典KEY
	private final static String CURRENTROLE_POST_KEY = "currentrole_post_key:";


	private Serializer serializer = new DefaultJDKSerializer();
	
	@Autowired
	private CacheManager cacheManager;
	@Autowired
	private IButtonService buttonService;
	@Autowired(required=false)
	private SessionManager sessionManager;
	/**
	 * 向redis中存放当前用户的角色id
	 * 
	 * @param userId
	 * @param
	 */
	public void putUserRoleCache(Long userId, Long tenantId, List<Long> roleIdList) {
		String key = CURRENTUSER_ROLE_KEY + userId + "-" + tenantId;
		cacheManager.removeCache(key);
		if(roleIdList != null && roleIdList.size() > 0) {
			cacheManager.piplineExecute(pipeline -> {
				pipeline.multi();
				for(Long roleId:roleIdList) {
					byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
	                byte[] fieldBytes = roleId.toString().getBytes(Charset.forName(DEFAULT_CHARSET));
					pipeline.sadd(keyBytes, fieldBytes);
				}
				pipeline.exec();
			});
		}
	}
	
	/**
	 * 从redis中获取该用户的角色ids
	 * 
	 * @param userId
	 * @return
	 */
	public List<Long> getUserRoleCache(Long userId, Long tenantId) {
		List<Long> resultList = new ArrayList<Long>();
		String key = CURRENTUSER_ROLE_KEY + userId + "-" + tenantId;
		Set<String> roleIds = cacheManager.execute(new CacheManager.JedisAction<Set<String>>() {
			@Override
			public Set<String> action(Jedis jedis) {
				return jedis.smembers(key);
			}
		});
		if(roleIds != null && !roleIds.isEmpty()) {
			for(String roleId:roleIds) {
				resultList.add(Long.parseLong(roleId));
			}
		}
		return resultList;
	}
	
	
	/**
	 * 向redis中存放当前角色的应用列表
	 * 
	 * @param roleId
	 * @param
	 */
	public void putRoleAppsCache(Long roleId, List<AppVO> appVos) {
		if(appVos != null && appVos.size() > 0) {
			cacheManager.piplineExecute(pipeline -> {
				pipeline.multi();
				String key = CURRENTROLE_APP_KEY + roleId;
				String keyButton = CURRENTROLE_APP_BUTTON_KEY + roleId;
				for(AppVO appVo:appVos) {
					byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
	                byte[] fieldBytes = appVo.getId().toString().getBytes(Charset.forName(DEFAULT_CHARSET));
	                byte[] contentBytes = serializer.marshalToByte(appVo.getId().toString());
					pipeline.hset(keyBytes, fieldBytes, contentBytes);
					if(ListUtil.isNotEmpty(appVo.getChildren())){
						for(ButtonVO btn:appVo.getChildren()){
							ButtonEntity buttonEntity =  buttonService.selectById(btn.getId());
							keyBytes = keyButton.getBytes(Charset.forName(DEFAULT_CHARSET));
							fieldBytes = (buttonEntity.getPageCode()+buttonEntity.getCode()).getBytes(Charset.forName(DEFAULT_CHARSET));
							contentBytes = serializer.marshalToByte(JSONObject.toJSONString(buttonEntity));
							pipeline.hset(keyBytes, fieldBytes, contentBytes);
						}
					}
				}
				pipeline.exec();
			});
		}
	}
	
	/**
	 * 删除redis中对应的角色下的应用
	 * 
	 * @param roleId
	 * @param appId
	 */
	public void removeRoleAppsCache(Long roleId, Long appId) {
		String key = CURRENTROLE_APP_KEY + roleId;
		cacheManager.hdel(key, appId.toString());
	}

	/**
	 * 删除redis中对应的角色下的应用按钮
	 *
	 * @param roleId
	 * @param btnKey
	 */
	public void removeRoleAppsBtnCache(Long roleId, String btnKey) {
		String key = CURRENTROLE_APP_BUTTON_KEY + roleId;
		cacheManager.hdel(key, btnKey);
	}
	
	/**
	 * 从redis中获取当前角色的所有应用
	 * 
	 * @param roleId
	 * @return
	 */
	public List<Long> getRoleAppsCache(Long roleId) {
		List<Long> resultList = new ArrayList<>();
		String key = CURRENTROLE_APP_KEY + roleId;
		Set<String> appIds = cacheManager.execute(new CacheManager.JedisAction<Set<String>>() {
			@Override
			public Set<String> action(Jedis jedis) {
				Set<String> resultSet = new HashSet<>();
				Map<String, String> resultMap = jedis.hgetAll(key);
				if(!resultMap.isEmpty()) {
					for(Map.Entry<String, String> entry: resultMap.entrySet()) {
						resultSet.add(entry.getKey());
					}
				}
				return resultSet;
			}
		});
		if(appIds != null && !appIds.isEmpty()) {
			for(String appId:appIds) {
				resultList.add(Long.parseLong(appId));
			}
		}
		return resultList;
	}
	
	/**
	 * 判断当前应用是否在 角色应用缓存中
	 * 
	 * @param roleId
	 * @param
	 * @return
	 */
	public boolean isExistInRoleAppBtnCache(Long roleId, String btnkey) {
		String key = CURRENTROLE_APP_BUTTON_KEY + roleId;
		return cacheManager.hexists(key, btnkey);
	}
	/**
	 * 判断当前应用是否在 角色应用缓存中
	 *
	 * @param roleId
	 * @param appId
	 * @return
	 */
	public boolean isExistInRoleAppsCache(Long roleId, Long appId) {
		String key = CURRENTROLE_APP_KEY + roleId;
		return cacheManager.hexists(key, appId.toString());
	}
	
	/**
	 * 向redis中存放当前角色的用户信息
	 * 
	 * @param roleId
	 * @param
	 */
	public void putOneRoleUsersCache(Long roleId, RoleUserRelationVO roleUserRelationVo) {
		String key = CURRENTROLE_USER_KEY + roleId;
			
		Map<String, String> resultMap = new HashMap<String, String>();
		String authOrgId = roleUserRelationVo.getAuthOrgId().toString();
		resultMap.put(roleUserRelationVo.getUserId().toString(), authOrgId);
		String cacheValue = cacheManager.hget(key, roleUserRelationVo.getUserId().toString());
		if(StringUtils.isNotBlank(cacheValue)) {
			if(cacheValue.indexOf(authOrgId) == -1) {
				resultMap.put(roleUserRelationVo.getUserId().toString(), authOrgId + "," + cacheValue);
			}
		}else{
			resultMap.put(roleUserRelationVo.getUserId().toString(), authOrgId+"");
		}
		
		cacheManager.piplineExecute(pipeline -> {
			pipeline.multi();
			for(Map.Entry<String, String> entry:resultMap.entrySet()) {
				byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
                byte[] fieldBytes = entry.getKey().getBytes(Charset.forName(DEFAULT_CHARSET));
                byte[] contentBytes = serializer.marshalToByte(entry.getValue());
				pipeline.hset(keyBytes, fieldBytes, contentBytes);
			}
			pipeline.exec();
		});
	}
	
	/**
	 * 向redis中存放当前角色的所有用户信息
	 * 
	 * @param roleId
	 * @param roleUserRelations
	 */
	public void putRoleUsersCache(Long roleId, List<RoleUserRelationEntity> roleUserRelations) {
		if(roleUserRelations != null && roleUserRelations.size() > 0) {
			String key = CURRENTROLE_USER_KEY + roleId;
			
			Map<String, String> resultMap = new HashMap<String, String>();
			for(RoleUserRelationEntity roleUserRelation:roleUserRelations) {
				String value = resultMap.get(roleUserRelation.getUserId().toString());
				if(StringUtils.isNotBlank(value)) {
					resultMap.put(roleUserRelation.getUserId().toString(), value+","+roleUserRelation.getAuthOrgId().toString());
				}else{
					resultMap.put(roleUserRelation.getUserId().toString(), roleUserRelation.getAuthOrgId().toString());
				}
				String cacheValue = cacheManager.hget(key, roleUserRelation.getUserId().toString());
				if(StringUtils.isNotBlank(cacheValue)) {
					resultMap.put(roleUserRelation.getUserId().toString(), resultMap.get(roleUserRelation.getUserId().toString()) + "," + cacheValue);
				}
			}
			
			cacheManager.piplineExecute(pipeline -> {
				pipeline.multi();
				for(Map.Entry<String, String> entry:resultMap.entrySet()) {
					byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
	                byte[] fieldBytes = entry.getKey().getBytes(Charset.forName(DEFAULT_CHARSET));
	                byte[] contentBytes = serializer.marshalToByte(entry.getValue());
					pipeline.hset(keyBytes, fieldBytes, contentBytes);
				}
				pipeline.exec();
			});
		}
	}
	
	/**
	 * 向redis中删除对应用户的缓存
	 * 
	 * @param roleId
	 * @param userId
	 */
	public void removeRoleUsersCache(Long roleId, Long userId, Long authOrgId) {
		String key = CURRENTROLE_USER_KEY + roleId;
		String cacheValue = cacheManager.hget(key, userId.toString());
		if(StringUtils.isNotBlank(cacheValue)) {
			if(authOrgId != null && authOrgId.toString().equals(cacheValue)) {
				cacheManager.hdel(key, userId.toString());
			}else{
				String[] values = cacheValue.split(",");
				String newOrgIds = "";
				for(String value:values) {
					if(authOrgId != null && !authOrgId.toString().equals(value)) {
						newOrgIds += value + ",";
					}
				}
				if(StringUtils.isNotBlank(newOrgIds)) {
					cacheManager.hset(key, userId.toString(), newOrgIds.substring(0, newOrgIds.length()-1));
				}
			}
		}
	}




	/**
	 * 向redis中存放当前角色的所有岗位信息
	 *
	 * @param roleId
	 *
	 */
	public void putRolePostsCache(Long roleId, List<RolePostRelationEntity> rolePostRelations) {
		if(rolePostRelations != null && rolePostRelations.size() > 0) {
			String key = CURRENTROLE_POST_KEY + roleId;
			cacheManager.piplineExecute(pipeline -> {
				pipeline.multi();
				for(RolePostRelationEntity rolePostRelation:rolePostRelations) {
					byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
					byte[] fieldBytes = String.valueOf(rolePostRelation.getPostId()).getBytes(Charset.forName(DEFAULT_CHARSET));
					pipeline.hset(keyBytes, fieldBytes,new byte[0]);
				}
				pipeline.exec();
			});
		}
	}

	/**
	 * 向redis中删除对应岗位的缓存
	 *
	 * @param roleId
	 *
	 */
	public void removeRolePostsCache(Long roleId, Long postId) {
		String key = CURRENTROLE_POST_KEY + roleId;
		cacheManager.hdel(key, postId.toString());
	}


	/**
	 * 获取用户的所有任职组织
	 * 
	 * @param roleId
	 * @param userId
	 * @return
	 */
	public List<Long> getRoleUserIdsCache(Long roleId, Long userId) {
		List<Long> resultList = new ArrayList<>();
		
		String key = CURRENTROLE_USER_KEY + roleId;
		String authOrgs = cacheManager.hget(key, userId.toString());
		
		if(StringUtils.isNotBlank(authOrgs)) {
			String[] orgIds = authOrgs.split(",");
			for(String orgId:orgIds) {
				resultList.add(Long.parseLong(orgId));
			}
		}
		UserContext userContext = sessionManager.getUserContext();
		JSONObject contextExt = userContext.getExt();
		String cacheKey = "user_role_post_cache_"+roleId+"_"+userId;
		if(contextExt != null) {
			String cacheValue = contextExt.getString(cacheKey);
			if(StringUtils.isNotBlank(cacheValue)) {
				String[] cacheAddr = cacheValue.split(",");
				for(String cacheStr:cacheAddr) {
					resultList.add(Long.parseLong(cacheStr));
				}
			}
		}
		return resultList;
	}
}
