package com.ejianc.business.income.controller;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.income.service.IContractService;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.ParsedStringTerms;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.ParsedSum;
import org.elasticsearch.search.aggregations.metrics.SumAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import java.io.IOException;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @description: HomeStatisticController
 * @author songlx
 * @date 2024/11/11
 * @version 1.0
 */
@Controller
@RequestMapping("homeStatistic")
public class HomeStatisticController implements Serializable {
    private static final long serialVersionUID = 1L;
    private Logger logger = LoggerFactory.getLogger(this.getClass());


    @Autowired
    private IContractService contractService;

    @Autowired(required = false)
    private RestHighLevelClient client;


    /**
     * @Description 合同额信息管理
     */
    @RequestMapping(value = "/contractInfo", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> contractInfo() {
        return CommonResponse.success("合同额信息管理：本年数据", contractService.contractInfo());
    }



    /**
     * @Description 投标数据
     */
    @RequestMapping(value = "/tenderInfo", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> tenderInfo() {
        return CommonResponse.success("目标成本管理-投标净利润、游标利润率、投标成本", contractService.tenderInfo());
    }

    /**
     * @Description 目标成本区域利润折线图
     * 投标净利润率（%）：施工合同增加字段    穿透施工合同
     *
     * 目标净利润率（%）：（预算书总金额累计-目标成本总金额累计）/预算书总金额累计
     *
     * 实际净利润率（%）：（项目实际金额-实际执行成本金额）/项目实际金额
     */
    @RequestMapping(value = "/costInfo", method = RequestMethod.GET)
    @ResponseBody
    private CommonResponse<List<JSONObject>> costInfo() {
        ArrayList<JSONObject> resVO = new ArrayList<>();
        List<String> sumList = Arrays.asList(
                "projectNum", // 项目个数、施工合同数
                "tenderProfitRate", // 投标利润率
                "baseTotalMoney", // 预算总额
                "targetCostMny", // 目标成本总额
                "CALCbec9d61477", // 项目实际金额
                "CALC1d3918a8dd" //  实际执行成本金额
        );

        SearchRequest searchRequest = new SearchRequest("project_dynamic_cost");
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid()));

        TermsAggregationBuilder termsAggregationBuilder = AggregationBuilders.terms("areaName").field("orgName.keyword");

        for (String key : sumList) {
            SumAggregationBuilder aggregation = AggregationBuilders
                    .sum("sum_" + key)
                    .field(key)
                    .missing(0);
            termsAggregationBuilder.subAggregation(aggregation);
        }
        sourceBuilder.aggregation(termsAggregationBuilder);
        sourceBuilder.query(boolQuery);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(60, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = response.getAggregations();

            ParsedStringTerms sourceWay = aggregations.get("areaName");
            for (Terms.Bucket bucket : sourceWay.getBuckets()) {
                JSONObject obj = new JSONObject();
                obj.put("areaName", bucket.getKeyAsString());
                for (String key : sumList) {
                    ParsedSum sum_mny = bucket.getAggregations().get("sum_" + key);
                    obj.put("sum_" + key, ComputeUtil.toBigDecimal(sum_mny.getValue()));
                }
                // 投标净利润率（%）：施工合同投标净利润率合计/施工合同数 求平均数
                BigDecimal tenderProfitRate = ComputeUtil.bigDecimalPercent(ComputeUtil.toBigDecimal(obj.get("sum_tenderProfitRate")), ComputeUtil.toBigDecimal(obj.get("sum_projectNum")), 2);
                obj.put("tenderProfitRate", tenderProfitRate);
                // 目标净利润率（%）：（预算书总金额累计-目标成本总金额累计）/预算书总金额累计
                BigDecimal targetSub = ComputeUtil.safeSub(ComputeUtil.toBigDecimal(obj.get("sum_tenderProfitRate")), ComputeUtil.toBigDecimal(obj.get("sum_projectNum")));
                BigDecimal targetProfitRate = ComputeUtil.bigDecimalPercent(targetSub, ComputeUtil.toBigDecimal(obj.get("sum_baseTotalMoney")), 2);
                obj.put("targetProfitRate", targetProfitRate);
                // 实际净利润率（%）：（项目实际金额-实际执行成本金额）/项目实际金额
                BigDecimal realSub = ComputeUtil.safeSub(ComputeUtil.toBigDecimal(obj.get("sum_CALCbec9d61477")), ComputeUtil.toBigDecimal(obj.get("sum_CALC1d3918a8dd")));
                BigDecimal realProfitRate = ComputeUtil.bigDecimalPercent(realSub, ComputeUtil.toBigDecimal(obj.get("sum_CALCbec9d61477")), 2);
                obj.put("realProfitRate", realProfitRate);
                resVO.add(obj);
            }

        } catch (IOException e) {
            e.printStackTrace();
        }

        return CommonResponse.success(resVO);
    }



}
