package com.ejianc.business.income.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.finance.api.IReceiveManageApi;
import com.ejianc.business.finance.vo.SumReceiveVO;
import com.ejianc.business.income.bean.*;
import com.ejianc.business.income.consts.EJCDateUtil;
import com.ejianc.business.income.consts.FinanceUseConsts;
import com.ejianc.business.income.consts.ProjectSurveyEnum;
import com.ejianc.business.income.history.ProjectProgressVo;
import com.ejianc.business.income.mapper.ContractMapper;
import com.ejianc.business.income.mapper.ProductionMapper;
import com.ejianc.business.income.mapper.QuoteMapper;
import com.ejianc.business.income.service.*;
import com.ejianc.business.income.utils.TreeNodeBUtil;
import com.ejianc.business.income.vo.*;
import com.ejianc.business.income.vo.comparator.ContractDetailComparatorVo;
import com.ejianc.business.income.vo.report.FinanceUseResSubVO;
import com.ejianc.business.income.vo.report.FinanceUseResVO;
import com.ejianc.business.income.vo.warn.IncomeContractWarnVo;
import com.ejianc.business.income.vo.warn.QualityAssuranceEndDateWarnVo;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.FilterOrgIdVo;
import com.ejianc.business.tax.api.IInvoiceApi;
import com.ejianc.business.tax.vo.InvoiceOpenRecordVO;
import com.ejianc.business.utils.BigDecimalUtils;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateUtil;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.DateFormater;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.metrics.ParsedSum;
import org.elasticsearch.search.aggregations.metrics.SumAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.io.IOException;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.ejianc.business.income.utils.AddressResolutionUtil.*;

/**
 * <p>
 * 施工合同表 服务实现类
 * </p>
 *
 * @author yuezx
 * @since 2020-05-28
 */
@Service("ContractService")
public class ContractServiceImpl extends BaseServiceImpl<ContractMapper, ContractEntity> implements IContractService {
    private static final String INCOME_CONTRACT_BILL_CODE = "INCOME_CONTRACT";
    @Autowired
    private ContractMapper contractMapper;
    @Autowired
    private IContractDetailService detailsService;
    @Autowired
    private IContractClauseService clauseService;
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IReceiveManageApi receiveManageApi;
    @Autowired
    private IInvoiceApi invoiceApi;
    @Autowired
    private IUserApi userApi;
    @Autowired
    private QuoteMapper quoteMapper;
    @Autowired
    private IQuoteService quoteService;

    @Autowired(required = false)
    private RestHighLevelClient client;

    @Autowired
    private IOrgApi iOrgApi;
    @Autowired
    private IContractFinishSettleService finishSettleService;
    @Autowired
    private IProjectApi projectApi;

    @Autowired
    private ProductionMapper productionMapper;

    @Override
    public CommonResponse<FinanceUseResVO> queryFinanceUse(String dateIn, List<Long> orgIds) {
        String startMonth = null;
        String endMonth = null;
        if (FinanceUseConsts.TIME_RANGE_THIS_YEAR.equals(dateIn)) {
            startMonth = EJCDateUtil.getYear() + "-01";
            endMonth = EJCDateUtil.getYear() + "-12";
        } else if (FinanceUseConsts.TIME_RANGE_3_YEAR.equals(dateIn)) {
            startMonth = EJCDateUtil.getMonth(-35);
            endMonth = EJCDateUtil.getMonth(0);
        } else if (FinanceUseConsts.TIME_RANGE_3_MONTH.equals(dateIn)) {
            startMonth = EJCDateUtil.getMonth(-2);
            endMonth = EJCDateUtil.getMonth(0);
        } else if (FinanceUseConsts.TIME_RANGE_6_MONTH.equals(dateIn)) {
            startMonth = EJCDateUtil.getMonth(-5);
            endMonth = EJCDateUtil.getMonth(0);
        } else {
            startMonth = EJCDateUtil.getLastYear() + "-01";
            endMonth = EJCDateUtil.getLastYear() + "-12";
        }

        Long tenantId = InvocationInfoProxy.getTenantid();

        List<FinanceUseResSubVO> proRes = productionMapper.totalProductionMoney(startMonth, endMonth, orgIds, tenantId);
        List<FinanceUseResSubVO> openRes = contractMapper.totalOpenInvince(startMonth, endMonth, orgIds, tenantId);

        List<FinanceUseResSubVO> recRes = contractMapper.totalReceiveInvince(startMonth, endMonth, orgIds, tenantId);

        List<FinanceUseResSubVO> quoteRes = quoteMapper.totalFinQuoteMoney(startMonth, endMonth, orgIds, tenantId);

        List<FinanceUseResSubVO> recMoneyRes = contractMapper.totalFinReceiveMoney(startMonth, endMonth, orgIds, tenantId);

        List<FinanceUseResSubVO> payRes = contractMapper.totalFinOutMoney(startMonth, endMonth, orgIds, tenantId);

        List<FinanceUseResSubVO> payContractRes = contractMapper.totalPayContractFinOutMoney(startMonth, endMonth, orgIds, tenantId);

        List<FinanceUseResSubVO> settleRes = contractMapper.totalFinSettleMoney(startMonth, endMonth, orgIds, tenantId);

        Map<String, BigDecimal> prodMap = proRes.stream().collect(Collectors.toMap(FinanceUseResSubVO::getMon, FinanceUseResSubVO::getTotal));

        Map<String, BigDecimal> openInvoiceMap = openRes.stream().collect(Collectors.toMap(FinanceUseResSubVO::getMon, FinanceUseResSubVO::getTotal));

        Map<String, BigDecimal> recInvoiceMap = recRes.stream().collect(Collectors.toMap(FinanceUseResSubVO::getMon, FinanceUseResSubVO::getTotal));

        Map<String, BigDecimal> quoteMap = quoteRes.stream().collect(Collectors.toMap(FinanceUseResSubVO::getMon, FinanceUseResSubVO::getTotal));

        Map<String, BigDecimal> recMoneyMap = recMoneyRes.stream().collect(Collectors.toMap(FinanceUseResSubVO::getMon, FinanceUseResSubVO::getTotal));

        Map<String, BigDecimal> payMap = payRes.stream().collect(Collectors.toMap(FinanceUseResSubVO::getMon, FinanceUseResSubVO::getTotal));

        Map<String, BigDecimal> settleMap = settleRes.stream().collect(Collectors.toMap(FinanceUseResSubVO::getMon, FinanceUseResSubVO::getTotal));


        Map<String, BigDecimal> payContractMap = payContractRes.stream().collect(Collectors.toMap(FinanceUseResSubVO::getMon, FinanceUseResSubVO::getTotal));


        FinanceUseResVO resVO = new FinanceUseResVO();
        List<String> monthData = EJCDateUtil.getMonthBetween(startMonth, endMonth);
        HashMap<String, FinanceUseResSubVO> subVOMap = new HashMap<>();

        /**
         * @param dateIn
         * @description: 公司领导门户资金使用
         * * "production":产值统计1
         * * "partAReport":甲方报量1
         * * "yingshouweishou":应收未收 = 甲方报量 - 累计收款
         * * "backMoney":回款 = 累计收款1
         * * "openInvince":"开票
         * * "jiesuanjine":结算金额1
         * * "receiveInvince":收票金额
         * * "payMoney":"付款金额1
         * * "yingfuweifu":"应付未付 = 结算金额 - 付款金额
         * * "zhangmianjine":账面金额 = 回款 - 付款金额
         * * "monthData":['2021-09', '2021-10', '2021-11', '2021-12', '2022-01', '2022-02', '2022-03']
         * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.income.vo.FinanceUseVO>
         * @author songlx
         * @date: 2022/4/14
         */

        //合计字段
        List<String> keyList = Arrays.asList(
                "production",
                "partAReport",
                "backMoney",
                "openInvince",
                "jiesuanjine",
                "receiveInvince",
                "payMoney",
                "yingshouweishou",
                "yingfuweifu",
                "zhangmianjine"
        );

        for (String mon : monthData) {
            BigDecimal monVal = BigDecimal.ZERO;
            for (String key : keyList) {
                //产值
                if (key.equals(keyList.get(0))) {
                    monVal = ComputeUtil.nullToZero(prodMap.get(mon));
                }
                //甲方报量
                if (key.equals(keyList.get(1))) {
                    monVal = ComputeUtil.nullToZero(quoteMap.get(mon));
                }
                //收款
                if (key.equals(keyList.get(2))) {
                    monVal = ComputeUtil.nullToZero(recMoneyMap.get(mon));
                }
                //开票
                if (key.equals(keyList.get(3))) {
                    monVal = ComputeUtil.nullToZero(openInvoiceMap.get(mon));
                }
                //结算
                if (key.equals(keyList.get(4))) {
                    monVal = ComputeUtil.nullToZero(settleMap.get(mon));
                }
                //收票
                if (key.equals(keyList.get(5))) {
                    monVal = ComputeUtil.nullToZero(recInvoiceMap.get(mon));
                }
                //付款
                if (key.equals(keyList.get(6))) {
                    monVal = ComputeUtil.nullToZero(payMap.get(mon));
                }
                //应收未收 = 甲方报量 - 累计收款
                if (key.equals(keyList.get(7))) {
                    monVal = ComputeUtil.safeSub(quoteMap.get(mon), recMoneyMap.get(mon));
                }

                // 应付未付 = 结算金额 - 合同付款金额
                if (key.equals(keyList.get(8))) {
                    monVal = ComputeUtil.safeSub(settleMap.get(mon), payContractMap.get(mon));
                }
                // 账面金额 = 回款 - 付款金额
                if (key.equals(keyList.get(9))) {
                    monVal = ComputeUtil.safeSub(recMoneyMap.get(mon), payMap.get(mon));
                }

                FinanceUseResSubVO subVO = subVOMap.get(key);
                if (subVO == null) {
                    subVO = new FinanceUseResSubVO();
                    subVOMap.put(key, subVO);
                }
                subVO.setTotal(ComputeUtil.safeAdd(subVO.getTotal(), monVal));
                List<BigDecimal> monthDataList = subVO.getMonthData();
                if (monthDataList == null) {
                    monthDataList = new ArrayList<>();
                    subVO.setMonthData(monthDataList);
                }
                monthDataList.add(monVal);
            }
        }

        resVO.setMonthData(monthData);
        resVO.setProduction(subVOMap.get(keyList.get(0)));
        resVO.setPartAReport(subVOMap.get(keyList.get(1)));
        resVO.setBackMoney(subVOMap.get(keyList.get(2)));
        resVO.setOpenInvince(subVOMap.get(keyList.get(3)));
        resVO.setSettleMoney(subVOMap.get(keyList.get(4)));
        resVO.setReceiveInvince(subVOMap.get(keyList.get(5)));
        resVO.setPayMoney(subVOMap.get(keyList.get(6)));
        resVO.setShouldRecNotRec(subVOMap.get(keyList.get(7)));
        resVO.setShouldPayNotPay(subVOMap.get(keyList.get(8)));
        resVO.setBillMoney(subVOMap.get(keyList.get(9)));

        return CommonResponse.success("查询资金使用数据成功！", resVO);
    }

    @Override
    public CommonResponse<ContractVo> saveOrUpdate(ContractVo contractVo) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        //新增
        if (StringUtils.isBlank(contractVo.getBillCode())) {
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(INCOME_CONTRACT_BILL_CODE, tenantId);
            if (billCode.isSuccess()) {
                contractVo.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        ContractEntity entity = BeanMapper.map(contractVo, ContractEntity.class);

        LambdaQueryWrapper<ContractEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(ContractEntity::getTenantId, tenantId);
        lambda.eq(ContractEntity::getBillCode, contractVo.getBillCode());
        lambda.ne(contractVo.getId() != null && contractVo.getId() > 0, ContractEntity::getId, contractVo.getId());
        List<ContractEntity> entities = super.list(lambda);
        if (entities != null && entities.size() > 0) {
            throw new BusinessException("存在相同编码，不允许保存!");
        }

        entity.setBaseTaxMoney(entity.getContractTaxMny());//初始版本金额(含税)
        entity.setBaseMoney(entity.getContractMny());//初始版本金额(不含税)
        entity.setBeforeChangeTaxMny(entity.getContractTaxMny());//变更前金额(含税)
        entity.setBeforeChangeMny(entity.getContractMny());//变更前金额(不含税)
        super.saveOrUpdate(entity);

        List<ContractDetailVo> detailsVos = contractVo.getCheckList();
        List<ContractDetailEntity> saveOrUpldates = new ArrayList<>();
        List<Long> deleteIds = new ArrayList<>();
        for (ContractDetailVo detailsVo : detailsVos) {
            detailsVo.setChangeType(0);//因为导入的时候设置为增补项，避免后面逻辑问题，这里修改一下
            if ("add".equals(detailsVo.getRowState())) {
                ContractDetailEntity detailslist = BeanMapper.map(detailsVo, ContractDetailEntity.class);
                detailslist.setContractId(entity.getId());
                detailslist.setId(null);
                saveOrUpldates.add(detailslist);
            } else if ("edit".equals(detailsVo.getRowState())) {
                ContractDetailEntity detailslist = BeanMapper.map(detailsVo, ContractDetailEntity.class);
                saveOrUpldates.add(detailslist);
            } else if ("del".equals(detailsVo.getRowState())) {
                deleteIds.add(detailsVo.getId());
            }
        }
        if (saveOrUpldates.size() > 0) {
            detailsService.saveOrUpdateBatch(saveOrUpldates, saveOrUpldates.size(), false);
            //维护父子级关系
            Map<String, Long> idMap = new HashMap<>();
            for (ContractDetailEntity cdEntity : saveOrUpldates) {
                idMap.put(cdEntity.getTid(), cdEntity.getId());
            }
            for (ContractDetailEntity cdEntity : saveOrUpldates) {
                if (StringUtils.isNotEmpty(cdEntity.getTpid())) {
                    cdEntity.setParentId(idMap.get(cdEntity.getTpid()));
                }
            }
            detailsService.saveOrUpdateBatch(saveOrUpldates, saveOrUpldates.size(), false);
        }
        if (deleteIds.size() > 0) {
            detailsService.removeByIds(deleteIds, false);
        }

        List<ContractClauseVo> clauseVos = contractVo.getClauseList();
        List<ContractClauseEntity> clauseList = new ArrayList<>();
        List<Long> deleteClauseIds = new ArrayList<>();
        for (ContractClauseVo clauseVo : clauseVos) {
            if ("add".equals(clauseVo.getRowState())) {
                ContractClauseEntity detailslist = BeanMapper.map(clauseVo, ContractClauseEntity.class);
                detailslist.setContractId(entity.getId());
                clauseList.add(detailslist);
            } else if ("edit".equals(clauseVo.getRowState())) {
                ContractClauseEntity detailslist = BeanMapper.map(clauseVo, ContractClauseEntity.class);
                clauseList.add(detailslist);
            } else if ("del".equals(clauseVo.getRowState())) {
                deleteClauseIds.add(clauseVo.getId());
            }
        }
        if (clauseList.size() > 0) {
            clauseService.saveOrUpdateBatch(clauseList, clauseList.size(), false);
        }
        if (deleteClauseIds.size() > 0) {
            clauseService.removeByIds(deleteClauseIds, false);
        }
        return CommonResponse.success(queryDetail(entity.getId()));
    }

    @Override
    public ContractVo queryDetail(Long id) {
        ContractEntity entity = contractMapper.selectById(id);
        if (entity != null) {
            ContractVo contractVo = BeanMapper.map(entity, ContractVo.class);
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractVo.getId()));
            queryParam.getParams().put("change_type", new Parameter(QueryParam.NE, 5));
            //queryParam.getOrderMap().put("detailIndex", "asc");
            List<ContractDetailEntity> purchaseDetailss = detailsService.queryList(queryParam, false);
            if (purchaseDetailss != null && purchaseDetailss.size() > 0) {
                for (ContractDetailEntity cdEntity : purchaseDetailss) {
                    cdEntity.setTid(cdEntity.getId().toString());
                    cdEntity.setTpid(cdEntity.getParentId() != null && cdEntity.getParentId() > 0 ? cdEntity.getParentId().toString() : "");
                    cdEntity.setRowState("edit");
                }
                //List<ContractDetailVo> purchaseDetailsVos = BeanMapper.mapList(purchaseDetailss, ContractDetailVo.class);
                List<ContractDetailVo> resultMapList = BeanMapper.mapList(purchaseDetailss, ContractDetailVo.class);
                //实现排序
                Collections.sort(resultMapList, new ContractDetailComparatorVo());
                contractVo.setCheckList(TreeNodeBUtil.buildTree(resultMapList));
                //contractVo.setDetailsList(purchaseDetailsVos);
            }

            QueryParam query = new QueryParam();
            query.getParams().put("contractId", new Parameter(QueryParam.EQ, contractVo.getId()));
            query.getOrderMap().put("createTime", QueryParam.ASC);
            List<ContractClauseEntity> clauses = clauseService.queryList(query, false);
            if (clauses != null && clauses.size() > 0) {
                List<ContractClauseVo> clauseVoList = BeanMapper.mapList(clauses, ContractClauseVo.class);
                contractVo.setClauseList(clauseVoList);
            }
            return contractVo;
        }
        return null;
    }

    @Override
    public void deleteContract(List<ContractVo> vos) {
        for (ContractVo vo : vos) {
            detailsService.deleteByContractId(vo.getId());
            clauseService.deleteByContractId(vo.getId());
            contractMapper.deleteById(vo.getId());
        }
    }

//    @Override
//    public void updateEntity(ContractEntity contractEntity) {
//        super.updateById(contractEntity);
//    }

    @Override
    public Map<String, Object> countContractAmount(QueryParam queryParam) {
        Map<String, Object> resp = new HashMap<>();
        QueryWrapper wrapper = changeToQueryWrapper(queryParam);
        wrapper.select("round(sum(base_tax_money),2) as originalAmount, round(sum(contract_tax_mny),2) as curAmount");
        resp = super.getMap(wrapper);
//        String originalAmount = resp.get("originalAmount").toString();
//        String curAmount = resp.get("curAmount").toString();
//        resp.put("originalAmount",originalAmount);
//        resp.put("curAmount",curAmount);
        return resp;
    }

    /**
     * 获取合同签订时间
     *
     * @param id
     * @return
     */
    @Override
    public Date getSignDate(Long id) {
        return contractMapper.getSignDate(id);
    }

    // 回写累计收款
    @Override
    public void updateCollectMny(Long contractId, BigDecimal collectMny, Boolean type) {
        BigDecimal sumCollectMny = BigDecimal.ZERO;
        if (type == null) {
            sumCollectMny = ComputeUtil.nullToZero(collectMny);
        } else {
            ContractEntity contractEntity = super.selectById(contractId);
            sumCollectMny = ComputeUtil.nullToZero(contractEntity.getSumCollectMny());
            if (type) {
                sumCollectMny = ComputeUtil.safeAdd(sumCollectMny, collectMny);
            } else {
                sumCollectMny = ComputeUtil.safeSub(sumCollectMny, collectMny);
            }
        }
        LambdaUpdateWrapper<ContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.set(ContractEntity::getSumCollectMny, sumCollectMny);
        updateWrapper.eq(ContractEntity::getId, contractId);
        this.update(updateWrapper);
    }

    // 回写累计开票
    @Override
    public void updateInvoicingMny(Long contractId, BigDecimal invoicingTaxMny, BigDecimal invoicingMny, Boolean type) {
        ContractEntity contractEntity = super.selectById(contractId);
        BigDecimal sumInvoicingMny = contractEntity.getSumInvoicingMny() == null ? BigDecimal.ZERO : contractEntity.getSumInvoicingMny();
        if (type) {
            sumInvoicingMny = sumInvoicingMny.add(invoicingMny);
        } else {
            sumInvoicingMny = sumInvoicingMny.subtract(invoicingMny);
        }
        BigDecimal sumInvoicingTaxMny = contractEntity.getSumInvoicingTaxMny() == null ? BigDecimal.ZERO : contractEntity.getSumInvoicingTaxMny();
        if (type) {
            sumInvoicingTaxMny = sumInvoicingTaxMny.add(invoicingTaxMny);
        } else {
            sumInvoicingTaxMny = sumInvoicingTaxMny.subtract(invoicingTaxMny);
        }
        contractEntity.setSumInvoicingTaxMny(sumInvoicingTaxMny);
        contractEntity.setSumInvoicingMny(sumInvoicingMny);
        super.updateById(contractEntity);
    }

    // 根据项目主键查询最新创建的合同
    @Override
    public ContractVo searchContract(Long projectId) {
        ContractVo contractVo = contractMapper.searchContract(projectId);
        return contractVo;
    }

    // 工程收款
    @Override
    public ContractEntity queryFinanceHistory(Long id) {
        ContractEntity contractEntity = baseMapper.selectById(id);
        CommonResponse<SumReceiveVO> sumReceiveVOList = receiveManageApi.getSumReceiveVOList(id);
        if (sumReceiveVOList.isSuccess()) {
            contractEntity.setReceiveVOList(sumReceiveVOList.getData().getReceiveVOList());
        } else {
            throw new BusinessException("获取收款登记数据失败， 请稍后再试");
        }

        LambdaQueryWrapper<ContractFinishSettleEntity> lambda = Wrappers.lambdaQuery();
        lambda.eq(ContractFinishSettleEntity::getContractId, id);
        List<ContractFinishSettleEntity> list = finishSettleService.list(lambda);

        BigDecimal sumCollectMny = contractEntity.getSumCollectMny();
        if (CollectionUtils.isNotEmpty(list)) {
            ContractFinishSettleEntity finishSettleEntity = list.get(0);
            BigDecimal endSettleMny = finishSettleEntity.getEndSettleMny();
            contractEntity.setEndSettleMny(endSettleMny);
            contractEntity.setEndSettleUnRecMny(ComputeUtil.safeSub(endSettleMny, sumCollectMny));
            // 如果做了竣工结算并且质保金不为空就取竣工结算的
            Date warrantyEndDate = finishSettleEntity.getWarrantyEndDate();
            if (warrantyEndDate != null) {
                contractEntity.setQualityAssuranceEndDate(warrantyEndDate);
            }
            BigDecimal warrantyAmount = finishSettleEntity.getWarrantyAmount();
            if (warrantyAmount != null) {
                contractEntity.setQualityAssuranceMny(warrantyAmount);
            }
        }

        CommonResponse<InvoiceOpenRecordVO> invoiceOpenRecord = invoiceApi.getInvoiceOpenRecord(id);
        if (invoiceOpenRecord.isSuccess()) {
            InvoiceOpenRecordVO data = invoiceOpenRecord.getData();
            contractEntity.setSumInvoicingTaxMny(data.getInvoiceTaxMny());
        }

        // 累计收入占比 = 累计收款金额 / 竣工结算金额
        BigDecimal endSettleMny = contractEntity.getEndSettleMny();
        if (sumCollectMny != null && ComputeUtil.isNotEmpty(endSettleMny)) {
            contractEntity.setSumCollectScale(ComputeUtil.bigDecimalPercent(sumCollectMny, endSettleMny, 2));
        }
        return contractEntity;
    }

    // 开票登记
    @Override
    public InvoiceOpenRecordVO queryTaxHistory(Long id) {
        ContractEntity contractEntity = baseMapper.selectById(id);
        CommonResponse<InvoiceOpenRecordVO> invoiceOpenRecord = invoiceApi.getInvoiceOpenRecord(id);
        if (invoiceOpenRecord.isSuccess()) {
            InvoiceOpenRecordVO data = invoiceOpenRecord.getData();
            data.setContractTaxMny(contractEntity.getContractTaxMny());
            data.setTaxRate(contractEntity.getTaxRate());
            return data;
        } else {
            throw new BusinessException("网络异常,查询发票记录失败,请稍后再试!");
        }
    }

    @Override
    public ContractVo contractDetail(Long id) {
        ContractEntity entity = contractMapper.selectById(id);
        ContractVo contractVo = BeanMapper.map(entity, ContractVo.class);

        ContractVo vo = new ContractVo();
        vo.setId(contractVo.getId());
        vo.setContractName(contractVo.getContractName());
        vo.setContractTaxMny(contractVo.getContractTaxMny());
        vo.setTaxRate(contractVo.getTaxRate());
        vo.setProjectId(contractVo.getProjectId());
        vo.setProjectName(contractVo.getProjectName());
        vo.setCustomerId(contractVo.getCustomerId());
        vo.setCustomerName(contractVo.getCustomerName());
        vo.setContractorUnit(contractVo.getContractorUnit());
        vo.setContractorUnitName(contractVo.getContractorUnitName());
        vo.setOrgId(contractVo.getOrgId());
        vo.setOrgName(contractVo.getOrgName());
        vo.setVersion(contractVo.getVersion());
        return vo;
    }

    @Override
    public CommonResponse<Map<String, BigDecimal>> capitalCount(Integer range, List<Long> orgIds) {
        String year = null;
        String startMonth = null;
        String endMonth = null;
        if (range == 2) {//年累
            year = Calendar.getInstance().get(Calendar.YEAR) + "";
            startMonth = EJCDateUtil.getYear() + "-01";
            endMonth = EJCDateUtil.getMonth(0);
        }
        Long tenantId = InvocationInfoProxy.getTenantid();
        BigDecimal inContractMoney = contractMapper.inContractMoney(range, year, orgIds, tenantId);//收入合同金额
        BigDecimal totalReceiveMoney = contractMapper.totalReceiveMoney(range, year, orgIds, tenantId);//累计收款
        BigDecimal outContractMoney = contractMapper.outContractMoney(range, year, orgIds, tenantId);//支出合同金额
        BigDecimal totalSettleMoney = contractMapper.totalSettleMoney(range, year, orgIds, tenantId);//累计结算金额
        BigDecimal totalOutMoney = contractMapper.totalOutMoney(range, year, orgIds, tenantId);//累计支出
        BigDecimal totalQuoteMoney = quoteService.totalQuoteMoney(range, year, orgIds, tenantId);//累计甲方报量金额

        BigDecimal totalContractOutMoney = contractMapper.totalContractOutMoney(range, year, orgIds, tenantId);//合同付款申请累计支出

        //应收未收金额
        BigDecimal shouldRecNotRec = ComputeUtil.safeSub(totalQuoteMoney, totalReceiveMoney);
        //应付未付金额 = 累计结算 - 合同付款申请累计支出
        BigDecimal shouldPayNotPay = ComputeUtil.safeSub(totalSettleMoney, totalContractOutMoney);
        Map<String, BigDecimal> map = new HashMap<>();
        map.put("inContractMoney", inContractMoney);
        map.put("shouldRecNotRec", shouldRecNotRec);
        map.put("shouldPayNotPay", shouldPayNotPay);
        map.put("totalReceiveMoney", totalReceiveMoney);
        map.put("outContractMoney", outContractMoney);
        map.put("totalSettleMoney", totalSettleMoney);
        map.put("totalOutMoney", totalOutMoney);
        map.put("bookMoney", getProjectSumMny());

        return CommonResponse.success("查询数据成功！", map);
    }

    private BigDecimal getProjectSumMny() {
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder();
        SumAggregationBuilder aggregation = AggregationBuilders
                .sum("sum_mny")
                .field("xiangmukeyongzijin")
                .missing(0);
        searchSourceBuilder.aggregation(aggregation);
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));
        boolQuery.must(QueryBuilders.termsQuery("projectDepartmentId", iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        /**
         * 设置索引以及填充语句
         */
        SearchRequest searchRequest = new SearchRequest("project_finance");
        searchSourceBuilder.query(boolQuery);
        searchSourceBuilder.trackTotalHits(true);
        searchSourceBuilder.timeout(new TimeValue(60, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(searchSourceBuilder);

        BigDecimal sumMny = null;
        try {
            SearchResponse response = null;
            try {
                response = client.search(searchRequest, RequestOptions.DEFAULT);
                Aggregations aggregations = response.getAggregations();
                ParsedSum sum_mny = aggregations.get("sum_mny");
                sumMny = ComputeUtil.toBigDecimal(sum_mny.getValue());
            } catch (IOException e) {
                e.printStackTrace();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return ComputeUtil.nullToZero(sumMny);
    }


    @Override
    public CommonResponse<List<Map>> getproject(List<Long> orgIds) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        List<Map> map = contractMapper.getproject(orgIds, tenantId);
        return CommonResponse.success("查询数据成功！", map);
    }

    @Override
    public CommonResponse<Map> getprogress(Long projectId) {
        Map map = contractMapper.getProjectDetail(projectId);
        Map Datemap = contractMapper.getProjectDate(projectId);
        SimpleDateFormat fm = new SimpleDateFormat("yyy-MM-dd");
        Date now = new Date();
        Date start = null;
        Date end = null;
        if (Datemap != null) {
            if (Datemap.get("endDate") != null) {
                end = (Date) Datemap.get("endDate");
            }
            if (Datemap.get("startDate") != null) {
                start = (Date) Datemap.get("startDate");
            }
        } else {
            if (map.get("plannedFinishDate") != null) {
                end = (Date) map.get("plannedFinishDate");
            }
            if (map.get("plannedCommencementDate") != null) {
                start = (Date) map.get("plannedCommencementDate");
            }
        }
        map.put("startDate", start != null ? fm.format(start) : "未知");
        map.put("endDate", end != null ? fm.format(end) : "未知");
        map.put("customerName", map.get("customerName"));
        map.put("projectState", map.get("projectState"));
        if (end != null && start != null) {
            //当当前日期超过竣工日期时，此值为100%
            if (DateUtil.daysBetween(end, now) >= 0) {
                map.put("dateProgress", "100%");
            } else {
                //当当前日期小于开工日期时，此值为0%
                int a = DateUtil.daysBetween(start, now);
                if (a <= 0) {
                    map.put("dateProgress", "0%");
                } else {
                    int b = DateUtil.daysBetween(start, end);
                    int c = a * 100 / b;
                    map.put("dateProgress", c + "%");
                }
            }

            int allDays = DateUtil.daysBetween(start, end);
            map.put("days", allDays >= 0 ? allDays : 0);
        } else {//开竣工日期有一方未定义时候，此值返回为0%
            map.put("dateProgress", "0%");
            map.put("days", "未知");
        }

        // EJC-D20221018-项目经理门户项目进度增加开工剩余天数
        if (end != null) {
            int subEnd = DateUtil.daysBetween(now, end);
            map.put("remainDays", subEnd >= 0 ? subEnd : 0);
        } else {
            map.put("remainDays", "未知");
        }

        BigDecimal outputMoney = contractMapper.outputMoney(projectId);
        BigDecimal contractMoney = contractMapper.contractMoney(projectId);
        map.put("contractMoney", contractMoney);
        map.put("outputMoney", outputMoney);
        if (contractMoney.compareTo(BigDecimal.ZERO) == 0) {
            map.put("outputProgress", "0%");
        } else {
            map.put("outputProgress", outputMoney.divide(contractMoney, 8, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal("100")).setScale(0, BigDecimal.ROUND_HALF_UP) + "%");
        }
        //查询项目成员
        CommonResponse<List<Map<String, Object>>> membersResp = userApi.getEmployeeList((Long) map.get("projectDepartmentId"));
        if (!membersResp.isSuccess()) {
            throw new BusinessException("查询失败，获取项目成员失败。");
        }
        map.put("managerNum", membersResp.getData().size());

        return CommonResponse.success("查询数据成功！", map);
    }

    @Override
    public CommonResponse<List<ProjectProgressVo>> getprogressList(List<Long> projectIds) {
        List<ProjectProgressVo> projectDetailList = contractMapper.getProjectDetailList(projectIds);
        for (ProjectProgressVo pro : projectDetailList
        ) {
            pro.setOutputMoney(contractMapper.outputMoney(pro.getProjectId()));
        }
        return CommonResponse.success("查询数据成功！", projectDetailList);
    }

    @Override
    public CommonResponse<List<Map>> costCount(Integer range, List<Long> orgIds) {
        String year = null;
        if (range == 2) {//年累
            year = Calendar.getInstance().get(Calendar.YEAR) + "";
        }
        Long tenantId = InvocationInfoProxy.getTenantid();
        List<Map> map = contractMapper.costCount(range, year, orgIds, tenantId);

        return CommonResponse.success("查询数据成功！", map);
    }

    @Override
    public CommonResponse<JSONObject> projectCount(String range, String name, String dateIn, String projectType, Long engineeringType,List<Long> orgIds, boolean queryBudgetCostAndVideo) {
//                allProjectNumber:总项目数,
//                allContractMoney:总合同额,
//                allInMoney:总收入,
//                allOutMoney:总支出,
//                bookMoney:账面资金 =总收入-总支出；
        Long tenantId = InvocationInfoProxy.getTenantid();
        String year = Calendar.getInstance().get(Calendar.YEAR) + "";
        String startDate = null;
        String endDate = null;
        String type = null;
        Calendar instance = Calendar.getInstance();
        instance.setTime(new Date());
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM");
        if (ProjectSurveyEnum.THISYEAR.getCode().equals(dateIn)) {
            dateIn = Calendar.getInstance().get(Calendar.YEAR) + "";
            type = "1";
        } else if (ProjectSurveyEnum.LASTYEAR.getCode().equals(dateIn)) {
            dateIn = Calendar.getInstance().get(Calendar.YEAR) - 1 + "";
            type = "1";
        } else if (ProjectSurveyEnum.NEAR6MON.getCode().equals(dateIn)) {
            instance.add(Calendar.MONTH, -5);
            type = "2";
        } else if (ProjectSurveyEnum.NEAR3MON.getCode().equals(dateIn)) {
            instance.add(Calendar.MONTH, -2);
            type = "2";
        } else if (ProjectSurveyEnum.THISMON.getCode().equals(dateIn)) {
            instance.add(Calendar.MONTH, 0);
            type = "2";
        }

        startDate = dateFormat.format(instance.getTime());
        endDate = dateFormat.format(new Date());
        List<ProjectCountVO> list = contractMapper.projectCount_project(range, year, startDate, endDate, dateIn, projectType, engineeringType, orgIds, tenantId, type);
        List<Long> ids = new ArrayList<>();
        BigDecimal allContractMoney = BigDecimal.ZERO;
        Set<String> provinceSet = new HashSet<>();
        Set<String> citySet = new HashSet<>();
        Set<String> countySet = new HashSet<>();
        String pname = null;
        for (ProjectCountVO vo : list) {
            ids.add(vo.getId());
            BigDecimal contractTaxMny = vo.getContractTaxMny() == null ? BigDecimal.ZERO : vo.getContractTaxMny();
            allContractMoney = allContractMoney.add(contractTaxMny);
            if (StringUtils.isNotEmpty(vo.getArea())) {
                if (vo.getArea().equals("台湾省")) {
                    vo.setProvince("台湾省");
                    vo.setCity("台湾省");
                    vo.setCounty("台湾省");

                    provinceSet.add("台湾省");
                    citySet.add("台湾省");
                    countySet.add("台湾省");
                } else {
                    Map<String, String> areaMmap = addressResolution(vo.getArea());
                    if (areaMmap != null) {
                        String province = areaMmap.get("province") == null ? "其他" : areaMmap.get("province");
                        String city = areaMmap.get("city") == null ? "其他" : areaMmap.get("city");
                        String county = areaMmap.get("county") == null ? "其他" : areaMmap.get("county");
                        vo.setProvince(province);
                        vo.setCity(city);
                        vo.setCounty(county);

                        provinceSet.add(province);
                        citySet.add(city);
                        countySet.add(county);

                        if (StringUtils.isNotBlank(name) && city.equals(name)) {
                            pname = province;
                        }
                    } else {
                        vo.setProvince("其他");
                        vo.setCity("其他");
                        vo.setCounty("其他");

                        provinceSet.add("其他");
                        citySet.add("其他");
                        countySet.add("其他");
                    }
                }
            } else {
                vo.setProvince("其他");
                vo.setCity("其他");
                vo.setCounty("其他");

                provinceSet.add("其他");
                citySet.add("其他");
                countySet.add("其他");
            }

        }

        String rangeType = null;
        List<ProjectDataVO> dataVOS = new ArrayList<>();

        if (StringUtils.isNotBlank(name)) {
            if (provinceSet.contains(name)) {// 如果选择省
                rangeType = getProvincePinYin(name);
                dataVOS = projectCountByName(name, list, 0);
            } else if (citySet.contains(name)) {
                rangeType = getProvincePinYin(pname) + "-" + getNamePinYinChar(name, 1);
                dataVOS = projectCountByName(name, list, 1);
            }

        } else {

            if (provinceSet.size() == 1) {//同一省份
                if (citySet.size() == 1) {// 如果是同一市
                    rangeType = getProvincePinYin(provinceSet.iterator().next()) + "-" + getNamePinYinChar(citySet.iterator().next(), 1);
                    Map<String, ProjectDataVO> countyMap = new HashMap<>();
                    for (ProjectCountVO vo : list) {
                        String county = vo.getCounty();
                        BigDecimal contractTaxMny = vo.getContractTaxMny() == null ? BigDecimal.ZERO : vo.getContractTaxMny();
                        if (!countyMap.containsKey(county)) {
                            ProjectDataVO pvo = new ProjectDataVO();
                            pvo.setName(county);
                            pvo.setValue(1);
                            pvo.setContractMoney(contractTaxMny);
                            countyMap.put(county, pvo);
                        } else {
                            ProjectDataVO projectDataVO = countyMap.get(county);
                            projectDataVO.setValue(projectDataVO.getValue() + 1);
                            projectDataVO.setContractMoney(projectDataVO.getContractMoney().add(contractTaxMny));
                            countyMap.put(county, projectDataVO);
                        }
                    }
                    dataVOS = new ArrayList<>(countyMap.values());
                    Map<String, List<ProjectCountVO>> map = list.stream().collect(Collectors.groupingBy(ProjectCountVO::getCounty));
                    for (ProjectDataVO dataVO : dataVOS) {
                        if (map.containsKey(dataVO.getName())) {
                            List<ProjectCountVO> projectCountVOS = map.get(dataVO.getName());
                            List<ProjectVo> projectVos = new ArrayList<>();
                            for (ProjectCountVO projectCountVO : projectCountVOS) {
                                ProjectVo projectVo = new ProjectVo();
                                projectVo.setId(projectCountVO.getId());
                                projectVo.setName(projectCountVO.getProjectName());
                                projectVo.setOrgId(projectCountVO.getOrgId());
                                projectVos.add(projectVo);
                            }
                            dataVO.setProjectList(projectVos);
                        }
                    }
                } else {
                    rangeType = getProvincePinYin(provinceSet.iterator().next());
                    Map<String, ProjectDataVO> cityMap = new HashMap<>();
                    for (ProjectCountVO vo : list) {
                        String city = vo.getCity();
                        BigDecimal contractTaxMny = vo.getContractTaxMny() == null ? BigDecimal.ZERO : vo.getContractTaxMny();
                        if (!cityMap.containsKey(city)) {
                            ProjectDataVO pvo = new ProjectDataVO();
                            pvo.setName(city);
                            pvo.setValue(1);
                            pvo.setContractMoney(contractTaxMny);
                            cityMap.put(city, pvo);
                        } else {
                            ProjectDataVO projectDataVO = cityMap.get(city);
                            projectDataVO.setValue(projectDataVO.getValue() + 1);
                            projectDataVO.setContractMoney(projectDataVO.getContractMoney().add(contractTaxMny));
                            cityMap.put(city, projectDataVO);
                        }
                    }
                    dataVOS = new ArrayList<>(cityMap.values());
                    Map<String, List<ProjectCountVO>> map = list.stream().collect(Collectors.groupingBy(ProjectCountVO::getCity));
                    for (ProjectDataVO dataVO : dataVOS) {
                        if (map.containsKey(dataVO.getName())) {
                            List<ProjectCountVO> projectCountVOS = map.get(dataVO.getName());
                            List<ProjectVo> projectVos = new ArrayList<>();
                            for (ProjectCountVO projectCountVO : projectCountVOS) {
                                ProjectVo projectVo = new ProjectVo();
                                projectVo.setId(projectCountVO.getId());
                                projectVo.setName(projectCountVO.getProjectName());
                                projectVo.setOrgId(projectCountVO.getOrgId());
                                projectVos.add(projectVo);
                            }
                            dataVO.setProjectList(projectVos);
                        }
                    }
                }

            } else {//不同省份
                rangeType = "China";
                Map<String, ProjectDataVO> provinceMap = new HashMap<>();
                for (ProjectCountVO vo : list) {
                    String province = vo.getProvince();
                    BigDecimal contractTaxMny = vo.getContractTaxMny() == null ? BigDecimal.ZERO : vo.getContractTaxMny();
                    if (!provinceMap.containsKey(province)) {
                        ProjectDataVO pvo = new ProjectDataVO();
                        pvo.setName(province);
                        pvo.setValue(1);
                        pvo.setContractMoney(contractTaxMny);
                        provinceMap.put(province, pvo);
                    } else {
                        ProjectDataVO projectDataVO = provinceMap.get(province);
                        projectDataVO.setValue(projectDataVO.getValue() + 1);
                        projectDataVO.setContractMoney(projectDataVO.getContractMoney().add(contractTaxMny));
                        provinceMap.put(province, projectDataVO);
                    }
                }
                Map<String, List<ProjectCountVO>> map = list.stream().collect(Collectors.groupingBy(ProjectCountVO::getProvince));
                dataVOS = new ArrayList<>(provinceMap.values());
                for (ProjectDataVO dataVO : dataVOS) {
                    if (map.containsKey(dataVO.getName())) {
                        List<ProjectCountVO> projectCountVOS = map.get(dataVO.getName());
                        List<ProjectVo> projectVos = new ArrayList<>();
                        for (ProjectCountVO projectCountVO : projectCountVOS) {
                            ProjectVo projectVo = new ProjectVo();
                            projectVo.setId(projectCountVO.getId());
                            projectVo.setName(projectCountVO.getProjectName());
                            projectVo.setOrgId(projectCountVO.getOrgId());
                            projectVos.add(projectVo);
                        }
                        dataVO.setProjectList(projectVos);
                    }
                }
            }
        }

        int allProjectNumber = list.size();
        BigDecimal allInMoney = ids.size() > 0 ? contractMapper.allInMoney(ids) : BigDecimal.ZERO;//总收入
        BigDecimal allOutMoney = ids.size() > 0 ? contractMapper.allOutMoney(ids, null) : BigDecimal.ZERO;//总收入
        BigDecimal bookMoney = allInMoney.subtract(allOutMoney);

        JSONObject jsonObject = new JSONObject();
        jsonObject.put("allProjectNumber", allProjectNumber);
        jsonObject.put("allContractMoney", allContractMoney);
        jsonObject.put("allInMoney", allInMoney);
        jsonObject.put("allOutMoney", allOutMoney);
        jsonObject.put("bookMoney", bookMoney);
        if (CollectionUtils.isNotEmpty(dataVOS)) {
            Collections.sort(dataVOS, (o1, o2) -> {
                int a = "其他".equals(o1.getName()) ? 100000 : o1.getValue();
                int b = "其他".equals(o2.getName()) ? 100000 : o1.getValue();
                return a - b;
            });
        }
        jsonObject.put("data", dataVOS);
        jsonObject.put("range", rangeType == null || rangeType.contains("null") ? "China" : rangeType);
        //获取预算和实际成本数据
        if(queryBudgetCostAndVideo){
            getBudgetAndCostData(ids, jsonObject);
            BigDecimal lirun = ComputeUtil.safeSub(jsonObject.getBigDecimal("allContractMoney"), jsonObject.getBigDecimal("shijichengben"));
            jsonObject.put("lirun", lirun);
            jsonObject.put("lirunRate", ComputeUtil.bigDecimalPercent(lirun, jsonObject.getBigDecimal("allContractMoney"), 2));
        }
        return CommonResponse.success("查询数据成功！", jsonObject);
    }


    /**
     * @description: 从预算与实际成本报表获取项目的预算和成本数据
     *
     * @param projectIds
     * @return {@link JSONObject}
     * @author songlx
     * @date: 2023/4/24
     */
    private JSONObject getBudgetAndCostData(List<Long> projectIds, JSONObject resVO) {

        List<String> sumList = Arrays.asList(
                "yusuan",
                "zhichuhetongjine",
                "shijichengben"
        );

        SearchRequest searchRequest = new SearchRequest("budgetpro_cost");
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termsQuery("projectId", projectIds));

        for (String key : sumList) {
            SumAggregationBuilder aggregation = AggregationBuilders
                    .sum("sum_" + key)
                    .field(key)
                    .missing(0);
            sourceBuilder.aggregation(aggregation);
        }

        sourceBuilder.query(boolQuery);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(60, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = response.getAggregations();

            for (String key : sumList) {
                ParsedSum sum_mny = aggregations.get("sum_" + key);
                resVO.put(key, ComputeUtil.toBigDecimal(sum_mny.getValue()));
            }

        } catch (IOException e) {
            e.printStackTrace();
        }

        return resVO;
    }


    // type:0-省，1-市
    private List<ProjectDataVO> projectCountByName(String name, List<ProjectCountVO> list, Integer type) {
        Map<String, ProjectDataVO> map = new HashMap<>();
        for (ProjectCountVO vo : list) {
            String voName = null;
            String childrenName = null;
            if (type == 0) {
                voName = vo.getProvince();
                childrenName = vo.getCity();
            } else {
                voName = vo.getCity();
                childrenName = vo.getCounty();
            }
            if (voName.equals(name)) {
                BigDecimal contractTaxMny = vo.getContractTaxMny() == null ? BigDecimal.ZERO : vo.getContractTaxMny();
                if (!map.containsKey(childrenName)) {
                    ProjectDataVO pvo = new ProjectDataVO();
                    pvo.setName(childrenName);
                    pvo.setValue(1);
                    pvo.setContractMoney(contractTaxMny);
                    map.put(childrenName, pvo);
                } else {
                    ProjectDataVO projectDataVO = map.get(childrenName);
                    projectDataVO.setValue(projectDataVO.getValue() + 1);
                    projectDataVO.setContractMoney(projectDataVO.getContractMoney().add(contractTaxMny));
                    map.put(childrenName, projectDataVO);
                }
            }
        }
        Map<String, List<ProjectCountVO>> projectCountVOMap = null;
        if (type == 0) {
            projectCountVOMap = list.stream().collect(Collectors.groupingBy(ProjectCountVO::getCity));
        } else {
            projectCountVOMap = list.stream().collect(Collectors.groupingBy(ProjectCountVO::getCounty));
        }
        ArrayList<ProjectDataVO> projectDataVOS = new ArrayList<>(map.values());
        for (ProjectDataVO dataVO : projectDataVOS) {
            if (map.containsKey(dataVO.getName())) {
                List<ProjectCountVO> projectCountVOS = projectCountVOMap.get(dataVO.getName());
                List<ProjectVo> projectVos = new ArrayList<>();
                for (ProjectCountVO projectCountVO : projectCountVOS) {
                    ProjectVo projectVo = new ProjectVo();
                    projectVo.setId(projectCountVO.getId());
                    projectVo.setName(projectCountVO.getProjectName());
                    projectVo.setOrgId(projectCountVO.getOrgId());
                    projectVos.add(projectVo);
                }
                dataVO.setProjectList(projectVos);
            }
        }
        return projectDataVOS;

    }

    @Override
    public IPage<ProjectInOutVO> proPageList(Map<String, Object> params) {
        Object orgIds = params.get("orgIds");
        if (orgIds == null) {
            return new Page<>();
        }
        List<ProjectInOutVO> voList = new ArrayList<>();
        IPage<ProjectInOutVO> page = new Page<>();
        page.setCurrent(Integer.valueOf(params.get("pageIndex").toString()));
        page.setSize(Integer.valueOf(params.get("pageSize").toString()));

        long total = contractMapper.count(params);
        page.setTotal(total);
        if (total == 0) {
            page.setRecords(voList);
            return page;
        }

        long startLine = (page.getCurrent() < 1 ? 0 : page.getCurrent() - 1) * page.getSize();
        params.put("startLine", startLine);
        params.put("pageSize", page.getSize());
        voList = contractMapper.getList(params);
        page.setRecords(voList);
        return page;
    }

    @Override
    public List<Map<String, Object>> incomeMnyWarn(List<SqlParam> sqlParamList) {
        return baseMapper.incomeMnyWarn(sqlParamList);
    }

    @Override
    public List<IncomeContractWarnVo> outIncomeMnyWarn(List<Long> tenantIds) {
        return baseMapper.outIncomeMnyWarn(tenantIds);
    }

    @Override
    public List<QualityAssuranceEndDateWarnVo> qualityAssuranceEndDateWarn(List<Long> tenantIds) {
        return baseMapper.qualityAssuranceEndDateWarn(tenantIds);
    }

    @Override
    public BigDecimal getInContractMoneySum(Map<String, Object> params) {
        return contractMapper.getInContractMoneySum(params);
    }

    @Override
    public void refreshInvoiceMny(Map<Long, BigDecimal> map) {
//        Set<Long> keys = map.keySet();
//        if (CollectionUtils.isNotEmpty(keys)){
//            keys.forEach(key->{
//                LambdaUpdateWrapper<ContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
//                updateWrapper.eq(ContractEntity::getId, key);
//                updateWrapper.set(ContractEntity::getSumInvoicingTaxMny, map.get(key));
//                contractService.update(updateWrapper);
//            });
//        }
    }

    @Override
    public IPage selectPage(Page<ContractEntity> pageData, QueryWrapper wrapper) {
        return baseMapper.selectPage(pageData, wrapper);
    }

    /**
     * 根据项目id批量修改详细地址
     *
     * @param projectId
     * @param address
     * @return
     */
    @Override
    public CommonResponse<String> refreshAddress(Long projectId, String address) {
        LambdaUpdateWrapper<ContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(ContractEntity::getProjectId, projectId);
        updateWrapper.set(ContractEntity::getAddress, address);
        super.update(updateWrapper);
        return CommonResponse.success("详细地址修改成功！");
    }

    @Override
    public List<ContractVo> projectSumCollectMny(List<Long> tids) {
        return baseMapper.projectSumCollectMny(tids);
    }


    //日期相减得到天数
    public static int subtractDay(String date1, String date2) {
        int rs = 0;
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        try {
            Date start = simpleDateFormat.parse(date1);
            Date end = simpleDateFormat.parse(date2);
            long sss = (start.getTime() - end.getTime()) / 1000;
            rs = (int) sss / (60 * 60 * 24);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return rs;
    }

    public static void main(String[] args) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        try {
            System.out.println(DateUtil.daysBetween(sdf.parse("2022-10-11"), new Date()));
            System.out.println(subtractDay("2022-10-19", "2022-10-11"));
        } catch (ParseException e) {
            e.printStackTrace();
        }
    }


    @Override
    public BigDecimal fetchSghtzje(Long projectId) {
        Assert.notNull(projectId, "项目id不能为空");
        return baseMapper.fetchSghtzje(projectId);
    }

  @Override
  public Map<String, BigDecimal> costSquareChart(Long projectId) {
    Assert.notNull(projectId, "项目id不能为空");

    // 项目合同造价
    QueryWrapper<ContractEntity> query = Wrappers.query();
    query
        .select("ifnull(sum(contract_tax_mny), 0) as mny")
        .eq("dr", 0)
        .in("bill_state", 1, 3)
        .eq("project_id", projectId);
    Map<String, Object> map = super.getMap(query);
    BigDecimal sumContractMny = new BigDecimal(map.get("mny").toString());

    // 工程结算总价
    QueryWrapper<QuoteEntity> queryWrapper = Wrappers.query();
    queryWrapper
        .select("ifnull(sum(quote_tax_mny), 0) as mny")
        .eq("dr", 0)
        .in("bill_state", 1, 3)
        .eq("project_id", projectId);
    Map<String, Object> map1 = quoteService.getMap(queryWrapper);
    BigDecimal sumSettleMny = new BigDecimal(map1.get("mny").toString());

    HashMap<String, BigDecimal> hashMap = new HashMap<>();
    hashMap.put("sumContractMny", sumContractMny);
    hashMap.put("sumSettleMny", sumSettleMny);
    return hashMap;
  }

    @Override
    public BiVo queryProjectMnyCollection(Long orgId, String dateIn, String projectStatus) {
        BigDecimal sumFinishTaxMny = new BigDecimal(0);
        BigDecimal sumContractTaxMny=new BigDecimal(0);
        BigDecimal amountReceived = new BigDecimal(0);//已收
        BigDecimal amountReceivable=new BigDecimal(0);//应收金额(总金额)
        BigDecimal uncollectedamount=new BigDecimal(0);//未收金额
        BigDecimal totalamount = new BigDecimal(0);//Bi项目付款已付金额
        BigDecimal amountdue=new BigDecimal(0);//Bi项目付款应付金额
        BigDecimal overpaymentAmount=new BigDecimal(0);//Bi项目付款超付金额
        BigDecimal ProductionProportion = new BigDecimal(0);
        FilterOrgIdVo filterOrgVo = new FilterOrgIdVo();
        filterOrgVo.setOrgId(orgId);
        filterOrgVo.setDateIn(dateIn);
        filterOrgVo.setProjectStatus(projectStatus);
        CommonResponse<List<Long>> listCommonResponse = projectApi.queryFilterOrgIds(filterOrgVo);
        List<Long> orgIds = listCommonResponse.getData();
        Long tenantId = InvocationInfoProxy.getTenantid();

        if(filterOrgVo.getOrgId()!=null){
            List<FinanceUseResSubVO> list = productionMapper.totalProductionMoney(null,null,orgIds,tenantId);
            if (CollectionUtils.isNotEmpty(list)) {
                for (FinanceUseResSubVO vo:list ){
                    sumFinishTaxMny = BigDecimalUtils.safeAdd(sumFinishTaxMny,vo.getTotal()==null?BigDecimal.ZERO :vo.getTotal());
                }
            }
            amountReceived = contractMapper.totalReceiveMoney(null,null,orgIds,tenantId)==null?BigDecimal.ZERO :contractMapper.totalReceiveMoney(null,null,orgIds,tenantId);
            List<FinanceUseResSubVO> list1 = contractMapper.totalOpenInvince(null, null, orgIds, tenantId);
            for (FinanceUseResSubVO vo:list1 ){
                amountReceivable = BigDecimalUtils.safeAdd(amountReceivable,vo.getTotal()==null?BigDecimal.ZERO :vo.getTotal());
            }
            uncollectedamount=ComputeUtil.safeSub(amountReceivable,amountReceived);
            sumContractTaxMny = contractMapper.inContractMoney(null, null, orgIds, tenantId)==null?BigDecimal.ZERO :contractMapper.inContractMoney(null, null, orgIds, tenantId);
            ProductionProportion=ComputeUtil.bigDecimalPercent(sumFinishTaxMny,sumContractTaxMny,2);
            totalamount = contractMapper.totalOutMoney(null,null,orgIds,tenantId)==null?BigDecimal.ZERO :contractMapper.totalOutMoney(null,null,orgIds,tenantId);
            amountdue = contractMapper.totalCurrentReceiveMnyTax(orgIds,tenantId)==null?BigDecimal.ZERO :contractMapper.totalCurrentReceiveMnyTax(orgIds,tenantId);
            overpaymentAmount=ComputeUtil.safeSub(totalamount,amountdue);
        }

        BiVo vo = new BiVo();
        vo.setSumFinishTaxMny(sumFinishTaxMny);
        vo.setSumContractTaxMny(ComputeUtil.nullToZero(sumContractTaxMny));
        vo.setProductionProportion(ProductionProportion);
        vo.setAmountReceived(ComputeUtil.nullToZero(amountReceived));
        vo.setAmountReceivable(ComputeUtil.nullToZero(amountReceivable));
        vo.setUncollectedamount(ComputeUtil.nullToZero(uncollectedamount));
        vo.setTotalamount(ComputeUtil.nullToZero(totalamount));
        vo.setAmountdue(ComputeUtil.nullToZero(amountdue));
        vo.setOverpaymentAmount(ComputeUtil.nullToZero(overpaymentAmount));
        return vo;
    }

    @Override
    public List<ProjectWarnVo> queryProjectOverDueWarn(Long orgId, String dateIn, String projectStatus) {
        FilterOrgIdVo filterOrgVo = new FilterOrgIdVo();
        filterOrgVo.setOrgId(orgId);
        filterOrgVo.setDateIn(dateIn);
        filterOrgVo.setProjectStatus(projectStatus);
        CommonResponse<List<Long>> listCommonResponse = projectApi.queryFilterOrgIds(filterOrgVo);
        List<Long> orgIds = listCommonResponse.getData();
        Long tenantId = InvocationInfoProxy.getTenantid();
        List<ProjectWarnVo> projectWarnVos = contractMapper.projectOverDueWarn(orgIds, tenantId);
//        for (ProjectWarnVo projectWarnVo : projectWarnVos) {
//            projectWarnVo.setWeishou(ComputeUtil.nullToZero(projectWarnVo.getWeishou()));
//            projectWarnVo.setKaipiao(ComputeUtil.nullToZero(projectWarnVo.getKaipiao()));
//            if (ComputeUtil.isLessThan(ComputeUtil.nullToZero(projectWarnVo.getWeishou()),BigDecimal.ZERO))
//                projectWarnVo.setWeishou(BigDecimal.ZERO);
//        }


        return projectWarnVos;
    }

    @Override
    public JSONObject contractInfo() {
        JSONObject obj = new JSONObject();
        /*
        合同额信息管理：本年数据

计划完成合同额（万元）：施工合同增加字段  穿透施工合同

当月新开合同额（万元）：施工合同当月金额   穿透施工合同  筛选本月

累计新开合同额（万元）：施工合同当年累计金额    穿透施工合同 筛选本年

已审批目标项目个数：已生效目标成本个数  穿透目标成本 已审批或已提交的列表

年度中标项目数量（个）：施工合同项目数  穿透施工合同
         */
        Calendar calendar = Calendar.getInstance();
        int year = calendar.get(Calendar.YEAR);
        QueryWrapper<ContractEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.select("COUNT(DISTINCT project_id) projectNum, " +
                "ROUND(IFNULL( SUM(plan_finish_contract_mny)/10000, 0), 2) planFinishContractMny, " +
                "ROUND(IFNULL( SUM(IF( DATE_FORMAT(sign_date, '%Y-%m') = DATE_FORMAT(NOW(), '%Y-%m') , contract_tax_mny,0 )), 0)/10000, 2) thisMonContractTaxMny, " +
                "ROUND(IFNULL( SUM(contract_tax_mny), 0)/10000, 2) thisYearContractTaxMny");
        queryWrapper.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        queryWrapper.eq("YEAR ( sign_date )", year);
        queryWrapper.eq("tenant_id", InvocationInfoProxy.getTenantid());
        queryWrapper.isNotNull("project_id");
        List<Map<String, Object>> maps = this.listMaps(queryWrapper);
        Map<String, Object> stringObjectMap = maps.get(0);
        for (String key : stringObjectMap.keySet()) {
            obj.put(key, stringObjectMap.get(key));
        }
        BigDecimal sumTenderProfitRate = ComputeUtil.toBigDecimal(obj.get("sumTenderProfitRate"));
        BigDecimal projectNum = ComputeUtil.toBigDecimal(obj.get("projectNum"));
        obj.put("tenderProfitRate", ComputeUtil.safeDiv(sumTenderProfitRate, projectNum));
        return obj;
    }

    @Override
    public JSONObject tenderInfo() {
        JSONObject obj = new JSONObject();
        QueryWrapper<ContractEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.select("COUNT(1) projectNum, " +
                "ROUND(IFNULL( SUM(tender_profit_rate), 0), 2) sumTenderProfitRate, " +
                "ROUND(IFNULL( SUM(tender_cost_mny)/10000, 0), 2) tenderCostMny, " +
                "ROUND(IFNULL( SUM(tender_profit_mny)/10000, 0), 2) tenderProfitMny ");
        queryWrapper.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
        queryWrapper.eq("tenant_id", InvocationInfoProxy.getTenantid());
        queryWrapper.isNotNull("project_id");
        List<Map<String, Object>> maps = this.listMaps(queryWrapper);
        Map<String, Object> stringObjectMap = maps.get(0);
        for (String key : stringObjectMap.keySet()) {
            obj.put(key, stringObjectMap.get(key));
        }
        BigDecimal sumTenderProfitRate = ComputeUtil.toBigDecimal(obj.get("sumTenderProfitRate"));
        BigDecimal projectNum = ComputeUtil.toBigDecimal(obj.get("projectNum"));
        obj.put("tenderProfitRate", ComputeUtil.safeDiv(sumTenderProfitRate, projectNum));
        return obj;
    }
}
