package com.ejianc.business.income.controller.api;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.income.bean.ClaimEntity;
import com.ejianc.business.income.bean.ContractEntity;
import com.ejianc.business.income.bean.ProductionEntity;
import com.ejianc.business.income.bean.QuoteEntity;
import com.ejianc.business.income.service.IClaimService;
import com.ejianc.business.income.service.IContractService;
import com.ejianc.business.income.service.IProductionService;
import com.ejianc.business.income.service.IQuoteService;
import com.ejianc.business.income.vo.ContractVo;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/api/contract/")
public class ContractApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IContractService contractService;

    @Autowired
    private IProductionService productionService;

    @Autowired
    private IClaimService claimServicel;

    @Autowired
    private IQuoteService quoteService;

    @Autowired
    private IProjectApi projectApi;

    @Autowired
    private IOrgApi orgApi;

    @Value("${common.env.base-host}")
    private String BASE_HOST;

    /**
     * 回写累计收款
     * @param contractId
     * @param collectMny
     * @param type,累加=true,累减=false
     * @return
     */
    @RequestMapping(value = "updateCollectMny", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> updateCollectMny(@RequestParam(value = "contractId") Long contractId,
                                                  @RequestParam(value = "collectMny") BigDecimal collectMny,
                                                  @RequestParam(value = "type") Boolean type) {
        contractService.updateCollectMny(contractId,collectMny,type);
        return CommonResponse.success("更新累计收款成功！");
    }
    /**
     * 回写累计开票
     * @param contractId
     * @param invoicingMny
     * @param type,累加=true,累减=false
     * @return
     */
    @RequestMapping(value = "updateInvoicingMny", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> updateInvoicingMny(@RequestParam(value = "contractId") Long contractId,
                                                  @RequestParam(value = "invoicingTaxMny") BigDecimal invoicingTaxMny,
                                                  @RequestParam(value = "invoicingMny") BigDecimal invoicingMny,
                                                  @RequestParam(value = "type") Boolean type) {
        contractService.updateInvoicingMny(contractId,invoicingTaxMny,invoicingMny,type);
        return CommonResponse.success("更新累计开票成功！");
    }

    /**
     * 根据项目主键查询最新创建的合同
     * @param projectId
     * @return
     */
    @RequestMapping(value = "searchContract", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ContractVo> searchContract(@RequestParam(value = "projectId") Long projectId) {
        return CommonResponse.success("根据项目主键查询最新创建的合同成功！",contractService.searchContract(projectId));
    }
    /**
     * 根据合同主键查询合同详情
     * @param contractId
     * @return
     */
    @RequestMapping(value = "queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ContractVo> queryDetail(@RequestParam(value = "contractId") Long contractId) {
        return CommonResponse.success("根据合同主键查询合同详情成功！",contractService.contractDetail(contractId));
    }

    /**
     * 根据租户主键查询生效的合同列表
     * @param tenantIds
     * @return
     */
    @RequestMapping(value = "searchContractByTenantIds", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<ContractVo>> searchContractByTenantIds(@RequestParam(value = "tenantIds") List<Long> tenantIds) {
        LambdaQueryWrapper<ContractEntity> lambda = new LambdaQueryWrapper<>();
        lambda.in(ContractEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.in(ContractEntity::getTenantId, tenantIds);
        List<ContractEntity> list = contractService.list(lambda);
        return CommonResponse.success("据租户主键查询生效的合同列表成功！", BeanMapper.mapList(list, ContractVo.class));
    }

    /**
     * 收入合同统计
     *
     * @param projectRange 查询范围（1-所有项目，2-在建项目，3-完工项目，4-本年新签）
     * @return：
     *          incomeContractAmtCount: 统计生效施工合同的含税金额
     *          claimAmtCount: 统计生效的变迁变更索赔单金额
     *          productionCount: 已生效的产值统计金额（含税）
     *          quoteCount: 已生效的甲方批量金额（含税）
     *          receiveCount: 已生效的有合同收款登记金额（含税）
     *          invoiceOpenCount: 已生效的有合同开票登记金额（含税）
     */
    @RequestMapping(value = "incomeCount", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> countIncomeContract(HttpServletRequest request, @RequestParam(value = "projectRange") int projectRange
            , @RequestParam(value = "orgId", required = false) Long orgId) {
        JSONObject data = new JSONObject();

        List<Long> orgIds = new ArrayList<>();
        List<Long> newIds = new ArrayList<>();
        if (orgId != null) {
            orgIds = orgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
            CommonResponse<List<ProjectRegisterVO>> listCommonResponse = projectApi.queryChildrenProjectByOrgId(orgId);
            if (listCommonResponse.isSuccess()) {
                List<ProjectRegisterVO> list = listCommonResponse.getData();
                newIds = list.stream().map(ProjectRegisterVO::getId).collect(Collectors.toList());
            }
        }

        List<Long> projectIds = new ArrayList<>();
        if(projectRange != 1) {
            CommonResponse<List<Long>> projectIdsResp = projectApi.getProjectIdsByProperties(projectRange);
            if(!projectIdsResp.isSuccess()) {
                logger.error("根据条件 projectRange-{} 查询项目Id列表失败，原因：{}", projectRange, projectIdsResp.getMsg());
                return CommonResponse.error("查询失败，查询匹配的项目信息失败。");
            }
            projectIds = projectIdsResp.getData();
            // 根据组织过滤projectIds
            if (orgId != null) {
                projectIds = projectIds.stream().filter(newIds::contains).collect(Collectors.toList());
            }
            if(CollectionUtils.isEmpty(projectIds)) {
                data.put("incomeContractAmtCount", 0);
                data.put("claimAmtCount", 0);
                data.put("productionCount", 0);
                data.put("quoteCount", 0);
                data.put("receiveCount", 0);
                data.put("invoiceOpenCount", 0);
                return CommonResponse.success(data);
            }
        }else {
            projectIds = newIds;
        }

        String authority = request.getHeader("authority");

        ExecutorService threadPool = Executors.newFixedThreadPool(6);
        //统计生效施工合同的含税金额
        Callable<BigDecimal> incomeCountCallable = new IncomeCountCallable(InvocationInfoProxy.getTenantid(), projectIds, RequestContextHolder.getRequestAttributes(), authority, orgIds);
        Future<BigDecimal> countIncomeAmt = threadPool.submit(incomeCountCallable);
        //统计生效的变迁变更索赔单金额
        Callable<Object> claimAmtCountCallable = new ClaimAmtCountCallable(InvocationInfoProxy.getTenantid(), projectIds, RequestContextHolder.getRequestAttributes(), authority, orgIds);
        Future<Object> claimAmtCount = threadPool.submit(claimAmtCountCallable);
        //已生效的产值统计金额（含税）
        Callable<Object> productionCountCallable = new ProductionCountCallable(InvocationInfoProxy.getTenantid(), projectIds, RequestContextHolder.getRequestAttributes(), authority, orgIds);
        Future<Object> productionCount = threadPool.submit(productionCountCallable);
        //已生效的甲方批量金额（含税）
        Callable<Object> quoteCountCallable = new QuoteCountCallable(InvocationInfoProxy.getTenantid(), projectIds, RequestContextHolder.getRequestAttributes(), authority, orgIds);
        Future<Object> quoteCount = threadPool.submit(quoteCountCallable);
        //已生效的有合同收款登记金额（含税）
        Callable<Object> receiveCountCallable = new ReceiveCountCallable(InvocationInfoProxy.getTenantid(), projectIds, RequestContextHolder.getRequestAttributes(), authority, orgIds);
        Future<Object> receiveCount = threadPool.submit(receiveCountCallable);
        //已生效的有合同开票登记金额（含税）
        Callable<Object> invoiceOpenCountCallable = new InvoiceOpenCountCallable(InvocationInfoProxy.getTenantid(), projectIds, RequestContextHolder.getRequestAttributes(), authority, orgIds);
        Future<Object> invoiceOpenCount = threadPool.submit(invoiceOpenCountCallable);


        try {

            data.put("incomeContractAmtCount", countIncomeAmt.get());
            data.put("claimAmtCount", claimAmtCount.get());
            data.put("productionCount", productionCount.get());
            data.put("quoteCount", quoteCount.get());
            data.put("receiveCount", receiveCount.get());
            data.put("invoiceOpenCount", invoiceOpenCount.get());

            return CommonResponse.success(data);
        } catch (Exception e) {
            logger.error("统计合同收入情况异常, ", e);
        } finally {
            threadPool.shutdown();
        }

        return CommonResponse.error("统计收入合同信息失败！");
    }

    /**
     * 统计生效施工合同的含税金额
     */
    class IncomeCountCallable implements Callable<BigDecimal> {

        private Long tenantId;
        private List<Long> proIds;
        private RequestAttributes context;
        private String authority;
        private List<Long> orgIds;

        public IncomeCountCallable(Long tenantId, List<Long> proIds, RequestAttributes context, String authority, List<Long> orgIds) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.context = context;
            this.authority = authority;
            this.orgIds = orgIds;
        }

        @Override
        public BigDecimal call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            QueryParam queryParam = new QueryParam();
            //本租户下
            queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, tenantId));
            if (CollectionUtils.isNotEmpty(orgIds)) {
                queryParam.getParams().put("org_id", new Parameter(QueryParam.IN, orgIds));
            }
            //符合条件的项目
            if(CollectionUtils.isNotEmpty(proIds)) {
                queryParam.getParams().put("project_id", new Parameter(QueryParam.IN, proIds));
            }
            /** 移动端  收入合同统计金额要去掉 补充协议的金额 2021年8月14日*/
            queryParam.getParams().put("supplementFlag", new Parameter(QueryParam.EQ, 0));
            //有效的单据
            queryParam.getParams().put("billState", new Parameter(QueryParam.IN,
                    Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()})));

            Map<String, Object> result = contractService.countContractAmount(queryParam);
            String num = (null != result && null != result.get("curAmount")) ? result.get("curAmount").toString() : "0";
            BigDecimal amt = new BigDecimal(num);
            amt.setScale(8, BigDecimal.ROUND_HALF_UP);

            return amt.divide(new BigDecimal(10000), 2, BigDecimal.ROUND_HALF_UP);
        }
    }


    /**
     * 统计生效的变迁变更索赔单金额
     */
    class ClaimAmtCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private RequestAttributes context;
        private String authority;
        private List<Long> orgIds;

        public ClaimAmtCountCallable(Long tenantId, List<Long> proIds, RequestAttributes context, String authority, List<Long> orgIds) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.context = context;
            this.authority = authority;
            this.orgIds = orgIds;
        }

        @Override
        public Object call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            QueryWrapper<ClaimEntity> queryWrapper = new QueryWrapper<>();
            //本租户下
            queryWrapper.eq("tenant_id", tenantId);
            if (CollectionUtils.isNotEmpty(orgIds)) {
                queryWrapper.in("org_id", orgIds);
            }
            //符合条件的项目
            if(CollectionUtils.isNotEmpty(proIds)) {
                queryWrapper.in("project_id", proIds);
            }
            //有效的单据
            queryWrapper.in("bill_state", Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()}));
            queryWrapper.select(" round(ifnull(sum(reply_mny),0) / 10000, 2) as amt");
            Map<String, Object> result = claimServicel.getMap(queryWrapper);

            return result.get("amt");
        }
    }

    /**
     * 已生效的产值统计金额（含税）
     */
    class ProductionCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private RequestAttributes context;
        private String authority;
        private List<Long> orgIds;

        public ProductionCountCallable(Long tenantId, List<Long> proIds, RequestAttributes context, String authority, List<Long> orgIds) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.context = context;
            this.authority = authority;
            this.orgIds = orgIds;
        }

        @Override
        public Object call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            QueryWrapper<ProductionEntity> queryWrapper = new QueryWrapper<>();
            //本租户下
            queryWrapper.eq("tenant_id", tenantId);
            if (CollectionUtils.isNotEmpty(orgIds)) {
                queryWrapper.in("org_id", orgIds);
            }
            //符合条件的项目
            if(CollectionUtils.isNotEmpty(proIds)) {
                queryWrapper.in("project_id", proIds);
            }
            //有效的单据
            queryWrapper.in("bill_state", Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()}));
            queryWrapper.select(" round(ifnull(sum(finish_tax_mny),0) / 10000, 2) as amt");

            Map<String, Object> result = productionService.getMap(queryWrapper);
            return result.get("amt");
        }
    }

    /**
     * 已生效的产值统计金额（含税）
     */
    class QuoteCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private RequestAttributes context;
        private String authority;
        private List<Long> orgIds;

        public QuoteCountCallable(Long tenantId, List<Long> proIds, RequestAttributes context, String authority, List<Long> orgIds) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.context = context;
            this.authority = authority;
            this.orgIds = orgIds;
        }

        @Override
        public Object call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            QueryWrapper<QuoteEntity> queryWrapper = new QueryWrapper<>();
            //本租户下
            queryWrapper.eq("tenant_id", tenantId);
            if (CollectionUtils.isNotEmpty(orgIds)) {
                queryWrapper.in("org_id", orgIds);
            }
            queryWrapper.eq("dr", BaseVO.DR_UNDELETE);
            //符合条件的项目
            if(CollectionUtils.isNotEmpty(proIds)) {
                queryWrapper.in("project_id", proIds);
            }
            //有效的单据
            queryWrapper.in("bill_state", Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()}));
            queryWrapper.select(" round(ifnull(sum(quote_tax_mny),0) / 10000, 2) as amt");

            Map<String, Object> result = quoteService.getMap(queryWrapper);

            return result.get("amt");
        }
    }

    /**
     * 已生效的有合同收款登记金额（含税）
     */
    class ReceiveCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private RequestAttributes context;
        private String authority;
        private List<Long> orgIds;

        public ReceiveCountCallable(Long tenantId, List<Long> proIds, RequestAttributes context, String authority, List<Long> orgIds) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.context = context;
            this.authority = authority;
            this.orgIds = orgIds;
        }

        @Override
        public Object call() throws Exception {
            Map<String, Object> param = new HashMap<>();
            param.put("tenantId", tenantId);
            param.put("projectIds", proIds);
            if (CollectionUtils.isNotEmpty(orgIds)) {
                param.put("orgIds", orgIds);
            }

            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            String resp = HttpTookit.postByJson(BASE_HOST + "ejc-finance-web/api/receive/countByProperties",
                    JSONObject.toJSONString(param));
            logger.info("查询租户-{}下[项目列表-{}]已生效的有合同收款登记金额（含税）结果：{}", tenantId, proIds, resp);

            try {
                CommonResponse<BigDecimal> res = JSONObject.parseObject(resp, CommonResponse.class);
                if(!res.isSuccess()) {
                    logger.error("查询已生效的有合同收款登记金额（含税）失败，原因：{}, ", res.getMsg());
                    return 0;
                }
                return res.getData();
            } catch (Exception e) {
                logger.error("查询已生效的有合同收款登记金额（含税）异常, ", e);
            }
            return 0;
        }
    }

    /**
     * 已生效的有合同开票登记金额（含税）
     */
    class InvoiceOpenCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private RequestAttributes context;
        private String authority;
        private List<Long> orgIds;

        public InvoiceOpenCountCallable(Long tenantId, List<Long> proIds, RequestAttributes context, String authority, List<Long> orgIds) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.context = context;
            this.authority = authority;
            this.orgIds = orgIds;
        }

        @Override
        public Object call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            Map<String, Object> param = new HashMap<>();
            param.put("tenantId", tenantId);
            param.put("projectIds", proIds);
            if (CollectionUtils.isNotEmpty(orgIds)) {
                param.put("orgIds", orgIds);
            }

            String resp = HttpTookit.postByJson(BASE_HOST + "ejc-tax-web/api/tax/countInvoiceOpenByProperties",
                    JSONObject.toJSONString(param));
            logger.info("查询租户-{}下[项目列表-{}]已生效的有合同开票登记金额（含税）结果：{}", tenantId, proIds, resp);

            try {
                CommonResponse<BigDecimal> res = JSONObject.parseObject(resp, CommonResponse.class);
                if(!res.isSuccess()) {
                    logger.error("查询已生效的有合同开票登记金额（含税）失败，原因：{}, ", res.getMsg());
                    return 0;
                }
                return res.getData();
            } catch (Exception e) {
                logger.error("查询已生效的有合同开票登记金额（含税）异常, ", e);
            }
            return 0;
        }
    }

    @PostMapping(value = "getListByQueryParam")
    public CommonResponse<JSONObject> getListByQueryParam(@RequestBody QueryParam queryParam) {
        List<ContractEntity> list = contractService.queryList(queryParam, false);
        JSONObject back = new JSONObject();
        back.put("records", list);
        return CommonResponse.success(back);
    }
}
