package com.ejianc.business.income.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.center.api.IWarnCenterApi;
import com.ejianc.business.center.vo.EarlyWarnTransVO;
import com.ejianc.business.equipment.api.IEquipmentWarnApi;
import com.ejianc.business.equipment.vo.warn.EquipmentWarnVo;
import com.ejianc.business.income.service.IContractService;
import com.ejianc.business.income.vo.SqlParam;
import com.ejianc.business.income.vo.warn.IncomeContractWarnVo;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.material.api.IMaterialWarnApi;
import com.ejianc.business.material.vo.warn.MaterialWarnVo;
import com.ejianc.business.sub.api.ISubWarnApi;
import com.ejianc.business.sub.vo.warn.SubWarnVo;
import com.ejianc.foundation.tenant.api.ITenantApi;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@RestController
@RequestMapping(value = "/contractWarn/")
public class ContractWarnController {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String PC_URL_INCOME = "/ejc-income-frontend/#/incomeContract/contractApprove?id=";
    private static final String PC_URL_PROJECT = "/ejc-report-frontend/#/proPeportDetail?id=";

    @Autowired
    private ITenantApi tenantApi;
    @Autowired
    private IWarnCenterApi warnCenterApi;
    @Autowired
    private IContractService contractService;
    @Autowired
    private ISubWarnApi subWarnApi;
    @Autowired
    private IMaterialWarnApi materialWarnApi;
    @Autowired
    private IEquipmentWarnApi equipmentWarnApi;
    @Autowired
    private IProjectApi projectApi;


    /**
     * 施工合同-【收款金额】小于【报量金额】
     *
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "incomeMnyWarn")
    public CommonResponse<String> payMnyWarn(@RequestBody JSONObject jsonParam) {
        logger.info("接收到Task参数：{}", jsonParam);
        List<Long> ignoreTenantIds = new ArrayList<>();
        if (StringUtils.isNotBlank(jsonParam.getString("ignoreTenantIds"))) {
            ignoreTenantIds.addAll(JSONArray.parseArray(jsonParam.getString("ignoreTenantIds"), Long.class));
        }
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));

        CommonResponse<List<Long>> tenantResp = tenantApi.getAllValidTenantId();
        if (!tenantResp.isSuccess()) {
            return CommonResponse.error("执行施工合同收款金额小于等于报量金额预警失败，获取有效的租户Id列表失败！");
        }

        List<Long> tenantIds = tenantResp.getData();
        if (CollectionUtils.isEmpty(tenantIds)) {
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }
        //1.去掉停用的预警对应的租户Id（不用任何处理）
        tenantIds.removeAll(ignoreTenantIds);
        if (CollectionUtils.isNotEmpty(ignoreTenantIds) && CollectionUtils.isEmpty(tenantIds)) {
            logger.error("执行施工合同收款金额小于等于报量金额预警完成，有效的租户Id列表为空！", tenantResp.getMsg());
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        //2.再去掉自定义租户ID，走自定义的预警条件
        List<SqlParam> sqlParamList = new ArrayList<>();
        List<SqlParam> warnParamList = null;
        List<Long> customTenantIds = new ArrayList<>();
        for (JSONObject param : billParams) {
            warnParamList = BeanMapper.mapList(param.getJSONArray("parameters"), SqlParam.class);
            for (SqlParam p : warnParamList) {
                p.setWarnLevel(param.getString("warnLevel"));
                p.setTenantId(Long.valueOf(param.getString("tenantId")));
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
            customTenantIds.add(Long.valueOf(param.getString("tenantId")));
        }

        //去除自定义预警设置的对应的租户Id
        tenantIds.removeAll(customTenantIds);

        //3.除了忽略和自定义，其他都走默认预警条件
        for (Long tenantId : tenantIds) {
            warnParamList = JSONArray.parseArray(defaultWarnInfo.getString("parameters"), SqlParam.class);
            for (SqlParam p : warnParamList) {
                p.setWarnLevel(defaultWarnInfo.getString("warnLevel"));
                p.setTenantId(tenantId);
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
        }

        List<Map<String, Object>> result = contractService.incomeMnyWarn(sqlParamList);
        if (CollectionUtils.isEmpty(result)) {
            logger.error("执行施工合同收款金额小于等于报量金额预警完成，未发现有匹配预警规则的合同信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        return sendWarnToTask(payTransToWarnVO(result, defaultWarnInfo), defaultWarnInfo.getLong("warnSetId"));
    }

    /**
     * 项目类-支出合同金额大于等于收入合同金额
     *
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "outIncomeMnyWarn")
    public CommonResponse<String> outIncomeMnyWarn(@RequestBody JSONObject jsonParam) {
        logger.info("接收到Task参数：{}", jsonParam);
        List<Long> ignoreTenantIds = new ArrayList<>();
        if (StringUtils.isNotBlank(jsonParam.getString("ignoreTenantIds"))) {
            ignoreTenantIds.addAll(JSONArray.parseArray(jsonParam.getString("ignoreTenantIds"), Long.class));
        }
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));

        CommonResponse<List<Long>> tenantResp = tenantApi.getAllValidTenantId();
        if (!tenantResp.isSuccess()) {
            return CommonResponse.error("执行项目类支出合同金额大于等于收入合同金额预警失败，获取有效的租户Id列表失败！");
        }

        List<Long> tenantIds = tenantResp.getData();
        if (CollectionUtils.isEmpty(tenantIds)) {
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }
        //1.去掉停用的预警对应的租户Id（不用任何处理）
        tenantIds.removeAll(ignoreTenantIds);
        if (CollectionUtils.isNotEmpty(ignoreTenantIds) && CollectionUtils.isEmpty(tenantIds)) {
            logger.error("执行项目类支出合同金额大于等于收入合同金额预警完成，有效的租户Id列表为空！", tenantResp.getMsg());
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        //2.再去掉自定义租户ID，走自定义的预警条件
        List<SqlParam> sqlParamList = new ArrayList<>();
        List<SqlParam> warnParamList = null;
        List<Long> customTenantIds = new ArrayList<>();
        for (JSONObject param : billParams) {
            warnParamList = BeanMapper.mapList(param.getJSONArray("parameters"), SqlParam.class);
            for (SqlParam p : warnParamList) {
                p.setWarnLevel(param.getString("warnLevel"));
                p.setTenantId(Long.valueOf(param.getString("tenantId")));
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
            customTenantIds.add(Long.valueOf(param.getString("tenantId")));
        }

        //去除自定义预警设置的对应的租户Id
        tenantIds.removeAll(customTenantIds);

        //3.除了忽略和自定义，其他都走默认预警条件
        for (Long tenantId : tenantIds) {
            warnParamList = JSONArray.parseArray(defaultWarnInfo.getString("parameters"), SqlParam.class);
            for (SqlParam p : warnParamList) {
                p.setWarnLevel(defaultWarnInfo.getString("warnLevel"));
                p.setTenantId(tenantId);
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
        }

        //分包合同、物资合同、设备采购合同、设备租赁
        List<Long> tids = sqlParamList.stream().map(SqlParam::getTenantId).collect(Collectors.toList());
        CommonResponse<List<SubWarnVo>> subCommon = subWarnApi.subProjectOutMny(tids);
        logger.debug("执行项目类支出合同金额大于等于收入合同金额，查询分包合同支出金额！");
        if (!subCommon.isSuccess()) {
            return CommonResponse.error("执行项目类支出合同金额大于等于收入合同金额预警失败，获取分包合同支出金额失败！");
        }
        CommonResponse<List<MaterialWarnVo>> materialCommon = materialWarnApi.materialProjectOutMny(tids);
        logger.debug("执行项目类支出合同金额大于等于收入合同金额，查询物资合同支出金额！");
        if (!materialCommon.isSuccess()) {
            return CommonResponse.error("执行项目类支出合同金额大于等于收入合同金额预警失败，获取物资合同支出金额失败！");
        }
        CommonResponse<List<EquipmentWarnVo>> equipmentCommon = equipmentWarnApi.equipmentProjectOutMny(tids);
        logger.debug("执行项目类支出合同金额大于等于收入合同金额，查询设备采购、设备租赁支出金额！");
        if (!equipmentCommon.isSuccess()) {
            return CommonResponse.error("执行项目类支出合同金额大于等于收入合同金额预警失败，获取设备采购、设备租赁支出金额失败！");
        }
        List<IncomeContractWarnVo> incomeList = contractService.outIncomeMnyWarn(tids);

        List<IncomeContractWarnVo> incomeContractWarnVos = outIncomeWarn(incomeList, sqlParamList, subCommon.getData(), materialCommon.getData(), equipmentCommon.getData());
        if (CollectionUtils.isEmpty(incomeContractWarnVos)) {
            logger.error("执行项目类支出合同金额大于等于收入合同金额预警完成，未发现有匹配预警规则的合同信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        return sendWarnToTask(payTransToWarnVO2(incomeContractWarnVos, defaultWarnInfo), defaultWarnInfo.getLong("warnSetId"));
    }

    private CommonResponse<String> sendWarnToTask(List<EarlyWarnTransVO> warnList, Long warnId) {
        CommonResponse<String> warnResp = warnCenterApi.sendToWarnCenter(warnList, warnId);
        if (!warnResp.isSuccess()) {
            logger.error("回调预警任务服务失败，{}", warnResp.getMsg());
            return CommonResponse.error("回调预警任务服务失败" + warnResp.getMsg());
        }

        return CommonResponse.success("任务接受处理成功！");
    }

    private List<EarlyWarnTransVO> payTransToWarnVO(List<Map<String, Object>> result, JSONObject defaultWarnInfo) {
        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");
        for (Map<String, Object> m : result) {
            EarlyWarnTransVO vo = new EarlyWarnTransVO();
            vo.setBillName((null != defaultWarnInfo.get("categoryName") ? defaultWarnInfo.get("categoryName").toString() + " - " : "") + m.get("contractCode").toString());
            vo.setPcTitle("施工合同");
            vo.setPcUrl(PC_URL_INCOME + m.get("contractId").toString());
            vo.setOrgId(Long.valueOf(m.get("orgId").toString()));
            vo.setOrgName(m.get("orgName").toString());
            vo.setSourceId(m.get("contractId").toString());
            vo.setTenantId(Long.valueOf(m.get("tenantId").toString()));
            vo.setWarnLevel(m.get("warnLevel").toString());
            vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
            vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

            tmpContent = defaultWarnInfo.getString("earlywarnContent");
            vo.setEarlywarnContent(
                    tmpContent.replaceAll("#contractName#", m.get("contractName").toString())
                            .replaceAll("#contractTaxMny#", df.format(getBigDecimal(m.get("contractTaxMny"))))
                            .replaceAll("#sumQuoteTaxMny#", df.format(getBigDecimal(m.get("sumQuoteTaxMny"))))
                            .replaceAll("#percent#", m.get("percent").toString())
                            .replaceAll("#overMny#", df.format(getBigDecimal(m.get("overMny"))))
            );

            resp.add(vo);
        }

        logger.info("发送预警参数：{}", JSON.toJSONString(resp));
        return resp;
    }

    private BigDecimal getBigDecimal(Object obj) {
        return null == obj ? BigDecimal.ZERO : new BigDecimal(obj.toString());
    }

    private List<IncomeContractWarnVo> outIncomeWarn(List<IncomeContractWarnVo> incomeVo, List<SqlParam> sqlParamList,
                                                     List<SubWarnVo> subVo, List<MaterialWarnVo> materialVo, List<EquipmentWarnVo> equipmentVo) {
        List<IncomeContractWarnVo> outList = new ArrayList<>();
        outList.addAll(BeanMapper.mapList(subVo, IncomeContractWarnVo.class));
        outList.addAll(BeanMapper.mapList(materialVo, IncomeContractWarnVo.class));
        outList.addAll(BeanMapper.mapList(equipmentVo, IncomeContractWarnVo.class));
        if (outList.size() < 1) {
            logger.debug("执行项目类支出合同金额大于等于收入合同金额预警完成，支出合同为空！");
            return null;
        }
        // 支出合同金额
        Map<Long, IncomeContractWarnVo> outMap = new HashMap<>();
        for (IncomeContractWarnVo outVo : outList) {
            if (outMap.containsKey(outVo.getProjectId())) {
                IncomeContractWarnVo incomeContractWarnVo = outMap.get(outVo.getProjectId());
                incomeContractWarnVo.setOutMny(incomeContractWarnVo.getOutMny().add(outVo.getOutMny()));
                outMap.put(outVo.getProjectId(), incomeContractWarnVo);
            } else {
                outMap.put(outVo.getProjectId(), outVo);
            }
        }
        // 收入合同金额
        Map<Long, IncomeContractWarnVo> inMap = new HashMap<>();
        for (IncomeContractWarnVo inVo : incomeVo) {
            inMap.put(inVo.getProjectId(), inVo);
        }
        //分租户
        Map<Long, SqlParam> tenantMap = new HashMap<>();
        for (SqlParam sqlVo : sqlParamList) {
            tenantMap.put(sqlVo.getTenantId(), sqlVo);
        }

        List<IncomeContractWarnVo> result = new ArrayList<>();
        for (IncomeContractWarnVo inoutVo : outMap.values()) {//遍历支出
            SqlParam sqlParam = tenantMap.get(inoutVo.getTenantId());
            if (inMap.containsKey(inoutVo.getProjectId())) {//有支出，有收入
                IncomeContractWarnVo inVo = inMap.get(inoutVo.getProjectId());//收入
                //支出合同金额>=收入合同金额*X%
                BigDecimal overMny = inVo.getContractTaxMny().multiply(new BigDecimal(sqlParam.getValue()));
                if (inoutVo.getOutMny().compareTo(overMny) >= 0) {
                    inoutVo.setContractTaxMny(inVo.getContractTaxMny());
                    inoutVo.setOverMny(overMny);
                    inoutVo.setWarnLevel(sqlParam.getWarnLevel());
                    inoutVo.setPercent(sqlParam.getValue() * 100);
                    result.add(inoutVo);
                }
            } else {//有支出，没收入
                inoutVo.setContractTaxMny(BigDecimal.ZERO);
                inoutVo.setOverMny(BigDecimal.ZERO);
                inoutVo.setWarnLevel(sqlParam.getWarnLevel());
                inoutVo.setPercent(sqlParam.getValue() * 100);
                result.add(inoutVo);
            }
        }
        // 项目编码
        if (result.size() > 0) {
            List<Long> projectIds = result.stream().map(IncomeContractWarnVo::getProjectId).collect(Collectors.toList());
            CommonResponse<List<ProjectRegisterVO>> res = projectApi.queryProjectByIds(projectIds);
            Map<Long, String> projectMap = new HashMap<>();
            if (res.isSuccess() && !res.getData().isEmpty()) {
                projectMap = res.getData().stream().collect(Collectors.toMap(ProjectRegisterVO::getId, ProjectRegisterVO::getCode));
            }
            for (IncomeContractWarnVo re : result) {
                re.setProjectCode(projectMap.get(re.getProjectId()));
            }
            return result;
        } else {
            logger.debug("执行项目类支出合同金额大于等于收入合同金额预警完成，符合预警的为空！");
            return null;
        }
    }

    private List<EarlyWarnTransVO> payTransToWarnVO2(List<IncomeContractWarnVo> result, JSONObject defaultWarnInfo) {
        logger.debug("执行项目类支出合同金额大于等于收入合同金额预警完成，发送预警！");
        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");
        for (IncomeContractWarnVo m : result) {
            EarlyWarnTransVO vo = new EarlyWarnTransVO();
            vo.setBillName((null != defaultWarnInfo.get("categoryName") ? defaultWarnInfo.get("categoryName").toString() + " - " : "") + m.getProjectCode());
            vo.setPcTitle("项目统计");
            vo.setPcUrl(PC_URL_PROJECT + m.getProjectId()+"&orgId="+m.getOrgId());
            vo.setOrgId(Long.valueOf(m.getOrgId()));
            vo.setOrgName(m.getOrgName());
            vo.setSourceId(m.getProjectId().toString());
            vo.setTenantId(Long.valueOf(m.getTenantId().toString()));
            vo.setWarnLevel(m.getWarnLevel());
            vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
            vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

            tmpContent = defaultWarnInfo.getString("earlywarnContent");
            vo.setEarlywarnContent(
                    tmpContent.replaceAll("#projectName#", m.getProjectName())
                            .replaceAll("#outMny#", df.format(getBigDecimal(m.getOutMny())))
                            .replaceAll("#contractTaxMny#", df.format(getBigDecimal(m.getContractTaxMny())))
                            .replaceAll("#percent#", m.getPercent().toString())
                            .replaceAll("#overMny#", df.format(getBigDecimal(m.getOverMny())))
            );

            resp.add(vo);
        }

        logger.info("发送预警参数：{}", JSON.toJSONString(resp));
        return resp;
    }
}
