package com.ejianc.business.income.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.center.api.IWarnCenterApi;
import com.ejianc.business.center.vo.EarlyWarnTransVO;
import com.ejianc.business.equipment.api.IEquipmentWarnApi;
import com.ejianc.business.equipment.vo.warn.EquipmentWarnVo;
import com.ejianc.business.finance.api.IPayContractApi;
import com.ejianc.business.finance.vo.FinanceWarnV0;
import com.ejianc.business.income.service.IContractService;
import com.ejianc.business.income.vo.ContractVo;
import com.ejianc.business.income.vo.SqlParam;
import com.ejianc.business.income.vo.warn.IncomeContractWarnVo;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.material.api.IMaterialWarnApi;
import com.ejianc.business.material.vo.warn.MaterialWarnVo;
import com.ejianc.business.sub.api.ISubWarnApi;
import com.ejianc.business.sub.vo.warn.SubWarnVo;
import com.ejianc.foundation.tenant.api.ITenantApi;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

@RestController
@RequestMapping(value = "/projectPayWarn/")
public class PayContractWarnController {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String PC_URL_INCOME = "/ejc-income-frontend/#/incomeContract/contractApprove?id=";
    private static final String PC_URL_PROJECT = "/ejc-report-frontend/#/proPeportDetail?id=";

    @Autowired
    private ITenantApi tenantApi;
    @Autowired
    private IWarnCenterApi warnCenterApi;
    @Autowired
    private IContractService contractService;
    @Autowired
    private ISubWarnApi subWarnApi;
    @Autowired
    private IMaterialWarnApi materialWarnApi;
    @Autowired
    private IEquipmentWarnApi equipmentWarnApi;
    @Autowired
    private IProjectApi projectApi;
    @Autowired
    private IPayContractApi payContractApi;



    /**
     * 未付款=该项目所有支出合同的合同总额（去掉集采合同）-合同的实际付款额；未收款=该项目所有收入合同的合同总额-合同的实际收款额；
     * 按照项目维度，未付款额大于未收款额*x%时，预警
     *
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "projectPayContractMnyWarn")
    public CommonResponse<String> outIncomeMnyWarn(@RequestBody JSONObject jsonParam) {
        logger.info("接收到Task参数：{}", jsonParam);
        List<Long> ignoreTenantIds = new ArrayList<>();
        if (StringUtils.isNotBlank(jsonParam.getString("ignoreTenantIds"))) {
            ignoreTenantIds.addAll(JSONArray.parseArray(jsonParam.getString("ignoreTenantIds"), Long.class));
        }
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));

        CommonResponse<List<Long>> tenantResp = tenantApi.getAllValidTenantId();
        if (!tenantResp.isSuccess()) {
            return CommonResponse.error("执行项目类支出合同金额大于等于收入合同金额预警失败，获取有效的租户Id列表失败！");
        }

        List<Long> tenantIds = tenantResp.getData();
        if (CollectionUtils.isEmpty(tenantIds)) {
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }
        //1.去掉停用的预警对应的租户Id（不用任何处理）
        tenantIds.removeAll(ignoreTenantIds);
        if (CollectionUtils.isNotEmpty(ignoreTenantIds) && CollectionUtils.isEmpty(tenantIds)) {
            logger.error("有效的租户Id列表为空！", tenantResp.getMsg());
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        //2.再去掉自定义租户ID，走自定义的预警条件
        List<SqlParam> sqlParamList = new ArrayList<>();
        List<SqlParam> warnParamList = null;
        List<Long> customTenantIds = new ArrayList<>();
        for (JSONObject param : billParams) {
            warnParamList = BeanMapper.mapList(param.getJSONArray("parameters"), SqlParam.class);
            for (SqlParam p : warnParamList) {
                p.setWarnLevel(param.getString("warnLevel"));
                p.setTenantId(Long.valueOf(param.getString("tenantId")));
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
            customTenantIds.add(Long.valueOf(param.getString("tenantId")));
        }


        tenantIds.removeAll(customTenantIds);

        //3.除了忽略和自定义，其他都走默认预警条件
        for (Long tenantId : tenantIds) {
            warnParamList = JSONArray.parseArray(defaultWarnInfo.getString("parameters"), SqlParam.class);
            for (SqlParam p : warnParamList) {
                p.setWarnLevel(defaultWarnInfo.getString("warnLevel"));
                p.setTenantId(tenantId);
                p.setValue(p.getValue() / 100d);
                sqlParamList.add(p);
            }
        }

        //分包合同、物资合同、设备采购合同、设备租赁
        List<Long> tids = sqlParamList.stream().map(SqlParam::getTenantId).collect(Collectors.toList());
        CommonResponse<List<SubWarnVo>> subCommon = subWarnApi.subProjectOutMny(tids);
        logger.debug("查询分包合同支出金额！");
        if (!subCommon.isSuccess()) {
            return CommonResponse.error("获取分包合同支出金额失败！");
        }
        CommonResponse<List<MaterialWarnVo>> materialCommon = materialWarnApi.materialProjectOutMny(tids);
        logger.debug("查询物资合同支出金额！");
        logger.info("tids：{}", tids);
        if (!materialCommon.isSuccess()) {
            return CommonResponse.error("获取物资合同支出金额失败！");
        }
        CommonResponse<List<EquipmentWarnVo>> equipmentCommon = equipmentWarnApi.equipmentProjectOutMny(tids);
        logger.debug("查询设备采购、设备租赁支出金额！");
        if (!equipmentCommon.isSuccess()) {
            return CommonResponse.error("获取设备采购、设备租赁支出金额失败！");
        }
        //该项目所有收入合同的合同总额
        List<IncomeContractWarnVo> incomeList = contractService.outIncomeMnyWarn(tids);
        //项目合同的实际付款额
        CommonResponse<List<FinanceWarnV0>> projectPayMny = payContractApi.getProjectPayMny();
        logger.debug("查询项目合同的实际付款额！");
        if (!projectPayMny.isSuccess()){
            return CommonResponse.error("获取项目合同的实际付款额失败！");
        }
        List<IncomeContractWarnVo> incomeContractWarnVos = outIncomeWarn(incomeList, sqlParamList, subCommon.getData(), materialCommon.getData(), equipmentCommon.getData(),projectPayMny.getData());
        if (CollectionUtils.isEmpty(incomeContractWarnVos)) {
            logger.error("执行未付款额大于未收款额预警完成，未发现有匹配预警规则的合同信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        return sendWarnToTask(payTransToWarnVO2(incomeContractWarnVos, defaultWarnInfo), defaultWarnInfo.getLong("warnSetId"));
    }

    private CommonResponse<String> sendWarnToTask(List<EarlyWarnTransVO> warnList, Long warnId) {
        CommonResponse<String> warnResp = warnCenterApi.sendToWarnCenter(warnList, warnId);
        if (!warnResp.isSuccess()) {
            logger.error("回调预警任务服务失败，{}", warnResp.getMsg());
            return CommonResponse.error("回调预警任务服务失败" + warnResp.getMsg());
        }

        return CommonResponse.success("任务接受处理成功！");
    }



    private BigDecimal getBigDecimal(Object obj) {
        return null == obj ? BigDecimal.ZERO : new BigDecimal(obj.toString());
    }

    private List<IncomeContractWarnVo> outIncomeWarn(List<IncomeContractWarnVo> incomeVo, List<SqlParam> sqlParamList,
                                                     List<SubWarnVo> subVo, List<MaterialWarnVo> materialVo, List<EquipmentWarnVo> equipmentVo,List<FinanceWarnV0> list) {
        List<IncomeContractWarnVo> outList = new ArrayList<>();
        outList.addAll(BeanMapper.mapList(subVo, IncomeContractWarnVo.class));
        outList.addAll(BeanMapper.mapList(materialVo, IncomeContractWarnVo.class));
        outList.addAll(BeanMapper.mapList(equipmentVo, IncomeContractWarnVo.class));
        if (outList.size() < 1) {
            logger.debug("执行该项目所有收入合同的合同总额预警完成，支出合同为空！");
            return null;
        }
        // 支出合同金额
        Map<Long, IncomeContractWarnVo> outMap = new HashMap<>();
        for (IncomeContractWarnVo outVo : outList) {
            if (outMap.containsKey(outVo.getProjectId())) {
                IncomeContractWarnVo incomeContractWarnVo = outMap.get(outVo.getProjectId());
                incomeContractWarnVo.setOutMny(incomeContractWarnVo.getOutMny().add(outVo.getOutMny()));
                outMap.put(outVo.getProjectId(), incomeContractWarnVo);
            } else {
                outMap.put(outVo.getProjectId(), outVo);
            }
        }
        //合同的实际付款额
        HashMap<Long, FinanceWarnV0> map = new HashMap<>();
        for (FinanceWarnV0 financeWarnV0 : list) {
            map.put(financeWarnV0.getProjectId(),financeWarnV0);
        }
        // 收入合同金额
        Map<Long, IncomeContractWarnVo> inMap = new HashMap<>();
        for (IncomeContractWarnVo inVo : incomeVo) {
            inMap.put(inVo.getProjectId(), inVo);
        }
        //分租户
        Map<Long, SqlParam> tenantMap = new HashMap<>();
        for (SqlParam sqlVo : sqlParamList) {
            tenantMap.put(sqlVo.getTenantId(), sqlVo);
        }
        List<IncomeContractWarnVo> result = new ArrayList<>();
        for (IncomeContractWarnVo inoutVo : outMap.values()) {//遍历支出
            SqlParam sqlParam = tenantMap.get(inoutVo.getTenantId());
            BigDecimal outMny = inoutVo.getOutMny();
            if (outMny==null){
                outMny=BigDecimal.ZERO;
            }
            if (map.containsKey(inoutVo.getProjectId())){
                inoutVo.setUncollectedMny(ComputeUtil.safeSub(outMny, map.get(inoutVo.getProjectId()).getSumPayMny()));
            }else {
                inoutVo.setUncollectedMny(outMny);
            }
            if (inMap.containsKey(inoutVo.getProjectId())) {//有支出，有收入
                //未付款
                BigDecimal uncollectedMny = inoutVo.getUncollectedMny();
                //该项目所有收入合同的合同总额
                IncomeContractWarnVo incomeContractWarnVo = inMap.get(inoutVo.getProjectId());
                BigDecimal inContractTaxMny = incomeContractWarnVo.getContractTaxMny();
                //合同的实际收款额
                BigDecimal sumCollectMny = incomeContractWarnVo.getSumCollectMny();
                //未收款金额
                BigDecimal overMny = ComputeUtil.safeMultiply(ComputeUtil.safeSub(inContractTaxMny, sumCollectMny), new BigDecimal(sqlParam.getValue()));

                if (uncollectedMny.compareTo(overMny) >= 0) {
                    inoutVo.setUncollectedMny(uncollectedMny);
                    inoutVo.setOverMny(overMny);
                    inoutVo.setDiff( ComputeUtil.safeSub(uncollectedMny, overMny));
                    inoutVo.setWarnLevel(sqlParam.getWarnLevel());
                    inoutVo.setPercent(sqlParam.getValue() * 100);
                    result.add(inoutVo);
                }
            } else {//有支出，没收入
                inoutVo.setOverMny(BigDecimal.ZERO);
                inoutVo.setDiff(inoutVo.getUncollectedMny());
                inoutVo.setWarnLevel(sqlParam.getWarnLevel());
                inoutVo.setPercent(sqlParam.getValue() * 100);
                result.add(inoutVo);
            }
        }
        // 项目编码
        if (result.size() > 0) {
            List<Long> projectIds = result.stream().map(IncomeContractWarnVo::getProjectId).collect(Collectors.toList());
            CommonResponse<List<ProjectRegisterVO>> res = projectApi.queryProjectByIds(projectIds);
            Map<Long, String> projectMap = new HashMap<>();
            if (res.isSuccess() && !res.getData().isEmpty()) {
                projectMap = res.getData().stream().collect(Collectors.toMap(ProjectRegisterVO::getId, ProjectRegisterVO::getCode));
            }
            for (IncomeContractWarnVo re : result) {
                re.setProjectCode(projectMap.get(re.getProjectId()));
            }
            return result;
        } else {
            logger.debug("执行未付款额大于未收款额预警完成，符合预警的为空！");
            return null;
        }
    }

    private List<EarlyWarnTransVO> payTransToWarnVO2(List<IncomeContractWarnVo> result, JSONObject defaultWarnInfo) {
        logger.debug("未付款=该项目所有支出合同的合同总额（去掉集采合同）-合同的实际付款额；未收款=该项目所有收入合同的合同总额-合同的实际收款额；\n" +
                "按照项目维度，未付款额大于未收款额预警完成，发送预警！");
        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");
        for (IncomeContractWarnVo m : result) {
            EarlyWarnTransVO vo = new EarlyWarnTransVO();
            vo.setBillName((null != defaultWarnInfo.get("categoryName") ? defaultWarnInfo.get("categoryName").toString() + " - " : "") + m.getProjectCode());
            vo.setPcTitle("未付款额大于未收款额");
            vo.setPcUrl(PC_URL_PROJECT + m.getProjectId()+"&orgId="+m.getOrgId());
            vo.setOrgId(Long.valueOf(m.getOrgId()));
            vo.setOrgName(m.getOrgName());
            vo.setSourceId(m.getProjectId().toString());
            vo.setTenantId(Long.valueOf(m.getTenantId().toString()));
            vo.setWarnLevel(m.getWarnLevel());
            vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
            vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

            tmpContent = defaultWarnInfo.getString("earlywarnContent");
            vo.setEarlywarnContent(
                    tmpContent.replaceAll("#projectName#", m.getProjectName())
                            .replaceAll("#uncollectedMny#", df.format(getBigDecimal(m.getUncollectedMny())))
                            .replaceAll("#percent#", m.getPercent().toString())
                            .replaceAll("#overMny#", df.format(getBigDecimal(m.getOverMny())))
                            .replaceAll("#diff#", df.format(getBigDecimal(m.getDiff())))
            );
            resp.add(vo);
        }

        logger.info("发送预警参数：{}", JSON.toJSONString(resp));
        return resp;
    }
}
