package com.ejianc.business.income.controller.api;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.ejianc.business.income.bean.*;
import com.ejianc.business.income.dto.ContractDto;
import com.ejianc.business.income.history.ProjectProgressVo;
import com.ejianc.business.income.service.*;
import com.ejianc.business.income.vo.ContractFinishSettleVO;
import com.ejianc.business.income.vo.ContractVo;
import com.ejianc.business.income.vo.QuoteVo;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/api/contract/")
public class ContractApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IContractService contractService;
    @Autowired
    private IContractFinishSettleService finishSettleService;

    @Autowired
    private IProductionService productionService;

    @Autowired
    private IClaimService claimServicel;

    @Autowired
    private IQuoteService quoteService;

    @Autowired
    private IProjectApi projectApi;

    @Value("${common.env.base-host}")
    private String BASE_HOST;
    @Autowired
    private IQuoteDetailService quoteDetailService;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IOrgApi orgApi;

    /**
     * 回写累计收款
     *
     * @param contractId
     * @param collectMny
     * @param type,累加=true,累减=false
     * @return
     */
    @RequestMapping(value = "updateCollectMny", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> updateCollectMny(@RequestParam(value = "contractId") Long contractId,
                                                   @RequestParam(value = "collectMny") BigDecimal collectMny,
                                                   @RequestParam(value = "type") Boolean type) {
        contractService.updateCollectMny(contractId, collectMny, type);
        return CommonResponse.success("更新累计收款成功！");
    }

    /**
     * 回写累计开票
     *
     * @param contractId
     * @param invoicingMny
     * @param type,累加=true,累减=false TODO
     * @return
     */
    @RequestMapping(value = "updateInvoicingMny", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> updateInvoicingMny(@RequestParam(value = "contractId") Long contractId,
                                                     @RequestParam(value = "invoicingTaxMny") BigDecimal invoicingTaxMny,
                                                     @RequestParam(value = "invoicingMny") BigDecimal invoicingMny,
                                                     @RequestParam(value = "type") Boolean type) {
        contractService.updateInvoicingMny(contractId, invoicingTaxMny, invoicingMny, type);
        return CommonResponse.success("更新累计开票成功！");
    }

    /**
     * 根据项目主键查询最新创建的合同
     *
     * @param projectId
     * @return
     */
    @RequestMapping(value = "searchContract", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ContractVo> searchContract(@RequestParam(value = "projectId") Long projectId) {
        return CommonResponse.success("根据项目主键查询最新创建的合同成功！", contractService.searchContract(projectId));
    }

    /**
     * 根据合同主键查询合同详情
     *
     * @param contractId
     * @return
     */
    @RequestMapping(value = "queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ContractVo> queryDetail(@RequestParam(value = "contractId") Long contractId) {
        return CommonResponse.success("根据合同主键查询合同详情成功！", contractService.contractDetail(contractId));
    }

    /**
     * 根据租户主键查询生效的合同列表
     *
     * @param tenantIds
     * @return
     */
    @RequestMapping(value = "searchContractByTenantIds", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<ContractVo>> searchContractByTenantIds(@RequestParam(value = "tenantIds") List<Long> tenantIds) {
        tenantIds.forEach(id->{
            logger.info("tenantIds----------------------------:"+id);
        });
        LambdaQueryWrapper<ContractEntity> lambda = new LambdaQueryWrapper<>();
        lambda.in(ContractEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.in(ContractEntity::getTenantId, tenantIds);
        List<ContractEntity> list = contractService.list(lambda);
        return CommonResponse.success("据租户主键查询生效的合同列表成功！", BeanMapper.mapList(list, ContractVo.class));
    }

    /**
     * 根据租户主键查询生效的合同列表
     * post请求
     *
     * @param tenantIds
     * @return
     */
    @RequestMapping(value = "postSearchContractByTenantIds", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<List<ContractDto>> postSearchContractByTenantIds(@RequestBody List<Long> tenantIds) {
        tenantIds.forEach(id->{
            logger.info("tenantIds----------------------------:"+id);
        });
        LambdaQueryWrapper<ContractEntity> lambda = new LambdaQueryWrapper<>();
        lambda.in(ContractEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.in(ContractEntity::getTenantId, tenantIds);
        List<ContractEntity> list = contractService.list(lambda);
        return CommonResponse.success("据租户主键查询生效的合同列表成功！", BeanMapper.mapList(list, ContractDto.class));
    }

    /**
     * 根据项目Id查询生效的竣工结算单
     * post请求
     * @param projectIds
     * @return
     */
    @RequestMapping(value = "queryFinishSettle", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<List<ContractFinishSettleVO>> queryFinishSettle(@RequestBody List<Long> projectIds) {
        LambdaQueryWrapper<ContractFinishSettleEntity> lambda = new LambdaQueryWrapper<>();
        lambda.in(ContractFinishSettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.in(ContractFinishSettleEntity::getProjectId, projectIds);
        List<ContractFinishSettleEntity> list = finishSettleService.list(lambda);
        return CommonResponse.success("根据项目ids查询生效的竣工结算列表成功！", BeanMapper.mapList(list, ContractFinishSettleVO.class));
    }

    /**
     * 收入合同统计
     *
     * @param projectRange 查询范围（1-所有项目，2-在建项目，3-完工项目，4-本年新签）
     * @return： incomeContractAmtCount: 统计生效施工合同的含税金额
     * claimAmtCount: 统计生效的变迁变更索赔单金额
     * productionCount: 已生效的产值统计金额（含税）
     * quoteCount: 已生效的甲方批量金额（含税）
     * receiveCount: 已生效的有合同收款登记金额（含税）
     * invoiceOpenCount: 已生效的有合同开票登记金额（含税）
     */
    @RequestMapping(value = "incomeCount", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> countIncomeContract(HttpServletRequest request, @RequestParam(value = "projectRange") int projectRange) {
        JSONObject data = new JSONObject();

        List<Long> projectIds = new ArrayList<>();
        if (projectRange != 1) {
            CommonResponse<List<Long>> projectIdsResp = projectApi.getProjectIdsByProperties(projectRange);
            if (!projectIdsResp.isSuccess()) {
                logger.error("根据条件 projectRange-{} 查询项目Id列表失败，原因：{}", projectRange, projectIdsResp.getMsg());
                return CommonResponse.error("查询失败，查询匹配的项目信息失败。");
            }
            projectIds = projectIdsResp.getData();

            if (CollectionUtils.isEmpty(projectIds)) {
                data.put("incomeContractAmtCount", 0);
                data.put("claimAmtCount", 0);
                data.put("productionCount", 0);
                data.put("quoteCount", 0);
                data.put("receiveCount", 0);
                data.put("invoiceOpenCount", 0);
                return CommonResponse.success(data);
            }
        }
        List<Long> orgIds = new ArrayList<>();
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            orgIds = authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        }  else {
            orgIds = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        }

        String authority = request.getHeader("authority");

        ExecutorService threadPool = Executors.newFixedThreadPool(6);
        //统计生效施工合同的含税金额
        Callable<BigDecimal> incomeCountCallable = new IncomeCountCallable(InvocationInfoProxy.getTenantid(), projectIds, orgIds, RequestContextHolder.getRequestAttributes(), authority);
        Future<BigDecimal> countIncomeAmt = threadPool.submit(incomeCountCallable);
        //统计生效的变迁变更索赔单金额
        Callable<Object> claimAmtCountCallable = new ClaimAmtCountCallable(InvocationInfoProxy.getTenantid(), projectIds,orgIds, RequestContextHolder.getRequestAttributes(), authority);
        Future<Object> claimAmtCount = threadPool.submit(claimAmtCountCallable);
        //已生效的产值统计金额（含税）
        Callable<Object> productionCountCallable = new ProductionCountCallable(InvocationInfoProxy.getTenantid(), projectIds,orgIds, RequestContextHolder.getRequestAttributes(), authority);
        Future<Object> productionCount = threadPool.submit(productionCountCallable);
        //已生效的甲方批量金额（含税）
        Callable<Object> quoteCountCallable = new QuoteCountCallable(InvocationInfoProxy.getTenantid(), projectIds, orgIds,RequestContextHolder.getRequestAttributes(), authority);
        Future<Object> quoteCount = threadPool.submit(quoteCountCallable);
        //已生效的有合同收款登记金额（含税）
        Callable<Object> receiveCountCallable = new ReceiveCountCallable(InvocationInfoProxy.getTenantid(), projectIds, orgIds,RequestContextHolder.getRequestAttributes(), authority);
        Future<Object> receiveCount = threadPool.submit(receiveCountCallable);
        //已生效的有合同开票登记金额（含税）
        Callable<Object> invoiceOpenCountCallable = new InvoiceOpenCountCallable(InvocationInfoProxy.getTenantid(), projectIds,orgIds, RequestContextHolder.getRequestAttributes(), authority);
        Future<Object> invoiceOpenCount = threadPool.submit(invoiceOpenCountCallable);


        try {

            data.put("incomeContractAmtCount", countIncomeAmt.get());
            data.put("claimAmtCount", claimAmtCount.get());
            data.put("productionCount", productionCount.get());
            data.put("quoteCount", quoteCount.get());
            data.put("receiveCount", receiveCount.get());
            data.put("invoiceOpenCount", invoiceOpenCount.get());

            return CommonResponse.success(data);
        } catch (Exception e) {
            logger.error("统计合同收入情况异常, ", e);
        } finally {
            threadPool.shutdown();
        }

        return CommonResponse.error("统计收入合同信息失败！");
    }

    /**
     * 统计生效施工合同的含税金额
     */
    class IncomeCountCallable implements Callable<BigDecimal> {

        private Long tenantId;
        private List<Long> proIds;
        private List<Long> orgIds;
        private RequestAttributes context;
        private String authority;

        public IncomeCountCallable(Long tenantId, List<Long> proIds, List<Long> orgIds, RequestAttributes context, String authority) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.orgIds = orgIds;
            this.context = context;
            this.authority = authority;
        }

        @Override
        public BigDecimal call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            QueryParam queryParam = new QueryParam();
            //本租户下
            queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, tenantId));
            //符合条件的项目
            if (CollectionUtils.isNotEmpty(proIds)) {
                queryParam.getParams().put("project_id", new Parameter(QueryParam.IN, proIds));
            }
            if (CollectionUtils.isNotEmpty(orgIds)) {
                queryParam.getParams().put("org_id", new Parameter(QueryParam.IN, orgIds));
            }
            //有效的单据
            queryParam.getParams().put("billState", new Parameter(QueryParam.IN,
                    Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()})));

            Map<String, Object> result = contractService.countContractAmount(queryParam);
            String num = (null != result && null != result.get("curAmount")) ? result.get("curAmount").toString() : "0";
            BigDecimal amt = new BigDecimal(num);
            amt.setScale(8, BigDecimal.ROUND_HALF_UP);

            return amt.divide(new BigDecimal(10000), 2, BigDecimal.ROUND_HALF_UP);
        }
    }


    /**
     * 统计生效的变迁变更索赔单金额
     */
    class ClaimAmtCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private List<Long> orgIds;
        private RequestAttributes context;
        private String authority;

        public ClaimAmtCountCallable(Long tenantId, List<Long> proIds,List<Long> orgIds, RequestAttributes context, String authority) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.orgIds=orgIds;
            this.context = context;
            this.authority = authority;
        }

        @Override
        public Object call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            QueryWrapper<ClaimEntity> queryWrapper = new QueryWrapper<>();
            //本租户下
            queryWrapper.eq("tenant_id", tenantId);
            //符合条件的项目
            if (CollectionUtils.isNotEmpty(proIds)) {
                queryWrapper.in("project_id", proIds);
            }
            if (CollectionUtils.isNotEmpty(orgIds)) {
                queryWrapper.in("org_id", orgIds);
            }
            //有效的单据
            queryWrapper.in("bill_state", Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()}));
            queryWrapper.select(" round(ifnull(sum(reply_mny),0) / 10000, 2) as amt");
            Map<String, Object> result = claimServicel.getMap(queryWrapper);

            return result.get("amt");
        }
    }

    /**
     * 已生效的产值统计金额（含税）
     */
    class ProductionCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private List<Long> orgIds;
        private RequestAttributes context;
        private String authority;

        public ProductionCountCallable(Long tenantId, List<Long> proIds,List<Long>orgIds, RequestAttributes context, String authority) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.orgIds=orgIds;
            this.context = context;
            this.authority = authority;
        }

        @Override
        public Object call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            QueryWrapper<ProductionEntity> queryWrapper = new QueryWrapper<>();
            //本租户下
            queryWrapper.eq("tenant_id", tenantId);
            //符合条件的项目
            if (CollectionUtils.isNotEmpty(proIds)) {
                queryWrapper.in("project_id", proIds);
            }
            if (CollectionUtils.isNotEmpty(orgIds)) {
                queryWrapper.in("org_id", orgIds);
            }
            //有效的单据
            queryWrapper.in("bill_state", Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()}));
            queryWrapper.select(" round(ifnull(sum(finish_tax_mny),0) / 10000, 2) as amt");

            Map<String, Object> result = productionService.getMap(queryWrapper);
            return result.get("amt");
        }
    }

    /**
     * 已生效的产值统计金额（含税）
     */
    class QuoteCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private List<Long> orgIds;
        private RequestAttributes context;
        private String authority;

        public QuoteCountCallable(Long tenantId, List<Long> proIds,List<Long> orgIds, RequestAttributes context, String authority) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.orgIds=orgIds;
            this.context = context;
            this.authority = authority;
        }

        @Override
        public Object call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            QueryWrapper<QuoteEntity> queryWrapper = new QueryWrapper<>();
            //本租户下
            queryWrapper.eq("tenant_id", tenantId);
            queryWrapper.eq("dr", BaseVO.DR_UNDELETE);
            //符合条件的项目
            if (CollectionUtils.isNotEmpty(proIds)) {
                queryWrapper.in("project_id", proIds);
            }
            if (CollectionUtils.isNotEmpty(orgIds)) {
                queryWrapper.in("org_id", orgIds);
            }
            //有效的单据
            queryWrapper.in("bill_state", Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()}));
            queryWrapper.select(" round(ifnull(sum(quote_tax_mny),0) / 10000, 2) as amt");

            Map<String, Object> result = quoteService.getMap(queryWrapper);

            return result.get("amt");
        }
    }

    /**
     * 已生效的有合同收款登记金额（含税）
     */
    class ReceiveCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private List<Long> orgIds;
        private RequestAttributes context;
        private String authority;

        public ReceiveCountCallable(Long tenantId, List<Long> proIds,List<Long> orgIds, RequestAttributes context, String authority) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.orgIds=orgIds;
            this.context = context;
            this.authority = authority;
        }

        @Override
        public Object call() throws Exception {
            Map<String, Object> param = new HashMap<>();
            param.put("tenantId", tenantId);
            param.put("projectIds", proIds);
            param.put("orgIds", orgIds);
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            String resp = HttpTookit.postByJson(BASE_HOST + "ejc-finance-web/api/receive/countByProperties",
                    JSONObject.toJSONString(param));
            logger.info("查询租户-{}下[项目列表-{}]已生效的有合同收款登记金额（含税）结果：{}", tenantId, proIds, resp);

            try {
                CommonResponse<BigDecimal> res = JSONObject.parseObject(resp, CommonResponse.class);
                if (!res.isSuccess()) {
                    logger.error("查询已生效的有合同收款登记金额（含税）失败，原因：{}, ", res.getMsg());
                    return 0;
                }
                return res.getData();
            } catch (Exception e) {
                logger.error("查询已生效的有合同收款登记金额（含税）异常, ", e);
            }
            return 0;
        }
    }

    /**
     * 已生效的有合同开票登记金额（含税）
     */
    class InvoiceOpenCountCallable implements Callable<Object> {
        private Long tenantId;
        private List<Long> proIds;
        private List<Long> orgIds;
        private RequestAttributes context;
        private String authority;

        public InvoiceOpenCountCallable(Long tenantId, List<Long> proIds, List<Long> orgIds,RequestAttributes context, String authority) {
            this.tenantId = tenantId;
            this.proIds = proIds;
            this.orgIds=orgIds;
            this.context = context;
            this.authority = authority;
        }

        @Override
        public Object call() throws Exception {
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);

            Map<String, Object> param = new HashMap<>();
            param.put("tenantId", tenantId);
            param.put("projectIds", proIds);
            param.put("orgIds", orgIds);
            String resp = HttpTookit.postByJson(BASE_HOST + "ejc-tax-web/api/tax/countInvoiceOpenByProperties",
                    JSONObject.toJSONString(param));
            logger.info("查询租户-{}下[项目列表-{}]已生效的有合同开票登记金额（含税）结果：{}", tenantId, proIds, resp);

            try {
                CommonResponse<BigDecimal> res = JSONObject.parseObject(resp, CommonResponse.class);
                if (!res.isSuccess()) {
                    logger.error("查询已生效的有合同开票登记金额（含税）失败，原因：{}, ", res.getMsg());
                    return 0;
                }
                return res.getData();
            } catch (Exception e) {
                logger.error("查询已生效的有合同开票登记金额（含税）异常, ", e);
            }
            return 0;
        }
    }

    @PostMapping(value = "getListByQueryParam")
    public CommonResponse<JSONObject> getListByQueryParam(@RequestBody QueryParam queryParam) {
        List<ContractEntity> list = contractService.queryList(queryParam, false);
        JSONObject back = new JSONObject();
        back.put("records", list);
        return CommonResponse.success(back);
    }

    /**
     * @param projectId
     * @description: 查询项目合同额，产值、产值进度、日期进度
     * @return: com.ejianc.framework.core.response.CommonResponse<java.util.Map>
     * @author songlx
     * @date: 2021/7/29
     */
    @RequestMapping(value = "projectProgress", method = RequestMethod.GET)
    @ResponseBody
    CommonResponse<JSONObject> projectProgress(@RequestParam(value = "projectId") Long projectId) {
        CommonResponse<Map> getprogress = contractService.getprogress(projectId);
        return CommonResponse.success("查询数据成功！", new JSONObject(getprogress.getData()));
    }

    /**
     * @param projectIds
     * @description: 查询项目合同额，产值、产值进度、日期进度
     * @return: com.ejianc.framework.core.response.CommonResponse<java.util.Map>
     * @author xuannl
     * @date: 2021/11/10
     */
    @RequestMapping(value = "projectProgressList", method = RequestMethod.GET)
    @ResponseBody
    CommonResponse<List<ProjectProgressVo>> projectProgressList(@RequestParam(value = "projectIds") List<Long> projectIds) {
        CommonResponse<List<ProjectProgressVo>> listCommonResponse = contractService.getprogressList(projectIds);
        return CommonResponse.success("查询数据成功！", listCommonResponse.getData());
    }

    /**
     * 回写累计开票
     *
     * @param map
     * @return
     */
    @RequestMapping(value = "refreshInvoiceMny", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> refreshInvoiceMny(Map<Long, BigDecimal> map) {
        contractService.refreshInvoiceMny(map);
        return CommonResponse.success("更新累计开票成功！");
    }

    /**
     * 根据项目id批量修改详细地址
     * @param projectId
     * @param address
     */
    @RequestMapping(value = "refreshAddress",method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> refreshAddress(@RequestParam(value = "projectId") Long projectId,@RequestParam(value = "address") String address){
        contractService.refreshAddress(projectId,address);
        return CommonResponse.success("更新累计开票成功！");
    }

    /**
     * 更新 本期报量收款金额 合同累计收款金额（含本期）
     * @param vos
     *
     */
    @RequestMapping(value = "/updateReceiveTaxMny", method = RequestMethod.POST)
    @ResponseBody
    CommonResponse<String> updateReceiveTaxMny(@RequestBody List<QuoteVo> vos){
        vos.forEach(vo->{
            Long id = vo.getId();
            QuoteEntity quoteEntity = quoteService.selectById(id);
            LambdaUpdateWrapper<QuoteEntity> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.eq(QuoteEntity::getId,id);
            updateWrapper.set(QuoteEntity::getTotalContractReceiveTaxMny,vo.getTotalContractReceiveTaxMny());
            updateWrapper.set(QuoteEntity::getTotalReceiveTaxMny,vo.getTotalReceiveTaxMny());
            quoteService.update(updateWrapper);
            //更新这个合同的 合同含本期累计收款金额
            LambdaUpdateWrapper<QuoteEntity> updateConWrapper = new LambdaUpdateWrapper<>();
            updateConWrapper.eq(QuoteEntity::getContractId,vo.getContractId());
            List<Integer> billStatus = new ArrayList<>();
            billStatus.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
            billStatus.add(BillStateEnum.PASSED_STATE.getBillStateCode());
            updateConWrapper.in(QuoteEntity::getBillState,billStatus);
            updateConWrapper.set(QuoteEntity::getTotalContractReceiveTaxMny,vo.getTotalContractReceiveTaxMny());
            quoteService.update(updateConWrapper);

            List<QuoteDetailEntity> checkList = quoteEntity.getCheckList();
            if (CollectionUtils.isNotEmpty(checkList)){
                for (QuoteDetailEntity quoteDetailEntity : checkList) {
                    BigDecimal ratio = ComputeUtil.safeDiv(vo.getTotalReceiveTaxMny(), quoteEntity.getQuoteTaxMny());
                    BigDecimal detailMny = ComputeUtil.safeMultiply(quoteDetailEntity.getQuoteMny(), ratio);
                    LambdaUpdateWrapper<QuoteDetailEntity> detailUpdateWrapper = new LambdaUpdateWrapper<>();
                    detailUpdateWrapper.eq(QuoteDetailEntity::getQuoteId,id);
                    detailUpdateWrapper.eq(QuoteDetailEntity::getLeafFlag,true);
                    detailUpdateWrapper.set(QuoteDetailEntity::getDetailMny,detailMny);
                    quoteDetailService.update(detailUpdateWrapper);
                }
            }
        });
        return CommonResponse.success("会写甲方报量成功！");
    }

}
