package com.ejianc.business.income.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.income.bean.ContractEntity;
import com.ejianc.business.income.vo.comparator.ClaimDetailComparatorVo;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.business.income.history.ClaimHistoryVo;
import com.ejianc.business.income.utils.TreeNodeBUtil;
import com.ejianc.business.income.vo.ClaimDetailVo;
import com.ejianc.business.income.vo.ClaimVo;
import com.ejianc.business.income.bean.ClaimDetailEntity;
import com.ejianc.business.income.bean.ClaimEntity;
import com.ejianc.business.income.mapper.ClaimMapper;
import com.ejianc.business.income.service.IClaimDetailService;
import com.ejianc.business.income.service.IClaimService;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.ejianc.business.income.service.IContractService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigDecimal;
import java.util.*;

/**
 * <p>
 * 签证洽商索赔 服务实现类
 * </p>
 *
 * @author yuezx
 * @since 2020-06-04
 */
@Service("ClaimService")
public class ClaimServiceImpl extends BaseServiceImpl<ClaimMapper, ClaimEntity> implements IClaimService {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private static final String CLAIM_BILL_CODE = "INCOME_CLAIM";
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IClaimDetailService claimDetailService;
    @Autowired
    private ClaimMapper claimMapper;
    @Autowired
    private IClaimDetailService detailsService;
    @Autowired
    private IContractService contractService;
    @Autowired
    private JedisPool jedisPool;

    @Override
    public CommonResponse<ClaimVo> saveOrUpdate(ClaimVo claimVo) {
        ContractEntity contractEntity = contractService.selectById(claimVo.getContractId());
        //保存时校验合同version是否一致
        if(claimVo.getContractVersion()!=null&&claimVo.getContractVersion()!=0){
            Jedis jedis = jedisPool.getResource();
            boolean locked = false;
            try{
                locked = RedisTool.tryLock(jedis, String.valueOf(claimVo.getContractId()), "saveOrUpdate", 1000);
                logger.info("判断单据单据锁结果------"+locked);
                if(locked){
                    Integer version = contractEntity.getVersion()==null?0:contractEntity.getVersion();
                    Integer conVersion = claimVo.getContractVersion();
                    if(!version.equals(conVersion)){
                        return CommonResponse.error("施工合同已被更新，请刷新后重做！");
                    }
                }else{
                    return CommonResponse.error("出现并发操作,请稍后重试！");
                }
            }catch (Exception e) {
                e.printStackTrace();
            } finally {
                if(locked) {
                    RedisTool.releaseLock(jedis, String.valueOf(claimVo.getContractId()), "saveOrUpdate");
                }
                jedis.close();
            }
        }

        Date signDate = contractEntity.getSignDate();
        if(null!=signDate&&claimVo.getOccurDate().getTime()<signDate.getTime()){
            throw new BusinessException("发生时间只能是大于等于施工合同的签约日期");
        }
        Long tenantId = InvocationInfoProxy.getTenantid();
        ClaimEntity entity = null;
        String operateType = null;
        if (claimVo.getId() != null && claimVo.getId() > 0) { //修改
            if(StringUtils.isEmpty(claimVo.getBillCode())){
                claimVo.setBillCode(null);
            }
            entity = BeanMapper.map(claimVo, ClaimEntity.class);
            operateType = "edit";
        } else {
            //新增
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(CLAIM_BILL_CODE, tenantId);
            if (billCode.isSuccess()) {
                claimVo.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }

            entity = BeanMapper.map(claimVo, ClaimEntity.class);
            operateType = "add";
        }
        if (claimVo.getId() != null && claimVo.getId() > 0) {
            //修改  校验合同编号是否重复
            LambdaQueryWrapper<ClaimEntity> lambda = Wrappers.<ClaimEntity>lambdaQuery();
            lambda.eq(ClaimEntity::getBillCode, claimVo.getBillCode());
            lambda.eq(ClaimEntity::getTenantId, tenantId);
            lambda.ne(ClaimEntity::getId, claimVo.getId());
            List<ClaimEntity> entities = super.list(lambda);
            if (entities != null && entities.size() > 0) {
                throw new BusinessException("存在相同编码，不允许保存!");
            }
        } else {
            //校验合同编号是否重复
            LambdaQueryWrapper<ClaimEntity> lambda = Wrappers.<ClaimEntity>lambdaQuery();
            lambda.eq(ClaimEntity::getTenantId, tenantId);
            lambda.eq(ClaimEntity::getBillCode, claimVo.getBillCode());
            List<ClaimEntity> entities = super.list(lambda);
            if (entities != null && entities.size() > 0) {
                throw new BusinessException("存在相同编码，不允许保存!");
            }
        }
        super.saveOrUpdate(entity);

        List<ClaimDetailVo> detailsVos = claimVo.getCheckList();
        List<ClaimDetailEntity> saveOrUpldates = new ArrayList<>();
        List<Long> deleteIds = new ArrayList<>();
        for (ClaimDetailVo detailsVo : detailsVos) {
            if ("add".equals(detailsVo.getRowState())) {
                ClaimDetailEntity detailslist = BeanMapper.map(detailsVo, ClaimDetailEntity.class);
                detailslist.setClaimId(entity.getId());
                detailslist.setId(null);
                saveOrUpldates.add(detailslist);
            } else if ("edit".equals(detailsVo.getRowState())) {
                ClaimDetailEntity detailslist = BeanMapper.map(detailsVo, ClaimDetailEntity.class);
                saveOrUpldates.add(detailslist);
            } else if ("del".equals(detailsVo.getRowState())) {
                deleteIds.add(detailsVo.getId());
            }
        }
        if (saveOrUpldates.size() > 0) {
            claimDetailService.saveOrUpdateBatch(saveOrUpldates, saveOrUpldates.size(), false);
            //维护父子级关系
            Map<String, Long> idMap = new HashMap<>();
            for (ClaimDetailEntity cdEntity : saveOrUpldates) {
                idMap.put(cdEntity.getTid(), cdEntity.getId());
            }
            for (ClaimDetailEntity cdEntity : saveOrUpldates) {
                if (StringUtils.isNotEmpty(cdEntity.getTpid())) {
                    cdEntity.setParentId(idMap.get(cdEntity.getTpid()));
                }
            }
            claimDetailService.saveOrUpdateBatch(saveOrUpldates, saveOrUpldates.size(), false);
        }
        if (deleteIds.size() > 0) {
            claimDetailService.removeByIds(deleteIds, false);
        }
        return CommonResponse.success(queryDetail(entity.getId()));
    }

    @Override
    public ClaimVo queryDetail(Long id) {
        ClaimEntity entity = claimMapper.selectById(id);
        if (entity != null) {
            ClaimVo vo = BeanMapper.map(entity, ClaimVo.class);
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("claimId", new Parameter(QueryParam.EQ, vo.getId()));
            //queryParam.getOrderMap().put("detailIndex", "asc");
            List<ClaimDetailEntity> details = detailsService.queryList(queryParam, false);
            if (details != null && details.size() > 0) {
                for (ClaimDetailEntity cdEntity : details) {
                    cdEntity.setTid(cdEntity.getId().toString());
                    cdEntity.setTpid(cdEntity.getParentId() != null && cdEntity.getParentId() > 0 ? cdEntity.getParentId().toString() : "");
                    cdEntity.setRowState("edit");
                }
                List<ClaimDetailVo> resultMapList = BeanMapper.mapList(details, ClaimDetailVo.class);
                //实现排序
                Collections.sort(resultMapList,new ClaimDetailComparatorVo());
                vo.setCheckList(TreeNodeBUtil.buildTree(resultMapList));
            }
            return vo;
        }
        return null;
    }

    @Override
    public void deleteClaim(List<ClaimVo> vos) {
        for (ClaimVo vo : vos) {
            detailsService.deleteByClaimId(vo.getId());
            claimMapper.deleteById(vo.getId());
        }
    }

    @Override
    public ClaimHistoryVo queryClaimHistory(Long id) {
        List<Integer> billStatus = new ArrayList<>();
        billStatus.add(1);
        billStatus.add(3);
        List<ClaimEntity> claimEntities = claimMapper.selectList(new QueryWrapper<ClaimEntity>()
                .eq("contract_id", id)
                .in("bill_state", billStatus).orderByDesc("occur_date","create_time"));
        ContractEntity contractEntity = contractService.selectById(id);
        ClaimHistoryVo vo = new ClaimHistoryVo();
        vo.setContractId(id);
        vo.setChangeStatus(contractEntity.getChangeStatus());
        vo.setContractStatus(contractEntity.getContractStatus());
        vo.setContractTaxMny(contractEntity.getContractTaxMny()== null ? BigDecimal.ZERO : contractEntity.getContractTaxMny());
        if (null != claimEntities && claimEntities.size() > 0) {
            List<ClaimVo> claimVos = BeanMapper.mapList(claimEntities, ClaimVo.class);
            //现场签证总金额
            BigDecimal sumVisaMny = BigDecimal.ZERO;
            //变更洽商总金额
            BigDecimal sumChangeMny = BigDecimal.ZERO;
            //索赔总金额
            BigDecimal sumClaimMny = BigDecimal.ZERO;
            //二次经营总金额
            BigDecimal sumSecondManage = BigDecimal.ZERO;
            for (ClaimVo cv : claimVos) {
                //变更类型，1-现场签证，2-变更洽商，3-索赔
                Integer changeType = cv.getChangeType();
                BigDecimal replyMny = cv.getReplyMny() == null ? BigDecimal.ZERO : cv.getReplyMny();
                if (1 == changeType) {
                    sumVisaMny = sumVisaMny.add(replyMny);
                }
                if (2 == changeType) {
                    sumChangeMny = sumChangeMny.add(replyMny);
                }
                if (3 == changeType) {
                    sumClaimMny = sumClaimMny.add(replyMny);
                }
            }
            sumSecondManage = sumVisaMny.add(sumChangeMny).add(sumClaimMny);
            vo.setSumVisaMny(sumVisaMny);
            vo.setSumChangeMny(sumChangeMny);
            vo.setSumClaimMny(sumClaimMny);
            vo.setSumSecondManage(sumSecondManage);
            BigDecimal secondManageRate = sumSecondManage.divide(vo.getContractTaxMny(), 8, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(100));
            vo.setSecondManageRate(secondManageRate);
            vo.setClaimRecord(claimVos);
        }
        return vo;
    }

    @Override
    public IPage selectPage(Page<ClaimEntity> page, QueryWrapper wrapper) {
        return baseMapper.selectPage(page,wrapper);
    }

}
