package com.ejianc.business.integration.base.module.auth.impl;


import com.ejianc.business.integration.base.datacenter.DataCenterUrlProvider;
import com.ejianc.business.integration.base.module.auth.OpenApiAuthProvider;
import com.ejianc.business.integration.base.module.common.OpenApiAccessToken;
import com.ejianc.business.integration.base.module.token.OpenApiTokenCacheProvider;
import com.ejianc.business.integration.base.network.OpenApiRequestEncryptor;
import com.ejianc.business.integration.base.properties.OpenApiProperties;
import com.ejianc.business.integration.base.module.auth.pojo.OpenApiAccessTokenResponse;
import com.ejianc.framework.core.exception.BusinessException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.TreeMap;

/**
 * 本类主要用于
 *
 * @author liuhaoi
 * @since Created At 2020/5/11 11:00 上午
 */
@Slf4j
@Service

public class OpenApiAuthProviderImpl implements OpenApiAuthProvider {

    private static final String URL_TOKEN = "open-auth/selfAppAuth/getAccessToken?signature=%s&appKey=%s&timestamp=%s";

    @Autowired
    private OpenApiProperties properties;

    @Autowired
    private OpenApiRequestEncryptor encryptor;
    @Autowired
    private RestTemplate restTemplate;

    @Resource
    OpenApiTokenCacheProvider cacheProvider;

    @Autowired
    DataCenterUrlProvider dataCenterUrlProvider;


    public OpenApiAccessToken buildTenantAuthToken() {
        OpenApiAccessToken token = cacheProvider.loadTokenFromCache(dataCenterUrlProvider.queryTenantId());
        if (token != null && token.getExpiredAt() > System.currentTimeMillis()) {
            return token;
        }
        return buildTenantAuthTokenFromRemote();

    }

    public OpenApiAccessToken buildTenantAuthTokenFromRemote() {

        long timestamp = System.currentTimeMillis();
        String appKey = properties.getAppKey();
        Map<String, String> params = new TreeMap<>();
        params.put("appKey", appKey);
        params.put("timestamp", String.valueOf(timestamp));
        String signature = encryptor.signature(params, properties.getAppSecret());

        String url = properties.concatURL(dataCenterUrlProvider.buildTokenUrl(), String.format(URL_TOKEN, signature, appKey, timestamp));

        URI uri;
        try {
            uri = new URI(url);
        } catch (URISyntaxException e) {
            throw new BusinessException("invalid open api uri format", e);
        }

        ResponseEntity<OpenApiAccessTokenResponse> responseEntity = restTemplate
                .getForEntity(uri, OpenApiAccessTokenResponse.class);

        OpenApiAccessTokenResponse body = responseEntity.getBody();
        if (body != null) {
            body.check();
            OpenApiAccessToken accessToken = body.getData().build();
            cacheProvider.saveTokenToCache(dataCenterUrlProvider.queryTenantId(), accessToken);
            return accessToken;
        }

        throw new BusinessException("Unexpected empty response encountered when request open api access token");
    }

    @Override
    public OpenApiAccessToken buildISVAuthToken() {
        return buildTenantAuthToken();
    }

}
