package com.ejianc.business.jlcost.finance.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.jlcost.cost.enums.CostTypeEnum;
import com.ejianc.business.jlcost.cost.service.ITargetService;
import com.ejianc.business.jlcost.cost.vo.QueryTargetDataVO;
import com.ejianc.business.jlcost.finance.consts.ArchiveProjectStatusEnum;
import com.ejianc.business.jlcost.finance.consts.ProjectStatusEnum;
import com.ejianc.business.jlcost.finance.mapper.ReimburseMapper;
import com.ejianc.business.jlcost.finance.vo.PayReimburseVO;
import com.ejianc.business.jlcost.payout.vo.SettleVO;
import com.ejianc.business.jlcost.payout.vo.SjCostReportVO;
import com.ejianc.foundation.share.api.IProjectArchiveApi;
import com.ejianc.foundation.share.vo.ProjectArchiveVO;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillParamVO;
import com.ejianc.foundation.support.vo.ParamsCheckDsVO;
import com.ejianc.foundation.support.vo.ParamsCheckVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.jsoup.Jsoup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;

import com.ejianc.business.jlcost.finance.mapper.PayReimburseMapper;
import com.ejianc.business.jlcost.finance.bean.PayReimburseEntity;
import com.ejianc.business.jlcost.finance.service.IPayReimburseService;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 资金管理-费用报销单
 *
 * @author generator
 */
@Service("payReimburseService")
public class PayReimburseServiceImpl extends BaseServiceImpl<PayReimburseMapper, PayReimburseEntity> implements IPayReimburseService {

    private static final String XCSGF_MNY_PARAM_CODE = "P-50GO4a0002";//【目标成本-现场施工费】管控【实际现场施工费】

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ITargetService targetService;

    @Autowired
    private ReimburseMapper reimburseMapper;

    @Autowired
    private IParamConfigApi paramConfigApi;

    @Autowired
    private IProjectArchiveApi projectArchiveApi;

    @Override
    public void updateProjectStatus() {
        // 查询所有报销单
        List<PayReimburseEntity> reimburseList = this.list();
        // 遍历报销单 将项目id存入集合 不重复
        HashSet<Long> projectIds = new HashSet<>();
        reimburseList.forEach(reimburse -> {
            projectIds.add(reimburse.getProjectId());
        });
        // Set转List
        ArrayList<Long> projectIdList = new ArrayList<>(projectIds);
        // 调rpc接口 根据id集合查询项目信息
        List<ProjectArchiveVO> projectList = projectArchiveApi.getProjectArchiveByIds(projectIdList).getData();
        // 遍历项目集合，使用Map集合保存项目中标状态，为Map赋值 statusMap<项目id,项目中标状态>
        Map<Long, String> statusMap = new HashMap<>();
        for (ProjectArchiveVO projectArchiveVO : projectList) {
            /*
             * 项目中标状态：根据项目阶段赋值，实时更新
             * 项目阶段：商机、投标，赋值：投标中
             * 项目阶段：未中标，赋值：未中标
             * 项目阶段：已中标、合同、生产、供货、结算、完工，赋值：已中标
             */
            if (projectArchiveVO.getProjectStatus().equals(ArchiveProjectStatusEnum.未中标.getCode())) {
                statusMap.put(projectArchiveVO.getId(), ProjectStatusEnum.未中标.getName());
            } else if (projectArchiveVO.getProjectStatus().equals(ArchiveProjectStatusEnum.商机阶段.getCode()) || projectArchiveVO.getProjectStatus().equals(ArchiveProjectStatusEnum.投标阶段.getCode())) {
                statusMap.put(projectArchiveVO.getId(), ProjectStatusEnum.投标中.getName());
            } else {
                statusMap.put(projectArchiveVO.getId(), ProjectStatusEnum.已中标.getName());
            }
        }
        // 遍历费用报销单 为项目中标状态赋值
        reimburseList.forEach(e -> {
            if (statusMap.get(e.getProjectId()) != null) {
                e.setProjectStatusName(statusMap.get(e.getProjectId()));
                e.setProjectStatus(ProjectStatusEnum.getCodeByName(statusMap.get(e.getProjectId())));
            }
        });
        // 数据库批量更新
        this.updateBatchById(reimburseList);
    }

    @Override
    public ParamsCheckVO checkParams(PayReimburseVO vo) {
        List<ParamsCheckVO> paramsCheckVOS = new ArrayList<>();
        ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
        paramsCheckVO.setWarnType("none");
        /*添加参数控制区域---*/
        paramsCheckVOS.addAll(this.checkParamsByMny(vo));//【目标成本-现场施工费】管控【实际现场施工费】
        /*添加参数控制区域---*/
        Map<String, List<ParamsCheckDsVO>> map = new HashMap<>();
        String[] paramsArray = {"alert", "warn", "none"};
        if(CollectionUtils.isNotEmpty(paramsCheckVOS)){
            for (ParamsCheckVO checkVO : paramsCheckVOS) {
                String warnType = checkVO.getWarnType();
                if(map.containsKey(warnType)){
                    List<ParamsCheckDsVO> checkDsVOS = map.get(warnType);
                    checkDsVOS.addAll(checkVO.getDataSource());
                    map.put(warnType,checkDsVOS);
                }else {
                    map.put(warnType,checkVO.getDataSource());
                }
            }
        }
        for (String s : paramsArray) {
            if(map.containsKey(s)){
                paramsCheckVO.setWarnType(s);
                paramsCheckVO.setDataSource(map.get(s));
                if(CollectionUtils.isEmpty(paramsCheckVO.getDataSource())){
                    paramsCheckVO.setWarnType("none");
                }else {
                    return paramsCheckVO;
                }
            }
        }
        return paramsCheckVO;
    }

    private List<ParamsCheckVO> checkParamsByMny(PayReimburseVO vo) {
        // 三种控制方式：不控制，提醒，无法保存 (默认为提醒)
        String[] paramsArray = {"none", "warn", "alert"};
        List<ParamsCheckVO> paramsCheckVOS = new ArrayList<>();

        BigDecimal mny = vo.getReimburseMny();//本次结算
        BigDecimal targetMny;//目标成本现场施工费金额
        BigDecimal totalMny = mny;//累计现场施工费  默认赋值本次

        //1.查询目标成本 现场施工费
        CommonResponse<List<QueryTargetDataVO>> targetData = targetService.getTargetDataByProjectId(vo.getProjectId());
        if (!targetData.isSuccess()){
            throw new BusinessException("获取目标成本费用信息失败！");
        }

        //未编制目标成本或目标成本中没有现场施工费时不管控
        List<QueryTargetDataVO> targetDataList = targetData.getData();
        if (CollectionUtils.isEmpty(targetDataList)){
            return paramsCheckVOS;
        }
        Map<Integer, BigDecimal> targetMap = targetDataList.stream().collect(Collectors.toMap(QueryTargetDataVO::getCostType, QueryTargetDataVO::getMny));
        if (!targetMap.containsKey(CostTypeEnum.现场施工费.getCode())){
            return paramsCheckVOS;
        }
        targetMny = targetMap.get(CostTypeEnum.现场施工费.getCode());

        //2.查询累计现场施工费
        List<SjCostReportVO> queryDataList = reimburseMapper.getMnyByProjectId(vo.getProjectId());
        if (CollectionUtils.isNotEmpty(queryDataList)){
            for (SjCostReportVO reportVO : queryDataList) {
                if (!(null != vo.getId() && reportVO.getId().compareTo(vo.getId()) == 0)){
                    totalMny = ComputeUtil.safeAdd(totalMny, reportVO.getMny());
                }
            }
        }

        CommonResponse<List<BillParamVO>> billParamByCode = paramConfigApi.getBillParamByCodeAndOrgId(XCSGF_MNY_PARAM_CODE, vo.getOrgId());

        if (billParamByCode.isSuccess() && null != billParamByCode.getData()) {
            List<BillParamVO> data = billParamByCode.getData();
            logger.info("【目标成本-现场施工费】管控【实际现场施工费】："+ JSONObject.toJSONString(data));
            if(CollectionUtils.isNotEmpty(data)){
                for (BillParamVO datum : data) {
                    ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
                    List<ParamsCheckDsVO> checkDsVOS = new ArrayList<>();
                    BigDecimal roleValue = datum.getRoleValue();
                    BigDecimal comMny = ComputeUtil.safeDiv(ComputeUtil.safeMultiply(targetMny, roleValue), new BigDecimal("100")).setScale(2, BigDecimal.ROUND_HALF_UP);
                    paramsCheckVO.setWarnType(paramsArray[datum.getControlType()]);
                    if (totalMny.compareTo(comMny) > 0) {
                        ParamsCheckDsVO paramsCheckDsVO = new ParamsCheckDsVO();
                        paramsCheckDsVO.setOrgName(datum.getOrgName());
                        paramsCheckDsVO.setWarnItem("现场施工费超额预警");
                        paramsCheckDsVO.setWarnName("项目现场施工费超目标成本");
                        StringBuffer stringBuffer = new StringBuffer();
                        String text = "超出金额："+ComputeUtil.safeSub(totalMny, comMny).setScale(2, BigDecimal.ROUND_HALF_UP)+"元";
                        String redText = Jsoup.parse("<font color=\"red\">" + text + "</font>").body().html();
                        stringBuffer.append("累计现场施工费金额：").append(totalMny.setScale(2, BigDecimal.ROUND_HALF_UP))
                                .append("元，目标成本现场施工费金额：").append(targetMny.setScale(2, BigDecimal.ROUND_HALF_UP))
                                .append("元，管控比例：").append(roleValue).append("%，")
                                .append(redText);
                        paramsCheckDsVO.setContent(stringBuffer.toString());
                        checkDsVOS.add(paramsCheckDsVO);
                    }
                    paramsCheckVO.setDataSource(checkDsVOS);
                    paramsCheckVOS.add(paramsCheckVO);
                }
            }
        }
        else {
            logger.info(billParamByCode.getMsg());
            throw new BusinessException("获取控制参数失败");
        }
        return paramsCheckVOS;
    }
}
