package com.ejianc.business.jlcost.cost.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.center.api.IWarnCenterApi;
import com.ejianc.business.center.vo.EarlyWarnTransVO;
import com.ejianc.business.jlcost.finance.vo.FinanceReportVO;
import com.ejianc.business.jlcost.payout.mapper.ContractMapper;
import com.ejianc.foundation.share.api.IProjectArchiveApi;
import com.ejianc.foundation.share.vo.ProjectArchiveVO;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * @Description: 预警控制类
 */
@RestController
@RequestMapping(value = "/warn/")
public class WarnController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String PROJECT_BALANCE_WARN_NAME = "项目虚拟资金账户余额不足";

    private static final String FINANCE_REPORT_URL = "/ejc-jlcost-frontend/#/financeReport?projectId=";// 项目虚拟资金账户统计详情

    @Autowired
    private IWarnCenterApi warnCenterApi;

    @Autowired
    private IProjectArchiveApi projectApi;

    @Autowired
    private ContractMapper contractMapper;


    /**
     * 项目虚拟资金账户余额不足
     * 定时预警
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "projectBalanceWarn")
    public CommonResponse<String> projectBalanceWarn(@RequestBody JSONObject jsonParam) {
        logger.info("接收到Task参数：{}", jsonParam);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);

        List<FinanceReportVO> recordVOList = contractMapper.queryProjectBalance();

        if(CollectionUtils.isEmpty(recordVOList)) {
            logger.error("执行" + PROJECT_BALANCE_WARN_NAME + "完成，未发现有匹配预警规则的单据信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        // 项目信息
        CommonResponse<List<ProjectArchiveVO>> res = projectApi.getProjectArchiveByIds(recordVOList.stream().map(FinanceReportVO::getProjectId).collect(Collectors.toList()));
        Map<Long, ProjectArchiveVO> projectMap = new HashMap<>();
        if(res.isSuccess() && !res.getData().isEmpty()){
            projectMap = res.getData().stream().collect(Collectors.toMap(ProjectArchiveVO::getId, Function.identity()));
        }

        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");
        for (FinanceReportVO reportVO : recordVOList) {
            String warnLevel = null;
            StringBuffer content = new StringBuffer();
            boolean flag = false;
            // 付款金额 > 收款金额 * X%
//            SqlParam param = paramMap.get(999999);
            warnLevel = defaultWarnInfo.getString("warnLevel");
            Double value = defaultWarnInfo.getDoubleValue("value");
            BigDecimal warnMny = ComputeUtil.safeMultiply(reportVO.getReceiveMny(), new BigDecimal(value));
            if(reportVO.getPaymentMny().compareTo(warnMny) > 0){
                flag = true;
                content.append("收款金额*");
                content.append(value * 100d);
                content.append("%：");
                content.append(df.format(warnMny != null ? warnMny : BigDecimal.ZERO));
                content.append("元");
            }

            if(flag){
                EarlyWarnTransVO vo = new EarlyWarnTransVO();
                vo.setBillName((null != defaultWarnInfo.get("categoryName") ? defaultWarnInfo.get("categoryName").toString() + " - " : "")  + reportVO.getProjectCode());
                vo.setPcTitle("项目虚拟资金账户余额不足");
                vo.setPcUrl(FINANCE_REPORT_URL + reportVO.getProjectId().toString());
                vo.setOrgId(projectMap.get(reportVO.getProjectId()).getOrgId());
                vo.setOrgName(projectMap.get(reportVO.getProjectId()).getOrgName());
                vo.setSourceId(reportVO.getProjectId().toString());
                vo.setTenantId(999999L);
                vo.setWarnLevel(warnLevel);
                vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
                vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

                tmpContent = defaultWarnInfo.getString("earlywarnContent");
                vo.setEarlywarnContent(
                        tmpContent.replaceAll("#projectName#", reportVO.getProjectName())
                                .replaceAll("#receiveMny#", df.format(this.getBigDecimal(reportVO.getReceiveMny())))
                                .replaceAll("#paymentMny#", df.format(this.getBigDecimal(reportVO.getPaymentMny())))
                );
                resp.add(vo);
            }
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }

    private CommonResponse<String> sendWarnToTask(List<EarlyWarnTransVO> warnList, Long warnId) {
        logger.info("发送任务：{}", JSONObject.toJSONString(warnList));
        CommonResponse<String> warnResp = warnCenterApi.sendToWarnCenter(warnList, warnId);
        if(!warnResp.isSuccess()) {
            logger.error("回调预警任务服务失败，{}", warnResp.getMsg());
            return CommonResponse.error("回调预警任务服务失败" + warnResp.getMsg());
        }

        return CommonResponse.success("任务接受处理成功！");
    }

    private BigDecimal getBigDecimal(Object obj){
        return null == obj ? BigDecimal.ZERO : new BigDecimal(obj.toString());
    }

}
