package com.ejianc.business.jlcost.cost.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.additional.query.impl.QueryChainWrapper;
import com.ejianc.business.center.api.IWarnCenterApi;
import com.ejianc.business.center.vo.EarlyWarnTransVO;
import com.ejianc.business.jlcost.cost.vo.ProductCostVO;
import com.ejianc.business.jlcost.cost.vo.TargetReportDetailVO;
import com.ejianc.business.jlcost.cost.vo.TargetVO;
import com.ejianc.business.jlcost.cost.bean.TargetEntity;
import com.ejianc.business.jlcost.cost.service.ITargetService;
import com.ejianc.business.jlcost.finance.vo.FinanceReportVO;
import com.ejianc.business.jlcost.payout.mapper.ContractMapper;
import com.ejianc.business.jlcost.payout.vo.SjCostReportVO;
import com.ejianc.foundation.share.api.IProjectArchiveApi;
import com.ejianc.foundation.share.vo.ProjectArchiveVO;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * @Description: 预警控制类
 */
@RestController
@RequestMapping(value = "/warn/")
public class WarnController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String PROJECT_BALANCE_WARN_NAME = "项目虚拟资金账户余额不足";
    private static final String PRODUCT_COST_WARN_NAME = "产品生产成本超额";
    private static final String PROJECT_SJCOST_WARN_NAME = "项目虚拟资金账户余额不足";

    private static final String FINANCE_REPORT_URL = "/ejc-jlcost-frontend/#/financeReport?projectId=";// 项目虚拟资金账户统计详情
    private static final String PRODUCT_COST_URL = "/ejc-jlcost-frontend/#/target?projectId=";// 产品生产成本
    private static final String SJCOST_REPORT_URL = "/ejc-jlcost-frontend/#/sjCostReport?projectId=";// 项目虚拟资金账户统计详情

    @Autowired
    private IWarnCenterApi warnCenterApi;

    @Autowired
    private IProjectArchiveApi projectApi;

    @Autowired
    private ITargetService targetService;

    @Autowired
    private ContractMapper contractMapper;


    /**
     * 项目虚拟资金账户余额不足
     * 定时预警
     *
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "projectBalanceWarn")
    public CommonResponse<String> projectBalanceWarn(@RequestBody JSONObject jsonParam) {
        //1.接收预警参数
        logger.info("接收到Task参数：{}", jsonParam);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);

        //2.获得项目虚拟资金信息
        List<FinanceReportVO> recordVOList = contractMapper.queryProjectBalance();

        if (CollectionUtils.isEmpty(recordVOList)) {
            logger.error("执行" + PROJECT_BALANCE_WARN_NAME + "完成，未发现有匹配预警规则的单据信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        logger.info("获得项目虚拟资金信息：{}", JSONObject.toJSONString(recordVOList));
        // 项目信息
        CommonResponse<List<ProjectArchiveVO>> res = projectApi.getProjectArchiveByIds(recordVOList.stream().map(FinanceReportVO::getProjectId).collect(Collectors.toList()));
        Map<Long, ProjectArchiveVO> projectMap = new HashMap<>();
        if (res.isSuccess() && !res.getData().isEmpty()) {
            projectMap = res.getData().stream().collect(Collectors.toMap(ProjectArchiveVO::getId, Function.identity()));
        }

        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");

        //3.筛出符合条件的数据发送信息
        for (FinanceReportVO reportVO : recordVOList) {
            String warnLevel = defaultWarnInfo.getString("warnLevel");
            // 付款金额 > 收款金额 * X%
            List<SqlParam> sqlParams = JSONArray.parseArray(defaultWarnInfo.getString("parameters"), SqlParam.class);
            Double value = sqlParams.get(0).getValue();
            BigDecimal warnMny = ComputeUtil.safeMultiply(reportVO.getReceiveMny(), new BigDecimal(value / 100d));
            if (reportVO.getPaymentMny().compareTo(warnMny) > 0) {

                ProjectArchiveVO projectArchiveVO = projectMap.get(reportVO.getProjectId());
                EarlyWarnTransVO vo = new EarlyWarnTransVO();
                vo.setBillName(projectArchiveVO.getName());
                vo.setPcTitle("项目虚拟资金账户余额不足");
                vo.setPcUrl(FINANCE_REPORT_URL + reportVO.getProjectId().toString() + "&projectName=" + projectArchiveVO.getName());
                vo.setOrgId(projectArchiveVO.getOrgId());
                vo.setOrgName(projectArchiveVO.getOrgName());
                vo.setSourceId(reportVO.getProjectId().toString());
                vo.setTenantId(999999L);
                vo.setWarnLevel(warnLevel);
                vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
                vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

                tmpContent = defaultWarnInfo.getString("earlywarnContent");
                vo.setEarlywarnContent(
                        tmpContent.replaceAll("#projectName#", projectArchiveVO.getName())
                                .replaceAll("#receiveMny#", df.format(this.getBigDecimal(reportVO.getReceiveMny())))
                                .replaceAll("#paymentMny#", df.format(this.getBigDecimal(reportVO.getPaymentMny())))
                );
                resp.add(vo);
            }
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }

    /**
     * 产品生产成本超额
     * 定时预警
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "productCostWarn")
    public CommonResponse<String> productCostWarn(@RequestBody JSONObject jsonParam) {
        //1.接收预警参数
        logger.info("接收到Task参数：{}", jsonParam);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);

        //2.获得项目目标成本
        List<ProductCostVO> productCostVOList = contractMapper.queryProductCost();

        if(CollectionUtils.isEmpty(productCostVOList)) {
            logger.error("执行" + PRODUCT_COST_WARN_NAME + "完成，未发现有匹配预警规则的单据信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        logger.info("获得项目目标成本信息：{}", JSONObject.toJSONString(productCostVOList));
        // 项目信息

        CommonResponse<List<ProjectArchiveVO>> res = projectApi.getProjectArchiveByIds(productCostVOList.stream().map(ProductCostVO::getProjectId).collect(Collectors.toList()));
        Map<Long, ProjectArchiveVO> projectMap = new HashMap<>();
        if(res.isSuccess() && !res.getData().isEmpty()){
            projectMap = res.getData().stream().collect(Collectors.toMap(ProjectArchiveVO::getId, Function.identity()));
        }

        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");

        //3.筛出符合条件的数据发送信息
        for (ProductCostVO productCostVO : productCostVOList) {
            String warnLevel = defaultWarnInfo.getString("warnLevel");
            // 【产品生产成本】>【产品目标成本】*X%
            List<SqlParam> sqlParams = JSONArray.parseArray(defaultWarnInfo.getString("parameters"), SqlParam.class);
            Double value = sqlParams.get(0).getValue();
            BigDecimal warnMny = ComputeUtil.safeMultiply(productCostVO.getMbMny(), new BigDecimal(value / 100d));
            System.out.println("+++++++++++++++++++++++++++++++");
            System.out.println("产品生产目标成本：" + warnMny);
            System.out.println("产品生产成本：" + productCostVO.getMny());
            System.out.println("+++++++++++++++++++++++++++++++");
            if(productCostVO.getMny().compareTo(warnMny) > 0){

                ProjectArchiveVO projectArchiveVO = projectMap.get(productCostVO.getProjectId());
                EarlyWarnTransVO vo = new EarlyWarnTransVO();
                vo.setBillName(projectArchiveVO.getName());
                vo.setPcTitle("该产品有成本超额风险");
                vo.setPcUrl(PRODUCT_COST_URL + productCostVO.getProjectId().toString() +"&projectName=" + projectArchiveVO.getName());
                vo.setOrgId(projectArchiveVO.getOrgId());
                vo.setOrgName(projectArchiveVO.getOrgName());
                vo.setSourceId(productCostVO.getProjectId().toString());
                vo.setTenantId(999999L);
                vo.setWarnLevel(warnLevel);
                vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
                vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

                tmpContent = defaultWarnInfo.getString("earlywarnContent");
                vo.setEarlywarnContent(
                        tmpContent.replaceAll("#projectName#", projectArchiveVO.getName())
                                .replaceAll("#wbsName#", productCostVO.getWbsName())
                                .replaceAll("#receiveMny#", df.format(this.getBigDecimal(productCostVO.getMny())))
                                .replaceAll("#paymentMny#", df.format(this.getBigDecimal(productCostVO.getMbMny())))
                );
                resp.add(vo);
            }
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }

    private CommonResponse<String> sendWarnToTask(List<EarlyWarnTransVO> warnList, Long warnId) {
        logger.info("发送任务：{}", JSONObject.toJSONString(warnList));
        CommonResponse<String> warnResp = warnCenterApi.sendToWarnCenter(warnList, warnId);
        if (!warnResp.isSuccess()) {
            logger.error("回调预警任务服务失败，{}", warnResp.getMsg());
            return CommonResponse.error("回调预警任务服务失败" + warnResp.getMsg());
        }

        return CommonResponse.success("任务接受处理成功！");
    }

    private BigDecimal getBigDecimal(Object obj) {
        return null == obj ? BigDecimal.ZERO : new BigDecimal(obj.toString());
    }

    /**
     * 项目总成本超额
     * 定时预警
     *
     * @param jsonParam
     * @return
     */
    @PostMapping(value = "projectSjCostWarn")
    public CommonResponse<String> projectSjCostWarn(@RequestBody JSONObject jsonParam) {
        //1.接收预警参数
        logger.info("接收到Task参数：{}", jsonParam);
        List<JSONObject> billParams = JSONArray.parseArray(jsonParam.getString("billParams"), JSONObject.class);
        logger.info("预警参数列表：{}", billParams);
        JSONObject defaultWarnInfo = JSON.parseObject(jsonParam.getString("defaultParam"));
        logger.info("预警默认设置：{}", defaultWarnInfo);

        List<SjCostReportVO> recordList = contractMapper.queryProjectSjCost();

        if (CollectionUtils.isEmpty(recordList)) {
            logger.error("执行" + PROJECT_SJCOST_WARN_NAME + "完成，未发现有匹配预警规则的单据信息！");
            return sendWarnToTask(new ArrayList<>(), defaultWarnInfo.getLong("warnSetId"));
        }

        //2.获得实际总成本信息
        logger.info("获得实际目标总成本信息：{}", JSONObject.toJSONString(recordList));
        List<Long> ids = recordList.stream().map(SjCostReportVO::getProjectId).collect(Collectors.toList());

        // 项目信息
        CommonResponse<List<ProjectArchiveVO>> res = projectApi.getProjectArchiveByIds(ids);
        Map<Long, ProjectArchiveVO> projectMap = new HashMap<>();
        if (res.isSuccess() && !res.getData().isEmpty()) {
            projectMap = res.getData().stream().collect(Collectors.toMap(ProjectArchiveVO::getId, Function.identity()));
        }
        //3.获取实际目标总成本信息
        List<TargetEntity> targetList = targetService.list(new QueryWrapper<TargetEntity>().in("project_id", ids));
        Map<Long, BigDecimal> targetMap = targetList.stream().collect(Collectors.toMap(TargetEntity::getProjectId, TargetEntity::getCostMny));

        List<EarlyWarnTransVO> resp = new ArrayList<>();
        String tmpContent = null;
        DecimalFormat df = new DecimalFormat("###,##0.00");

        //3.筛出符合条件的数据发送信息
        for (SjCostReportVO reportVO : recordList) {
            String warnLevel = defaultWarnInfo.getString("warnLevel");
            // 付款金额 > 收款金额 * X%
            List<SqlParam> sqlParams = JSONArray.parseArray(defaultWarnInfo.getString("parameters"), SqlParam.class);
            Double value = sqlParams.get(0).getValue();
            BigDecimal targetMny = targetMap.get(reportVO.getProjectId());
            if (targetMny == null) continue;
            BigDecimal warnMny = ComputeUtil.safeMultiply(targetMny, new BigDecimal(value / 100d));
            if (reportVO.getMny().compareTo(warnMny) > 0) {

                ProjectArchiveVO projectArchiveVO = projectMap.get(reportVO.getProjectId());
                EarlyWarnTransVO vo = new EarlyWarnTransVO();
                vo.setBillName((null != defaultWarnInfo.get("categoryName") ? defaultWarnInfo.get("categoryName").toString() + " - " : "") + projectArchiveVO.getName());
                vo.setPcTitle("项目总成本超额");
                vo.setPcUrl(SJCOST_REPORT_URL + reportVO.getProjectId().toString() + "&projectName" + projectArchiveVO.getName());
                vo.setOrgId(projectArchiveVO.getOrgId());
                vo.setOrgName(projectArchiveVO.getOrgName());
                vo.setSourceId(reportVO.getProjectId().toString());
                vo.setTenantId(999999L);
                vo.setWarnLevel(warnLevel);
                vo.setWarnSetId(defaultWarnInfo.getLong("warnSetId"));
                vo.setEarlywarnName(defaultWarnInfo.getString("earlywarnName"));

                tmpContent = defaultWarnInfo.getString("earlywarnContent");
                vo.setEarlywarnContent(
                        tmpContent.replaceAll("#projectName#", projectArchiveVO.getName())
                                .replaceAll("#targetMny#", df.format(this.getBigDecimal(targetMny))
                                        .replaceAll("#costMny#", df.format(this.getBigDecimal(reportVO.getMny())))
                                ));
                resp.add(vo);
            }
        }
        return sendWarnToTask(resp, defaultWarnInfo.getLong("warnSetId"));
    }

}
