package com.ejianc.business.market.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.income.api.IIncomeContractApi;
import com.ejianc.business.income.utils.TreeNodeBUtil;
import com.ejianc.business.income.vo.ProductionClaimVo;
import com.ejianc.business.income.vo.ProductionDetailVo;
import com.ejianc.business.income.vo.ProductionOtherVO;
import com.ejianc.business.income.vo.ProductionVo;
import com.ejianc.business.income.vo.comparator.ProductionDetailComparatorVo;
import com.ejianc.business.market.bean.ProjectBidPeopleEntity;
import com.ejianc.business.market.bean.ProjectOuterPeopleEntity;
import com.ejianc.business.market.bean.ProjectRegisterEntity;
import com.ejianc.business.market.service.*;
import com.ejianc.business.market.vo.ProjectAwardsVO;
import com.ejianc.business.market.vo.ProjectBidPeopleVO;
import com.ejianc.business.market.vo.ProjectOuterPeopleVO;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.oa.api.IOaApi;
import com.ejianc.business.oa.vo.AwardVO;
import com.ejianc.foundation.orgcenter.api.IEmployeeApi;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.vo.CustomerVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.api.ICustomerApi;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.core.util.ImportTemplate;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.*;
import java.util.stream.Collectors;

@RestController
@RequestMapping("projectRegister")
public class ProjectResigerController {

    private final String TEMPLATE_KEY_PREFIX = "template_data_cache_market_project";
    private final String TEMPLATE_KEY_SEPRATOR = ":";
    private final int TIMEOUT = 1800;
    private final String PROJECT_BILL_TYPE_CODE = "BT200519000000002";
    private final static Random random = new Random();
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IProjectRegisterService projectRegisterService;
    @Autowired
    private IProjectBidPeopleService bidPeopleService;
    @Autowired
    private IProjectOuterPeopleService outerPeopleService;
    @Autowired
    private IOaApi oaApi;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IUserApi userApi;

    @Autowired
    private IEmployeeApi employeeApi;

    @Autowired
    private IBusinessOpportunityService businessOpportunityService;

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private CacheManager cacheManager;

    @Autowired
    private ICustomerApi customerApi;

    @Value("${common.env.base-host}")
    private String BASE_HOST;

    @Autowired
    private IBillTypeApi billTypeApi;

    @Autowired
    private IIncomeContractApi incomeContractApi;

    /**
     * 保存、更新项目登记信息
     *
     * @param projectRegisterVO
     * @return
     */
    @PostMapping(value = "saveProject")
    public CommonResponse<ProjectRegisterVO> saveProject(@RequestBody ProjectRegisterVO projectRegisterVO) {

        projectRegisterVO.setName(projectRegisterVO.getName().replaceAll(" ", ""));

        LambdaQueryWrapper<ProjectRegisterEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(ProjectRegisterEntity::getTenantId, InvocationInfoProxy.getTenantid());
        queryWrapper.eq(ProjectRegisterEntity::getName, projectRegisterVO.getName());
        queryWrapper.ne(null != projectRegisterVO.getId(), ProjectRegisterEntity::getId, projectRegisterVO.getId());
        int count = projectRegisterService.count(queryWrapper);
        if (count > 0) {
            return CommonResponse.error("当前项目名称已使用，请重新输入！");
        }

        if (StringUtils.isNotBlank(projectRegisterVO.getCode())) {
            projectRegisterVO.setCode(projectRegisterVO.getCode().replaceAll(" ", ""));
            boolean codeCheck = projectRegisterService.codeUsedCheck(projectRegisterVO);
            if (codeCheck) {
                return CommonResponse.error("保存失败，编码重复！");
            }
        }
        if (CollectionUtils.isNotEmpty(projectRegisterVO.getBidPeopleList())){
            List<ProjectBidPeopleVO> bidPeopleList = projectRegisterVO.getBidPeopleList();
            for (ProjectBidPeopleVO bidPeopleVO : bidPeopleList) {
                bidPeopleVO.setEndDateOne(bidPeopleVO.getChangeDateTwo());
                bidPeopleVO.setEndDateTwo(bidPeopleVO.getChangeDateThree());
                bidPeopleVO.setEndDateThree(bidPeopleVO.getChangeDateFour());
            }
        }
        ProjectRegisterVO resp = projectRegisterService.saveProject(projectRegisterVO);
        List<Map<String, Object>> list =(List<Map<String, Object>>) getRespData(userApi.getEmployeeList(resp.getProjectDepartmentId()), false, null);
        for (Map<String,Object> map :list){
            JSONObject jsonObj = new JSONObject();
            jsonObj.put("id",map.get("postId"));
            jsonObj.put("name",map.get("postName"));
            map.put("postId",jsonObj);
        }
        //查询项目成员
        resp.setMembersList(list);
        //根据项目id，同步施工合同中该项目的详细地址
        if(resp.getId()!=null){
            incomeContractApi.refreshAddress(resp.getId(), resp.getAddress());
        }

        return CommonResponse.success("保存成功！", resp);
    }

    private Object getRespData(CommonResponse resp, boolean isMustSuc, String errMsg) {
        if (isMustSuc && !resp.isSuccess()) {
            throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
        }
        return resp.getData();
    }

    /**
     * 分页查询项目登记列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping("page")
    public CommonResponse<JSONObject> queryPageList(@RequestBody QueryParam queryParam) {
        JSONObject resp = new JSONObject();
        queryParam.getFuzzyFields().add("name");
        queryParam.getFuzzyFields().add("code");
        queryParam.getFuzzyFields().add("orgName");
        //添加简称项目名称模糊查询
        queryParam.getFuzzyFields().add("shortName");
        queryParam.getFuzzyFields().add("createUserName");
        queryParam.getFuzzyFields().add("contractorUnit");
        queryParam.getFuzzyFields().add("projectManagerName");
        queryParam.getFuzzyFields().add("projectOwner");
        queryParam.getFuzzyFields().add("pubUnit");
        queryParam.getFuzzyFields().add("contractorUnit");
        queryParam.getFuzzyFields().add("pubUnit");
        Map<String, Integer> countProjectNum = null;
        IPage<ProjectRegisterEntity> pageData = null;
        List<OrgVO> orgVOList = (List<OrgVO>) getRespData(orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()), true, "查询失败，获取当前本下组织信息失败。");

        if (CollectionUtils.isEmpty(orgVOList)) {
            pageData = new Page<>();
            pageData.setRecords(new ArrayList<>());
            pageData.setTotal(0);
            pageData.setSize(queryParam.getPageSize());
            pageData.setCurrent(queryParam.getPageIndex());
            pageData.setPages(0);
            countProjectNum = new HashMap<>();
            countProjectNum.put("buildingNum", 0);
            countProjectNum.put("finishNum", 0);

        } else {
            List<Long> commonOrgIds = new ArrayList<>();
            List<Long> departmentIds = new ArrayList<>();
            orgVOList.stream().forEach(org -> {
                if (5 == org.getOrgType()) {
                    //项目部
                    departmentIds.add(org.getId());
                } else {
                    //普通组织
                    commonOrgIds.add(org.getId());
                }
            });
            if (CollectionUtils.isNotEmpty(commonOrgIds)) {
                queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
            } else {
                queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, departmentIds));
            }

            pageData = projectRegisterService.queryPage(queryParam, false);
            //查询项目的总数、在建项目数和已完工数量
            countProjectNum = projectRegisterService.countProjectNum(queryParam);
        }


        resp.put("records", pageData.getRecords());
        resp.put("total", pageData.getTotal());
        resp.put("size", pageData.getSize());
        resp.put("current", pageData.getCurrent());
        resp.put("pages", pageData.getPages());
        resp.put("projectCount", countProjectNum);

        return CommonResponse.success("分页查询登记项目列表成功！", resp);
    }

    /**
     * 根据主键Id查询项目登记详情
     *
     * @param id
     * @return
     */
    @GetMapping("detail")
    public CommonResponse<ProjectRegisterVO> detail(@RequestParam Long id) {
        ProjectRegisterVO resp = BeanMapper.map(projectRegisterService.queryById(id), ProjectRegisterVO.class);

        //查询项目成员
        CommonResponse<List<Map<String, Object>>> membersResp = userApi.getEmployeeList(resp.getProjectDepartmentId());
        if (!membersResp.isSuccess()) {
            return CommonResponse.error("查询失败，获取项目成员失败。");
        }
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("pid", new Parameter(QueryParam.EQ, resp.getId()));
        List<ProjectBidPeopleEntity> bidPeopleEntityList = bidPeopleService.queryList(queryParam, false);
        resp.setBidPeopleList(BeanMapper.mapList(bidPeopleEntityList, ProjectBidPeopleVO.class));
        List<ProjectOuterPeopleEntity> outerPeopleList = outerPeopleService.queryList(queryParam, false);
        resp.setOuterPeopleList(BeanMapper.mapList(outerPeopleList, ProjectOuterPeopleVO.class));
        CommonResponse<List<AwardVO>> listCommonResponse = oaApi.queryAwardByProjectId(id);
        if (listCommonResponse.isSuccess()&&CollectionUtils.isNotEmpty(listCommonResponse.getData())){
            List<ProjectAwardsVO> list = new ArrayList();
            for (AwardVO vo : listCommonResponse.getData()) {
                ProjectAwardsVO projectAwardsVO = new ProjectAwardsVO();
                projectAwardsVO.setAwardId(vo.getId());
                projectAwardsVO.setId(vo.getId());
                projectAwardsVO.setAwardsCode(vo.getBillCode());
                projectAwardsVO.setAwardLevelId(vo.getAwardLevelId());
                projectAwardsVO.setAwardLevelName(vo.getAwardLevelName());
                projectAwardsVO.setAwardName(vo.getAwardName());
                projectAwardsVO.setAwardRankingId(vo.getAwardRankingId());
                projectAwardsVO.setAwardRankingName(vo.getAwardRankingName());
                projectAwardsVO.setPid(id);
                projectAwardsVO.setRatifyData(vo.getRatifyData());
                list.add(projectAwardsVO);
            }
            resp.setAwardsList(BeanMapper.mapList(list, ProjectAwardsVO.class));
        }
        resp.setMembersList(membersResp.getData());
        for (Map<String,Object> map :resp.getMembersList()){
            JSONObject jsonObj = new JSONObject();
            jsonObj.put("id",map.get("postId"));
            jsonObj.put("name",map.get("postName"));
            map.put("postId",jsonObj);
        }
        //查询项目附件
        return CommonResponse.success("查询项目登记详情成功！", resp);
    }

    /**
     * 根据主键Id删除对应的登记项目
     *
     * @return
     */
    @PostMapping("delById")
    public CommonResponse<String> deleteById(@RequestBody List<Long> ids) {
        Long projectId = ids.get(0);
        logger.info("执行项目删除操作，删除项目[id-{}]", projectId);

        String userType = InvocationInfoProxy.getUserType();
        if (!("1".equals(userType) || "3".equals(userType))) {
            return CommonResponse.error("请联系管理员执行项目删除操作。");
        }

        //引用查询，若该项目已被下游引用，则无法进行删除操作
        CommonResponse<String> resp = billTypeApi.checkQuote(PROJECT_BILL_TYPE_CODE, projectId);
        if (!resp.isSuccess()) {
            logger.info("项目立项【Id-{}】无法进行删除操作，原因：{}", projectId, resp.getMsg());
            return CommonResponse.error("校验项目下游引用状况失败，删除操作中止.");
        }

        //执行项目删除操作
        String result = projectRegisterService.delProject(projectId);
        if (StringUtils.isNotBlank(result)) {
            return CommonResponse.error(result);
        }
        return CommonResponse.success("删除成功！");
    }

    @PostMapping("changeValidStatus")
    public CommonResponse<String> changeValidStatus(@RequestBody ProjectRegisterVO projectRegisterVO) {
        projectRegisterService.changeValidStatus(projectRegisterVO);

        return CommonResponse.success("操作成功！");
    }

    /**
     * 查询指定组织本下的所有项目信息列表
     *
     * @param pageNumber
     * @param pageSize
     * @param condition
     * @param searchText
     * @return
     */
    @GetMapping("/projectListRefer")
    public CommonResponse<IPage<ProjectRegisterVO>> projectListRefe(@RequestParam(defaultValue = "1") Integer pageNumber,
                                                                    @RequestParam(defaultValue = "10") Integer pageSize,
                                                                    @RequestParam(value = "condition", required = false) String condition,
                                                                    @RequestParam(value = "searchText", required = false) String searchText) {

        QueryParam queryParam = new QueryParam();
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.setSearchText(searchText);
        queryParam.getFuzzyFields().add("code");
        queryParam.getFuzzyFields().add("name");
        queryParam.getFuzzyFields().add("orgName");
        //添加简称项目名称模糊查询
        queryParam.getFuzzyFields().add("shortName");
        queryParam.getFuzzyFields().add("createUserName");

        //只查询启用的项目
        queryParam.getParams().put("validStatus", new Parameter(QueryParam.EQ, '0'));
        //租户过滤
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.getOrderMap().put("createTime", QueryParam.DESC);

        List<Long> commonOrgIds = new ArrayList<>();
        List<Long> departmentIds = new ArrayList<>();
        Long orgId = InvocationInfoProxy.getOrgId();
        Boolean tenantLeave = false;
        /** 移动端参照标识 */
        Boolean mobileLeave = false;

        if (StringUtils.isNotBlank(condition)) {
            Map<String, Object> conditionMap = JSONObject.parseObject(condition, Map.class);

            if (null != conditionMap.get("orgId")) {
                orgId = Long.valueOf(conditionMap.get("orgId").toString());
            }
            if (null != conditionMap.get("tenantLeave")) {
                //租户级别数据
                tenantLeave = true;
            }
            if (null != conditionMap.get("mobileLeave")) {
                //移动端数据
                mobileLeave = true;
            }

            if (null != conditionMap.get("province")) {
                queryParam.getParams().put("province", new Parameter(QueryParam.EQ, conditionMap.get("province")));
            }
            if (null != conditionMap.get("city")) {
                queryParam.getParams().put("city", new Parameter(QueryParam.EQ, conditionMap.get("city")));
            }
            if (null != conditionMap.get("region")) {
                queryParam.getParams().put("region", new Parameter(QueryParam.EQ, conditionMap.get("region")));
            }
            // 不包含这个项目
            if (null != conditionMap.get("projectId")) {
                queryParam.getParams().put("id", new Parameter(QueryParam.NE, conditionMap.get("projectId")));
            }
        }

        if (mobileLeave) {
            //移动端
            CommonResponse<List<OrgVO>> response = orgApi.findProjectOrgsByUserId(InvocationInfoProxy.getUserid());
            if (response.isSuccess() && ListUtil.isNotEmpty(response.getData())) {
                queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, response.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
            } else {
                IPage<ProjectRegisterVO> pageData = new Page<>(pageNumber, pageSize, 0);
                return CommonResponse.success("该用户没有任何项目权限！", pageData);
            }

        } else if (!tenantLeave) {
            //组织本下
            CommonResponse<List<OrgVO>> orgListResp = orgApi.findChildrenByParentId(orgId);
            if (!orgListResp.isSuccess()) {
                IPage<ProjectRegisterVO> pageData = new Page<>(pageNumber, pageSize, 0);
                return CommonResponse.success("查询数表参照失败，项目组织信息查询失败！", pageData);
            }
            List<OrgVO> orgList = orgListResp.getData();

            if (CollectionUtils.isNotEmpty(orgList)) {
                orgList.forEach(orgVO -> {
                    if (5 == orgVO.getOrgType()) {
                        //项目部
                        departmentIds.add(orgVO.getId());
                    } else {
                        //普通组织
                        commonOrgIds.add(orgVO.getId());
                    }
                });
                if (CollectionUtils.isNotEmpty(commonOrgIds)) {
                    queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
                } else if (CollectionUtils.isNotEmpty(departmentIds)) {
                    queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.EQ, departmentIds.get(0)));
                }
            }
        }

        IPage<ProjectRegisterEntity> pageData = projectRegisterService.queryPage(queryParam, false);
        IPage<ProjectRegisterVO> result = new Page<>(pageData.getCurrent(), pageData.getSize(), pageData.getTotal());
        result.setRecords(BeanMapper.mapList(pageData.getRecords(), ProjectRegisterVO.class));
        return CommonResponse.success("项目参照查询成功！", result);
    }

    /**
     * 查询项目名称在当前租户下是否重复
     *
     * @param projectRegisterVO
     * @return
     */
    @PostMapping("checkProjectName")
    public CommonResponse<String> checkProjectName(@RequestBody ProjectRegisterVO projectRegisterVO) {
        LambdaQueryWrapper<ProjectRegisterEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(ProjectRegisterEntity::getTenantId, InvocationInfoProxy.getTenantid());
        queryWrapper.eq(ProjectRegisterEntity::getName, projectRegisterVO.getName());
        queryWrapper.ne(null != projectRegisterVO.getId(), ProjectRegisterEntity::getId, projectRegisterVO.getId());
        int count = projectRegisterService.count(queryWrapper);
        if (count > 0) {
            return CommonResponse.error("当前项目名称已使用，请重新输入。");
        }
        return CommonResponse.success("项目名称未被使用，可以使用。");
    }

    @PostMapping("/excelExport")
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        queryParam.getFuzzyFields().add("name");
        queryParam.getFuzzyFields().add("code");
        queryParam.setPageSize(10000); //默认导出全部

        List<OrgVO> orgVOList = (List<OrgVO>) getRespData(orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()), true, "查询失败，获取当前本下组织信息失败。");
        List<Long> commonOrgIds = new ArrayList<>();
        List<Long> departmentIds = new ArrayList<>();
        orgVOList.stream().forEach(org -> {
            if (5 == org.getOrgType()) {
                //项目部
                departmentIds.add(org.getId());
            } else {
                //普通组织
                commonOrgIds.add(org.getId());
            }
        });
        if (CollectionUtils.isNotEmpty(commonOrgIds)) {
            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
        } else if (CollectionUtils.isNotEmpty(departmentIds)) {
            queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, departmentIds));
        }

        IPage<ProjectRegisterEntity> pageData = projectRegisterService.queryPage(queryParam, false);
        Map<String, Object> beans = new HashMap<String, Object>();
        beans.put("records", BeanMapper.mapList(pageData.getRecords(), ProjectRegisterVO.class));
        ExcelExport.getInstance().export("project-export.xlsx", beans, response);
    }

    /**
     * 导入模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping("/download")
    @ResponseBody
    public void download(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "project-import.xlsx", "项目导入模板");
    }

    @PostMapping("/saveImportTemplate")
    public CommonResponse<JSONObject> saveImportData(@RequestBody String templatekey) {
        try {
            String jsonData = cacheManager.get(templatekey);
            if (StringUtils.isBlank(jsonData)) {
                return CommonResponse.error("数据导入失败，未找到可导入的数据内容！");
            }
            List<ProjectRegisterVO> proList = JSONObject.parseArray(jsonData, ProjectRegisterVO.class);
            if (CollectionUtils.isEmpty(proList)) {
                return CommonResponse.error("数据导入失败，没有可导入的数据！");
            }

            projectRegisterService.saveImportProject(proList);
            return CommonResponse.success("数据导入保存成功！");
        } finally {
            //删除缓存模版数据
            cacheManager.removeCache(templatekey);
        }
    }

    @PostMapping("/excelImport")
    public CommonResponse<JSONObject> excelImport(HttpServletRequest request, HttpServletResponse resp) {
        JSONObject respData = new JSONObject();
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        List<ProjectRegisterVO> saveList = new ArrayList<>();
        List<JSONObject> errorList = new ArrayList<>();
        boolean isFailed = false;
        MultipartFile mf = null;
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            logger.info("接收到项目导入文件：[{}]", originalFileName);
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }

        if (isFailed) {
            return CommonResponse.error("文件格式不合法");
        } else {
            List<List<String>> result = ExcelReader.readExcel(mf);
            logger.info("项目导入文件行数(包含非数据行)：[{}]", result.size());
            logger.info("项目导入文件内容：[{}]", JSONObject.toJSONString(result));
            if (CollectionUtils.isNotEmpty(result)) {
                List<String> rowData = null;
                List<ProjectRegisterEntity> dbList = null;
                List<String> saveNameList = new ArrayList<>();
                List<String> saveCodeList = new ArrayList<>();
                ProjectRegisterVO proInfo = null;

                //每次仅处理前100条数据
                for (int i = 1; (i < result.size() && i < 100); i++) {
                    proInfo = new ProjectRegisterVO();
                    //从第二行开始读取数据
                    rowData = result.get(i);
                    JSONObject errInfo = rowDataCheck(i, rowData, saveNameList, saveCodeList, proInfo);
                    logger.info("第{}行校验结果：{}", i + 1, null != errInfo ? errInfo.get("errMsg") : "校验通过！");
                    if (null != errInfo) {
                        proInfo = null;
                        errorList.add(errInfo);
                        continue;
                    }
                    saveList.add(proInfo);
                }

                if (CollectionUtils.isNotEmpty(saveList)) {
                    String key = getCacheKey();
                    respData.put("templateKey", key);
                    cacheManager.setex(key, JSONObject.toJSONString(saveList), TIMEOUT);
                }
            }
            respData.put("importableNum", saveList.size());
            respData.put("errorList", errorList);
            return CommonResponse.success(respData);
        }
    }

    private String getCacheKey() {
        return TEMPLATE_KEY_PREFIX + TEMPLATE_KEY_SEPRATOR +
                InvocationInfoProxy.getUserid() + TEMPLATE_KEY_SEPRATOR +
                System.currentTimeMillis();
    }

    private JSONObject generateErrorRow(int rowNum, List<String> rowData, String msg) {
        JSONObject error = new JSONObject();
        error.put("rowNum", rowNum);
        error.put("code", rowData.get(0));
        error.put("shortName", rowData.get(1));
        error.put("name", rowData.get(2));
        error.put("pubUnitName", rowData.get(3));
        error.put("orgName", rowData.get(4));
        error.put("status", rowData.get(5));
        error.put("errMsg", msg);
        return error;
    }

    /**
     * 项目编码重复校验
     *
     * @return
     */
    @PostMapping("codeUsedCheck")
    public CommonResponse<String> codeUsedCheck(@RequestBody ProjectRegisterVO project) {
        boolean checkResult = projectRegisterService.codeUsedCheck(project);
        if (checkResult) {
            return CommonResponse.error("编码重复！");
        }
        return CommonResponse.success("编码可以使用！");
    }

    private JSONObject rowDataCheck(int rowNum, List<String> rowData, List<String> curVlidProNameList, List<String> curVlidProCodeList, ProjectRegisterVO proInfo) {
        String name = rowData.get(2);
        String shortName = rowData.get(1);
        String code = rowData.get(0);
        String orgName = rowData.get(4);
        Long tenantId = InvocationInfoProxy.getTenantid();
        if (StringUtils.isNotBlank(shortName)) {
            if (shortName.length() > 50) {
                return generateErrorRow(rowNum, rowData, "项目简称最多50字");
            }
            shortName = shortName.replaceAll(" ", ""); //替换所有空格
        }

        if (StringUtils.isBlank(name)) {
            return generateErrorRow(rowNum, rowData, "项目名称为空");
        }
        name = name.replaceAll(" ", ""); //替换所有空格
        if (name.length() > 50) {
            return generateErrorRow(rowNum, rowData, "项目名称最多50字");
        }
        if (StringUtils.isNotBlank(code)) {
            code = code.replaceAll(" ", "");
            if (code.length() > 64) {
                return generateErrorRow(rowNum, rowData, "项目编码最多64个字符");
            }
        }


        if (StringUtils.isBlank(orgName)) {
            return generateErrorRow(rowNum, rowData, "所属组织为空");
        }
        orgName = orgName.replaceAll(" ", "");

        if (StringUtils.isBlank(rowData.get(5))) {
            return generateErrorRow(rowNum, rowData, "项目状态为空");
        }
        if (StringUtils.isBlank(rowData.get(6))) {
            return generateErrorRow(rowNum, rowData, "项目类型为空");
        }

        //名称重复校验
        if (curVlidProNameList.contains(name)) {
            return generateErrorRow(rowNum, rowData, "项目名称重复");
        }
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, tenantId));
        queryParam.getParams().put("name", new Parameter(QueryParam.EQ, name));
        List<ProjectRegisterEntity> dbList = projectRegisterService.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(dbList)) {
            return generateErrorRow(rowNum, rowData, "项目名称重复");
        }

        //编码唯一校验
        boolean codeCheck = curVlidProCodeList.contains(code);
        if (!codeCheck && StringUtils.isNotBlank(code)) {
            proInfo.setTenantId(tenantId);
            proInfo.setCode(code);
            codeCheck = projectRegisterService.codeUsedCheck(proInfo);
        }
        if (codeCheck) {
            return generateErrorRow(rowNum, rowData, "编码重复");
        }

        //判断发包单位是否存在
        if (StringUtils.isNotBlank(rowData.get(3))) {
            CustomerVO customer = (CustomerVO) getRespData(customerApi.findOneByName(rowData.get(2), InvocationInfoProxy.getTenantid()), true, "文件校验失败，查询发包单位信息失败！");
            if (null == customer) {
                return generateErrorRow(rowNum, rowData, "发包单位不存在");
            }
            proInfo.setPubUnit(customer.getId());
            proInfo.setPubUnitName(customer.getName());
        }
        //判断所属组织是否存在
        OrgVO orgVO = (OrgVO) getRespData(orgApi.findByNameAndTenantId(orgName, InvocationInfoProxy.getTenantid()),
                true, "文件校验失败，查询组织信息失败");
        if (null == orgVO) {
            return generateErrorRow(rowNum, rowData, "所属组织不存在");
        }
        if(orgVO.getOrgType() == 5){
            return generateErrorRow(rowNum, rowData, "所属组织不能是项目部组织");
        }

        curVlidProNameList.add(name);
        if (StringUtils.isNotBlank(code)) {
            curVlidProCodeList.add(code);
            proInfo.setCode(code);
        }
        proInfo.setOrgId(orgVO.getId());
        proInfo.setOrgName(orgVO.getName());
        proInfo.setName(name);
        proInfo.setShortName(shortName);
        proInfo.setType("自营".equals(rowData.get(6)) ? "1" : "2");

        if (StringUtils.isNotBlank(rowData.get(5))) {
            proInfo.setStatus("未开工".equals(rowData.get(5)) ? "1" : "在建".equals(rowData.get(5)) ? "2" : "3");
        }

        return null;
    }

    class Rule {
        private String type;
        private String pattern;
        private String message;

        public Rule(String type, String pattern, String message) {
            this.type = type;
            this.pattern = pattern;
            this.message = message;
        }

        public String getType() {
            return type;
        }

        public void setType(String type) {
            this.type = type;
        }

        public String getPattern() {
            return pattern;
        }

        public void setPattern(String pattern) {
            this.pattern = pattern;
        }

        public String getMessage() {
            return message;
        }

        public void setMessage(String message) {
            this.message = message;
        }
    }

    /**
     * 移动端项目数量统计(租户级别)
     *
     * @return
     */
    @GetMapping(value = "/maIndexCount")
    public JSONObject maIndexProjectsCount(HttpServletRequest req) {
        JSONObject resp = new JSONObject();
        QueryParam queryParam = new QueryParam();
        //查询本租户下的项目
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        //只查询启用的项目
        queryParam.getParams().put("validStatus", new Parameter(QueryParam.EQ, '0'));

        //查询项目的总数、在建项目数和已完工数量
        Map<String, Integer> numMap = projectRegisterService.countProjectNum(queryParam);
        numMap.put("total", numMap.get("buildingNum") + numMap.get("finishNum"));

        queryParam.getComplexParams().clear();
        List<Long> projectIds = projectRegisterService.queryProIdsByProperties(queryParam, true);
        numMap.put("newNumCurYear", projectIds.size());


        resp.put("code", 0);
        resp.put("msg", "查询成功！");
        resp.put("data", numMap);
        return resp;
    }

    /**
     * 根据当前组织，如果当前组织是项目，则部查询其对应的项目
     *
     * @return
     */
    @GetMapping("getProjectByCurrentOrg")
    public CommonResponse<ProjectRegisterVO> getProjectByCurrentOrg(){
        QueryParam param = new QueryParam();
        param.getParams().put("projectDepartmentId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getOrgId()));
        List<ProjectRegisterEntity> entities = projectRegisterService.queryList(param,false);
        if(ListUtil.isNotEmpty(entities)){
//            return CommonResponse.success(BeanMapper.map(entities.get(0),ProjectRegisterVO.class));
        }
        return CommonResponse.error("没有查到项目");
    }

    /**
     * 更改项目状态
     *
     * @param projectRegisterVO
     * @return
     */
    @PostMapping(value = "changeStatus")
    public CommonResponse<ProjectRegisterVO> changeStatus(@RequestBody ProjectRegisterVO projectRegisterVO) {
        ProjectRegisterVO resp = projectRegisterService.changeStatus(projectRegisterVO);
        return CommonResponse.success("保存成功！", resp);
    }

    /**
     * 企业微信和钉钉第三方应用初始化时候，生成默认项目
     * body{
     *     tenantId：xxx
     *     orgId：xxx
     *     orgName:xxx
     *     enterpriseId:xxx
     * }
     * */
    @PostMapping(value = "no_auth/initQYWXAndDDDefaultProject")
    public CommonResponse<JSONObject> initQYWXAndDDDefaultProject(@RequestBody JSONObject body){
        if(body == null){
            return CommonResponse.error("参数不能为空！");
        }
        if(body.getLong("tenantId") == null){
            return CommonResponse.error("租户不能为空！");
        }
        if(body.getLong("enterpriseId") == null){
            return CommonResponse.error("企业不能为空！");
        }
        if(body.getLong("orgId") == null || StringUtils.isEmpty(body.getString("orgName"))){
            return CommonResponse.error("组织不能为空！");
        }
        JSONObject res = new JSONObject();
        QueryParam param = new QueryParam();
        param.getParams().put("tenantId",new Parameter(QueryParam.EQ,body.getLong("tenantId")));
        param.getParams().put("orgId",new Parameter(QueryParam.EQ,body.getLong("orgId")));
        List<ProjectRegisterEntity> projectRegisterEntities = projectRegisterService.queryList(param,false);
        if(ListUtil.isNotEmpty(projectRegisterEntities)){
            res.put("exist",true);
            /** 有项目，不新增*/
            logger.info("企业微信和钉钉第三方应用初始化时候，生成默认项目===>> 有项目，不新增,body={} ",body);
            return CommonResponse.success(res);
        }
        res.put("exist",false);
        ProjectRegisterEntity entity = new ProjectRegisterEntity();
        entity.setCode("INIT-PRO-"+ DateFormatUtil.formatDate("yyyyMMddHHmmsss",System.currentTimeMillis()+random.nextInt(3000000)));
        entity.setName("我的第一个项目");
        entity.setShortName("我的第一个项目");
        entity.setId(IdWorker.getId());
        entity.setTenantId(body.getLong("tenantId"));
        OrgVO projectDepartment = new OrgVO();
        projectDepartment.setId(IdWorker.getId());
        projectDepartment.setCode("INIT"+projectDepartment.getId());
        projectDepartment.setParentId(body.getLong("orgId"));
        projectDepartment.setTenantId(body.getLong("tenantId"));
        projectDepartment.setEnterpriseId(body.getLong("enterpriseId"));
        projectDepartment.setIsParent(false);
        projectDepartment.setName(entity.getName());
        projectDepartment.setShortName(entity.getName());
        projectDepartment.setState(1);
        projectDepartment.setProjectState(1);
        projectDepartment.setProjectCode(entity.getCode());
        projectDepartment.setNote("系统创建");
        //项目部类型
        projectDepartment.setOrgType(OrgVO.ORG_TYPE_DEPARTMENT);
        entity.setProjectDepartmentId(projectDepartment.getId());
        entity.setCreateUserName("系统创建");
        //启用状态
        entity.setValidStatus("0");
        //项目状态-在建
        entity.setStatus("2");
        /** 立项日期 */
        entity.setSignDate(new Date());
        /** 项目类型：1-自营，2-联营 */
        entity.setType("1");
        /** 所属组织 */
        entity.setOrgName(body.getString("orgName"));
        entity.setOrgId(body.getLong("orgId"));
        projectRegisterService.saveOrUpdateNoES(entity);
        res.put("orgDeptInfo",projectDepartment);
        return CommonResponse.success(res);
    }

    public static void main(String[] args) {
        System.out.println("INIT-PRO-"+ DateFormatUtil.formatDate("yyyyMMddHHmmsss",System.currentTimeMillis()+random.nextInt(3000000)));
        System.out.println("INIT-PRO-"+ DateFormatUtil.formatDate("yyyyMMddHHmmsss",System.currentTimeMillis()+random.nextInt(300000)));
        System.out.println("INIT-PRO-"+ DateFormatUtil.formatDate("yyyyMMddHHmmsss",System.currentTimeMillis()+random.nextInt(300000)));
        System.out.println("INIT-PRO-"+ DateFormatUtil.formatDate("yyyyMMddHHmmsss",System.currentTimeMillis()+random.nextInt(300000)));
    }
}
