package com.ejianc.business.market.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.market.bean.ProjectRegisterEntity;
import com.ejianc.business.market.service.IProjectRegisterService;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.vo.CustomerVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.api.ICustomerApi;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.core.util.ImportTemplate;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.*;
import java.util.stream.Collectors;

@RestController
@RequestMapping("projectRegister")
public class ProjectResigerController {

    private final String TEMPLATE_KEY_PREFIX = "template_data_cache_market_project";
    private final String TEMPLATE_KEY_SEPRATOR = ":";
    private final int TIMEOUT = 1800;
    private final String PROJECT_BILL_TYPE_CODE = "BT200519000000002";
    private final static Random random = new Random();
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IProjectRegisterService projectRegisterService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IUserApi userApi;

    @Autowired
    private CacheManager cacheManager;

    @Autowired
    private ICustomerApi customerApi;

    @Value("${common.env.base-host}")
    private String BASE_HOST;

    @Autowired
    private IBillTypeApi billTypeApi;


    /**
     * 保存、更新项目登记信息
     *
     * @param projectRegisterVO
     * @return
     */
    @PostMapping(value = "saveProject")
    public CommonResponse<ProjectRegisterVO> saveProject(@RequestBody ProjectRegisterVO projectRegisterVO) {
        return projectRegisterService.saveProject(projectRegisterVO);
    }

    private Object getRespData(CommonResponse resp, boolean isMustSuc, String errMsg) {
        if (isMustSuc && !resp.isSuccess()) {
            throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
        }
        return resp.getData();
    }

    /**
     * 分页查询项目登记列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping("page")
    public CommonResponse<JSONObject> queryPageList(@RequestBody QueryParam queryParam) {
        JSONObject resp = new JSONObject();
        queryParam.getFuzzyFields().add("name");
        queryParam.getFuzzyFields().add("code");
        queryParam.getFuzzyFields().add("orgName");
        //添加简称项目名称模糊查询
        queryParam.getFuzzyFields().add("shortName");
        queryParam.getFuzzyFields().add("contractorUnitName");
        queryParam.getFuzzyFields().add("projectManagerName");
        queryParam.getFuzzyFields().add("pubUnitName");
        Map<String, Integer> countProjectNum = null;
        IPage<ProjectRegisterEntity> pageData = null;
        List<OrgVO> orgVOList = (List<OrgVO>) getRespData(orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()), true, "查询失败，获取当前本下组织信息失败。");

        if (CollectionUtils.isEmpty(orgVOList)) {
            pageData = new Page<>();
            pageData.setRecords(new ArrayList<>());
            pageData.setTotal(0);
            pageData.setSize(queryParam.getPageSize());
            pageData.setCurrent(queryParam.getPageIndex());
            pageData.setPages(0);
            countProjectNum = new HashMap<>();
            countProjectNum.put("buildingNum", 0);
            countProjectNum.put("finishNum", 0);

        } else {
            List<Long> commonOrgIds = new ArrayList<>();
            List<Long> departmentIds = new ArrayList<>();
            orgVOList.stream().forEach(org -> {
                if (5 == org.getOrgType()) {
                    //项目部
                    departmentIds.add(org.getId());
                } else {
                    //普通组织
                    commonOrgIds.add(org.getId());
                }
            });
            if (CollectionUtils.isNotEmpty(commonOrgIds)) {
                queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
            } else {
                queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, departmentIds));
            }

            pageData = projectRegisterService.queryPage(queryParam, false);
            //查询项目的总数、在建项目数和已完工数量
            countProjectNum = projectRegisterService.countProjectNum(queryParam);
        }


        resp.put("records", pageData.getRecords());
        resp.put("total", pageData.getTotal());
        resp.put("size", pageData.getSize());
        resp.put("current", pageData.getCurrent());
        resp.put("pages", pageData.getPages());
        resp.put("projectCount", countProjectNum);

        return CommonResponse.success("分页查询登记项目列表成功！", resp);
    }

    /**
     * 根据主键Id查询项目登记详情
     *
     * @param id
     * @return
     */
    @GetMapping("detail")
    public CommonResponse<ProjectRegisterVO> detail(@RequestParam Long id) {
        ProjectRegisterVO resp = BeanMapper.map(projectRegisterService.queryById(id), ProjectRegisterVO.class);

        //查询项目成员
        CommonResponse<List<Map<String, Object>>> membersResp = userApi.getEmployeeList(resp.getProjectDepartmentId());
        if (!membersResp.isSuccess()) {
            return CommonResponse.error("查询失败，获取项目成员失败。");
        }
        QueryParam queryParam = new QueryParam();
        resp.setMembersList(membersResp.getData());
        for (Map<String,Object> map :resp.getMembersList()){
            JSONObject jsonObj = new JSONObject();
            jsonObj.put("id",map.get("postId"));
            jsonObj.put("name",map.get("postName"));
            map.put("postId",jsonObj);
        }
        //查询项目附件
        return CommonResponse.success("查询项目登记详情成功！", resp);
    }

    /**
     * 根据主键Id删除对应的登记项目
     *
     * @return
     */
    @PostMapping("delById")
    public CommonResponse<String> deleteById(@RequestBody List<Long> ids) {
        Long projectId = ids.get(0);
        logger.info("执行项目删除操作，删除项目[id-{}]", projectId);

        String userType = InvocationInfoProxy.getUserType();
        if (!("1".equals(userType) || "3".equals(userType))) {
            return CommonResponse.error("请联系管理员执行项目删除操作。");
        }

        //引用查询，若该项目已被下游引用，则无法进行删除操作
        CommonResponse<String> resp = billTypeApi.checkQuote(PROJECT_BILL_TYPE_CODE, projectId);
        if (!resp.isSuccess()) {
            logger.info("项目立项【Id-{}】无法进行删除操作，原因：{}", projectId, resp.getMsg());
            return CommonResponse.error("校验项目下游引用状况失败，删除操作中止.");
        }

        //执行项目删除操作
        String result = projectRegisterService.delProject(projectId);
        if (StringUtils.isNotBlank(result)) {
            return CommonResponse.error(result);
        }
        return CommonResponse.success("删除成功！");
    }

    /**
     * 查询指定组织本下的所有项目信息列表
     *
     * @param pageNumber
     * @param pageSize
     * @param condition
     * @param searchText
     * @return
     */
    @GetMapping("/projectListRefer")
    public CommonResponse<IPage<ProjectRegisterVO>> projectListRefe(@RequestParam(defaultValue = "1") Integer pageNumber,
                                                                    @RequestParam(defaultValue = "10") Integer pageSize,
                                                                    @RequestParam(value = "condition", required = false) String condition,
                                                                    @RequestParam(value = "searchText", required = false) String searchText) {

        QueryParam queryParam = new QueryParam();
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.setSearchText(searchText);
        queryParam.getFuzzyFields().add("code");
        queryParam.getFuzzyFields().add("name");
        queryParam.getFuzzyFields().add("orgName");
        //添加简称项目名称模糊查询
        queryParam.getFuzzyFields().add("shortName");

        //只查询启用的项目
        queryParam.getParams().put("validStatus", new Parameter(QueryParam.EQ, '0'));
        //租户过滤
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.getOrderMap().put("createTime", QueryParam.DESC);

        List<Long> commonOrgIds = new ArrayList<>();
        List<Long> departmentIds = new ArrayList<>();
        Long orgId = InvocationInfoProxy.getOrgId();
        Boolean tenantLeave = false;
        /** 移动端参照标识 */
        Boolean mobileLeave = false;

        if (StringUtils.isNotBlank(condition)) {
            Map<String, Object> conditionMap = JSONObject.parseObject(condition, Map.class);

            if (null != conditionMap.get("orgId")) {
                orgId = Long.valueOf(conditionMap.get("orgId").toString());
            }
            if (null != conditionMap.get("tenantLeave")) {
                //租户级别数据
                tenantLeave = true;
            }
            if (null != conditionMap.get("mobileLeave")) {
                //移动端数据
                mobileLeave = true;
            }

            if (null != conditionMap.get("province")) {
                queryParam.getParams().put("province", new Parameter(QueryParam.EQ, conditionMap.get("province")));
            }
            if (null != conditionMap.get("city")) {
                queryParam.getParams().put("city", new Parameter(QueryParam.EQ, conditionMap.get("city")));
            }
            if (null != conditionMap.get("region")) {
                queryParam.getParams().put("region", new Parameter(QueryParam.EQ, conditionMap.get("region")));
            }
            // 不包含这个项目
            if (null != conditionMap.get("projectId")) {
                queryParam.getParams().put("id", new Parameter(QueryParam.NE, conditionMap.get("projectId")));
            }
        }

        if (mobileLeave) {
            //移动端
            CommonResponse<List<OrgVO>> response = orgApi.findProjectOrgsByUserId(InvocationInfoProxy.getUserid());
            if (response.isSuccess() && ListUtil.isNotEmpty(response.getData())) {
                queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.IN, response.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
            } else {
                IPage<ProjectRegisterVO> pageData = new Page<>(pageNumber, pageSize, 0);
                return CommonResponse.success("该用户没有任何项目权限！", pageData);
            }

        } else if (!tenantLeave) {
            //组织本下
            CommonResponse<List<OrgVO>> orgListResp = orgApi.findChildrenByParentId(orgId);
            if (!orgListResp.isSuccess()) {
                IPage<ProjectRegisterVO> pageData = new Page<>(pageNumber, pageSize, 0);
                return CommonResponse.success("查询数表参照失败，项目组织信息查询失败！", pageData);
            }
            List<OrgVO> orgList = orgListResp.getData();

            if (CollectionUtils.isNotEmpty(orgList)) {
                orgList.forEach(orgVO -> {
                    if (5 == orgVO.getOrgType()) {
                        //项目部
                        departmentIds.add(orgVO.getId());
                    } else {
                        //普通组织
                        commonOrgIds.add(orgVO.getId());
                    }
                });
                if (CollectionUtils.isNotEmpty(commonOrgIds)) {
                    queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, commonOrgIds));
                } else if (CollectionUtils.isNotEmpty(departmentIds)) {
                    queryParam.getParams().put("projectDepartmentId", new Parameter(QueryParam.EQ, departmentIds.get(0)));
                }
            }
        }

        IPage<ProjectRegisterEntity> pageData = projectRegisterService.queryPage(queryParam, false);
        IPage<ProjectRegisterVO> result = new Page<>(pageData.getCurrent(), pageData.getSize(), pageData.getTotal());
        result.setRecords(BeanMapper.mapList(pageData.getRecords(), ProjectRegisterVO.class));
        return CommonResponse.success("项目参照查询成功！", result);
    }


    private JSONObject generateErrorRow(int rowNum, List<String> rowData, String msg) {
        JSONObject error = new JSONObject();
        error.put("rowNum", rowNum);
        error.put("code", rowData.get(0));
        error.put("shortName", rowData.get(1));
        error.put("name", rowData.get(2));
        error.put("pubUnitName", rowData.get(3));
        error.put("orgName", rowData.get(4));
        error.put("status", rowData.get(5));
        error.put("errMsg", msg);
        return error;
    }

    /**
     * 根据当前组织，如果当前组织是项目，则部查询其对应的项目
     *
     * @return
     */
    @GetMapping("getProjectByCurrentOrg")
    public CommonResponse<ProjectRegisterVO> getProjectByCurrentOrg(){
        QueryParam param = new QueryParam();
        param.getParams().put("projectDepartmentId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getOrgId()));
        List<ProjectRegisterEntity> entities = projectRegisterService.queryList(param,false);
        if(ListUtil.isNotEmpty(entities)){
//            return CommonResponse.success(BeanMapper.map(entities.get(0),ProjectRegisterVO.class));
        }
        return CommonResponse.error("没有查到项目");
    }

    /**
     * 更改项目状态
     *
     * @param projectRegisterVO
     * @return
     */
    @PostMapping(value = "changeStatus")
    public CommonResponse<ProjectRegisterVO> changeStatus(@RequestBody ProjectRegisterVO projectRegisterVO) {
        ProjectRegisterVO resp = projectRegisterService.changeStatus(projectRegisterVO);
        return CommonResponse.success("保存成功！", resp);
    }

    public static void main(String[] args) {
        System.out.println("INIT-PRO-"+ DateFormatUtil.formatDate("yyyyMMddHHmmsss",System.currentTimeMillis()+random.nextInt(3000000)));
        System.out.println("INIT-PRO-"+ DateFormatUtil.formatDate("yyyyMMddHHmmsss",System.currentTimeMillis()+random.nextInt(300000)));
        System.out.println("INIT-PRO-"+ DateFormatUtil.formatDate("yyyyMMddHHmmsss",System.currentTimeMillis()+random.nextInt(300000)));
        System.out.println("INIT-PRO-"+ DateFormatUtil.formatDate("yyyyMMddHHmmsss",System.currentTimeMillis()+random.nextInt(300000)));
    }
}
