package com.ejianc.business.material.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.material.bean.*;
import com.ejianc.business.material.pub.MaterialStoreType;
import com.ejianc.business.material.service.*;
import com.ejianc.business.material.utlis.ArchivesUtil;
import com.ejianc.business.material.vo.*;
import com.ejianc.business.utils.DateUtil;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.vo.MaterialInsertArchiveVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.*;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 采购订单
 *
 * @author generator
 *
 */
@Controller
@RequestMapping("purchaseOrder")
public class PurchaseOrderController implements Serializable {
    private static final long serialVersionUID = 1L;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi iOrgApi;

    private static final String BILL_CODE = "PURCHASE_ORDER";

    @Autowired
    private IPurchaseOrderService service;

    @Autowired
    private IPurchaseOrderDetailService detailService;

    @Autowired
    IPurchaseSettlementService purchaseSettlementService;

    @Autowired
    IMaterialContractService contractService;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private ArchivesUtil archivesUtil;

    @Autowired
    IStoreService storeService;

    @Autowired
    IInstoreService instoreService;

    @Autowired
    IInstoreMaterialService instoreMaterialService;


    @Autowired
    IUseApplyService useApplyService;
    @Autowired
    IUseApplySubService useApplySubService;

    /**
     * @description: 用料申请待分配生成订单
     *
     * @param purchaseOrderVO
     * @return {@link CommonResponse< PurchaseOrderVO>}
     * @author songlx
     * @date: 2024/6/17
     */
    @RequestMapping(value = "/addByUseApplyAssign", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PurchaseOrderVO> addByUseApplyAssign(@RequestBody PurchaseOrderVO purchaseOrderVO) {
        Long applyId = purchaseOrderVO.getApplyId();
        UseApplyEntity entity = useApplyService.getById(applyId);
        PurchaseOrderVO orderVO = BeanMapper.map(entity, PurchaseOrderVO.class);
        //默认待确认
        orderVO.setOrderState(0);
        orderVO.setApplyId(applyId);
        orderVO.setId(null);
        orderVO.setBillCode(null);
        orderVO.setCreateUserCode(null);
        orderVO.setCreateTime(null);
        orderVO.setUpdateUserCode(null);
        orderVO.setUpdateTime(null);
        List<PurchaseOrderDetailVO> purchaseOrderDetailList = purchaseOrderVO.getPurchaseOrderDetailList();
        PurchaseOrderDetailVO detailVO = purchaseOrderDetailList.get(0);
        orderVO.setContractId(detailVO.getPurchaseContractId());
        orderVO.setContractName(detailVO.getPurchaseContractName());
        for (PurchaseOrderDetailVO purchaseOrderDetailVO : purchaseOrderDetailList) {
            purchaseOrderDetailVO.setRowState("add");
            purchaseOrderDetailVO.setSourceId(applyId);
            purchaseOrderDetailVO.setSourceDetailId(purchaseOrderDetailVO.getId());
            purchaseOrderDetailVO.setSourceType(2);
        }
        orderVO.setPurchaseOrderDetailList(purchaseOrderDetailList);
        return CommonResponse.success("查询详情数据成功！", purchaseOrderVO);
    }


    /**
     * @description: 采购任务退回、需要重新指派接收人
     *
     * @param purchaseOrderVO
     * @return {@link CommonResponse< PurchaseOrderVO>}
     * @author songlx
     * @date: 2024/6/17
     */
    @RequestMapping(value = "/back", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PurchaseOrderVO> back(@RequestBody PurchaseOrderVO purchaseOrderVO) {
        List<PurchaseOrderDetailVO> detailList = purchaseOrderVO.getPurchaseOrderDetailList();
        if(CollectionUtils.isNotEmpty(detailList)){
            UserContext userContext = sessionManager.getUserContext();
            List<Long> applySubIds = detailList.stream().map(PurchaseOrderDetailVO::getId).collect(Collectors.toList());
            LambdaUpdateWrapper<UseApplySubEntity> updateWrapper = Wrappers.lambdaUpdate();
            updateWrapper.in(UseApplySubEntity::getId, applySubIds);
            updateWrapper.set(UseApplySubEntity::getTaskState, "unassign");
            updateWrapper.set(UseApplySubEntity::getReceiverId, null);
            updateWrapper.set(UseApplySubEntity::getReceiverName, null);
            updateWrapper.set(UseApplySubEntity::getReturnerId, userContext.getEmployeeId());
            updateWrapper.set(UseApplySubEntity::getReturnerName, userContext.getEmployeeName());
            useApplySubService.update(updateWrapper);
        }
        return CommonResponse.success("查询详情数据成功！", purchaseOrderVO);
    }


    /**
     * @description: 零星采购
     *
     * @param purchaseOrderVO
     * @return {@link CommonResponse< PurchaseOrderVO>}
     * @author songlx
     * @date: 2024/6/17
     */
    @RequestMapping(value = "/sporadicPurchase", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PurchaseOrderVO> sporadicPurchase(@RequestBody PurchaseOrderVO purchaseOrderVO) {
        List<PurchaseOrderDetailVO> detailList = purchaseOrderVO.getPurchaseOrderDetailList();
        if(CollectionUtils.isNotEmpty(detailList)){
            UserContext userContext = sessionManager.getUserContext();
            List<Long> applySubIds = detailList.stream().map(PurchaseOrderDetailVO::getId).collect(Collectors.toList());
            LambdaUpdateWrapper<UseApplySubEntity> updateWrapper = Wrappers.lambdaUpdate();
            updateWrapper.in(UseApplySubEntity::getId, applySubIds);
            updateWrapper.set(UseApplySubEntity::getTaskState, "back");
            updateWrapper.set(UseApplySubEntity::getReceiverId, null);
            updateWrapper.set(UseApplySubEntity::getReceiverName, null);
            updateWrapper.set(UseApplySubEntity::getReturnerId, userContext.getEmployeeId());
            updateWrapper.set(UseApplySubEntity::getReturnerName, userContext.getEmployeeName());
            useApplySubService.update(updateWrapper);
        }
        return CommonResponse.success("查询详情数据成功！", purchaseOrderVO);
    }


    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PurchaseOrderVO> saveOrUpdate(@RequestBody PurchaseOrderVO saveOrUpdateVO) {
        PurchaseOrderEntity entity = BeanMapper.map(saveOrUpdateVO, PurchaseOrderEntity.class);
        if (entity.getId() == null || entity.getId() == 0) {
            UserContext userContext = sessionManager.getUserContext();
            String userName = userContext.getUserName();
            entity.setName(userName + DateUtil.getCurrentDay(null) + "采购订单");
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        long id = IdWorker.getId();
        if (entity.getId() == null) {
            entity.setId(id);
        }
        List<PurchaseOrderDetailVO> purchaseOrderDetailList = saveOrUpdateVO.getPurchaseOrderDetailList();
        if (CollectionUtils.isNotEmpty(purchaseOrderDetailList)) {
            for (PurchaseOrderDetailVO purchaseOrderDetailVO : purchaseOrderDetailList) {
                String onlykey = purchaseOrderDetailVO.getMaterialTypeName() + purchaseOrderDetailVO.getMaterialName() + purchaseOrderDetailVO.getUnit() + purchaseOrderDetailVO.getModel()+purchaseOrderDetailVO.getMaterialCode();
                purchaseOrderDetailVO.setOnlyKey(onlykey);
            }
            MaterialInsertArchiveVO archiveVO = new MaterialInsertArchiveVO(null, entity.getId(), entity.getBillCode(), "采购订单");
            JSONObject jsonObject = archivesUtil.batchSaveArchive(archiveVO, purchaseOrderDetailList, "materialTypeId", "materialId",
                    "materialTypeName", "materialName", "materialCode","unit", "model", "errorMessage");
            String sourceList = JSON.toJSONString(jsonObject.get("sourceList"), SerializerFeature.WriteMapNullValue, SerializerFeature.PrettyFormat);
            List<PurchaseOrderDetailVO> list = JSONObject.parseArray(sourceList, PurchaseOrderDetailVO.class);
            List<PurchaseOrderDetailEntity> purchaseOrderDetailEntities = BeanMapper.mapList(list, PurchaseOrderDetailEntity.class);
            entity.setPurchaseOrderDetailList(purchaseOrderDetailEntities);
        }
        service.saveOrUpdate(entity, false);
        PurchaseOrderVO vo = BeanMapper.map(entity, PurchaseOrderVO.class);
        return CommonResponse.success("保存或修改单据成功！", vo);
    }

    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<PurchaseOrderVO> queryDetail(Long id) {
        PurchaseOrderEntity entity = service.selectById(id);
        PurchaseOrderVO vo = BeanMapper.map(entity, PurchaseOrderVO.class);
        return CommonResponse.success("查询详情数据成功！", vo);
    }

    /**
     * @Description delete 批量删除单据
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<PurchaseOrderVO> vos) {
        service.removeByIds(vos.stream().map(PurchaseOrderVO::getId).collect(Collectors.toList()), true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<PurchaseOrderVO>> queryList(@RequestBody QueryParam param) {

        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("contractName", "projectName", "supplierName","billCode"));
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        /** 数据隔离 本下 */
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (org.apache.commons.lang.StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    iOrgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }

        IPage<PurchaseOrderEntity> page = service.queryPage(param, false);
        IPage<PurchaseOrderVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), PurchaseOrderVO.class));
        for (PurchaseOrderVO record : pageData.getRecords()) {
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("pid", new Parameter(QueryParam.EQ, record.getId()));
            List<PurchaseOrderDetailVO> detailEntityList = service.querySubData(null, queryParam);
            if (CollectionUtils.isEmpty(detailEntityList)) {
                record.setInstoreOver(1);
            }else {
                record.setInstoreOver(2);
            }
        }

        return CommonResponse.success("查询列表数据成功！", pageData);
    }

    /**
     * 获取RPC数据
     * resp 返回值
     * isMustSuc 是否必须成功
     * errMsg 失败提示
     */
    private Object getRespData(CommonResponse<?> resp, boolean isMustSuc, String errMsg) {
        if (isMustSuc && !resp.isSuccess()) {
            throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
        }
        return resp.getData();
    }


    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.addAll(Arrays.asList("contractName", "projectName", "supplierName","billCode"));
        param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.setPageIndex(1);
        param.setPageSize(-1);
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        List<PurchaseOrderEntity> list = service.queryList(param);
        List<PurchaseOrderVO> purchaseOrderVOS = BeanMapper.mapList(list, PurchaseOrderVO.class);
        if (CollectionUtils.isNotEmpty(purchaseOrderVOS)) {
            purchaseOrderVOS.forEach(t -> {
                t.setBillStateName(BillStateEnum.getEnumByStateCode(t.getBillState()).getDescription());
            });
        }
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", purchaseOrderVOS);
        ExcelExport.getInstance().exportWithTrans("purchaseOrder-export.xlsx", beans, response);
    }

    /**
     * @Description 参照
     */
    @RequestMapping(value = "/refPurchaseOrderData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<PurchaseOrderVO>> refPurchaseOrderData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                                       String condition,
                                                                       String searchObject,
                                                                       String searchText) {
        QueryParam param = new QueryParam();
        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        if (StringUtils.isNotEmpty(condition)) {
            /** 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
        }

        IPage<PurchaseOrderEntity> page = service.queryPage(param, false);
        IPage<PurchaseOrderVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), PurchaseOrderVO.class));

        return CommonResponse.success("查询参照数据成功！", pageData);
    }

    /**
     * @description: 采购订单tree参照
     *
     * @param pageNumber
     * @param pageSize
     * @param condition
     * @return {@link List< Map< String, Object>>}
     * @author songlx
     * @date: 2023/4/19
     */
    @RequestMapping(value = "/referPurchaseOrderTree", method = RequestMethod.GET)
    @ResponseBody
    public List<Map<String, Object>> referPurchaseOrderTree(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                            @RequestParam String condition
    ) {
        if (StringUtils.isEmpty(condition) || !condition.contains("projectId") || !condition.contains("contractId")) {
            throw new BusinessException("查询参数不正确！");
        }
        JSONObject conditionJson = JSONObject.parseObject(condition);
        Long contractId = conditionJson.getLong("contractId");
        Long projectId = conditionJson.getLong("projectId");
        QueryParam queryParam = new QueryParam();
        LinkedHashMap<String, String> orderMap = new LinkedHashMap<>();
        orderMap.put("createTime", "desc");
        if (null != projectId) {
            queryParam.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
        }
        if (null != contractId) {
            queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        }
        queryParam.setOrderMap(orderMap);
        List<String> fuzzyFields = queryParam.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("projectName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("name");
        /** 已提交的数据 或者 审批通过的数据 */
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));
        List<PurchaseOrderEntity> purchaseOrderEntityList = service.queryList(queryParam, false);
        List<Map> resultMapList = new ArrayList<>();
        Map map = null;
        for (PurchaseOrderEntity entity : purchaseOrderEntityList) {
            QueryParam queryApplySubParam = new QueryParam();
            queryApplySubParam.getParams().put("pid", new Parameter(QueryParam.EQ, entity.getId()));
            List<PurchaseOrderDetailVO> purchaseOrderDetailVOList = service.querySubData(null, queryApplySubParam);
            if (CollectionUtils.isEmpty(purchaseOrderDetailVOList)) {
                continue;
            }
            map = new HashMap<>();
            map.put("id", entity.getId());
            map.put("extdata", null);
            map.put("isLeaf", true);
            map.put("selectable", true);
            map.put("key", entity.getId());
            map.put("name", entity.getName());
            map.put("code", entity.getBillCode());
            map.put("parentId", null);
            resultMapList.add(map);
        }
        return ResultAsTree.createTreeData(resultMapList);
    }

    /**
     * @description: 采购订单tree-- 子表数据
     *
     * @param pageNumber
     * @param pageSize
     * @param condition
     * @return {@link List< Map< String, Object>>}
     * @author songlx
     * @date: 2023/4/19
     */
    @RequestMapping(value = "/refPurchaseOrderDetailData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<PurchaseOrderDetailVO>> refPurchaseOrderDetailData(@RequestParam Integer pageNumber,
                                                                                   @RequestParam Integer pageSize,
                                                                                   String relyCondition,
                                                                                   @RequestParam(required = false) String condition,
                                                                                   String relyNode,
                                                                                   String searchText,
                                                                                   @RequestParam(required = false) String searchObject) {
        QueryParam queryParam = new QueryParam();
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.setSearchText(searchText);
        List<String> fuzzyFields = queryParam.getFuzzyFields();
        fuzzyFields.add("materialTypeName");
        fuzzyFields.add("materialName");
        fuzzyFields.add("model");
        fuzzyFields.add("unit");
        fuzzyFields.add("supplierName");
        Map<String, Parameter> param = queryParam.getParams();
        if (StringUtils.isEmpty(relyCondition)) {
            throw new BusinessException("请选择采购订单！");
        }

        String pid = relyCondition.split("=")[1];
        param.put("pid", new Parameter(QueryParam.EQ, pid));

        if (StringUtils.isNotBlank(searchObject)) {
            JSONObject search = JSONObject.parseObject(searchObject);
            for (String key : search.keySet()) {
                queryParam.getParams().put(key, new Parameter(QueryParam.LIKE, search.get(key).toString()));
            }
        }

        IPage<PurchaseOrderDetailEntity> pageData = new Page<>(queryParam.getPageIndex(), queryParam.getPageSize());
        List<PurchaseOrderDetailVO> list = service.querySubData(pageData, queryParam);

        IPage<PurchaseOrderDetailVO> voPage = null;
        if (pageData != null) {
            voPage = new Page<>();
            voPage.setCurrent(pageData.getCurrent());
            voPage.setPages(pageData.getPages());
            voPage.setTotal(pageData.getTotal());
            voPage.setSize(queryParam.getPageSize());
            voPage.setRecords(list);
        }
        return CommonResponse.success("查询数表参照成功", voPage);
    }


    /**
     * 导入模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping(value = "/download")
    @ResponseBody
    public void download(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "purchaseOrder-import.xlsx", "采购订单物料导入模板");
    }

    /**
     * excel导入
     *
     * @param request
     * @return
     */
    @RequestMapping(value = "/excelImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelImport(HttpServletRequest request, HttpServletResponse response) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }

        if (isFailed) {
            return CommonResponse.error("文件格式不合法");
        } else {
            List<List<String>> result = ExcelReader.readExcel(mf);
            List<PurchaseOrderDetailVO> successList = new ArrayList<>();
            List<PurchaseOrderDetailVO> errorList = new ArrayList<>();
            if (result != null && result.size() > 0) {
                if (result.get(0).size() != 9) {
                    throw new BusinessException("请按照导入模板导入数据");
                }
                for (int i = 0; i < result.size(); i++) {
                    List<String> datas = result.get(i);
                    PurchaseOrderDetailVO purchaseOrderDetailVO = new PurchaseOrderDetailVO();
                    if (StringUtils.isBlank(datas.get(0))) {
                        purchaseOrderDetailVO.setErrorMessage("物资分类名称为必填项");
                    } else {
                        purchaseOrderDetailVO.setMaterialTypeName(datas.get(0).trim());
                    }
                    if (StringUtils.isBlank(datas.get(1))) {
                        purchaseOrderDetailVO.setErrorMessage("物资名称为必填项");
                    } else if (datas.get(1).length() > 20) {
                        purchaseOrderDetailVO.setErrorMessage("物资名称填写长度为0~20字");
                    } else {
                        purchaseOrderDetailVO.setMaterialName(datas.get(1).trim());
                    }
                    if (StringUtils.isBlank(datas.get(2))) {
                        purchaseOrderDetailVO.setErrorMessage("规格型号为必填项");
                    } else if (datas.get(2).length() > 200) {
                        purchaseOrderDetailVO.setErrorMessage("规格型号填写长度为0~200字");
                    } else {
                        purchaseOrderDetailVO.setModel(datas.get(2).trim());
                    }
                    if (StringUtils.isNotEmpty(datas.get(3))) {
                        purchaseOrderDetailVO.setUnit(datas.get(3).trim());
                    }
                    purchaseOrderDetailVO.setSupplierName(datas.get(4));

                    if (StringUtils.isNotBlank(datas.get(5))) {
                        try {
                            purchaseOrderDetailVO.setOrderNum(new BigDecimal(datas.get(5)));
                            if (purchaseOrderDetailVO.getOrderNum().longValue() < 0) {
                                purchaseOrderDetailVO.setErrorMessage("订单数量必须大于0");
                            }
                        } catch (Exception e) {
                            purchaseOrderDetailVO.setErrorMessage("订单数量必须为数字");
                        }
                    }
                    if (org.apache.commons.lang3.StringUtils.isNotBlank(datas.get(6))) {
                        try {
                            purchaseOrderDetailVO.setPrice(new BigDecimal(datas.get(6)));
                            if (purchaseOrderDetailVO.getPrice().doubleValue() < 0) {
                                purchaseOrderDetailVO.setErrorMessage("单价必须大于0");
                            }
                        } catch (Exception e) {
                            purchaseOrderDetailVO.setErrorMessage("单价为必须为数字");
                        }
                    } else {
                        purchaseOrderDetailVO.setPrice(new BigDecimal(0));
                    }
                    purchaseOrderDetailVO.setTaxMny(MathUtil.safeMultiply(purchaseOrderDetailVO.getOrderNum(),purchaseOrderDetailVO.getPrice()));
                    if (StringUtils.isNotBlank(datas.get(7)) && datas.get(7).length() > 64) {
                        purchaseOrderDetailVO.setErrorMessage("物料编码填写长度为0~64字");
                    } else {
                        purchaseOrderDetailVO.setMaterialCode(datas.get(7));
                    }

                    if (StringUtils.isNotBlank(datas.get(8)) && datas.get(8).length() > 50) {
                        purchaseOrderDetailVO.setErrorMessage("备注填写长度为0~50字");
                    } else {
                        purchaseOrderDetailVO.setMemo(datas.get(8));
                    }


                    if (StringUtils.isNotEmpty(purchaseOrderDetailVO.getErrorMessage())) {
                        errorList.add(purchaseOrderDetailVO);
                    } else {
                        String onlykey = purchaseOrderDetailVO.getMaterialTypeName() + purchaseOrderDetailVO.getMaterialName() + purchaseOrderDetailVO.getUnit() + purchaseOrderDetailVO.getModel();
                        purchaseOrderDetailVO.setOnlyKey(onlykey);
                        successList.add(purchaseOrderDetailVO);
                    }
                }
            }
            if (CollectionUtils.isNotEmpty(successList)) {
                JSONObject match = archivesUtil.findMatch(successList, "materialTypeId", "materialId",
                        "materialTypeName", "materialName", "materialCode" ,"unit", "model", "errorMessage");
                successList = (List<PurchaseOrderDetailVO>) match.get("sourceList");
                List<PurchaseOrderDetailVO> errorList1 = (List<PurchaseOrderDetailVO>) match.get("errorList");
                errorList.addAll(errorList1);
            }
            JSONObject json = new JSONObject();
            json.put("successList", successList);
            json.put("errorList", errorList);
            return CommonResponse.success(json);
        }
    }

    /***
     * @description: 参数校验
     */
    @RequestMapping(value = "/checkParams", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<ParamsCheckVO> checkParams(@RequestBody PurchaseOrderVO vo) {
        ParamsCheckVO paramsCheckVO = service.checkParams(vo);
        return CommonResponse.success("参数校验成功！", paramsCheckVO);
    }


    /**
     * @description: 采购订单跳转新增收料入库or直入直出
     *
     * @param purchaseOrderIds
     * @param instoreType 1收料 2直入直出
     * @return {@link CommonResponse< PurchaseSettlementVO>}
     * @author songlx
     * @date: 2023/4/20
     */
    @GetMapping(value = "/getInstoreDataByPurchaseOrder")
    @ResponseBody
    public CommonResponse<InstoreVO> getInstoreDataByPurchaseOrder(@RequestParam String purchaseOrderIds, @RequestParam Integer instoreType) {
        if (!StringUtils.isNotBlank(purchaseOrderIds)) {
            logger.info("======com.ejianc.business.material.controller.PurchaseOrderController.getInstoreDataByPurchaseOrder[purchaseOrderIds:{}]", purchaseOrderIds);
            return CommonResponse.error("未获取到采购订单数据!");
        }

        UserContext userContext = sessionManager.getUserContext();

        String[] orderIds = purchaseOrderIds.split(",");
        List<Long> orderIdList = Arrays.stream(orderIds).map(s -> Long.parseLong(s.trim())).collect(Collectors.toList());

        PurchaseOrderEntity order = service.getById(orderIdList.get(0));

        InstoreVO vo = new InstoreVO();
        List<InstoreMaterialVO> instoreMaterialList = new ArrayList<>();
        Date curDate = new Date();
        Long contractId = order.getContractId();
        Long orgId = order.getOrgId();
        MaterialContractEntity contract = contractService.getById(contractId);
        if (contract != null) {
            vo.setTaxRate(ComputeUtil.toBigDecimal(contract.getTaxRate()));
            vo.setProjectCode(contract.getProjectCode());
            vo.setContractPurchaseMode(contract.getPurchaseMode());
        }
        StoreVO storeVO = storeService.queryNewestStoreByOrgId(orgId);
        if (storeVO != null) {
            vo.setStoreId(storeVO.getId());
            vo.setStoreName(storeVO.getName());
        }

        vo.setOrgId(order.getOrgId());
        vo.setOrgName(order.getOrgName());
        vo.setContractId(contractId);
        vo.setContractName(order.getContractName());
        vo.setProjectId(order.getProjectId());
        vo.setProjectName(order.getProjectName());
        vo.setProjectFlag("1");
        vo.setSupplierId(order.getSupplierId());
        vo.setSupplierName(order.getSupplierName());
        vo.setInstoreDate(curDate);
        vo.setInstoreType(instoreType);
        vo.setContractType("hasContract");
        vo.setEmployeeId(userContext.getEmployeeId());
        vo.setEmployeeName(userContext.getEmployeeName());
        vo.setInstoreMaterialList(instoreMaterialList);

        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("pid", new Parameter(QueryParam.IN, orderIdList));
        List<PurchaseOrderDetailVO> detailEntityList = service.querySubData(null, queryParam);

        if (CollectionUtils.isEmpty(detailEntityList)) {
            return CommonResponse.success("这些采购订单的物资已全部入库!", vo);
        }

        for (PurchaseOrderDetailVO detail : detailEntityList) {
            InstoreMaterialVO instoreMaterialVO = new InstoreMaterialVO();
            instoreMaterialVO.setMaterialCategoryId(detail.getMaterialTypeId());
            instoreMaterialVO.setMaterialCategoryName(detail.getMaterialTypeName());
            instoreMaterialVO.setMaterialId(detail.getMaterialId());
            instoreMaterialVO.setMaterialName(detail.getMaterialName());
            instoreMaterialVO.setMaterialCode(detail.getMaterialCode());
            instoreMaterialVO.setMaterialSpec(detail.getModel());
            instoreMaterialVO.setMaterialUnit(detail.getUnit());
            instoreMaterialVO.setMaterialSupplierName(detail.getSupplierName());
            instoreMaterialVO.setUnitPrice(detail.getPrice());
            instoreMaterialVO.setInstoreNumber(detail.getSurplusNum());
            instoreMaterialVO.setAmount(ComputeUtil.safeMultiply(detail.getSurplusNum(), detail.getPrice()));
            instoreMaterialVO.setSourceType("purchaseOrder");
            instoreMaterialVO.setSourceId(detail.getId());
            instoreMaterialVO.setSourceMainId(detail.getPid());
            instoreMaterialVO.setRowState("add");
            instoreMaterialVO.setId(detail.getId());
            instoreMaterialList.add(instoreMaterialVO);
        }
        return CommonResponse.success(vo);
    }

    /**
     * @description: 采购订单跳转新增结算单, 查询采购订单的入库单据数据带入
     *
     * @param purchaseOrderIds
     * @return {@link CommonResponse< PurchaseSettlementVO>}
     * @author songlx
     * @date: 2023/4/20
     */
    @GetMapping(value = "/getSettleDataByPurchaseOrder")
    @ResponseBody
    public CommonResponse<PurchaseSettlementVO> getSettleDataByPurchaseOrder(@RequestParam String purchaseOrderIds) {
        PurchaseSettlementVO vo = new PurchaseSettlementVO();

        if (!StringUtils.isNotBlank(purchaseOrderIds)) {
            logger.info("======com.ejianc.business.material.controller.PurchaseOrderController.getSettleDataByPurchaseOrder[purchaseOrderIds:{}]", purchaseOrderIds);
            return CommonResponse.error("未获取到采购订单数据!");
        }
        UserContext userContext = sessionManager.getUserContext();
        String[] orderIds = purchaseOrderIds.split(",");
        List<Long> orderIdList = Arrays.stream(orderIds).map(s -> Long.parseLong(s.trim())).collect(Collectors.toList());
        PurchaseOrderEntity order = service.getById(orderIdList.get(0));
        List<InstoreMaterialVO> instoreMaterialVOList = service.getInstoreMaterialByPurchaseOrder(orderIdList);

        List<PurchaseSettlementMaterialDetailVO> detailVOS = new ArrayList<>();
        /**
         id: item.id,
         rowState: 'add',
         sourceType: '2',
         sourceTypeName: bill.instoreDate + bill.instoreTypeName + bill.billCode,
         materialTypeId: item.materialCategoryId,
         materialTypeName: item.materialCategoryName,
         materialId: item.materialId,
         materialName: item.materialName,
         materialInfo: item.materialInfo ? item.materialInfo : item.materialName,
         model: item.materialSpec,
         unit: item.materialUnit,
         brandName: !isProBudget ? item.materialSupplierName : null,
         num: item.instoreNumber,
         price: item.unitPrice,
         money: item.amount,
         sourceMainId: bill.id,
         sourceSubId: item.id,
         */
        for (InstoreMaterialVO s : instoreMaterialVOList) {
            PurchaseSettlementMaterialDetailVO d = new PurchaseSettlementMaterialDetailVO();
            d.setId(s.getId());
            d.setRowState("add");
            //来源入库
            d.setSourceType("2");
            d.setSourceTypeName(DateUtil.format(s.getInstoreDate(), DateUtil.DATE) + MaterialStoreType.getStoreTypeNameByCode(s.getInstoreType()) + s.getBillCode());
            d.setMaterialTypeId(s.getMaterialCategoryId());
            d.setMaterialTypeName(s.getMaterialCategoryName());
            d.setMaterialId(s.getMaterialId());
            d.setMaterialName(s.getMaterialName());
            d.setMaterialCode(s.getMaterialCode());
            d.setMaterialInfo(Optional.ofNullable(s.getMaterialInfo()).orElse(s.getMaterialName()));
            d.setModel(s.getMaterialSpec());
            d.setUnit(s.getMaterialUnit());
            d.setBrandName(s.getMaterialSupplierName());
            d.setNum(s.getInstoreNumber());
            d.setPrice(s.getUnitPrice());
            d.setMoney(s.getAmount());
            d.setSourceMainId(s.getInstoreId());
            d.setSourceSubId(s.getId());
            detailVOS.add(d);
        }

        Long contractId = order.getContractId();
        Long orgId = order.getOrgId();
        MaterialContractEntity contract = contractService.getById(contractId);
        if (contract != null) {
            vo.setTaxRate(ComputeUtil.toBigDecimal(contract.getTaxRate()));
            vo.setProjectCode(contract.getProjectCode());
            vo.setPurchaseMethod(contract.getPurchaseModeName());
            vo.setContractAmount(contract.getAmountWithTax());
        }
        vo.setOrgId(order.getOrgId());
        vo.setOrgName(order.getOrgName());
        vo.setContractId(contractId);
        vo.setContractName(order.getContractName());
        vo.setProjectId(order.getProjectId());
        vo.setProjectName(order.getProjectName());
        vo.setSupplierId(order.getSupplierId());
        vo.setSupplierName(order.getSupplierName());

        vo.setBelongToProject("1");
        vo.setSettlementDate(new Date());
        //默认过程结算
        vo.setSettleType("0");
        vo.setEmployeeId(userContext.getEmployeeId());
        vo.setEmployeeName(userContext.getEmployeeName());
        vo.setMaterialDetails(detailVOS);

        return CommonResponse.success(vo);
    }
}
