package com.ejianc.business.storeCheck.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.storeCheck.bean.StoreCheckEntity;
import com.ejianc.business.storeCheck.bean.StoreCheckSubEntity;
import com.ejianc.business.storeCheck.service.IStoreCheckService;
import com.ejianc.business.storeCheck.vo.StoreCheckExportVO;
import com.ejianc.business.storeCheck.vo.StoreCheckSubExportVO;
import com.ejianc.business.storeCheck.vo.StoreCheckSubVO;
import com.ejianc.business.storeCheck.vo.StoreCheckVO;
import com.ejianc.business.utils.ComputeUtil;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IMaterialApi;
import com.ejianc.foundation.share.vo.MaterialCategoryVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.core.util.ImportTemplate;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2021/3/12 16:15
 */
@RestController
@RequestMapping(value = "/storeCheck/")
public class StoreCheckController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IStoreCheckService storeCheckService;

    private final String BILL_CODE = "STORE_CHECK";

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private SessionManager sessionManager;
    @Autowired
    private IMaterialApi materialApi;
    @Autowired
    private IOrgApi orgApi;

    /**
     * 仓库盘点-保存、更新
     *
     * @param storeCheckVO
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<StoreCheckVO> saveOrUpdate(@RequestBody StoreCheckVO storeCheckVO) {
        if(StringUtils.isNotBlank(storeCheckVO.getBillCode())) {
            //编码重复校验
            StoreCheckVO voByCode = storeCheckService.queryByCode(storeCheckVO.getBillCode());
            if(null != voByCode && (null == storeCheckVO.getId() || !voByCode.getId().equals(storeCheckVO.getId()))) {
                return CommonResponse.error("保存失败，编码重复！");
            }
        } else {
            CommonResponse<String> codeResp = billCodeApi.getCodeBatchByRuleCode(BILL_CODE, InvocationInfoProxy.getTenantid());
            if(!codeResp.isSuccess()) {
                return CommonResponse.error("保存失败，获取自动编码失败！");
            }
            storeCheckVO.setBillCode(codeResp.getData());
        }

        //校验仓库下是否有对应未生效的盘点单
        StoreCheckVO voByStoreId = storeCheckService.queryInvalidBillByStoreId(storeCheckVO.getStoreId());
        if(null != voByStoreId && (null == storeCheckVO.getId() || !voByStoreId.getId().equals(storeCheckVO.getId()))) {
            return CommonResponse.error("保存失败，该仓库下已存在未生效的盘点单据！");
        }

        //检查是否存在晚于当前盘点单盘点单日期的已生效盘点单
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        StoreCheckVO latestStoreCheck = storeCheckService.getLatestStoreCheck(storeCheckVO.getStoreId());

        if(null != latestStoreCheck) {
            try {
                if(!sdf.parse(sdf.format(storeCheckVO.getCheckDate())).after(sdf.parse(sdf.format(latestStoreCheck.getCheckDate())))) {
                    return CommonResponse.error("保存失败，该仓库下已存在已生效的且晚于当前盘点日期的盘点单据！");
                }
            } catch (Exception e) {
                logger.error("校验盘点单据[{}]盘点日期异常，", JSONObject.toJSONString(storeCheckVO), e);
                return CommonResponse.error("保存失败，校验盘点单据盘点日期失败！");
            }
        }


        StoreCheckEntity saveEntity = null;
        if(CollectionUtils.isNotEmpty(storeCheckVO.getMaterialSubList())) {
            for(StoreCheckSubVO subVo : storeCheckVO.getMaterialSubList()) {
                subVo.setInventory(subVo.getCheckNum() - subVo.getAccNum());
            }
        }

        UserContext userContext = sessionManager.getUserContext();
        if(null == storeCheckVO.getId()) {
            saveEntity = BeanMapper.map(storeCheckVO, StoreCheckEntity.class);
            //设置为自由态
            saveEntity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
            //设置编制人
            saveEntity.setCreateUserName(userContext.getUserName());
        } else {
            saveEntity = storeCheckService.getById(storeCheckVO.getId());
            saveEntity.setModifyUserName(userContext.getUserName());
            saveEntity.setCheckDate(storeCheckVO.getCheckDate());
            saveEntity.setMaterialSubList(BeanMapper.mapList(storeCheckVO.getMaterialSubList(), StoreCheckSubEntity.class));
        }

        storeCheckVO = storeCheckService.saveCheckStore(saveEntity);

        return CommonResponse.success("保存成功!", storeCheckVO);
    }

    /**
     * 分页查询物资总计划列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "pageList")
    public CommonResponse<IPage<StoreCheckVO>> pageList(@RequestBody QueryParam queryParam) {
        queryParam.getFuzzyFields().add("storeName");
        queryParam.getFuzzyFields().add("billCode");
        queryParam.getFuzzyFields().add("orgName");
        queryParam.getFuzzyFields().add("createUserName");
        //查询当前上下文本下的物资总计划
        queryParam.getComplexParams().add(getPageQueryParam(null));

        //按创建时间排序
        queryParam.getOrderMap().put("createTime", QueryParam.DESC);
        IPage<StoreCheckEntity> pageData = storeCheckService.queryPage(queryParam);

        IPage<StoreCheckVO> resp = new Page<>();
        resp.setSize(pageData.getSize());
        resp.setRecords(BeanMapper.mapList(pageData.getRecords(), StoreCheckVO.class));
        resp.setCurrent(pageData.getCurrent());
        resp.setTotal(pageData.getTotal());
        resp.setPages(pageData.getPages());

        return CommonResponse.success("分页盘点列表查询成功！", resp);
    }

    public ComplexParam getPageQueryParam(Long orgId) {
        ComplexParam c1 = new ComplexParam();
        c1.setLogic(ComplexParam.AND);

        if(null == orgId) {
            orgId = InvocationInfoProxy.getOrgId();
        }
        CommonResponse<List<OrgVO>> childOrgResp = null;

        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)){
            childOrgResp  = orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong)
                            .collect(Collectors.toList()));
        }else{
            childOrgResp= orgApi.findChildrenByParentId(orgId);
        }
        List<Long> childIds = new ArrayList<>();
        //总计划直接跟项目关联，只查询本心的所有项目部即可
        childIds.addAll(childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));

        ComplexParam c2 = new ComplexParam();
        c2.setLogic(ComplexParam.OR);
        c2.getParams().put("org_id", new Parameter(QueryParam.IN, childIds));
        c1.getComplexParams().add(c2);

        return c1;
    }

    @GetMapping("queryDetail")
    public CommonResponse<StoreCheckVO> queryDetail(@RequestParam(value = "id") Long id) {
        StoreCheckVO vo = null;
        StoreCheckEntity dbEntity = storeCheckService.selectById(id);
        if(null != dbEntity) {
            vo = BeanMapper.map(dbEntity, StoreCheckVO.class);
        }

        return CommonResponse.success("查询详情成功！", vo);
    }

    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<StoreCheckVO> vos) {
        if(CollectionUtils.isNotEmpty(vos)) {
            List<Long> ids = vos.stream().map(StoreCheckVO::getId).collect(Collectors.toList());

            storeCheckService.removeByIds(ids, false);
        }
        return CommonResponse.success("删除成功！");
    }

    /**
     　* @Description: 仓库盘点列表导出
     　* @param [queryParam, response]
     　* @return void
     　* @throws
     　* @author CJ
     　* @date 2020/6/8 10:55
     　*/
    @PostMapping("excelExport")
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        JSONObject resp = new JSONObject();
        queryParam.getFuzzyFields().add("storeName");
        queryParam.getFuzzyFields().add("billCode");
        queryParam.getFuzzyFields().add("orgName");
        queryParam.setPageIndex(1);
        queryParam.setPageSize(10000);
        queryParam.getComplexParams().add(getPageQueryParam(null));

        //按创建时间排序
        queryParam.getOrderMap().put("createTime", QueryParam.DESC);

        IPage<StoreCheckEntity> pageData = storeCheckService.queryPage(queryParam, false);
        Map<String, Object> beans = new HashMap<String, Object>();
        List<StoreCheckExportVO> list= new ArrayList<>();
        List<StoreCheckEntity> data = pageData.getRecords();
        if(CollectionUtils.isNotEmpty(data)) {
            data.forEach(storeCheck -> {
                StoreCheckExportVO vo = BeanMapper.map(storeCheck, StoreCheckExportVO.class);
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(storeCheck.getBillState()).getDescription());
                list.add(vo);
            });
        }
        beans.put("records", list);
        ExcelExport.getInstance().export("storeCheck-export.xlsx", beans, response);
    }

    /**
     * 仓库盘点物资列表导出
     *
     * @param response
     */
    @PostMapping("excelExportMaterialList")
    public void excelExportMaterialList(@RequestBody StoreCheckVO storeCheckVO, HttpServletResponse response) {
        List<StoreCheckSubExportVO> list = new ArrayList<>();
        Map<String, Object> beans = new HashMap<>();
        Long storeCheckVOId = storeCheckVO.getId();
        if (storeCheckVOId != null) {
            StoreCheckEntity entity = storeCheckService.selectById(storeCheckVOId);
            DecimalFormat df = new DecimalFormat("#,##0.00");
            List<StoreCheckSubEntity> data = entity.getMaterialSubList();
            if (CollectionUtils.isNotEmpty(data)) {
                data.forEach(storeCheckSub -> {
                    StoreCheckSubExportVO vo = BeanMapper.map(storeCheckSub, StoreCheckSubExportVO.class);
                    vo.setAccAmountStr(df.format(vo.getAccAmount()));
                    list.add(vo);
                });
            }
        }
        beans.put("records", list);
        ExcelExport.getInstance().export("storeCheckMaterialList-export.xlsx", beans, response);
    }


    /**
     * 导入模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping(value = "/download")
    @ResponseBody
    public void download(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "storeCheckMaterialList-import.xlsx", "物料导入模板");
    }

    /**
     * excel导入
     *
     * @param request
     *
     * @return
     */
    @RequestMapping(value = "/excelImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelImport(HttpServletRequest request, HttpServletResponse response) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }

        if (isFailed) {
            return CommonResponse.error("文件格式不合法");
        } else {
            List<List<String>> result = ExcelReader.readExcel(mf);
            List<StoreCheckSubVO> successList = new ArrayList<>();
            List<StoreCheckSubVO> errorList = new ArrayList<>();
            if (result != null && result.size() > 0) {
                if (result.get(0).size() != 10) {
                    throw new BusinessException("请按照导入模板导入数据");
                }
                for (int i = 0; i < result.size(); i++) {
                    List<String> datas = result.get(i);
                    StoreCheckSubVO storeCheckSubVO = new StoreCheckSubVO();
                    if (StringUtils.isBlank(datas.get(0))) {
                        storeCheckSubVO.setErrorMessage("物料分类为必填项");
                    } else {
                        String materialCategoryName = datas.get(0).toString().trim();
                        storeCheckSubVO.setMaterialCategoryName(materialCategoryName);
                    }
                    if (StringUtils.isBlank(datas.get(1))) {
                        storeCheckSubVO.setErrorMessage("物资名称为必填项");
                    } else if (datas.get(1).length() > 30) {
                        storeCheckSubVO.setErrorMessage("物资名称填写长度为0~30字");
                    } else {
                        storeCheckSubVO.setMaterialName(datas.get(1).trim());
                    }
                    if (StringUtils.isBlank(datas.get(2))) {
                        storeCheckSubVO.setErrorMessage("规格型号为必填项");
                    } else if (datas.get(2).length() > 200) {
                        storeCheckSubVO.setErrorMessage("规格型号填写长度为0~200字");
                    } else {
                        storeCheckSubVO.setSpec(datas.get(2).trim());
                    }
                    if(StringUtils.isBlank(datas.get(3))){
                        storeCheckSubVO.setErrorMessage("品牌为必填项");
                    }else {
                        storeCheckSubVO.setBrandName(datas.get(3).trim());

                    }
                    if (!Objects.isNull(datas.get(4))) {
                        storeCheckSubVO.setUnit(datas.get(4).trim());
                    }

                    if (StringUtils.isNotBlank(datas.get(5)) && datas.get(5).length() > 64) {
                        storeCheckSubVO.setErrorMessage("物料编码填写长度为0~64字");

                    } else {
                        storeCheckSubVO.setMaterialCode(datas.get(5));
                    }








                    if (StringUtils.isNotBlank(datas.get(6))) {
                        try {
                            storeCheckSubVO.setAccNum(Double.parseDouble(datas.get(6)));
                        } catch (Exception e) {
                            storeCheckSubVO.setErrorMessage("账面数量为必须为数字");
                        }
                    } else {
                        storeCheckSubVO.setAccNum(Double.parseDouble("0"));
                    }
                    if (StringUtils.isNotBlank(datas.get(7))) {
                        try {
                            storeCheckSubVO.setCheckNum(Double.parseDouble(datas.get(7)));
                        } catch (Exception e) {
                            storeCheckSubVO.setErrorMessage("盘点数量为必须为数字");
                        }
                    } else {
                        storeCheckSubVO.setCheckNum(Double.parseDouble("0"));
                    }

                    if (StringUtils.isNotBlank(datas.get(8))) {
                        try {
                            storeCheckSubVO.setCheckNum(Double.parseDouble(datas.get(8)));
                        } catch (Exception e) {
                            storeCheckSubVO.setErrorMessage("盘点亏盈为必须为数字");
                        }
                    } else {
                        storeCheckSubVO.setInventory(Double.parseDouble("0"));
                    }

                    if (storeCheckSubVO.getAccNum() != null && storeCheckSubVO.getCheckNum() != null) {
                        double inventory = BigDecimal.valueOf(storeCheckSubVO.getCheckNum()).subtract(BigDecimal.valueOf(storeCheckSubVO.getAccNum())).doubleValue();
                        storeCheckSubVO.setInventory(inventory);
                    }
                    if (StringUtils.isNotBlank(datas.get(9)) && datas.get(8).length() > 50) {
                        storeCheckSubVO.setErrorMessage("备注填写长度为0~50字");
                    } else {
                        storeCheckSubVO.setRemark(datas.get(9));
                    }

                    if (StringUtils.isBlank(storeCheckSubVO.getErrorMessage())) {
                        successList.add(storeCheckSubVO);
                        storeCheckSubVO.setId((long) i);
                    } else {
                        errorList.add(storeCheckSubVO);
                    }
                }
            }
            JSONObject json = new JSONObject();
            json.put("successList", successList);
            json.put("errorList", errorList);
            return CommonResponse.success(json);
        }
    }


}
