package com.ejianc.business.material.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.material.service.IReportService;
import com.ejianc.business.material.vo.BuyPriceDetailVO;
import com.ejianc.business.material.vo.BuyPriceVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IMaterialApi;
import com.ejianc.foundation.share.vo.MaterialCategoryVO;
import com.ejianc.foundation.share.vo.SupplierVO;
import com.ejianc.foundation.support.api.ISupplierApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.CollectionUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.Requests;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.elasticsearch.index.query.TermQueryBuilder;
import org.elasticsearch.index.reindex.BulkByScrollResponse;
import org.elasticsearch.index.reindex.DeleteByQueryRequest;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import static org.elasticsearch.common.xcontent.XContentFactory.jsonBuilder;


/**
 * @author songlx
 * @version 1.0
 * @description: 物资统筹  材料成本统计、采购价格分析
 * @date 2021-04-07
 */
@Controller
@RequestMapping("/report/")
public class ReportController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());


    @Autowired
    IReportService reportService;


    @Autowired
    IProjectApi projectApi;


    @Autowired
    ISupplierApi supplierApi;


    @Autowired
    IMaterialApi materialApi;

    @Autowired
    IOrgApi orgApi;

    private final static Integer BATCH_SIZE = 1000;

    private final static Integer QUERY_TIMEOUT = 60;

    private final static String INDEX_NAME_DETAIL = "ejc_material_report_buy_price_detail";

    private final static String INDEX_NAME = "ejc_material_report_buy_price";

    private final static String INDEX_NAME_MATERIAL_CATEGORY = "ejc_material_report_material_category";

    /***
     {
     "ejc_material_report_buy_price_detail”: {
     "properties": {
     " billCode": {
     "type": "text"
     },
     "contractName": {
     "type": "text"
     },
     "contractId": {￼
     "type": "long"
     },
     "supplierName": {
     "type": "text"
     },
     "supplierId": {
     "type": "long"
     },
     "orgName": {
     "type": "text"
     },
     "orgId": {￼
     "type": "long"
     },
     "settlementDate": {
     "type": "date",
     "format": "yyyy-MM-dd"
     },
     "contractPrice": {￼
     "type": "long"
     },
     "pid": {￼
     "type": "long"
     },
     "materialTypeName": {￼
     "type": "text"
     },
     "materialTypeId": {￼
     "type": "long"
     },
     "materialName": {￼
     "type": "text"
     },
     "materialId": {￼
     "type": "long"
     },
     "model": {￼
     "type": "text"
     },
     "unit": {￼
     "type": "text"
     },
     "num": {￼
     "type": "long"
     },
     "price": {￼
     "type": "long"
     },
     "money": {￼
     "type": "long"
     },
     "taxRate": {￼
     "type": "long"
     },
     "province": {
     "type": "text",
     },
     "city": {
     "type": "text",
     },
     "region": {
     "type": "text",
     }
     }

     }
     }

     */

    @Autowired(required = false)
    private RestHighLevelClient client;


    @RequestMapping(path = "/inites", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> initEs() {
        try {
            XContentBuilder mapping = jsonBuilder().startObject().startObject(INDEX_NAME_DETAIL).startObject("properties");
            mapping.startObject("billCode").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("contractName").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("contractId").field("type", "text").endObject();
            mapping.startObject("supplierName").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("supplierId").field("type", "text").endObject();
            mapping.startObject("orgName").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("orgId").field("type", "text").endObject();
            mapping.startObject("settlementDate").field("type", "date").field("format", "yyyy-MM-dd").endObject();
            mapping.startObject("contractPrice").field("type", "long").endObject();
            mapping.startObject("pId").field("type", "text").endObject();
            mapping.startObject("materialTypeName").field("type", "text").endObject();
            mapping.startObject("materialTypeId").field("type", "text").endObject();
            mapping.startObject("materialName").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("materialId").field("type", "text").endObject();
            mapping.startObject("model").field("type", "text").endObject();
            mapping.startObject("unit").field("type", "text").endObject();
            mapping.startObject("num").field("type", "long").endObject();
            mapping.startObject("price").field("type", "long").endObject();
            mapping.startObject("money").field("type", "long").endObject();
            mapping.startObject("taxRate").field("type", "long").endObject();
            mapping.startObject("province").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("city").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("region").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("projectId").field("type", "text").endObject();
            mapping.startObject("area").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("tenantId").field("type", "text").endObject();

            mapping.endObject().endObject().endObject();
            PutMappingRequest mappingRequest = Requests.putMappingRequest(INDEX_NAME_DETAIL).type(INDEX_NAME_DETAIL).source(mapping);
            client.indices().putMapping(mappingRequest, RequestOptions.DEFAULT);
            return CommonResponse.success();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return CommonResponse.error("初始化失败");
    }


    @RequestMapping(path = "/init", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> init() {
        try {
            XContentBuilder mapping = jsonBuilder().startObject().startObject(INDEX_NAME).startObject("properties");
            mapping.startObject("tenantId").field("type", "text").endObject();
            mapping.startObject("materialTypeName").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("materialTypeId").field("type", "text").endObject();
            mapping.startObject("materialCode").field("type", "text").endObject();
            mapping.startObject("materialName").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("materialId").field("type", "text").endObject();
            mapping.startObject("model").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("unit").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("brandName").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("totalNum").field("type", "long").endObject();
            mapping.startObject("totalMoney").field("type", "long").endObject();
            mapping.startObject("minContractPrice").field("type", "long").endObject();
            mapping.startObject("maxContractPrice").field("type", "long").endObject();
            mapping.startObject("avgPrice").field("type", "long").endObject();
            mapping.startObject("minPriceArea").field("type", "long").endObject();
            mapping.startObject("maxPriceArea").field("type", "long").endObject();
            mapping.startObject("buyPrice").field("type", "long").endObject();
            mapping.startObject("settlementDate").field("type", "date").field("format", "yyyy-MM-dd").endObject();

            mapping.endObject().endObject().endObject();
            PutMappingRequest mappingRequest = Requests.putMappingRequest(INDEX_NAME).type(INDEX_NAME).source(mapping);
            client.indices().putMapping(mappingRequest, RequestOptions.DEFAULT);
            return CommonResponse.success();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return CommonResponse.error("初始化失败");
    }


    @RequestMapping(path = "/initCat", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> initCat() {
        try {
            XContentBuilder mapping = jsonBuilder().startObject().startObject(INDEX_NAME_MATERIAL_CATEGORY).startObject("properties");
            mapping.startObject("tenantId").field("type", "text").endObject();
            mapping.startObject("id").field("type", "text").endObject();
            mapping.startObject("parentId").field("type", "text").endObject();
            mapping.startObject("code").field("type", "text").field("fielddata", true).endObject();
            mapping.startObject("name").field("type", "text").field("analyzer", "ik_max_word").field("search_analyzer", "ik_smart").endObject();
            mapping.startObject("innerCode").field("type", "text").endObject();
            mapping.startObject("enabled").field("type", "text").endObject();
            mapping.startObject("isLeaf").field("type", "text").endObject();


            mapping.endObject().endObject().endObject();
            PutMappingRequest mappingRequest = Requests.putMappingRequest(INDEX_NAME_MATERIAL_CATEGORY).type(INDEX_NAME_MATERIAL_CATEGORY).source(mapping);
            client.indices().putMapping(mappingRequest, RequestOptions.DEFAULT);
            return CommonResponse.success();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return CommonResponse.error("初始化失败");
    }


    /**
     * @description: 判断当前用户是否有权限查看结算单
     * @return: com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     * @author songlx
     * @date: 2021-04-15
     */
    @RequestMapping(path = "/isViewSettle", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> isViewSettle(Long orgId) {
        if (null != orgId) {
            CommonResponse<List<OrgVO>> projectOrgsByUserId = orgApi.findProjectOrgsByUserId(InvocationInfoProxy.getUserid());
            if (null != projectOrgsByUserId && CollectionUtil.isNotEmpty(projectOrgsByUserId.getData())) {
                List<OrgVO> orgVOS = projectOrgsByUserId.getData();
                if (CollectionUtil.isNotEmpty(orgVOS)) {
                    for (OrgVO vo : orgVOS) {
                        if (orgId.equals(vo.getId()))
                            return CommonResponse.success("该用户有权限查看次结算单!");
                    }
                }

            }
        }
        return CommonResponse.error("该用户没有此结算单所属组织权限,无法查看!");
    }


    @RequestMapping(value = "/queryCategory", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<MaterialCategoryVO>> queryCategory(Long id, Long zid, Long tid) {
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        if (id != null) {
            boolQuery.must(QueryBuilders.termQuery("parentId", id));
        }
        if (zid != null) {
            boolQuery.must(QueryBuilders.termQuery("id", zid));
        }
        if (tid != null) {
            boolQuery.must(QueryBuilders.termQuery("tenantId", tid.toString()));
        }


        SearchRequest searchRequest = new SearchRequest(INDEX_NAME_MATERIAL_CATEGORY);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.query(boolQuery);
        sourceBuilder.sort("code.keyword", SortOrder.ASC);
        sourceBuilder.size(500);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        List<MaterialCategoryVO> list = new ArrayList<>();
        SearchResponse response = null;
        try {
            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, MaterialCategoryVO.class));
            }

        } catch (IOException e) {
            try {
                response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = response.getHits();
                for (SearchHit hit : hits) {
                    String sourceAsString = hit.getSourceAsString();
                    list.add(JSON.parseObject(sourceAsString, MaterialCategoryVO.class));
                }
            } catch (IOException e1) {
                logger.error(e1.getMessage());
            }

        }

        return CommonResponse.success("查询列表数据成功！", list);
    }


    /**
     * @param id 物料分类父id
     * @description: 加载此物料分类的直接下级
     * @return: com.ejianc.framework.core.response.CommonResponse<java.util.List < com.ejianc.foundation.share.vo.MaterialCategoryVO>>
     * @author songlx
     * @date: 2021-04-14
     */
    @RequestMapping(value = "/lazyMaterialCategory", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<MaterialCategoryVO>> lazyMaterialCategory(Long id) {
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        if (id != null) {
            boolQuery.must(QueryBuilders.termQuery("parentId", id));
        } else {
            boolQuery.mustNot(QueryBuilders.existsQuery("parentId"));
        }

        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));


        SearchRequest searchRequest = new SearchRequest(INDEX_NAME_MATERIAL_CATEGORY);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.query(boolQuery);
        sourceBuilder.sort("code.keyword", SortOrder.ASC);
        sourceBuilder.size(500);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        List<MaterialCategoryVO> list = new ArrayList<>();
        SearchResponse response = null;
        try {
            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, MaterialCategoryVO.class));
            }

        } catch (IOException e) {
            try {
                response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = response.getHits();
                for (SearchHit hit : hits) {
                    String sourceAsString = hit.getSourceAsString();
                    list.add(JSON.parseObject(sourceAsString, MaterialCategoryVO.class));
                }
            } catch (IOException e1) {
                logger.error(e1.getMessage());
            }

        }

        return CommonResponse.success("查询列表数据成功！", list);
    }


    private List<MaterialCategoryVO> queryAllMaterialCategoryChildren(String materialTypeId) {

        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("innerCode", materialTypeId));

        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));

        SearchRequest searchRequest = new SearchRequest(INDEX_NAME_MATERIAL_CATEGORY);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.query(boolQuery);
        sourceBuilder.sort("code.keyword", SortOrder.ASC);
        sourceBuilder.size(500);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        List<MaterialCategoryVO> list = new ArrayList<>();
        SearchResponse response = null;
        try {
            response = client.search(searchRequest, RequestOptions.DEFAULT);
        } catch (IOException e) {
            e.printStackTrace();
        }
        SearchHits hits = response.getHits();
        for (SearchHit hit : hits) {
            String sourceAsString = hit.getSourceAsString();
            list.add(JSON.parseObject(sourceAsString, MaterialCategoryVO.class));
        }
        return list;
    }


    /**
     * @param param
     * @description: 采购价格分析列表
     * @return: com.ejianc.framework.core.response.CommonResponse<com.baomidou.mybatisplus.core.metadata.IPage < com.ejianc.business.material.vo.MaterialCostVO>>
     * @author songlx
     * @date: 2021-04-08
     */
    @RequestMapping(value = "/queryBuyPrice", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<BuyPriceVO>> queryBuyPrice(@RequestBody QueryParam param) {
        //物料分类采用查询所有子集
        String materialTypeId = getParamValue(param.getParams(), "materialTypeId");
        List<MaterialCategoryVO> categoryVOS = this.queryAllMaterialCategoryChildren(materialTypeId);
        if (categoryVOS.size() > 0) {
            List<String> categoryIds = new ArrayList<>();
            for (MaterialCategoryVO vo : categoryVOS) {
                categoryIds.add(String.valueOf(vo.getId()));
            }
            param.getParams().put("materialTypeIds", new Parameter(QueryParam.IN, categoryIds));
            param.getParams().remove("materialTypeId");
        }

        Page<BuyPriceVO> page = new Page<>((long) param.getPageIndex(), (long) param.getPageSize());
        List<BuyPriceVO> list = reportService.queryBuyPriceList(page, param);
        IPage<BuyPriceVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(list);
        return CommonResponse.success("查询列表数据成功！", pageData);
    }


    public List<BuyPriceDetailVO> getBuyPriceDetail(Map<String, Parameter> params, List<String> materialIds) {
        int pageNumber = 1;
        int pageSize = 10000;
        IPage<BuyPriceDetailVO> page = new Page<>();
        page.setCurrent(pageNumber);
        page.setSize(pageSize);
        SearchRequest searchRequest = new SearchRequest(INDEX_NAME_DETAIL);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.from(pageNumber <= 0 ? 0 : (pageNumber - 1) * pageSize);

        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));

        boolQuery.must(QueryBuilders.termsQuery("materialId", materialIds));
        String startDate = getParamValue(params, "startDate");
        String endDate = getParamValue(params, "endDate");
        if (startDate != null || endDate != null) {
            RangeQueryBuilder settlementDate = QueryBuilders.rangeQuery("settlementDate");
            if (startDate != null) {
                params.remove("startDate");
                settlementDate.gte(startDate +" 00:00:00");
            }
            if (endDate != null) {
                params.remove("endDate");
                settlementDate.lte(endDate+" 00:00:00");
            }
            boolQuery.must(settlementDate);
        }

        sourceBuilder.size(pageSize);
        sourceBuilder.query(boolQuery);

        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        List<BuyPriceDetailVO> list = new ArrayList<>();
        SearchResponse response = null;
        try {
            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, BuyPriceDetailVO.class));
            }
            page.setRecords(list);
            page.setTotal(hits.getTotalHits().value);
        } catch (IOException e) {
            e.printStackTrace();
        }

        return list;
    }

    /**
     * @param param
     * @description: 采购价格分析列表
     * @return: com.ejianc.framework.core.response.CommonResponse<com.baomidou.mybatisplus.core.metadata.IPage < com.ejianc.business.material.vo.MaterialCostVO>>
     * 查询条件
     * materialTypeId
     * startDate
     * endDate
     * materialTypeName
     * materialName
     * model
     * unit
     * @author songlx
     * @date: 2021-04-08
     */
    @RequestMapping(value = "/queryBuyPriceList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<BuyPriceVO>> queryBuyPriceList(@RequestBody QueryParam param) {
        boolean flag = false;
        int pageNumber = param.getPageIndex();
        int pageSize = param.getPageSize();
        IPage<BuyPriceVO> page = new Page<>();
        page.setCurrent(pageNumber);
        page.setSize(pageSize);
        SearchRequest searchRequest = new SearchRequest(INDEX_NAME);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.from(pageNumber <= 0 ? 0 : (pageNumber - 1) * pageSize);

        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));
        Map<String, Parameter> params = param.getParams();

        Map<String, Parameter> detailParam = new HashMap<>();
        if (params != null && params.size() > 0) {

            //物料分类采用查询所有子集
            String materialTypeId = getParamValue(params, "materialTypeId");
            List<MaterialCategoryVO> categoryVOS = this.queryAllMaterialCategoryChildren(materialTypeId);
            if (categoryVOS.size() > 0) {
                List<String> categoryIds = new ArrayList<>();
                for (MaterialCategoryVO vo : categoryVOS) {
                    categoryIds.add(String.valueOf(vo.getId()));
                }
                boolQuery.must(QueryBuilders.termsQuery("materialTypeId", categoryIds));
                params.remove("materialTypeId");
            }


            String startDate = getParamValue(params, "startDate");
            String endDate = getParamValue(params, "endDate");
            if (startDate != null || endDate != null) {
                flag = true;
                //RangeQueryBuilder settlementDate = QueryBuilders.rangeQuery("settlementDate");
                if (startDate != null) {
                    detailParam.put("startDate", new Parameter(QueryParam.GT, startDate));
                    params.remove("startDate");
                    //settlementDate.gte(startDate);
                }
                if (endDate != null) {
                    detailParam.put("endDate", new Parameter(QueryParam.LE, endDate));
                    params.remove("endDate");
                    //settlementDate.lte(endDate);
                }
                //boolQuery.must(settlementDate);
            }


            for (Map.Entry<String, Parameter> p : params.entrySet()) {
                String key = p.getKey();
                String value = getParamValue(params, key);
                if (StringUtils.isEmpty(value)) continue;
                boolQuery.must(QueryBuilders.matchPhraseQuery(key, value));
            }
        }

        sourceBuilder.size(pageSize);
        sourceBuilder.query(boolQuery);
        LinkedHashMap<String, String> orderMap = param.getOrderMap();
        if (orderMap.size() > 0) {
            for (Map.Entry<String, String> entry : orderMap.entrySet()) {
                if (SortOrder.ASC.toString().equalsIgnoreCase(entry.getValue())) {
                    sourceBuilder.sort(entry.getKey(), SortOrder.ASC);
                } else {
                    sourceBuilder.sort(entry.getKey(), SortOrder.DESC);
                }
            }
        }

        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        List<BuyPriceVO> list = new ArrayList<>();
        SearchResponse response = null;
        try {
            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, BuyPriceVO.class));
            }
            if (flag && CollectionUtil.isNotEmpty(list)) {
                List<String> materialIds = new ArrayList<>();
                list.stream().forEach(item ->
                        materialIds.add(String.valueOf(item.getMaterialId()))
                );
                List<BuyPriceDetailVO> buyPriceDetail = this.getBuyPriceDetail(detailParam, materialIds);
                Map<Long, BuyPriceVO> buyPriceMap = new HashMap<>();
                if (CollectionUtil.isNotEmpty(buyPriceDetail)) {
                    Map<Long, List<BuyPriceDetailVO>> buyPriceDetailMap = buyPriceDetail.stream().collect(Collectors.groupingBy(BuyPriceDetailVO::getMaterialId));
                    for (Map.Entry<Long, List<BuyPriceDetailVO>> m : buyPriceDetailMap.entrySet()) {
                        List<BuyPriceDetailVO> value = m.getValue();
                        if (value == null || value.size() == 0) continue;
                        BuyPriceVO buyPriceVO = new BuyPriceVO();
                        BigDecimal sumNum = BigDecimal.ZERO;
                        BigDecimal sumMoney = BigDecimal.ZERO;
                        List<BigDecimal> conPriceList = new ArrayList<>();
                        List<BigDecimal> priceList = new ArrayList<>();
                        for (BuyPriceDetailVO vo : value) {
                            sumNum = MathUtil.safeAdd(sumNum, vo.getNum());
                            sumMoney = MathUtil.safeAdd(sumMoney, vo.getMoney());
                            if (vo.getContractPrice() != null) conPriceList.add(vo.getContractPrice());
                            if (vo.getPrice() != null) priceList.add(vo.getPrice());
                        }
                        Collections.sort(conPriceList);
                        Collections.sort(priceList);
                        if (conPriceList.size() > 0) {
                            buyPriceVO.setMinContractPrice(conPriceList.get(0));
                            buyPriceVO.setMaxContractPrice(conPriceList.get(conPriceList.size() - 1));
                        } else {
                            buyPriceVO.setMinContractPrice(null);
                            buyPriceVO.setMaxContractPrice(null);

                        }
                        if (priceList.size() > 0) {
                            buyPriceVO.setMinPriceArea(priceList.get(0));
                            buyPriceVO.setMaxPriceArea(priceList.get(priceList.size() - 1));
                        } else {
                            buyPriceVO.setMinPriceArea(null);
                            buyPriceVO.setMaxPriceArea(null);
                        }
                        buyPriceVO.setTotalNum(sumNum);
                        buyPriceVO.setTotalMoney(sumMoney);
                        buyPriceVO.setAvgPrice(MathUtil.safeDiv(sumMoney, sumNum));
                        buyPriceMap.put(m.getKey(), buyPriceVO);

                    }
                }
                for (BuyPriceVO b : list) {
                    Long materialId = b.getMaterialId();
                    BuyPriceVO p = buyPriceMap.get(materialId);
                    if (p != null) {
                        b.setMinContractPrice(p.getMinContractPrice());
                        b.setMaxContractPrice(p.getMaxContractPrice());
                        b.setMinPriceArea(p.getMinPriceArea());
                        b.setMaxPriceArea(p.getMaxPriceArea());
                        b.setTotalNum(p.getTotalNum());
                        b.setTotalMoney(p.getTotalMoney());
                        b.setAvgPrice(p.getAvgPrice());
                    } else {
                        b.setMinContractPrice(BigDecimal.ZERO);
                        b.setMaxContractPrice(BigDecimal.ZERO);
                        b.setMinPriceArea(BigDecimal.ZERO);
                        b.setMaxPriceArea(BigDecimal.ZERO);
                        b.setTotalNum(BigDecimal.ZERO);
                        b.setTotalMoney(BigDecimal.ZERO);
                        b.setAvgPrice(BigDecimal.ZERO);
                        b.setBuyPrice(BigDecimal.ZERO);
                    }
                }
            }
            page.setRecords(list);
            page.setTotal(hits.getTotalHits().value);
        } catch (IOException e) {
            e.printStackTrace();
        }

        return CommonResponse.success("查询列表数据成功！", page);
    }


    private List<Long> loadBuyPriceList2ES(QueryParam param) {
        BulkRequest bulkRequest = new BulkRequest();
        List<BuyPriceVO> list = reportService.queryBuyPriceList(param);
        List<Long> materialTypeIds = new ArrayList<>();
        for (BuyPriceVO vo : list) {
            materialTypeIds.add(vo.getMaterialTypeId());
            SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            Map<String, Object> dataMap = new HashMap<>();
            dataMap.put("tenantId", vo.getTenantId());
            dataMap.put("materialTypeName", vo.getMaterialTypeName());
            dataMap.put("materialTypeId", vo.getMaterialTypeId());
            dataMap.put("materialCode", null);
            dataMap.put("materialName", vo.getMaterialName());
            dataMap.put("materialId", vo.getMaterialId());
            dataMap.put("model", vo.getModel());
            dataMap.put("unit", vo.getUnit());
            dataMap.put("brandId", vo.getBrandId());
            dataMap.put("brandName", vo.getBrandName());
            dataMap.put("totalNum", vo.getTotalNum());
            dataMap.put("totalMoney", vo.getTotalMoney());
            dataMap.put("minContractPrice", vo.getMinContractPrice());
            dataMap.put("maxContractPrice", vo.getMaxContractPrice());
            dataMap.put("avgPrice", vo.getAvgPrice());
            dataMap.put("minPriceArea", vo.getMinPriceArea());
            dataMap.put("maxPriceArea", vo.getMaxPriceArea());
            dataMap.put("buyPrice", vo.getBuyPrice());
            dataMap.put("settlementDate", format.format(vo.getSettlementDate()));

            IndexRequest indexRequest = new IndexRequest(INDEX_NAME);
            indexRequest.id(String.valueOf(vo.getId()));
            indexRequest.source(dataMap, XContentType.JSON);
            bulkRequest.add(indexRequest);
        }
        if (bulkRequest.requests().size() > 0) {
            try {
                client.bulk(bulkRequest, RequestOptions.DEFAULT);
            } catch (Exception e) {
                logger.error(e.getMessage());
            }
        }
        logger.info("采购价格分析同步结算汇总数量: " + bulkRequest.requests().size());
        return materialTypeIds;
    }


    /**
     * @param param
     * @description: 查询物资结算详情
     * @return: com.ejianc.framework.core.response.CommonResponse<com.baomidou.mybatisplus.core.metadata.IPage < com.ejianc.business.material.vo.BuyPriceDetailVO>>
     * 查询条件
     * materialId
     * startDate
     * endDate
     * province
     * city
     * region
     * contractName
     * supplierName
     * orgName
     * area
     * @author songlx
     * @date: 2021-04-12
     */
    @RequestMapping(value = "/queryBuyPriceDetailList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryBuyPriceDetailList(@RequestBody QueryParam param) {
        JSONObject obj = new JSONObject();
        int pageNumber = param.getPageIndex();
        int pageSize = param.getPageSize();
        IPage<BuyPriceDetailVO> page = new Page<>();
        page.setCurrent(pageNumber);
        page.setSize(pageSize);
        SearchRequest searchRequest = new SearchRequest(INDEX_NAME_DETAIL);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.from(pageNumber <= 0 ? 0 : (pageNumber - 1) * pageSize);

        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));
        Map<String, Parameter> params = param.getParams();
        if (params != null && params.size() > 0) {
            String startDate = getParamValue(params, "startDate");
            String endDate = getParamValue(params, "endDate");
            if (startDate != null || endDate != null) {
                RangeQueryBuilder settlementDate = QueryBuilders.rangeQuery("settlementDate");
                if (startDate != null) {
                    params.remove("startDate");
                    settlementDate.gte(startDate +" 00:00:00");
                }
                if (endDate != null) {
                    params.remove("endDate");
                    settlementDate.lte(endDate+" 00:00:00");
                }
                boolQuery.must(settlementDate);
            }
            for (Map.Entry<String, Parameter> p : params.entrySet()) {
                String key = p.getKey();
                String value = getParamValue(params, key);
                if (StringUtils.isEmpty(value)) continue;
                boolQuery.must(QueryBuilders.matchQuery(key, value));
            }
        }

        sourceBuilder.size(pageSize);
        sourceBuilder.query(boolQuery);
        LinkedHashMap<String, String> orderMap = param.getOrderMap();
        if (orderMap.size() > 0) {
            for (Map.Entry<String, String> entry : orderMap.entrySet()) {
                if (SortOrder.ASC.toString().equalsIgnoreCase(entry.getValue())) {
                    sourceBuilder.sort(entry.getKey(), SortOrder.ASC);
                } else {
                    sourceBuilder.sort(entry.getKey(), SortOrder.DESC);
                }
            }
        }

        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);

        List<BuyPriceDetailVO> list = new ArrayList<>();
        BuyPriceVO buyPriceVO = new BuyPriceVO();
        SearchResponse response = null;
        try {
            response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, BuyPriceDetailVO.class));
            }
            page.setRecords(list);
            page.setTotal(hits.getTotalHits().value);

            if (CollectionUtil.isNotEmpty(list)) {

                BigDecimal sumNum = BigDecimal.ZERO;
                BigDecimal sumMoney = BigDecimal.ZERO;
                List<BigDecimal> conPriceList = new ArrayList<>();
                List<BigDecimal> priceList = new ArrayList<>();
                for (BuyPriceDetailVO vo : list) {
                    buyPriceVO.setMaterialName(vo.getMaterialName());
                    buyPriceVO.setModel(vo.getModel());
                    buyPriceVO.setUnit(vo.getUnit());
                    sumNum = MathUtil.safeAdd(sumNum, vo.getNum());
                    sumMoney = MathUtil.safeAdd(sumMoney, vo.getMoney());
                    if (vo.getContractPrice() != null) conPriceList.add(vo.getContractPrice());
                    if (vo.getPrice() != null) priceList.add(vo.getPrice());
                }
                Collections.sort(conPriceList);
                Collections.sort(priceList);
                if (conPriceList.size() > 0) {
                    buyPriceVO.setMinContractPrice(conPriceList.get(0));
                    buyPriceVO.setMaxContractPrice(conPriceList.get(conPriceList.size() - 1));
                } else {
                    buyPriceVO.setMinContractPrice(null);
                    buyPriceVO.setMaxContractPrice(null);

                }
                if (priceList.size() > 0) {
                    buyPriceVO.setMinPriceArea(priceList.get(0));
                    buyPriceVO.setMaxPriceArea(priceList.get(priceList.size() - 1));
                } else {
                    buyPriceVO.setMinPriceArea(null);
                    buyPriceVO.setMaxPriceArea(null);
                }
                buyPriceVO.setTotalNum(sumNum);
                buyPriceVO.setTotalMoney(sumMoney);
                buyPriceVO.setAvgPrice(MathUtil.safeDiv(sumMoney, sumNum));


            } else {
                buyPriceVO.setMinContractPrice(BigDecimal.ZERO);
                buyPriceVO.setMaxContractPrice(BigDecimal.ZERO);
                buyPriceVO.setMinPriceArea(BigDecimal.ZERO);
                buyPriceVO.setMaxPriceArea(BigDecimal.ZERO);
                buyPriceVO.setAvgPrice(BigDecimal.ZERO);
            }

        } catch (IOException e) {
            e.printStackTrace();
        }
        obj.put("obj", buyPriceVO);
        obj.put("pageData", page);
        return CommonResponse.success("查询列表数据成功！", obj);
    }

    private String getParamValue(Map<String, Parameter> params, String paramKey) {
        if (params.get(paramKey) != null) {
            Object value = params.get(paramKey).getValue();
            return value != null ? String.valueOf(value) : null;
        }
        return null;
    }


    /**
     * @param param
     * @description: 加载物资到ES
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.material.vo.BuyPriceDetailVO>
     * @author songlx
     * @date: 2021-04-12
     */
    @RequestMapping(value = "/loadBuyPrice2ES", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<BuyPriceDetailVO> loadBuyPrice2ES(@RequestBody QueryParam param) {

        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        try {
            this.deleteESData(INDEX_NAME, param);
            List<Long> materialTypeIds = this.loadBuyPriceList2ES(param);
            if (CollectionUtil.isNotEmpty(materialTypeIds)) {
                this.deleteESData(INDEX_NAME_MATERIAL_CATEGORY, param);
                CommonResponse<MaterialCategoryVO> materialCategoryVOCommonResponse = this.loadMaterialCategoryList2ES(materialTypeIds);
                if (!materialCategoryVOCommonResponse.isSuccess()) {
                    return CommonResponse.error(materialCategoryVOCommonResponse.getMsg());
                }

                this.deleteESData(INDEX_NAME_DETAIL, param);
                this.loadBuyPriceDetailList2ES(param);
            }

        } catch (IOException e) {
            e.printStackTrace();
        }

        return CommonResponse.success("重新加载数据成功！");
    }


    private Long queryIndexSize(String indexName) {
        Long resultSize = 0L;
        SearchRequest searchRequest = new SearchRequest(indexName);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();

        // 查询参数
        TermQueryBuilder termQueryBuilder = QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid());

        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        boolQueryBuilder.must(termQueryBuilder);

        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); // 设置超时时间

        searchRequest.source(sourceBuilder);

        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            resultSize = hits.getTotalHits().value;
        } catch (IOException e) {
            try { //  重试一次
                SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
                SearchHits hits = response.getHits();
                resultSize = hits.getTotalHits().value;
            } catch (IOException e1) {
                e1.printStackTrace();
                throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG：" + e1.getMessage());
            }
        }
        return resultSize;
    }

    private long deleteESData(String indexName, QueryParam param) throws IOException {
        Long resultSize = queryIndexSize(indexName);
        Long deleteTimes = resultSize % BATCH_SIZE;
        for (int i = 0; i < deleteTimes + 1; i++) {
            DeleteByQueryRequest deleteByQueryRequest = new DeleteByQueryRequest(indexName);
            deleteByQueryRequest.setQuery(new TermQueryBuilder("tenantId", InvocationInfoProxy.getTenantid()));
            deleteByQueryRequest.setBatchSize(BATCH_SIZE);
            BulkByScrollResponse response = client.deleteByQuery(deleteByQueryRequest, RequestOptions.DEFAULT);
            return response.getDeleted();
        }
        return 0;
    }

  /*  @RequestMapping(value = "/toES", method = RequestMethod.POST)
    @ResponseBody
    private CommonResponse<MaterialCategoryVO> toES() {
        String json = "[{'code':'WL001','innerCode':'1278232660993966192','tenantId':'999999','name':'全部物料分类','id':'1278232660993966192','isLeaf':'null','enabled':'1'},{'code':'WL101','innerCode':'1278232660993966192|1278232496145235969','tenantId':'999999','name':'黑色金属','id':'1278232496145235969','isLeaf':'null','parentId':'1278232660993966192','enabled':'1'}                                                                             "
                + ",{'code':'WL10101','innerCode':'1278232660993966192|1278232496145235969|1278232496224927745','tenantId':'999999','name':'线螺','id':'1278232496224927745','isLeaf':'null','parentId':'1278232496145235969','enabled':'1'}"
                + ",{'code':'WL10101001','innerCode':'1278232660993966192|1278232496145235969|1278232496224927745|1278232496250093569','tenantId':'999999','name':'高线','id':'1278232496250093569','isLeaf':'null','parentId':'1278232496224927745','enabled':'1'}"
                + ",{'code':'WL103','innerCode':'1278232660993966192|1278232503392993282','tenantId':'999999','name':'商品砼','id':'1278232503392993282','isLeaf':'null','parentId':'1278232660993966192','enabled':'1'}"
                + ",{'code':'WL10301','innerCode':'1278232660993966192|1278232503392993282|1278232503422353409','tenantId':'999999','name':'泵送商品砼','id':'1278232503422353409','isLeaf':'null','parentId':'1278232503392993282','enabled':'1'}"
                + ",{'code':'WL10301008','innerCode':'1278232660993966192|1278232503392993282|1278232503422353409|1278232503699177473','tenantId':'999999','name':'（泵送）商品砼','id':'1278232503699177473','isLeaf':'null','parentId':'1278232503422353409','enabled':'1'}"
                + ",{'code':'WL104','innerCode':'1278232660993966192|1278232509655089154','tenantId':'999999','name':'水泥及掺料','id':'1278232509655089154','isLeaf':'null','parentId':'1278232660993966192','enabled':'1'}"
                + ",{'code':'WL10401','innerCode':'1278232660993966192|1278232509655089154|1278232509676060673','tenantId':'999999','name':'水泥','id':'1278232509676060673','isLeaf':'null','parentId':'1278232509655089154','enabled':'1'}"
                + ",{'code':'WL10401001','innerCode':'1278232660993966192|1278232509655089154|1278232509676060673|1278232509692837889','tenantId':'999999','name':'硅酸盐水泥','id':'1278232509692837889','isLeaf':'null','parentId':'1278232509676060673','enabled':'1'}"
                + ",{'code':'WL116','innerCode':'1278232660993966192|1278232608837795841','tenantId':'999999','name':'机电设备','id':'1278232608837795841','isLeaf':'null','parentId':'1278232660993966192','enabled':'1'}"
                + ",{'code':'WL11602','innerCode':'1278232660993966192|1278232608837795841|1278232609341112322','tenantId':'999999','name':'空调、通风设备','id':'1278232609341112322','isLeaf':'null','parentId':'1278232608837795841','enabled':'1'}"
                + ",{'code':'WL11602006','innerCode':'1278232660993966192|1278232608837795841|1278232609341112322|1278232609462747137','tenantId':'999999','name':'空调机组','id':'1278232609462747137','isLeaf':'null','parentId':'1278232609341112322','enabled':'1'}"
                + ",{'code':'WL301','innerCode':'1278232660993966192|1278232660993966081','tenantId':'999999','name':'周转材料及料具','id':'1278232660993966081','isLeaf':'null','parentId':'1278232660993966192','enabled':'1'}"
                + ",{'code':'WL30104','innerCode':'1278232660993966192|1278232660993966081|1278232663552491522','tenantId':'999999','name':'其它周转材料','id':'1278232663552491522','isLeaf':'null','parentId':'1278232660993966081','enabled':'1'}"
                + ",{'code':'WL30104018','innerCode':'1278232660993966192|1278232660993966081|1278232663552491522|1278232664869502978','tenantId':'999999','name':'其它类','id':'1278232664869502978','isLeaf':'null','parentId':'1278232663552491522','enabled':'1'}"
                + ",{'code':'WL30104018014','innerCode':'1278232660993966192|1278232660993966081|1278232663552491522|1278232664869502978|1278232665108578306','tenantId':'999999','name':'安全体验材料','id':'1278232665108578306','isLeaf':'null','parentId':'1278232664869502978','enabled':'1'}"
                + ",{'code':'WL109','innerCode':'1278232660993966192|1278232526021263362','tenantId':'999999','name':'水暧','id':'1278232526021263362','isLeaf':'null','parentId':'1278232660993966192','enabled':'1'}"
                + ",{'code':'WL10940','innerCode':'1278232660993966192|1278232526021263362|1278232546816622594','tenantId':'999999','name':'其它水卫材料','id':'1278232546816622594','isLeaf':'null','parentId':'1278232526021263362','enabled':'1'}"
                + ",{'code':'WL10940111','innerCode':'1278232660993966192|1278232526021263362|1278232546816622594|1278232548838277122','tenantId':'999999','name':'风球','id':'1278232548838277122','isLeaf':'null','parentId':'1278232546816622594','enabled':'1'}]";
        List<MaterialCategoryVO> object = JSONArray.parseArray(json, MaterialCategoryVO.class);
        BulkRequest bulkRequest = new BulkRequest();
        for (MaterialCategoryVO vo : object) {
            Map<String, Object> dataMap = new HashMap<>();
            dataMap.put("tenantId", String.valueOf(vo.getTenantId()));
            dataMap.put("id", String.valueOf(vo.getId()));
            dataMap.put("parentId", null == vo.getParentId() ? null : String.valueOf(vo.getParentId()));
            dataMap.put("code", vo.getCode());
            dataMap.put("name", vo.getName());
            dataMap.put("innerCode", vo.getInnerCode());
            dataMap.put("enabled", String.valueOf(vo.getEnabled()));
            dataMap.put("isLeaf", String.valueOf(vo.getIsLeaf()));
            IndexRequest indexRequest = new IndexRequest(INDEX_NAME_MATERIAL_CATEGORY);
            indexRequest.id(String.valueOf(vo.getId()));
            indexRequest.source(dataMap, XContentType.JSON);
            bulkRequest.add(indexRequest);
            String json1 = JSON.toJSONString(dataMap);
            System.out.println("物资" + json1);
        }
        try {
            bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
            System.out.println("同步es....");
            client.bulk(bulkRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            logger.error(e.getMessage());
        }
        logger.info("采购价格分析同步物料分类数量: " + bulkRequest.requests().size());
        return CommonResponse.success("重新加载物料分类数据成功！");
    }*/


    private CommonResponse<MaterialCategoryVO> loadMaterialCategoryList2ES(List<Long> materialTypeIds) {
        if (CollectionUtil.isEmpty(materialTypeIds)) {
            logger.info("未查询到物资结算明物料分类数据!");
            return CommonResponse.error("未查询到物资结算明物料分类数据！");
        }
        List<Long> ids = materialTypeIds.stream().distinct().collect(Collectors.toList());
        CommonResponse<List<MaterialCategoryVO>> categoryList = materialApi.queryCategoryListByChildren(ids);
        if (CollectionUtil.isEmpty(categoryList.getData())) {
            logger.info("获取当前租户物料分类树失败!");
            return CommonResponse.error("获取当前租户物料分类树失败！");
        }

        BulkRequest bulkRequest = new BulkRequest();
        for (MaterialCategoryVO vo : categoryList.getData()) {
            Map<String, Object> dataMap = new HashMap<>();
            dataMap.put("tenantId", InvocationInfoProxy.getTenantid());
            dataMap.put("id", vo.getId());
            dataMap.put("parentId", null == vo.getParentId() ? null : vo.getParentId());
            dataMap.put("code", vo.getCode());
            dataMap.put("name", vo.getName());
            dataMap.put("innerCode", vo.getInnerCode());
            dataMap.put("enabled", vo.getEnabled());
            dataMap.put("isLeaf", vo.getIsLeaf());
            IndexRequest indexRequest = new IndexRequest(INDEX_NAME_MATERIAL_CATEGORY);
            indexRequest.id(String.valueOf(InvocationInfoProxy.getTenantid()) + vo.getId());
            indexRequest.source(dataMap, XContentType.JSON);
            bulkRequest.add(indexRequest);
        }
        try {
            bulkRequest.setRefreshPolicy(WriteRequest.RefreshPolicy.IMMEDIATE);
            System.out.println("同步es....");
            client.bulk(bulkRequest, RequestOptions.DEFAULT);
        } catch (Exception e) {
            logger.error(e.getMessage());
        }
        logger.info("采购价格分析同步物料分类数量: " + bulkRequest.requests().size());
        return CommonResponse.success("重新加载物料分类数据成功！");
    }


    private void loadBuyPriceDetailList2ES(QueryParam param) {

        List<BuyPriceDetailVO> list = reportService.queryBuyPriceDetailList(param);

        List<Long> projectIds = list.stream().map(BuyPriceDetailVO::getProjectId).distinct().collect(Collectors.toList());
        Map<Long, ProjectRegisterVO> projectMap = new HashMap<>();
        Map<Long, SupplierVO> supplierMap = new HashMap<>();
        if (!CollectionUtil.isEmpty(projectIds)) {
            List<List<Long>> partition = Lists.partition(projectIds, 300);
            List<ProjectRegisterVO> projects=new ArrayList<>();
            for (int i = 0; i < partition.size(); i++) {
                CommonResponse<List<ProjectRegisterVO>> response = projectApi.queryProjectByIds(partition.get(i));
                if (!response.isSuccess()) {
                    throw new BusinessException(response.getMsg());
                }
                projects.addAll(response.getData());
            }
            if (!CollectionUtil.isEmpty(projects)) {
                projectMap = projects.stream().collect(Collectors.toMap(ProjectRegisterVO::getId, Function.identity()));
            }
        }

        List<Long> supplierIds = list.stream().map(BuyPriceDetailVO::getSupplierId).distinct().collect(Collectors.toList());
        if (!CollectionUtil.isEmpty(supplierIds)) {
            List<List<Long>> partition = Lists.partition(supplierIds, 300);
            List<SupplierVO> supplierVOS=new ArrayList<>();
            for (int i = 0; i < partition.size(); i++) {
                CommonResponse<List<SupplierVO>> listCommonResponse = supplierApi.querySupplierByIds(partition.get(i));
                if (!listCommonResponse.isSuccess()) {
                    throw new BusinessException(listCommonResponse.getMsg());
                }
                supplierVOS.addAll(listCommonResponse.getData());
            }
            if (!CollectionUtil.isEmpty(supplierVOS)) {
                supplierMap = supplierVOS.stream().collect(Collectors.toMap(SupplierVO::getId, Function.identity()));
            }
        }
        boolean insertEsFlag = false;
        BulkRequest bulkRequest = new BulkRequest();
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        for (BuyPriceDetailVO vo : list) {
            Long projectId = vo.getProjectId();
            Long supplierId = vo.getSupplierId();
            ProjectRegisterVO pro = projectMap.get(projectId);
            SupplierVO su = supplierMap.get(supplierId);

            if (su != null && StringUtils.isNotEmpty(su.getProvince())) {
                vo.setProvince(su.getProvince());
                vo.setCity(su.getCity());
                vo.setRegion(su.getRegion());
                vo.setArea(su.getArea());
            } else {
                if (null != pro) {
                    vo.setProvince(pro.getProvince());
                    vo.setCity(pro.getCity());
                    vo.setRegion(pro.getRegion());
                    vo.setArea(pro.getArea());
                }
            }


            Map<String, Object> dataMap = new HashMap<>();
            dataMap.put("billCode", vo.getBillCode());
            dataMap.put("contractName", vo.getContractName());
            dataMap.put("contractId", vo.getContractId());
            dataMap.put("supplierName", vo.getSupplierName());
            dataMap.put("supplierId", vo.getSupplierId());
            dataMap.put("orgName", vo.getOrgName());
            dataMap.put("orgId", vo.getOrgId());
            dataMap.put("settlementDate", format.format(vo.getSettlementDate()));
            dataMap.put("contractPrice", vo.getContractPrice());
            dataMap.put("pId", vo.getpId());
            dataMap.put("materialTypeName", vo.getMaterialTypeName());
            dataMap.put("materialTypeId", vo.getMaterialTypeId());
            dataMap.put("materialName", vo.getMaterialName());
            dataMap.put("materialId", vo.getMaterialId());
            dataMap.put("model", vo.getModel());
            dataMap.put("unit", vo.getUnit());
            dataMap.put("brandId", vo.getBrandId());
            dataMap.put("num", vo.getNum());
            dataMap.put("price", vo.getPrice());
            dataMap.put("money", vo.getMoney());
            dataMap.put("taxRate", vo.getTaxRate());
            dataMap.put("province", vo.getProvince());
            dataMap.put("city", vo.getCity());
            dataMap.put("region", vo.getRegion());
            dataMap.put("projectId", vo.getProjectId());
            dataMap.put("tenantId", vo.getTenantId());
            dataMap.put("area", vo.getArea());

            IndexRequest indexRequest = new IndexRequest(INDEX_NAME_DETAIL);
            indexRequest.id(String.valueOf(vo.getId()));
            indexRequest.source(dataMap, XContentType.JSON);
            bulkRequest.add(indexRequest);
            insertEsFlag = true;

        }
        logger.info("采购价格分析同步物料明细数量: " + bulkRequest.requests().size());
        if (insertEsFlag) {
            try {
                client.bulk(bulkRequest, RequestOptions.DEFAULT);
            } catch (Exception e) {
                logger.error(e.getMessage());
            }
        }
    }


    /**
     * @param param
     * @param response
     * @description: 导出
     * @return: void
     * @author songlx
     * @date: 2021-04-14
     */
    @PostMapping("excelExport")
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {

        param.setPageIndex(1);
        param.setPageSize(10000);

        CommonResponse<IPage<BuyPriceVO>> page = this.queryBuyPriceList(param);
        List<BuyPriceVO> list = page.getData().getRecords();
        Map<String, Object> beans = new HashMap<String, Object>();
        if (CollectionUtils.isNotEmpty(list)) {
            beans.put("records", list);
            ExcelExport.getInstance().export("buyPriceListExport.xlsx", beans, response);
        }
    }


}
