package com.ejianc.business.material.controller;

import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.equipment.api.ISettlementApi;
import com.ejianc.business.equipment.vo.RentContractPayVO;
import com.ejianc.business.finance.api.IPayContractApi;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.vo.PayContractVO;
import com.ejianc.business.finance.vo.SumPayMnyVO;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.material.bean.MaterialContractDetailSubEntity;
import com.ejianc.business.material.bean.MaterialContractEntity;
import com.ejianc.business.material.bean.PurchaseSettlementEntity;
import com.ejianc.business.material.pub.ContractStateEnum;
import com.ejianc.business.material.service.IMaterialContractService;
import com.ejianc.business.material.service.IPurchaseSettlementService;
import com.ejianc.business.material.utlis.ArchivesUtil;
import com.ejianc.business.material.vo.*;
import com.ejianc.business.other.api.IOtherContractApi;
import com.ejianc.business.other.vo.OtherProjectReportVo;
import com.ejianc.business.rmat.api.IRmatContractApi;
import com.ejianc.business.rmat.vo.RmatProjectReportVo;
import com.ejianc.business.sub.api.ISubReportApi;
import com.ejianc.business.tax.api.IInvoiceApi;
import com.ejianc.business.tax.vo.InvoiceReceiveRecordVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IMaterialApi;
import com.ejianc.foundation.share.vo.MaterialInsertArchiveVO;
import com.ejianc.foundation.share.vo.MaterialVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.*;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import com.ejianc.framework.skeleton.template.BaseVO;
import com.ejianc.support.idworker.util.IdWorker;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2020/6/8 10:47
 */
@RestController
@RequestMapping("/materialContract/")
public class MaterialContractController {
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    private static volatile HttpServletRequest request;

    @Autowired
    private IMaterialContractService materialContractService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IProjectApi iProjectApi;

    @Autowired
    private IInvoiceApi iInvoiceApi;

    @Autowired
    private IPayContractApi payContractApi;

    @Autowired
    private ISettlementApi settlementApi;

    @Autowired
    private ISubReportApi subReportApi;

    @Autowired
    private IPurchaseSettlementService purchaseSettlementService;

    @Value("${common.env.base-host}")
    public String BASE_HOST;

    private final String DEFAULT_RULE_CODE = "materialContract";
    @Autowired
    private ArchivesUtil archivesUtil;
    @Autowired
    private IAttachmentApi attachmentApi;
    @Autowired
    private IInvoiceApi invoiceApi;
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private SessionManager sessionManager;
    @Autowired
    private IRmatContractApi rmatContractApi;
    @Autowired
    private IOtherContractApi otherContractApi;
    @Resource
    private IMaterialApi materialApi;

    @Autowired
    private IBillTypeApi billTypeApi;

    /**
     * @description: 查询物资合同价
     *
     * @return {@link CommonResponse< ParamsCheckVO>}
     * @author songlx
     * @date: 2024/6/18
     */
    @GetMapping("queryMaterialContractPrice")
    public CommonResponse<List<MaterialContractPriceResultVO>> queryMaterialContractPrice(@RequestParam Long projectId, @RequestParam Long materialId) {
        MaterialContractPriceQueryVO queryVO = new MaterialContractPriceQueryVO();
        queryVO.setProjectId(projectId);
        queryVO.setMaterialIds(Arrays.asList(materialId));
        Map<Long, List<MaterialContractPriceResultVO>> map = materialContractService.queryMaterialContractPrice(queryVO);
        return CommonResponse.success("查询物资合同价成功！", map.get(materialId));
    }


    /***
     * @description: 合同/变更参数校验
     *
     * @param vo
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.material.vo.ParamsCheckVO>
     * @author songlx
     * @date: 2021-06-08
     */
    // 【物资总计划量】控制【物资合同量】P-8N9Au207 集采不受控制
    // 总计划价格【物资总计划价】控制【物资合同价】P-2Ux27w08  集采不受控制
    // 结算历史价格区间【历史价】控制【物资合同价】P-126hYD10
    //控制方式 0：不控制，1、提醒，2、无法保存
    @PostMapping("checkParams/{isJc}")
    public CommonResponse<ParamsCheckVO> checkParams(@PathVariable(name = "isJc") Integer isJc,
                                                     @RequestBody MaterialPriceVO vo) {

        //ParamsCheckVO paramsCheckVO = materialContractService.checkParams(isJc, vo.getContractId(), vo);
        ParamsCheckVO pc = new ParamsCheckVO();
        pc.setWarnType("none");
        pc.setDataSource(null);
        return CommonResponse.success("参数校验成功！", pc);
    }


    /***
     * @description: 单据参数控制
     *      * 查询总计划单价和历史价格区间
     * @param vo
     * @return: com.ejianc.framework.core.response.CommonResponse<java.util.List < com.ejianc.business.material.vo.MaterialPriceVO>>
     * @author songlx
     * @date: 2021-05-26
     */
    @PostMapping("queryPrice/{isJc}")
    public CommonResponse<List<MaterialPriceVO>> queryPrice(@PathVariable(name = "isJc") Integer isJc,
                                                            @RequestBody MaterialPriceVO vo) {
        materialContractService.queryPrice(isJc, null, vo);
        return CommonResponse.success("查询成功！", vo.getDetail());
    }


    /**
     * 　* @Description: 物资合同分页列表查询, 查询范围：本下的所有合同+上级的审批通过的集采合同
     * 　* @param [queryParam]
     * 　* @return com.ejianc.framework.core.response.CommonResponse<com.alibaba.fastjson.JSONObject>
     * 　* @throws
     * 　* @author CJ
     * 　* @date 2020/6/8 11:22
     */
    @PostMapping(value = "pageList")
    public CommonResponse<JSONObject> pageList(@RequestBody QueryParam queryParam) {
        JSONObject resp = new JSONObject();
        queryParam.getFuzzyFields().add("name");
        queryParam.getFuzzyFields().add("code");
        queryParam.getFuzzyFields().add("supplierName");
        queryParam.getFuzzyFields().add("orgName");
        queryParam.getFuzzyFields().add("projectCode");
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("contractorEnterpriseName");
        queryParam.getFuzzyFields().add("createUserName");
        queryParam.getFuzzyFields().add("amountWithTax");

        queryParam.getComplexParams().add(getPageQueryParam(null));

        IPage<MaterialContractEntity> pageData = materialContractService.queryPage(queryParam, false);

        //页面统计，查询原合同金额，现合同金额
        Map<String, Object> contractAmountMap = materialContractService.countContractAmount(queryParam);

        resp.put("records", pageData.getRecords());
        resp.put("total", pageData.getTotal());
        resp.put("size", pageData.getSize());
        resp.put("current", pageData.getCurrent());
        resp.put("pages", pageData.getPages());
        resp.put("materialCount", contractAmountMap);

        return CommonResponse.success("合同分页列表查询成功！", resp);
    }

    public ComplexParam getPageQueryParam(Long orgId) {
        ComplexParam c1 = new ComplexParam();
        c1.setLogic(ComplexParam.AND);

        if (null == orgId) {
            orgId = InvocationInfoProxy.getOrgId();
        }
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(orgId);
        CommonResponse<List<OrgVO>> parentOrgResp = orgApi.findParentsByOrgId(orgId);
        if (!childOrgResp.isSuccess() || !parentOrgResp.isSuccess()) {
            throw new BusinessException("合同分页列表查询失败, 查询组织信息失败！");
        }

        List<Long> parentOrgIds = new ArrayList<>();
        List<Long> childIds = new ArrayList<>();
        childIds.addAll(childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        parentOrgIds.addAll(parentOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        parentOrgIds.remove(orgId);

        ComplexParam c2 = new ComplexParam();
        c2.setLogic(ComplexParam.OR);
        c2.getParams().put("org_id", new Parameter(QueryParam.IN, childIds));
        c1.getComplexParams().add(c2);

        if (CollectionUtils.isNotEmpty(parentOrgIds)) {
            ComplexParam c3 = new ComplexParam();
            c3.setLogic(ComplexParam.OR);
            c3.getParams().put("depend_on_project", new Parameter(QueryParam.EQ, "0"));
            c3.getParams().put("org_id", new Parameter(QueryParam.IN, parentOrgIds));
            c1.getComplexParams().add(c3);
            c3.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        }
        return c1;
    }

    /**
     * 　* @Description: 物资合同更新、保存
     * 　* @param [materialContractVO]
     * 　* @return com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.material.vo.MaterialContractVO>
     * 　* @throws
     * 　* @author CJ
     * 　* @date 2020/6/8 10:54
     */
    @PostMapping("saveOrUpdate")
    public CommonResponse<MaterialContractVO> save(@RequestBody MaterialContractVO materialContractVO) {
        archivesUtil.checkHaveTempMaterial(materialContractVO.getMaterialDetailList());
        if (StringUtils.isBlank(materialContractVO.getCode())) {
            String dependOnProject = materialContractVO.getDependOnProject();
            if ("1".equals(dependOnProject)) {
                //自动生成编码
                materialContractVO.setCode(getAutoCode(materialContractVO));
            } else {
                throw new BusinessException("集采合同请输入合同编码！");
            }
        }
        // 合同名称：【产品名称-供应商名称-项目编码-项目名称-合同金额-发起人】
        //例如：窗台石门槛石栏杆等-济宁纵诚商贸有限公司-2.1.0230-济宁市公共卫生中医医疗中心项目医疗净化专项-25000.00-刘严
        if (StringUtils.isBlank(materialContractVO.getName())) {
            LinkedList<String> namelist = new LinkedList<>();
            if (StringUtils.isNotBlank(materialContractVO.getProductName())) {
                namelist.add(materialContractVO.getProductName());
            }
            if (StringUtils.isNotBlank(materialContractVO.getSupplierName())) {
                namelist.add(materialContractVO.getSupplierName());
            }
            if (StringUtils.isNotBlank(materialContractVO.getProjectCode())) {
                namelist.add(materialContractVO.getProjectCode());
            }
            if (StringUtils.isNotBlank(materialContractVO.getProjectName())) {
                namelist.add(materialContractVO.getProjectName());
            }
            BigDecimal amountWithTax = materialContractVO.getAmountWithTax();
            if (amountWithTax != null) {
                namelist.add(String.valueOf(ComputeUtil.scaleTwo(amountWithTax)));
            }
            namelist.add(materialContractVO.getEmployeeName());
            materialContractVO.setName(String.join("-", namelist));
        }

        boolean checkResult = materialContractService.codeCheck(materialContractVO.getId(), materialContractVO.getCode());
        if (!checkResult) {
            return CommonResponse.error("保存失败，合同编码重复");
        }
        List<MaterialContractDetailSubVO> materialDetailList = materialContractVO.getMaterialDetailList();

        long id = IdWorker.getId();
        if (materialContractVO.getId() == null) {
            materialContractVO.setId(id);
        }
        if (CollectionUtils.isNotEmpty(materialDetailList)) {
            for (MaterialContractDetailSubVO materialContractDetailSubVO : materialDetailList) {
                String onlykey = materialContractDetailSubVO.getMaterialTypeName() + materialContractDetailSubVO.getMaterialName() + materialContractDetailSubVO.getMeasureUnit() + materialContractDetailSubVO.getSpec() + materialContractDetailSubVO.getMaterialCode();
                materialContractDetailSubVO.setOnlyKey(onlykey);
            }
            MaterialInsertArchiveVO archiveVO = new MaterialInsertArchiveVO(null, materialContractVO.getId(), materialContractVO.getCode(), materialContractVO.getProjectName(),"物资采购合同");
            JSONObject jsonObject = archivesUtil.batchSaveArchive(archiveVO, materialDetailList, "materialTypeId", "materialId",
                    "materialTypeName", "materialName", "materialCode","measureUnit", "spec", "errorMessage");
            String sourceList = JSON.toJSONString(jsonObject.get("sourceList"), SerializerFeature.WriteMapNullValue, SerializerFeature.PrettyFormat);
            List<MaterialContractDetailSubVO> list = JSONObject.parseArray(sourceList, MaterialContractDetailSubVO.class);
            materialContractVO.setMaterialDetailList(list);

            // 根据物资来源ids查询物资档案
            List<Long> sourceIds = list.stream().map(materialContractDetailSubVO -> Long.parseLong(materialContractDetailSubVO.getMaterialId())).collect(Collectors.toList());
            List<List<Long>> partition = Lists.partition(sourceIds, 300);
            List<MaterialVO> materialList=new ArrayList<>();
            for (int i = 0; i < partition.size(); i++) {
                CommonResponse<List<MaterialVO>> r = materialApi.queryMaterialByIds(partition.get(i));
                if (!r.isSuccess()) {
                    throw new BusinessException("保存合同信息失败，查询物资档案失败！");
                }
                materialList.addAll(r.getData());
            }
            Map<Long, MaterialVO> materialMap = materialList.stream().collect(Collectors.toMap(MaterialVO::getId, Function.identity(), (key1, key2) -> key2));
            for (MaterialContractDetailSubVO detailSubVO : materialContractVO.getMaterialDetailList()) {
                if (materialMap.containsKey(Long.parseLong(detailSubVO.getMaterialId()))) {
                    MaterialVO vo = materialMap.get(Long.parseLong(detailSubVO.getMaterialId()));
                    String assistUnitName = vo.getAssistUnitName();
                    BigDecimal unitAssistUnit = vo.getUnitAssistUnit();
                    if (StringUtils.isNotBlank(assistUnitName) && unitAssistUnit != null) {
                        detailSubVO.setAssistUnitName(assistUnitName);
                        detailSubVO.setUnitAssistUnit(unitAssistUnit);
                        detailSubVO.setNumAssistUnit(NumberUtil.mul(detailSubVO.getCount(), unitAssistUnit));
                        detailSubVO.setPriceAssistUnit(NumberUtil.div(detailSubVO.getTotalAmount(), detailSubVO.getNumAssistUnit(), 4, RoundingMode.HALF_UP));
                    }
                }
            }
        }

        MaterialContractVO resp = materialContractService.save(materialContractVO);
        this.queryMaterialPrice(resp);
        return CommonResponse.success("保存成功！", resp);
    }

    /**
     * 获取自动生成编码并校验重复
     *
     * @return
     * @param materialContractVO
     */
    private String getAutoCode(MaterialContractVO materialContractVO) {
        String code = null;

        while (StringUtils.isBlank(code)) {
            BillCodeParam billCodeParam = BillCodeParam
                    .build(DEFAULT_RULE_CODE, InvocationInfoProxy.getTenantid(), materialContractVO);
            CommonResponse<String> resp = billCodeApi.generateBillCode(billCodeParam);
            if (!resp.isSuccess()) {
                throw new BusinessException("保存合同信息失败，自动生成编码失败！");
            }
            code = resp.getData();
        }

        return code;
    }

    /**
     * 　* @Description: 物资合同列表导出
     * 　* @param [queryParam, response]
     * 　* @return void
     * 　* @throws
     * 　* @author CJ
     * 　* @date 2020/6/8 10:55
     */
    @PostMapping("excelExport")
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        queryParam.getFuzzyFields().add("name");
        queryParam.getFuzzyFields().add("code");
        queryParam.getFuzzyFields().add("supplierName");
        queryParam.getFuzzyFields().add("orgName");
        queryParam.getFuzzyFields().add("projectCode");
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("contractorEnterpriseName");
        queryParam.getFuzzyFields().add("createUserName");
        queryParam.getFuzzyFields().add("amountWithTax");
        queryParam.setPageIndex(1);
        queryParam.setPageSize(-1);
        queryParam.getComplexParams().add(getPageQueryParam(null));

        IPage<MaterialContractEntity> pageData = materialContractService.queryPage(queryParam, false);
        Map<String, Object> beans = new HashMap<String, Object>();
        List<ContractExportVO> list = new ArrayList<>();
        List<MaterialContractEntity> data = pageData.getRecords();
        if (CollectionUtils.isNotEmpty(data)) {
            data.forEach(contract -> {
                ContractExportVO vo = BeanMapper.map(contract, ContractExportVO.class);
                vo.setCreateTime(DateFormatUtil.formatDate("yyyy-MM-dd HH:mm:ss", contract.getCreateTime()));
                vo.setSignDate(DateFormatUtil.formatDate("yyyy-MM-dd", contract.getSignDate()));
                vo.setChangeStateName(contract.getChangeState().equals(MaterialContractVO.CONTRACT_CHANGE_STATE_UNCHANGED) ? "未变更" : contract.getChangeState().equals(MaterialContractVO.CONTRACT_CHANGE_STATE_CHANGING) ? "变更中" : "已变更");
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(contract.getBillState()).getDescription());
                list.add(vo);
            });
        }
        beans.put("records", list);
        ExcelExport.getInstance().export("contract-export.xlsx", beans, response);
    }

    @PostMapping("delete")
    public CommonResponse<String> delete(@RequestBody List<MaterialContractVO> vos) {
        if (CollectionUtils.isNotEmpty(vos)) {
            List<Long> ids = vos.stream().map(MaterialContractVO::getId).collect(Collectors.toList());

            //删除前先获取到根据定标单新增的
            LambdaQueryWrapper<MaterialContractEntity> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.in(MaterialContractEntity::getId, ids);
            queryWrapper.isNotNull(MaterialContractEntity::getEnquiryPriceBillId);
            List<MaterialContractEntity> materialContractEntities = materialContractService.list(queryWrapper);

            boolean b = materialContractService.removeByIds(ids, false);
            if (b && CollectionUtils.isNotEmpty(materialContractEntities)) {
                List<Long> enquiryPriceBillIds = materialContractEntities.stream().map(MaterialContractEntity::getEnquiryPriceBillId).collect(Collectors.toList());
                materialContractService.countContractNumAndMnyByEnquiry(enquiryPriceBillIds.toArray(new Long[enquiryPriceBillIds.size()]));
            }
        }
        return CommonResponse.success("删除成功！");
    }

    @GetMapping("queryDetail")
    public CommonResponse<MaterialContractVO> queryDetail(@RequestParam(value = "id") Long id) {
        MaterialContractVO vo = null;
        MaterialContractEntity dbEntity = materialContractService.selectById(id);
        if (null != dbEntity) {
            vo = BeanMapper.map(dbEntity, MaterialContractVO.class);
            this.queryMaterialPrice(vo);
        }

        return CommonResponse.success("查询物资合同详情成功！", vo);
    }


    @GetMapping("queryDetailAndPrice")
    public CommonResponse<MaterialContractVO> queryDetailAndPrice(@RequestParam(value = "id") Long id) {
        MaterialContractVO vo = null;
        MaterialContractEntity dbEntity = materialContractService.selectById(id);
        if (null != dbEntity) {
            vo = BeanMapper.map(dbEntity, MaterialContractVO.class);
            this.queryMaterialPrice(vo);

        }
        vo.setId(IdWorker.getId());
        attachmentApi.copyFilesFromSourceBillToTargetBill(String.valueOf(id),"BT200608000000001","materialContractBill", String.valueOf(vo.getId()), "BT200613000000001","materialContractChange");
        return CommonResponse.success("查询物资合同详情成功！", vo);
    }

    private void queryMaterialPrice(MaterialContractVO vo) {
        //根据单据参数控制查询总计划量\价,历史价格区间
        List<MaterialContractDetailSubVO> materialDetailList = vo.getMaterialDetailList();
        if (CollectionUtils.isNotEmpty(materialDetailList)) {
            MaterialPriceVO priceVO = new MaterialPriceVO(vo.getProjectId());
            List<MaterialPriceVO> detail = new ArrayList<>();
            materialDetailList.forEach(item -> {
                if (StringUtils.isNotEmpty(item.getMaterialId())) {
                    MaterialPriceVO materialPriceVO = new MaterialPriceVO();
                    materialPriceVO.setMaterialId(Long.valueOf(item.getMaterialId()));
                    detail.add(materialPriceVO);
                }
            });
            priceVO.setDetail(detail);
            //1集采 2项目自采
            Integer isJc = "1".equals(vo.getPurchaseMode()) ? 1 : 0;
            MaterialPriceVO materialPriceVO = materialContractService.queryPrice(isJc, vo.getId(), priceVO);
            List<MaterialPriceVO> dd = materialPriceVO.getDetail();
            if (CollectionUtils.isNotEmpty(dd)) {
                Map<Long, MaterialPriceVO> priceVOMap = dd.stream().collect(Collectors.toMap(MaterialPriceVO::getMaterialId, account -> account, (v1, v2) -> v2));
                for (MaterialContractDetailSubVO v : materialDetailList) {
                    String materialId = v.getMaterialId();
                    if (StringUtils.isEmpty(materialId)) {
                        continue;
                    }
                    MaterialPriceVO p = priceVOMap.get(Long.valueOf(materialId));
                    if (null != p) {
                        v.setPlanNum(p.getPlanNum());
                        v.setPlanPrice(p.getPlanPrice());
                        v.setMinPrice(p.getMinPrice());
                        v.setMaxPrice(p.getMaxPrice());
                        v.setPriceArea(p.getPriceArea());
                    }
                }
            }
        }
    }

    /**
     * 物资合同列表参照
     *
     * @return
     */
    @GetMapping("contractListRefer")
    public CommonResponse<IPage<MaterialContractEntity>> contractListRefer(@RequestParam(required = false) String condition,
                                                                           @RequestParam(required = false) String searchText,
                                                                           @RequestParam int pageSize,
                                                                           @RequestParam int pageNumber,
                                                                           @RequestParam(required = false) String searchObject) {
        QueryParam queryParam = new QueryParam();
        queryParam.getFuzzyFields().add("name");
        queryParam.getFuzzyFields().add("code");
        queryParam.getFuzzyFields().add("supplier_name");
        //按照合同审批通过时间倒序排列
        queryParam.getOrderMap().put("effective_date", QueryParam.DESC);
        queryParam.setPageSize(pageSize);
        queryParam.setPageIndex(pageNumber);
        //只查询审批通过的合同
        List<Integer> billStateList = new ArrayList<>();
        billStateList.add(BillStateEnum.PASSED_STATE.getBillStateCode());
        billStateList.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.IN, billStateList));

        if (StringUtils.isNotBlank(searchText)) {
            queryParam.setSearchText(searchText);
        }

        Long orgId = InvocationInfoProxy.getOrgId();
        // 已封账，不能被参照到
        queryParam.getParams().put("state", new Parameter(QueryParam.NOT_IN, Arrays.asList(ContractStateEnum.已封账.getCode(), ContractStateEnum.已作废.getCode())));
        if (StringUtils.isNotBlank(condition)) {
            Map<String, Object> conditionMap = JSONObject.parseObject(condition, Map.class);
            if (null != conditionMap.get("type") && "pay".equals(conditionMap.get("type").toString())) {
                //收票和付款可以参照 已封账 但不能参照已作废
                queryParam.getParams().put("state", new Parameter(QueryParam.NE, ContractStateEnum.已作废.getCode()));
            }
            if (null != conditionMap.get("mobileLeave")) {
                /** 移动端 有组织id 逻辑同pc端，没组织id 查租户下所有集采 */
                if (null != conditionMap.get("orgId")) {
                    orgId = Long.valueOf(conditionMap.get("orgId").toString());
                    queryParam.getComplexParams().add(getPageQueryParam(orgId));
                }
                // 移动端付款申请，选择不属于项目，需要参照租户下所有集采合同
                if (null != conditionMap.get("dependOnProject")) {
                    Integer dependOnProject = Integer.valueOf(conditionMap.get("dependOnProject").toString());
                    queryParam.getParams().put("dependOnProject", new Parameter(QueryParam.EQ, dependOnProject));
                }
            } else {
                if (null != conditionMap.get("orgId")) {
                    orgId = Long.valueOf(conditionMap.get("orgId").toString());
                }

                // 移动端收票登记，不属于项目，参照集采
                if (conditionMap.containsKey("isJc") && conditionMap.get("isJc").equals("1")) {
                    queryParam.getParams().put("purchase_mode", new Parameter(QueryParam.EQ, "1"));
                }

                if (null != conditionMap.get("supplierId")) {
                    queryParam.getParams().put("supplier_id", new Parameter(QueryParam.EQ, Long.valueOf(conditionMap.get("supplierId").toString())));
                }
                if (StringUtils.isNotBlank(searchObject)) {
                    JSONObject search = JSONObject.parseObject(searchObject);
                    for (String key : search.keySet()) {
                        queryParam.getParams().put(key, new Parameter(QueryParam.LIKE, search.get(key).toString()));
                    }
                }
                queryParam.getComplexParams().add(getPageQueryParam(orgId));
            }
            if (null != conditionMap.get("projectId")) {
                /** 项目关联合同以及集采合同 */
                Long projectId = Long.valueOf(conditionMap.get("projectId").toString());
                ComplexParam in = new ComplexParam();
                in.setLogic(ComplexParam.AND);

                ComplexParam c = new ComplexParam();
                c.setLogic(ComplexParam.AND);
                c.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));

                ComplexParam p = new ComplexParam();
                p.setLogic(ComplexParam.OR);
                Long orgId1 = InvocationInfoProxy.getOrgId();
                List<Long> collect = orgApi.findParentsByOrgId(orgId1).getData().stream().map(e -> e.getId()).collect(Collectors.toList());
                p.getParams().put("orgId", new Parameter(QueryParam.IN, collect));
                ComplexParam t = new ComplexParam();
                t.setLogic(ComplexParam.AND);
                t.getParams().put("purchaseMode", new Parameter(QueryParam.EQ, 1));
                in.getComplexParams().add(c);
                in.getComplexParams().add(p);
                in.getComplexParams().add(t);
                queryParam.getComplexParams().add(in);
            } else {
                queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
            }
        } else {
            if (StringUtils.isNotBlank(searchObject)) {
                JSONObject search = JSONObject.parseObject(searchObject);
                for (String key : search.keySet()) {
                    queryParam.getParams().put(key, new Parameter(QueryParam.LIKE, search.get(key).toString()));
                }
            }
            queryParam.getComplexParams().add(getPageQueryParam(orgId));
        }
        IPage<MaterialContractEntity> pageData = materialContractService.queryPage(queryParam, false);
        for (MaterialContractEntity materialContractEntity : pageData.getRecords()) {
            materialContractEntity.setContractNameExtend(materialContractEntity.getCode()+"-"+materialContractEntity.getName()+"-"+materialContractEntity.getSupplierName()+"-"+ComputeUtil.scale(materialContractEntity.getAmountWithTax(),2));
        }
        return CommonResponse.success("查询合同列表成功！", pageData);
    }

    /**
     * 合同状态修改
     *
     * @param materialContractVO
     * @return
     */
    @PostMapping("changeState")
    public CommonResponse<String> changeState(@RequestBody MaterialContractVO materialContractVO) {
        Long billId = materialContractVO.getId();
        String state = materialContractVO.getState();
        MaterialContractEntity contract = materialContractService.selectById(billId);
        if (ContractStateEnum.已作废.getCode().equals(state)) {
            CommonResponse<String> resp = billTypeApi.checkQuote("BT200608000000001", billId);
            if (!resp.isSuccess()) {
                return CommonResponse.error("单据已被下游引用不能作废：" + resp.getMsg());
            }
        }
        contract.setState(state);
        contract.setStateName(materialContractVO.getStateName());
        materialContractService.saveOrUpdate(contract, false);

        return CommonResponse.success("修改合同状态成功！");
    }

    /**
     * 合同编码重复校验
     *
     * @return
     */
    @PostMapping("codeCheck")
    public CommonResponse<String> checkContractCode(@RequestBody MaterialContractVO contract) {
        boolean result = materialContractService.codeCheck(contract.getId(), contract.getCode());
        if (!result) {
            return CommonResponse.error("编码重复，请重新输入！");
        }
        return CommonResponse.success("编码可以使用！");
    }

    /**
     * 查询合同收票统计记录
     *
     * @param id 合同ID
     * @return
     */
    @RequestMapping("invoiceRecRecords")
    public CommonResponse<JSONObject> queryContractInvoiceRecRecords(@RequestParam(value = "id") Long id) {
        JSONObject data = new JSONObject();

//        QueryParam queryParam = new QueryParam();
//        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, id));
//        queryParam.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
//        MaterialContractEntity contract = materialContractService.selectById(id);
//        if(!MaterialContractVO.CONTRACT_BELONG_PROJECT.equals(contract.getDependOnProject())) {
//            // 集采查询组织本下审批通过、已提交的结算
//            Long orgId = InvocationInfoProxy.getOrgId();
//            CommonResponse<List<OrgVO>> orgResp = orgApi.findChildrenByParentId(orgId);
//            if(!orgResp.isSuccess()) {
//                return CommonResponse.error("查询合同收票记录失败，获取组信息失败！");
//            }
//            List<Long> orgIds = orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
//            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgIds));
//        }

        //累计付款金额
        BigDecimal totalPayAmount = BigDecimal.ZERO.setScale(8);
        CommonResponse<SumPayMnyVO> payResp = payContractApi.getSumPayMnyVOList(id, InvocationInfoProxy.getOrgId());
        if (!payResp.isSuccess()) {
            return CommonResponse.error("查询合同收票记录失败，获合同付款信息失败！");
        }
        SumPayMnyVO payRecords = payResp.getData();
        if (null != payRecords.getSumPayMny()) {
            totalPayAmount = payRecords.getSumPayMny();
        }

        //累计结算金额
        BigDecimal totalSettlementAmount = getTotalSettleAmount(id);
        //欠收发票金额
        BigDecimal dueInInvoiceAmount = null;

        CommonResponse<InvoiceReceiveRecordVO> records = iInvoiceApi.getInvoiceReceiveRecord(id);
        if (!records.isSuccess()) {
            return CommonResponse.error("查询合同收票统计记录失败！");
        }
        InvoiceReceiveRecordVO irr = records.getData();

        dueInInvoiceAmount = MathUtil.safeSub(totalPayAmount, irr.getInvoiceTaxMny());

        BigDecimal settleNoRecMny = MathUtil.safeSub(totalSettlementAmount,irr.getInvoiceTaxMny());
        //合同Id
        data.put("contractId", id);
        //累计税金
        data.put("totalTaxAmount", irr.getTaxMny());
        //累计收票金额（含税）
        data.put("totalInvoiceRecTaxAmount", irr.getInvoiceTaxMny());
        //累计收票金额
        data.put("totalInvoiceRecAmount", irr.getInvoiceMny());
        //收票记录列表
        data.put("invoiceRecRecords", irr.getInvoiceReceiveVOListAllState());
        //累计结算金额
        data.put("cumulativePayAmount", totalSettlementAmount);
        //累计付款金额
        data.put("totalPayAmount", totalPayAmount);
        //欠收发票金额
        data.put("dueInInvoiceAmount", dueInInvoiceAmount);
        //已结算未收票金额
        data.put("settleNoRecMny",settleNoRecMny);
        return CommonResponse.success("查询合同收票记录成功！", data);
    }

    /**
     * 合同付款记录查询
     *
     * @param id 合同Id
     * @return
     */
    @GetMapping("paymentRecords")
    public CommonResponse<JSONObject> queryPaymentRecords(@RequestParam(value = "id") Long id) {
        JSONObject data = new JSONObject();

        MaterialContractEntity contractEntity = materialContractService.selectById(id);

        //累计付款金额
        BigDecimal totalPayAmount = BigDecimal.ZERO.setScale(8);
        CommonResponse<SumPayMnyVO> payResp = payContractApi.getSumPayMnyVOList(id, InvocationInfoProxy.getOrgId());
        if (!payResp.isSuccess()) {
            return CommonResponse.error("查询合同付款记录失败，获合同付款信息失败！");
        }
        SumPayMnyVO payRecords = payResp.getData();
        if (null != payRecords.getSumPayMny()) {
            totalPayAmount = payRecords.getSumPayMny();
        }

        //累计结算金额
        BigDecimal totalSettlementAmount = getTotalSettleAmount(id);

        //合同Id
        data.put("contractId", id);
        //累计付款金额
        data.put("totalPayAmount", totalPayAmount);
        //累计结算金额
        data.put("totalSettlementAmount", totalSettlementAmount);
        //约定支付比例
        data.put("aggreedPaymentRatio", contractEntity.getAggreedPaymentRatio());
        data.put("sumApplyMny",payResp.getData().getSumApplyMny()==null?BigDecimal.ZERO: payResp.getData().getSumApplyMny());
        //约定支付金额
        data.put("aggreedPaymentAmount", totalSettlementAmount.multiply(new BigDecimal((null != contractEntity.getAggreedPaymentRatio() ? contractEntity.getAggreedPaymentRatio() : 100) / 100).setScale(8, BigDecimal.ROUND_HALF_UP)));
        //累计已付比例
        Double totalPayRatio = 0d;
        if (totalSettlementAmount.compareTo(BigDecimal.ZERO.setScale(8)) != 0) {
            totalPayRatio = totalPayAmount.divide(totalSettlementAmount, 8, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(100).setScale(8)).doubleValue();
        }
        data.put("totalPayRatio", totalPayRatio);
        data.put("payRecords", payRecords.getContractVOList());

        return CommonResponse.success("查询合同付款记录成功！", data);
    }



    /**
     * 根据主键ID查询付款记录
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryPayDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<RentContractPayVO> queryPayDetail(@RequestParam Long id) {
        MaterialContractEntity entity = materialContractService.selectById(id);
        RentContractPayVO rentContractPayVO = BeanMapper.map(entity, RentContractPayVO.class);
        CommonResponse<SumPayMnyVO> sumPayMnyVOList = payContractApi.getSumPayMnyVOList(id, null);
        if(!sumPayMnyVOList.isSuccess()) {
            throw new BusinessException("获取付款信息失败！");
        }
        BigDecimal sumPayMny = sumPayMnyVOList.getData().getSumPayMny()==null?BigDecimal.ZERO:sumPayMnyVOList.getData().getSumPayMny();
        BigDecimal sumDeductionMny = sumPayMnyVOList.getData().getSumDeductionMny();
        rentContractPayVO.setContractId(sumPayMnyVOList.getData().getContractId());
        rentContractPayVO.setSumPayMny(sumPayMny);
        rentContractPayVO.setSumDeductionMny(sumDeductionMny);
        rentContractPayVO.setContractVOList(sumPayMnyVOList.getData().getContractVOList());
        rentContractPayVO.setSumApplyMny(sumPayMnyVOList.getData().getSumApplyMny()==null?BigDecimal.ZERO: sumPayMnyVOList.getData().getSumApplyMny());
        rentContractPayVO.setContractTaxMny(entity.getAmountWithTax());
        //累计结算金额
        BigDecimal sumSettlementTaxMny = getTotalSettleAmount(id);
        rentContractPayVO.setSumSettlementTaxMny(sumSettlementTaxMny);
        //获取累计收票
        CommonResponse<InvoiceReceiveRecordVO>  res = invoiceApi.getInvoiceReceiveRecord(id);
        if(!res.isSuccess()) {
            throw new BusinessException("获取收票信息失败！");
        }
        rentContractPayVO.setSumInvoiceTaxMny(res.getData().getInvoiceTaxMny());

        //累计未付金额 = 累计结算金额 - 累计付款金额 - 累计扣款金额
        rentContractPayVO.setSumUnPayMny(ComputeUtil.safeSub(sumSettlementTaxMny,sumPayMny, sumDeductionMny));
        boolean freeFlag = false;
        //查询是否存在预付款
        for (PayContractVO vo :sumPayMnyVOList.getData().getContractVOList()) {
            if (vo.getFeeType()==1){
                freeFlag = true;
                break;
            }
        }
        if (!freeFlag){
            QueryWrapper<PurchaseSettlementEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.select("IFNULL(SUM(offset_amount_in_current_period),0) as offsetAmountInCurrentPeriod");
            queryWrapper.eq("contract_id",id);
            queryWrapper.in("bill_state", new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()});
            queryWrapper.in("org_id",orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList()) );
            PurchaseSettlementEntity one = purchaseSettlementService.getOne(queryWrapper);
            rentContractPayVO.setSumUnPayMny(MathUtil.safeSub(rentContractPayVO.getSumUnPayMny(),one.getOffsetAmountInCurrentPeriod()));
        }
        return CommonResponse.success(rentContractPayVO);
    }

    private BigDecimal getTotalSettleAmount(Long contractId) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
        MaterialContractEntity contract = materialContractService.selectById(contractId);
        if (!MaterialContractVO.CONTRACT_BELONG_PROJECT.equals(contract.getDependOnProject())) {
            // 集采查询组织本下审批通过、已提交的结算
            Long orgId = InvocationInfoProxy.getOrgId();
            CommonResponse<List<OrgVO>> orgResp = orgApi.findChildrenByParentId(orgId);
            if (!orgResp.isSuccess()) {
                throw new BusinessException("查询合同收票记录失败，获取组信息失败！");
            }
            List<Long> orgIds = orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgIds));
        }

        return purchaseSettlementService.calculateTotalSettlement(queryParam);
    }

    @GetMapping("getOutContractTotalAmount")
    public CommonResponse<JSONObject> getOutContractTotalAmount(@RequestParam Integer type,@RequestParam(required = false) Long engineeringType, HttpServletRequest req) {
        request = req;
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));//租户隔离
        queryParam.getParams().put("dependOnProject", new Parameter(QueryParam.EQ, "1"));//属于项目
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));//单据状态已提交和审批通过
        List<Long> orgIds = new ArrayList<>();
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            orgIds = authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        } else {
            orgIds = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        }
        List<Long> projectIds = new ArrayList<>();
        if (null!=type) {//全部项目的合同
            CommonResponse<List<Long>> commonResponse = iProjectApi.getProjectIdsByEngineeringType(type,null==engineeringType?null:engineeringType);
            if (commonResponse.isSuccess()) {
                projectIds = commonResponse.getData();
                if (ListUtil.isNotEmpty(projectIds)) {
                    queryParam.getParams().put("projectId", new Parameter(QueryParam.IN, projectIds));
                } else {
                    JSONObject back = new JSONObject();
                    back.put("contract", "0.00");
                    back.put("settle", "0.00");
                    back.put("invoiceTaxMny", "0.00");
                    back.put("payApplyAmount", "0.00");
                    return CommonResponse.success(back);
                }
            }
        }
        queryParam.getParams().put("org_id", new Parameter(QueryParam.IN, orgIds));
        List<MaterialContractEntity> contractEntities = materialContractService.queryList(queryParam, false);
        queryParam.getParams().remove("dependOnProject");
        queryParam.getParams().put("belongToProject", new Parameter(QueryParam.EQ, "1"));//属于项目
        List<PurchaseSettlementEntity> settlementEntities = purchaseSettlementService.queryList(queryParam, false);
        BigDecimal contract = new BigDecimal("0.00");
        BigDecimal settle = new BigDecimal("0.00");
        BigDecimal invoiceTaxMny = new BigDecimal("0.00");
        BigDecimal payApplyAmount = new BigDecimal("0.00");
        if (ListUtil.isNotEmpty(contractEntities)) {
            for (MaterialContractEntity c : contractEntities) {
                contract = ComputeUtil.safeAdd(contract, c.getAmountWithTax());
            }
        }
        if (ListUtil.isNotEmpty(settlementEntities)) {
            for (PurchaseSettlementEntity c : settlementEntities) {
                settle = ComputeUtil.safeAdd(settle, c.getCurrentSettlementAmountTax());
            }
        }
        queryParam.getParams().remove("belongToProject");
        CommonResponse<BigDecimal> sub = subReportApi.querySubSettleMoneyByTypeAndProjectIds(queryParam);
        logger.info("移动首页，查询支出结算，分包结算：参数：{}，结果：{}", JSONObject.toJSONString(queryParam), JSONObject.toJSONString(sub));
        if (sub.isSuccess()) {
            settle = settle.add(sub.getData());
        }
        sub = settlementApi.querySubSettleMoneyByTypeAndProjectIds(queryParam);
        logger.info("移动首页，查询支出结算，设备采购和租赁结算：参数：{}，结果：{}", JSONObject.toJSONString(queryParam), JSONObject.toJSONString(sub));
        if (sub.isSuccess()) {
            settle = settle.add(sub.getData());
        }
        JSONObject back = getOtherContractAmount(projectIds);
        contract = contract.add(back.getBigDecimal("contract")).divide(new BigDecimal("10000")).setScale(2, BigDecimal.ROUND_HALF_UP);
        settle = settle.add(back.getBigDecimal("settle")).divide(new BigDecimal("10000")).setScale(2, BigDecimal.ROUND_HALF_UP);
        invoiceTaxMny = invoiceTaxMny.add(back.getBigDecimal("invoiceTaxMny")).divide(new BigDecimal("10000")).setScale(2, BigDecimal.ROUND_HALF_UP);
        payApplyAmount = payApplyAmount.add(back.getBigDecimal("payApplyAmount")).divide(new BigDecimal("10000")).setScale(2, BigDecimal.ROUND_HALF_UP);
        back = new JSONObject();
        back.put("contract", contract);
        back.put("settle", settle);
        back.put("invoiceTaxMny", invoiceTaxMny);
        back.put("payApplyAmount", payApplyAmount);
        return CommonResponse.success(back);
    }

    private JSONObject getOtherContractAmount(List<Long> projectIds) {
        //创建一个固定数量的线程池
        ExecutorService threadPool = Executors.newFixedThreadPool(4);
        BigDecimal contract = new BigDecimal("0.00");
        BigDecimal settle = new BigDecimal("0.00");
        BigDecimal invoiceTaxMny = new BigDecimal("0.00");
        BigDecimal payApplyAmount = new BigDecimal("0.00");
        Callable<JSONObject> equipmentContractAmountNum = new PostDataCallable(BASE_HOST + "ejc-equipment-web/purchaseContract/getEquipmentContractTotalAmount", projectIds);
        Future<JSONObject> future1 = threadPool.submit(equipmentContractAmountNum);
        Callable<JSONObject> subContractAmountNum = new PostDataCallable(BASE_HOST + "ejc-sub-web/contract/getSubContractTotalAmount", projectIds);
        Future<JSONObject> future2 = threadPool.submit(subContractAmountNum);
        Callable<JSONObject> tax = new PostDataCallable(BASE_HOST + "ejc-tax-web/invoiceReceive/getTotalInvoiceTaxMnyByProjectIds", projectIds);
        Future<JSONObject> future3 = threadPool.submit(tax);
        Callable<JSONObject> finance = new PostDataCallable(BASE_HOST + "ejc-finance-web/payContract/getTotalPayApplyAmountByProjectIds", projectIds);
        Future<JSONObject> future4 = threadPool.submit(finance);
        try {
            JSONObject c1 = future1.get();
            if (c1.getInteger("code") == 0) {
                JSONObject data = c1.getJSONObject("data");
                contract = contract.add(data.getBigDecimal("contract"));
                settle = settle.add(data.getBigDecimal("settle"));
            }
            JSONObject c2 = future2.get();
            if (c1.getInteger("code") == 0) {
                JSONObject data = c2.getJSONObject("data");
                contract = contract.add(data.getBigDecimal("contract"));
                settle = settle.add(data.getBigDecimal("settle"));
            }
            JSONObject c3 = future3.get();
            if (c3.getInteger("code") == 0) {
                JSONObject data = c3.getJSONObject("data");
                invoiceTaxMny = invoiceTaxMny.add(data.getBigDecimal("invoiceTaxMny"));
            }
            JSONObject c4 = future4.get();
            if (c4.getInteger("code") == 0) {
                JSONObject data = c4.getJSONObject("data");
                payApplyAmount = payApplyAmount.add(data.getBigDecimal("payApplyAmount"));
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            threadPool.shutdown();
        }
        CommonResponse<RmatProjectReportVo> rmatCommonResponse = rmatContractApi.getContractMnyByProjectIds(projectIds);
        if (rmatCommonResponse.isSuccess()) {
            BigDecimal mny = rmatCommonResponse.getData().getMny();
            BigDecimal settleMny = rmatCommonResponse.getData().getSettleMny();
            contract = ComputeUtil.safeAdd(contract, mny);
            settle = ComputeUtil.safeAdd(settle, settleMny);
        }
        CommonResponse<OtherProjectReportVo> otherResponse = otherContractApi.getContractMnyByProjectIds(projectIds);
        if (otherResponse.isSuccess()) {
            BigDecimal mny = otherResponse.getData().getMny();
            BigDecimal settleMny = otherResponse.getData().getSettleMny();
            contract = ComputeUtil.safeAdd(contract, mny);
            settle = ComputeUtil.safeAdd(settle, settleMny);
        }


        JSONObject back = new JSONObject();
        back.put("contract", contract);
        back.put("settle", settle);
        back.put("invoiceTaxMny", invoiceTaxMny);
        back.put("payApplyAmount", payApplyAmount);
        return back;
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/9/22
     * @Description
     * @Param
     * @Return
     */
    class PostDataCallable implements Callable<JSONObject> {
        private String url;
        private Object data;

        public PostDataCallable(String url, Object data) {
            this.data = data;
            this.url = url;
        }

        @Override
        public JSONObject call() throws Exception {
            System.out.println(InvocationInfoProxy.getTenantid());
            String responseStr = ReferHttpClientUtils.postByJson(url, JSON.toJSONString(data), request);
            System.out.println(responseStr);
            return JSONObject.parseObject(responseStr);
        }

    }

    /**
     * 统计匹配条件的（属于项目的）物资合同含税现合同金额总值(单位：万元）
     *
     * @param param projectIds: 指定所属的项目的id列表
     * @return
     */
    @PostMapping(value = "analysisContract")
    public CommonResponse<BigDecimal> analysisContract(@RequestBody Map<String, Object> param) {
        QueryParam queryParam = new QueryParam();
        //查询本租户下的
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.getParams().put("dr", new Parameter(QueryParam.EQ, BaseVO.DR_UNDELETE));
        //属于项目
        queryParam.getParams().put("dependOnProject", new Parameter(QueryParam.EQ, 1));
        //已审批生效的
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN,
                Arrays.asList(new Integer[]{BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()})));
        //在指定项目范围内的
        if (null != param && null != param.get("projectIds") && CollectionUtils.isNotEmpty((List<Long>) param.get("projectIds"))) {
            queryParam.getParams().put("projectId", new Parameter(QueryParam.IN, param.get("projectIds")));
        }
        List<Long> orgIds = new ArrayList<>();
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            orgIds = authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        } else {
            orgIds = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        }
        if (CollectionUtils.isNotEmpty(orgIds)) {
            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgIds));
        }
        Map<String, Object> result = materialContractService.countContractAmount(queryParam);
        String dataStr = null != result.get("curAmount") ? result.get("curAmount").toString() : "0";
        BigDecimal total = new BigDecimal(dataStr);

        return CommonResponse.success(total.divide(new BigDecimal(10000), 2, BigDecimal.ROUND_HALF_UP));
    }

    /**
     * 物资采购合同明细统计查询参数
     * @return
     */
    @RequestMapping(value = "getMaterialDetailQueryParam", method=RequestMethod.GET)
    public CommonResponse<QueryParam> getMaterialDetailQueryParam() {
        try {
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN,orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
            String orgType = InvocationInfoProxy.getOrgType();
            if ("5".equals(orgType)){
                queryParam.getParams().put("dependOnProject", new Parameter(QueryParam.EQ,1));
            }
            return CommonResponse.success(queryParam);
        } catch (Exception e) {
            logger.error("系统异常："+e.getMessage());
            e.printStackTrace();
        }
        return CommonResponse.error("查询失败");
    }

    /**
     * @description: 根据协议合同查询协议价
     *
     * @return {@link CommonResponse< ParamsCheckVO>}
     * @author yqls
     * @date: 2024/10/12
     */
    @PostMapping("queryPriceByContractId")
    public CommonResponse<List<MaterialContractPriceResultVO>> queryPriceByContractId(@RequestBody MaterialPriceVO vo) {
        Long contractId = vo.getContractId();
        List<String> materialIds = vo.getDetail().stream().map(x->String.valueOf(x.getMaterialId())).collect(Collectors.toList());
        MaterialContractEntity entity = materialContractService.selectById(contractId);
        List<MaterialContractDetailSubEntity> detailList = entity.getMaterialDetailList();
        Map<String, MaterialContractDetailSubEntity> detailMap = detailList.stream().filter(x->materialIds.contains(x.getMaterialId())).
                collect(Collectors.toMap(x->x.getMaterialId(), x->x, (v1, v2)->v1));
        List<MaterialContractPriceResultVO> result = new ArrayList<>();
        for(MaterialContractDetailSubEntity detail : detailMap.values()){
            MaterialContractPriceResultVO data = new MaterialContractPriceResultVO();
            data.setId(entity.getId());
            data.setName(entity.getName());
            data.setMaterialId(Long.valueOf(detail.getMaterialId()));
            data.setPrice(detail.getUnitPrice());
            result.add(data);
        }
        return CommonResponse.success("查询物资合同价成功！", result);
    }

}

