package com.ejianc.business.material.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.vo.ParamsCheckVO;
import com.ejianc.business.material.bean.MaterialContractEntity;
import com.ejianc.business.material.bean.MaterialSupplementEntity;
import com.ejianc.business.material.bean.PurchaseSettlementEntity;
import com.ejianc.business.material.service.IMaterialContractService;
import com.ejianc.business.material.service.IPurchaseSettlementMaterialDetailService;
import com.ejianc.business.material.service.IPurchaseSettlementOtherFeeService;
import com.ejianc.business.material.service.IPurchaseSettlementService;
import com.ejianc.business.material.vo.*;
import com.ejianc.business.utils.NCUtil;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.*;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author mrsir_wxp
 * @Date 2020/6/6 采购结算控制器
 * @Description PurchaseSettlementController
 */
@Controller
@RequestMapping("/purchaseSettlement/")
public class PurchaseSettlementController implements Serializable {

    private static final long serialVersionUID = 2779265683203439571L;
    private static final String BILL_CODE = "purchaseSettlement";

    @Autowired
    IPurchaseSettlementService iPurchaseSettlementService;
    @Autowired
    IPurchaseSettlementOtherFeeService iPurchaseSettlementOtherFeeService;
    @Autowired
    IPurchaseSettlementMaterialDetailService iPurchaseSettlementMaterialDetailService;
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi iOrgApi;
    @Autowired
    private IMaterialContractService materialContractService;

    /**
     * @Author mrsir_wxp
     * @Date 2020/6/1  保存修改
     * @Description saveOrUpdate
     * @Param [vo]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.material.vo. PurchaseSettlementVO>
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PurchaseSettlementVO> saveOrUpdate(@RequestBody PurchaseSettlementVO vo) {
        beforeSave(vo);
        PurchaseSettlementEntity entity = BeanMapper.map(vo, PurchaseSettlementEntity.class);
        List<PurchaseSettlementEntity>  entities = getHasPurchased(entity.getContractId(),entity.getOrgId());
        if(entity.getId() == null || entity.getId() == 0){
            if(ListUtil.isNotEmpty(entities)){
                throw new BusinessException("所选合同和组织存在未完成的结算单，不可再次新增结算单，如需新增，请先完成单据编号为【"+entities.get(0).getBillCode()+"】的结算单。");
            }
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(BILL_CODE, InvocationInfoProxy.getTenantid());
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }else {
            if(ListUtil.isNotEmpty(entities)){
                if(entities.size()==1){
                    if(!Objects.equals(entities.get(0).getId(),entity.getId())){
                        throw new BusinessException("不能根据所选合同和组织修改保存该结算单，因为存在相同合同和组织未完成的结算单【"+entities.get(0).getBillCode()+"】，如确须修改，请先完成单据编号为【"+entities.get(0).getBillCode()+"】的结算单再修改。");
                    }
                }
            }
        }
        MaterialContractEntity contractEntity = materialContractService.selectById(vo.getContractId());
        //判断集采合同
        if("1".equals(contractEntity.getPurchaseMode())&&vo.getSettlementType()==1){
            return CommonResponse.error("集采合同不能做最终结算!");
        }
        //判断未生效的补充协议
        List<MaterialSupplementEntity> supplementEntityList = materialContractService.querySupplement(contractEntity.getId());
        for (MaterialSupplementEntity supplementEntity : supplementEntityList) {
            if(!(supplementEntity.getBillState() ==1 || supplementEntity.getBillState() ==3)){
                return CommonResponse.error("含有未生效的补充协议，不允许结算!");
            }
        }
        //判断合同状态
        if(contractEntity.getChangeState()==1 && vo.getSettlementType() == 1){
            return CommonResponse.error("合同处于变更中，不允许最终结算!");
        }
        //结算类型判断
        if (null != vo.getId() && !"".equals(vo.getId())){
            // 最终结算
            if (iPurchaseSettlementService.queryById(vo.getId(),vo.getContractId()) > 0) {
                return CommonResponse.error("合同存在最终结算!");
            }
        }else {
            if (iPurchaseSettlementService.queryByContractId(vo.getContractId()) > 0) {
                return CommonResponse.error("合同存在最终结算!");
            }
        }
        
        materialContractService.checkContract(entity.getContractId());
        entity.setRemainCanApplyAmount(entity.getCurrentSettlementAmountTax().subtract(entity.getOffsetAmountInCurrentPeriod()).setScale(2,BigDecimal.ROUND_HALF_UP));
        materialContractService.updateSettlementType(vo.getContractId(),vo.getSettlementType());
        iPurchaseSettlementService.saveOrUpdate(entity,false);
        PurchaseSettlementVO backVo = BeanMapper.map(entity, PurchaseSettlementVO.class);
        return CommonResponse.success("保存或修改单据成功！",backVo);
    }

    private void beforeSave(PurchaseSettlementVO vo){
        if(vo.getTaxRate() == null){
            vo.setTaxRate(new BigDecimal("0.00"));
        }
        if(ListUtil.isNotEmpty(vo.getFeeEntities())){
            vo.getFeeEntities().forEach(sub->sub.setTaxRate(vo.getTaxRate()));
        }
        if(ListUtil.isNotEmpty(vo.getMaterialDetails())){
            vo.getMaterialDetails().forEach(sub->sub.setTaxRate(vo.getTaxRate()));
        }
    }


    /***
     * @Author mrsir_wxp
     * @Date 2020/12/22 保存 关联 领料退库使用
     * @Description pushToCost
     * @Param [vo]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.material.vo.InstoreVO>
     */
    @RequestMapping(value = "pushToCost", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<PurchaseSettlementVO> pushToCost(@RequestBody PurchaseSettlementVO vo) {
        beforeSave(vo);
        PurchaseSettlementEntity entity = BeanMapper.map(vo, PurchaseSettlementEntity.class);
        if(vo.getBillState()==BillStateEnum.COMMITED_STATE.getBillStateCode()||vo.getBillState()==BillStateEnum.PASSED_STATE.getBillStateCode()){
            iPurchaseSettlementService.processCost(entity);
        }
        iPurchaseSettlementService.saveOrUpdate(entity,false);
        PurchaseSettlementVO backVo = BeanMapper.map(entity, PurchaseSettlementVO.class);
        return CommonResponse.success("保存关联成功！",backVo);
    }


    /**
     * @Author mrsir_wxp
     * @Date 2020/6/22 根据合同和组织查询未完成的采购结算单
     * @Description getHasPurchased
     * @Param [contractId, orgId]
     * @Return java.util.List<com.ejianc.business.material.bean.PurchaseSettlementEntity>
     */
    public List<PurchaseSettlementEntity> getHasPurchased(Long contractId,Long orgId){
        QueryParam param = new QueryParam();
        param.getParams().put("contractId",new Parameter(QueryParam.EQ,contractId));
        param.getParams().put("orgId",new Parameter(QueryParam.EQ,orgId));
        param.getParams().put("billState",new Parameter(QueryParam.IN, Arrays.asList(
                BillStateEnum.UNCOMMITED_STATE.getBillStateCode(),
                BillStateEnum.APPROVING_UNEXAM_STATE.getBillStateCode(),
                BillStateEnum.APPROVING_HAS_STATE.getBillStateCode()
        )));
        return  iPurchaseSettlementService.queryList(param,false);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/6/2 查询详情
     * @Description queryDetail
     * @Param [id]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.material.vo. PurchaseSettlementVO>
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse< PurchaseSettlementVO> queryDetail(Long id) {
        PurchaseSettlementEntity entity = iPurchaseSettlementService.selectById(id);
        PurchaseSettlementVO vo = null;
        if(entity != null){
            vo = BeanMapper.map(entity, PurchaseSettlementVO.class);
            //根据单据参数控制查询总计划量\价,历史价格区间
            List<PurchaseSettlementMaterialDetailVO> materialDetailList = vo.getMaterialDetails();


            if (CollectionUtils.isNotEmpty(materialDetailList)) {
                MaterialPriceVO priceVO = new MaterialPriceVO(entity.getProjectId());
                List<MaterialPriceVO> detail = new ArrayList<>();
                materialDetailList.forEach(item -> {
                    MaterialPriceVO materialPriceVO = new MaterialPriceVO();
                    if(item.getMaterialId() == null){
                        return;
                    }
                    materialPriceVO.setMaterialId(Long.valueOf(item.getMaterialId()));
                    detail.add(materialPriceVO);
                });
                priceVO.setDetail(detail);
                Integer isJc = entity.getBelongToProject().equals("0")  ? 1 : 0;
                MaterialPriceVO materialPriceVO = materialContractService.queryPrice(isJc, entity.getContractId(), priceVO);
                List<MaterialPriceVO> dd = materialPriceVO.getDetail();
                if (CollectionUtils.isNotEmpty(dd)) {
                    Map<Long, MaterialPriceVO> priceVOMap = dd.stream().collect(Collectors.toMap(MaterialPriceVO::getMaterialId, account -> account,(v1, v2) -> v2));
                    for (PurchaseSettlementMaterialDetailVO v : materialDetailList) {
                        Long materialId = v.getMaterialId();
                        MaterialPriceVO p = priceVOMap.get(materialId);
                        if (null != p) {
                            v.setPlanNum(p.getPlanNum());
                            v.setPlanPrice(p.getPlanPrice());
                            v.setMinPrice(p.getMinPrice());
                            v.setMaxPrice(p.getMaxPrice());
                            v.setPriceArea(p.getMinPrice().setScale(2,BigDecimal.ROUND_HALF_UP) + "~" + p.getMaxPrice().setScale(2,BigDecimal.ROUND_HALF_UP));
                        }
                    }
                }
            }
        }
        return CommonResponse.success("查询详情数据成功！",vo);
    }

    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage< PurchaseSettlementVO>> queryList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("orgName");
        fuzzyFields.add("purchaseMethod");
        fuzzyFields.add("creatorName");
        fuzzyFields.add("projectName");
        fuzzyFields.add("projectCode");

        /** 数据隔离 本下 */
        param.getParams().put("orgId",new Parameter(QueryParam.IN,iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        // 是否生成凭证
        NCUtil.voucherParam(param);
        return CommonResponse.success("查询列表数据成功！",iPurchaseSettlementService.queryForList(param,false));
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/6/2 查询打印数据
     * @Description queryPrint
     * @Param [id]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.alibaba.fastjson.JSONObject>
     */
    @RequestMapping(value = "/queryPrint", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryPrint(Long id) {
        JSONObject json = new JSONObject();
        json.put("PurchaseSettlement",iPurchaseSettlementService.selectById(id));
        return CommonResponse.success("查询打印数据成功！",json);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/6/2 删除
     * @Description delete
     * @Param [vos]
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List< PurchaseSettlementVO> vos) {
        for (PurchaseSettlementVO vo : vos) {
            PurchaseSettlementEntity purchaseSettlementEntity = iPurchaseSettlementService.selectById(vo.getId());
            materialContractService.updateSettlementType(purchaseSettlementEntity.getContractId(),purchaseSettlementEntity.getSettlementType()==1?0:purchaseSettlementEntity.getSettlementType());
        }
        iPurchaseSettlementService.removeByIds(vos.stream().map( PurchaseSettlementVO::getId).collect(Collectors.toList()),true);
        return CommonResponse.success("删除成功！");
    }


    /**
     * @Author mrsir_wxp
     * @Date 2020/6/2 导出
     * @Description excelExport
     * @Param [queryParam, response]
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.setPageIndex(0);
        queryParam.setPageSize(10000);
        List<String> fuzzyFields = queryParam.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("orgName");
        fuzzyFields.add("creatorName");
        fuzzyFields.add("purchaseMethod");

        /** 数据隔离 本下 */
        queryParam.getParams().put("orgId",new Parameter(QueryParam.IN,iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        // 是否生成凭证
        NCUtil.voucherParam(queryParam);
        IPage< PurchaseSettlementVO>  voPage = iPurchaseSettlementService.queryForList(queryParam,false);
        List< PurchaseSettlementVO> list = voPage.getRecords();
        List<PurchaseSettlementExportVO> exportVOS = new ArrayList<>();
        list.forEach(vo -> {
            PurchaseSettlementExportVO exportVO = BeanMapper.map(vo, PurchaseSettlementExportVO.class);
            exportVO.setCreateTime(DateFormatUtil.formatDate("yyyy-MM-dd HH:mm:ss",vo.getCreateTime()));
            if(vo.getSettlementDate()!=null){
                exportVO.setSettlementDate(DateFormatUtil.formatDate("yyyy-MM-dd",vo.getSettlementDate()));
                exportVO.setBillState(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            }
            exportVOS.add(exportVO);
        });
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", exportVOS);
        ExcelExport.getInstance().export("purchaseSettlementListExport.xlsx", beans, response);
    }

    /**
     * 查询用户当前组织本下的指定合同的所有结算记录统计
     *
     * @param id 物资合同Id
     * @return
     */
    @GetMapping("/querySettlementRecords")
    @ResponseBody
    public CommonResponse<JSONObject> querySettlementRecords(@RequestParam(value = "id") Long id) {
        JSONObject resp = new JSONObject();
        List<PurchaseSettlementVO> changeRecords = new ArrayList<>();
        MaterialContractEntity contract = materialContractService.selectById(id);
        BigDecimal totalSettlementAmount = new BigDecimal(0).setScale(8, BigDecimal.ROUND_HALF_UP);
        BigDecimal settlementRatio = new BigDecimal(0).setScale(8, BigDecimal.ROUND_HALF_UP);

        Long orgId = InvocationInfoProxy.getOrgId();
        CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentId(orgId);
        if(!orgResp.isSuccess()) {
            return CommonResponse.error("查询合同结算记录失败，获取组信息失败！");
        }
        List<Long> orgIds = orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());

        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, id));
        queryParam.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));

        //集采查询组织本下审批通过、已提交的结算
        if(!MaterialContractVO.CONTRACT_BELONG_PROJECT.equals(contract.getDependOnProject())) {
            queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgIds));
            //新增结算使用
            resp.put("contractId", id);
        }

        List<PurchaseSettlementEntity> settleList = iPurchaseSettlementService.queryList(queryParam, false);
        if(CollectionUtils.isNotEmpty(settleList)) {
            changeRecords = BeanMapper.mapList(settleList, PurchaseSettlementVO.class);
            totalSettlementAmount = iPurchaseSettlementService.calculateTotalSettlement(queryParam);
        }

        if(MaterialContractVO.CONTRACT_BELONG_PROJECT.equals(contract.getDependOnProject())) {
            //自采查询本下未生效的结算记录
            List<PurchaseSettlementEntity> list = getHasPurchased(contract.getId(), contract.getOrgId());
            if(CollectionUtils.isNotEmpty(list)) {
                resp.put("settlementId", list.get(0).getId());
            } else {
                resp.put("contractId", id);
            }
        }
        if(null != contract.getAmountWithTax() && contract.getAmountWithTax().doubleValue() != 0) {
            settlementRatio = totalSettlementAmount.divide(contract.getAmountWithTax(), 4, BigDecimal.ROUND_HALF_UP).multiply(new BigDecimal(100));
        }

        resp.put("id", id);
        resp.put("contractAmountWithTax", contract.getAmountWithTax());
        resp.put("totalSettlementAmount", totalSettlementAmount);
        resp.put("settlementRatio", settlementRatio);
        resp.put("changeRecords", changeRecords);
        resp.put("settlementType",contract.getSettlementType());
        resp.put("changeState",contract.getChangeState());

        return CommonResponse.success("查询成功！", resp);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/6/20 根据合同id和组织id查询直属结算单累计结算金额和
     * @Description calculateAlreadyPurchaseMoney
     * @Param [contractId, orgId]
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @GetMapping("/calculateAlreadyPurchaseMoney")
    @ResponseBody
    public CommonResponse<JSONObject> calculateAlreadyPurchaseMoney(@RequestParam Long contractId,@RequestParam Long orgId){
        if(contractId == null || orgId == null){
            return CommonResponse.error("参数错误！");
        }
        QueryParam param = new QueryParam();
        param.getParams().put("contractId",new Parameter(QueryParam.EQ,contractId));
        param.getParams().put("orgId",new Parameter(QueryParam.EQ,orgId));
        param.getOrderMap().put("settlementDate",QueryParam.DESC);
        param.getParams().put("billState",new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(),BillStateEnum.PASSED_STATE.getBillStateCode())));
        List<PurchaseSettlementEntity> entities = iPurchaseSettlementService.queryList(param,false);
        String lastTime = null;
        final BigDecimal[] alreadyPurchaseMoney = {new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP),new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP)};
        if(ListUtil.isNotEmpty(entities)){
            lastTime = DateFormatUtil.formatDate("yyyy-MM-dd",entities.get(0).getSettlementDate());
            entities.forEach(e-> {
                alreadyPurchaseMoney[0] = alreadyPurchaseMoney[0].add(e.getCurrentSettlementAmountTax() == null ?new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP):e.getCurrentSettlementAmountTax());
                alreadyPurchaseMoney[1] = alreadyPurchaseMoney[1].add(e.getOffsetAmountInCurrentPeriod() == null ?new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP):e.getOffsetAmountInCurrentPeriod());
            });
        }
        MaterialContractEntity materialContractEntity = materialContractService.selectById(contractId);

        JSONObject data = new JSONObject();
        data.put("purchaseMoney",alreadyPurchaseMoney[0].toString());
        data.put("offsetAmountInCurrentPeriod",alreadyPurchaseMoney[1].toString());
        data.put("lastTime",lastTime);
        data.put("stageList", materialContractEntity.getStageList());
        return CommonResponse.success(data);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/6/23 根据合同和组织查询未完成的结算单 返回为空：传合同id新增；不为空：传单据id查询
     * @Description hasNoFinishedPurchase
     * @Param [contractId, orgId]
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.Long>
     */
    @GetMapping("/hasNoFinishedPurchase")
    @ResponseBody
    public CommonResponse<Long> hasNoFinishedPurchase(@RequestParam Long contractId,@RequestParam Long orgId){
        if(contractId == null || orgId == null){
            return CommonResponse.error("参数错误！");
        }
        List<PurchaseSettlementEntity> entities = getHasPurchased(contractId,orgId);
        if(ListUtil.isEmpty(entities)){
            return CommonResponse.success("查询成功！");
        }else {
            return CommonResponse.success("查询成功！",entities.get(0).getId());
        }
    }


    /**
     * @Author mrsir_wxp
     * @Date 2020/6/24 采购结算 列表参照
     * @Description refPurchaseData
     * @Param [pageNumber, pageSize, condition, searchText]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.baomidou.mybatisplus.core.metadata.IPage<com.ejianc.business.material.vo.PurchaseSettlementVO>>
     */
    @RequestMapping(value = "/refPurchaseData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<PurchaseSettlementRefVO>> refPurchaseData(@RequestParam Integer pageNumber,
                                                                          @RequestParam Integer pageSize,
                                                                          String condition,
                                                                          String searchText) {
        QueryParam queryParam = new QueryParam();
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.setSearchText(searchText);
        List<String> fuzzyFields = queryParam.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("creatorName");
        if(StringUtils.isNotBlank(condition)){
            JSONObject con = JSONObject.parseObject(condition);
            if(condition.contains("contractId")){
                queryParam.getParams().put("contractId",new Parameter(QueryParam.EQ,con.getLong("contractId")));
            }
            if(condition.contains("orgId")){
                queryParam.getParams().put("orgId",new Parameter(QueryParam.EQ,con.getLong("orgId")));
            }
            if(condition.contains("supplierId")){
                queryParam.getParams().put("supplierId",new Parameter(QueryParam.EQ,con.getLong("supplierId")));
            }
        }
        /** 剩余可申请金额  不等于0 或者 为空的 */
        List<ComplexParam> pList = queryParam.getComplexParams();
        ComplexParam p = new ComplexParam();
        p.setLogic(ComplexParam.AND);
        pList.add(p);
        List<ComplexParam> cList = p.getComplexParams();
        ComplexParam p1 = new ComplexParam();
        p1.setLogic(ComplexParam.OR);
        p1.getParams().put("remainCanApplyAmount",new Parameter(QueryParam.NE,0));
        ComplexParam p2 = new ComplexParam();
        p2.setLogic(ComplexParam.OR);
        p2.getParams().put("remainCanApplyAmount",new Parameter(QueryParam.EQ,null));
        cList.add(p1);
        cList.add(p2);

        queryParam.getParams().put("tenantId",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));
        queryParam.getParams().put("billState",new Parameter(QueryParam.IN, Arrays.asList(
                BillStateEnum.COMMITED_STATE.getBillStateCode(),
                BillStateEnum.PASSED_STATE.getBillStateCode()
        )));
        IPage<PurchaseSettlementVO>  voPage = iPurchaseSettlementService.queryForList(queryParam,false);
        
        // 业财一体化改造
        List<PurchaseSettlementRefVO> refVOS = BeanMapper.mapList(voPage.getRecords(), PurchaseSettlementRefVO.class);
        for (PurchaseSettlementRefVO refVO : refVOS) {
            // 已核销金额(无税) paidVerifyMny
            refVO.setPaidVerifyMny(refVO.getTotalInvoiceVerifyMny());
            // 已核销金额(含税) paidVerifyTaxMny
            refVO.setPaidVerifyTaxMny(refVO.getTotalInvoiceVerifyTaxMny());
            // 本次核销金额(无税)
            refVO.setCurrentVerifyMny(MathUtil.safeSub(refVO.getCurrentSettlementAmount(), refVO.getTotalInvoiceVerifyMny()));
            // 本次核销金额(含税)
            refVO.setCurrentVerifyTaxMny(MathUtil.safeSub(refVO.getCurrentSettlementAmountTax(), refVO.getTotalInvoiceVerifyTaxMny()));
            // 税额
            refVO.setVerifyTaxMny(MathUtil.safeSub(refVO.getCurrentVerifyTaxMny(), refVO.getCurrentVerifyMny()));
            // 累计核销金额(无税)(含本次)
            refVO.setCurrentTotalVerifyMny(refVO.getCurrentSettlementAmount());
            // 累计核销金额(含税)(含本次)
            refVO.setCurrentTotalVerifyTaxMny(refVO.getCurrentSettlementAmountTax());
        }
        
        IPage<PurchaseSettlementRefVO> page = new Page<>();
        page.setSize(voPage.getSize());
        page.setRecords(refVOS);
        page.setPages(voPage.getPages());
        page.setCurrent(voPage.getCurrent());
        page.setTotal(voPage.getTotal());
        return CommonResponse.success("查询结算参照成功！",page);
    }
    // 【物资总计划量】控制【物资合同量】
    // 总计划价格【物资总计划价】控制【物资合同价】
    // 结算历史价格区间【历史价】控制【物资合同价】
    // 【合同金额】控制【结算金额】
    // 控制方式 0：不控制，1、提醒，2、无法保存
    @PostMapping("checkParams/{isJc}/{contractId}")
    @ResponseBody
    public CommonResponse<ParamsCheckVO> checkParams(@PathVariable(name = "isJc") Integer isJc,
                                                     @PathVariable(name = "contractId") Long contractId,
                                                     @RequestBody MaterialPriceVO vo) {
        BigDecimal currentSettlementAmount = vo.getCurrentSettlementAmount();
        BigDecimal contractAmount = vo.getContractAmount();
        BigDecimal settledAmount = vo.getSettledAmount();
        String purchaseMethod = vo.getPurchaseMethod();

        ParamsCheckVO paramsCheckVO = iPurchaseSettlementService.checkParams(isJc, contractId,settledAmount, currentSettlementAmount,contractAmount,purchaseMethod,vo);

        return CommonResponse.success("参数校验成功！", paramsCheckVO);
    }

    @PostMapping("queryPrice/{isJc}/{contractId}")
    @ResponseBody
    public CommonResponse<List<MaterialPriceVO>> queryPrice(@PathVariable(name = "isJc") Integer isJc,
                                                            @PathVariable(name = "contractId") Long contractId, @RequestBody MaterialPriceVO vo) {
        materialContractService.queryPrice(isJc, contractId, vo);
        return CommonResponse.success("查询成功！", vo.getDetail());
    }
}
