package com.ejianc.business.storeCheck.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.material.bean.InstoreEntity;
import com.ejianc.business.material.bean.InstoreMaterialEntity;
import com.ejianc.business.material.bean.OutStoreEntity;
import com.ejianc.business.material.bean.OutStoreSubEntity;
import com.ejianc.business.material.pub.MaterialStoreType;
import com.ejianc.business.material.service.IInstoreMaterialService;
import com.ejianc.business.material.service.IInstoreService;
import com.ejianc.business.material.service.IOutStoreService;
import com.ejianc.business.material.vo.InstoreMaterialVO;
import com.ejianc.business.plan.bean.MaterialMasterPlanEntity;
import com.ejianc.business.storeCheck.bean.StoreCheckEntity;
import com.ejianc.business.storeCheck.bean.StoreCheckSubEntity;
import com.ejianc.business.storeCheck.service.IStoreCheckService;
import com.ejianc.business.storeCheck.service.IStoreCheckSubService;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.beanutils.BeanMap;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 仓库盘点审批回调服务
 *
 * @author CJ
 * @Description:
 * @date 2021/3/15 14:02
 */
@Service(value = "storeCheck")
public class StoreCheckBpmCallbackServiceImpl implements ICommonBusinessService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IStoreCheckService storeCheckService;

    @Autowired
    private IStoreCheckSubService storeCheckSubService;

    @Autowired
    private IInstoreService instoreService;

    @Autowired
    private IOutStoreService outStoreService;

    @Autowired
    private IInstoreMaterialService instoreMaterialService;

    @Autowired
    private IBillCodeApi billCodeApi;

    private static final String IN_STORE_BILL_CODE = "MATERIAL-INSTORE";

    private static final String OUT_STORE_BILL_CODE = "MATERIAL-OUTSTORE";

    private static final String IN_STORE_BILL_TYPE_CODE = "BT200611000000015";

    @Autowired
    private IBillTypeApi billTypeApi;

    @Override
    public CommonResponse<String> beforeSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        return submitCheck(billId, state, billTypeCode);
    }

    @Override
    public CommonResponse<String> beforeApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        return submitCheck(billId, state, billTypeCode);
    }

    private CommonResponse<String> submitCheck(Long billId, Integer state, String billTypeCode) {
        StoreCheckEntity e = storeCheckService.selectById(billId);
        if(e == null) {
            return CommonResponse.success("盘点单据提交失败，系统中未找到匹配的单据信息！");
        }
        logger.info("盘点单据-【{}】执行提交前校验逻辑！", JSONObject.toJSONString(e));
        if(CollectionUtils.isEmpty(e.getMaterialSubList())) {
            return CommonResponse.error("盘点单据提交失败, 盘点单据中缺少要盘点的物资信息！");
        }

        return CommonResponse.success("盘点单据执行提交前校验完成，校验通过！");
    }

    @Override
    public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        StoreCheckEntity e = storeCheckService.selectById(billId);
        logger.info("盘点单据-【{}】执行提交处理逻辑！", JSONObject.toJSONString(e));
        UserContext userContext = sessionManager.getUserContext();

        e.setCommitDate(new Date());
        e.setCommitUserCode(userContext.getUserCode());
        e.setCommitUserName(userContext.getUserName());
        storeCheckService.saveOrUpdate(e, false);

        return CommonResponse.success("仓库盘点审批回调处理成功！");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        StoreCheckEntity e = storeCheckService.selectById(billId);

        logger.info("盘点单据-【{}】执行审批通过处理逻辑！", JSONObject.toJSONString(e));
        e.setEffectiveDate(new Date());
        if(state == 1) {
            //直审时记录
            UserContext userContext = sessionManager.getUserContext();
            e.setCommitDate(new Date());
            e.setCommitUserCode(userContext.getUserCode());
            e.setCommitUserName(userContext.getUserName());
        }

        List<InstoreMaterialEntity> inStoreList = new ArrayList<>();
        InstoreEntity instoreEntity = null;
        OutStoreEntity outStoreEntity = null;
        List<OutStoreSubEntity> outStoreList = new ArrayList<>();
        //若盘盈则生成入库单据，若盘亏则生成出库单据
        if(CollectionUtils.isNotEmpty(e.getMaterialSubList())) {
            for(StoreCheckSubEntity sub : e.getMaterialSubList()) {
                if(sub.getInventory() > 0) {
                    inStoreList.add(generateInStoreBill(sub, e));
                } else if(sub.getInventory() < 0) {
                    generateOutStoreBill(sub, e, outStoreList);
                }
            }

            if(CollectionUtils.isNotEmpty(inStoreList)) {
                List<String> materialNames = new ArrayList<>();
                BigDecimal totalAmt = BigDecimal.ZERO.setScale(8);
                BigDecimal totalCount = BigDecimal.ZERO.setScale(8);
                for(InstoreMaterialEntity item : inStoreList) {
                    materialNames.add(item.getMaterialName());
                    totalAmt = totalAmt.add(item.getAmount());
                    totalCount = totalCount.add(item.getInstoreNumber());
                }

                //创建入库单
                instoreEntity = new InstoreEntity();
                CommonResponse<String> billCodeResp = billCodeApi.getCodeBatchByRuleCode(IN_STORE_BILL_CODE, InvocationInfoProxy.getTenantid());
                if(!billCodeResp.isSuccess()) {
                    return CommonResponse.error("仓库盘点审批回调处理失败，生成入库单编码失败！");
                }
                //设置编码
                instoreEntity.setBillCode(billCodeResp.getData());
                //设置单据状态
                instoreEntity.setBillState(BillStateEnum.PASSED_STATE.getBillStateCode());
                //设置入库日期
                instoreEntity.setInstoreDate(e.getCheckDate());
                instoreEntity.setInstoreMaterialList(inStoreList);
                //入库类型：盘盈入库
                instoreEntity.setInstoreType(MaterialStoreType.INVENTORY_PROFIT_IN_STORE.getCode());
                instoreEntity.setOrgId(e.getOrgId());
                instoreEntity.setOrgName(e.getOrgName());
                //物资种类
                instoreEntity.setMaterialName(StringUtils.join(materialNames, ","));
                //无合同入库
                instoreEntity.setContractType("noContract");
                //仓库
                instoreEntity.setStoreId(e.getStoreId());
                //仓库名称
                instoreEntity.setStoreName(e.getStoreName());
                //税率
                instoreEntity.setTaxRate(new BigDecimal(0));
                //入库金额 含税, 注：税率为0
                instoreEntity.setTotalAmount(totalAmt);
                //入库金额 不含税
                instoreEntity.setTotalAmountNoTax(totalAmt);
                //入库数量
                instoreEntity.setTotalCount(totalCount);
                //备注
                instoreEntity.setNote(MaterialStoreType.INVENTORY_PROFIT_IN_STORE.getDescription());
                instoreEntity.setPurpose(MaterialStoreType.INVENTORY_PROFIT_IN_STORE.getDescription());
            }
            if(CollectionUtils.isNotEmpty(outStoreList)) {
                List<String> outMaterialNames = new ArrayList<>();
                BigDecimal outTotalAmt = BigDecimal.ZERO.setScale(8);
                BigDecimal outTotalCount = BigDecimal.ZERO.setScale(8);
                for(InstoreMaterialEntity item : inStoreList) {
                    outMaterialNames.add(item.getMaterialName());
                    outTotalAmt = outTotalAmt.add(item.getAmount());
                    outTotalCount = outTotalCount.add(item.getInstoreNumber());
                }
                //创建出库单
                outStoreEntity = new OutStoreEntity();
                CommonResponse<String> billCodeResp = billCodeApi.getCodeBatchByRuleCode(OUT_STORE_BILL_CODE, InvocationInfoProxy.getTenantid());
                if(!billCodeResp.isSuccess()) {
                    return CommonResponse.error("仓库盘点审批回调处理失败，生成入库单编码失败！");
                }
                //设置编码
                outStoreEntity.setBillCode(billCodeResp.getData());
                //设置单据状态
                outStoreEntity.setBillState(BillStateEnum.PASSED_STATE.getBillStateCode());
                //出库日期
                outStoreEntity.setOutDate(e.getCheckDate());
                outStoreEntity.setOutStoreSubEntities(outStoreList);
                //出库类型：盘盈入库
                outStoreEntity.setStoreType(MaterialStoreType.INVENTORY_LOSSES_OUT_STORE.getCode());
                outStoreEntity.setOrgId(e.getOrgId());
                outStoreEntity.setOrgName(e.getOrgName());
                //物资种类
                outStoreEntity.setMaterialNames(StringUtils.join(outMaterialNames, ","));
                //不属于项目
                outStoreEntity.setProjectFlag("0");
                //仓库
                outStoreEntity.setStoreId(e.getStoreId());
                //仓库名称
                outStoreEntity.setStoreName(e.getStoreName());
                //入库金额 含税, 注：税率为0
                outStoreEntity.setOutMoney(outTotalAmt);
                //入库金额 不含税
                outStoreEntity.setOutMoneyNoTax(outTotalAmt);
                //入库数量
                outStoreEntity.setOutNum(outTotalCount);
                //用途
                outStoreEntity.setUseFor(MaterialStoreType.INVENTORY_LOSSES_OUT_STORE.getDescription());
                //备注
                outStoreEntity.setMemo(MaterialStoreType.INVENTORY_LOSSES_OUT_STORE.getDescription());
                //税率
                outStoreEntity.setTaxRate(BigDecimal.ZERO.setScale(8));
            }
        }

        if(null != instoreEntity) {
            instoreService.saveOrUpdate(instoreEntity, false);
            e.setRelationInStoreId(instoreEntity.getId());
        } else {
            e.setRelationInStoreId(null);
        }
        if(null != outStoreEntity) {
            outStoreService.saveOrUpdate(outStoreEntity, false);
            e.setRelationOutStoreId(outStoreEntity.getId());
        } else {
            e.setRelationOutStoreId(null);
        }

        storeCheckService.saveOrUpdate(e, false);
        return CommonResponse.success("仓库盘点审批回调处理成功！");
    }

    /**
     * 根据仓库盘点生成入库单子表（盘盈入库）
     *
     * @param materialEntity
     * @return
     */
    private InstoreMaterialEntity generateInStoreBill(StoreCheckSubEntity materialEntity, StoreCheckEntity e) {
        InstoreMaterialEntity resp = new InstoreMaterialEntity();

        BigDecimal instorNumber = new BigDecimal(null != materialEntity.getInventory() ? materialEntity.getInventory().toString() : "0");

        //入库日期
        resp.setInstoreDate(e.getCheckDate());
        //金额
        resp.setAmount(null != materialEntity.getPrice() ? materialEntity.getPrice().multiply(instorNumber) : BigDecimal.ZERO.setScale(8));
        //仓库id
        resp.setStoreId(e.getStoreId());
        //入库数量
        resp.setInstoreNumber(instorNumber);
        //已入库
        resp.setStoreState(3);
        //物资名称
        resp.setMaterialName(materialEntity.getMaterialName());
        //物资id
        resp.setMaterialId(materialEntity.getMaterialId());
        //物资分类id
        resp.setMaterialCategoryId(materialEntity.getMaterialCategoryId());
        //物资分类名称
        resp.setMaterialCategoryName(materialEntity.getMaterialCategoryName());
        //规格型号
        resp.setMaterialSpec(materialEntity.getSpec());
        //计量单位
        resp.setMaterialUnit(materialEntity.getUnit());
        //入库价格
        resp.setUnitPrice(materialEntity.getPrice());
        //备注
        resp.setRemark(materialEntity.getRemark());
        //税率
        resp.setTaxRate(BigDecimal.ZERO.setScale(8));
        //入库类型
        resp.setInstoreType(MaterialStoreType.INVENTORY_PROFIT_IN_STORE.getCode());
        resp.setSourceId(materialEntity.getId());
        resp.setSourceType(MaterialStoreType.INVENTORY_PROFIT_IN_STORE.getDescription());
        resp.setSourceMainId(e.getId().toString());

        return resp;
    }

    /**
     * 根据仓库盘点生成出库单子表（盘亏出库）
     *
     * @param materialEntity
     * @return
     */
    private void generateOutStoreBill(StoreCheckSubEntity materialEntity, StoreCheckEntity e, List<OutStoreSubEntity> outStoreList) {
        List<InstoreMaterialVO> usableMaterialList = instoreMaterialService.querySubMaterialStoreInfo(e.getStoreId(), e.getCheckDate(),
                materialEntity.getMaterialId(), Math.abs(materialEntity.getInventory()));

        BigDecimal outNum = new BigDecimal(Math.abs(materialEntity.getInventory()));

        OutStoreSubEntity resp = new OutStoreSubEntity();
        //仓库id
        resp.setStoreId(e.getStoreId());
        //扣减库存态：1
        resp.setStoreState(1);
        //出库类型
        resp.setStoreType(MaterialStoreType.INVENTORY_LOSSES_OUT_STORE.getCode());
        //出库日期
        resp.setOutDate(e.getCheckDate());
        //物资名称
        resp.setMaterialName(materialEntity.getMaterialName());
        //物资id
        resp.setMaterialId(materialEntity.getMaterialId());
        //物资分类id
        resp.setMaterialCategoryId(materialEntity.getMaterialCategoryId());
        //物资分类名称
        resp.setMaterialCategoryName(materialEntity.getMaterialCategoryName());
        //规格型号
        resp.setMaterialSpec(materialEntity.getSpec());
        //计量单位
        resp.setMaterialUnit(materialEntity.getUnit());
        //备注
        resp.setRemark(materialEntity.getRemark());
        //税率
        resp.setTaxRate(BigDecimal.ZERO.setScale(8));
        //入库价格
        resp.setUnitPrice(materialEntity.getPrice());

        int compareResult = 0;
        OutStoreSubEntity tmp = null;
        for(InstoreMaterialVO instoreMaterialVO : usableMaterialList) {
            compareResult = instoreMaterialVO.getInstoreNumber().compareTo(outNum);
            tmp = BeanMapper.map(resp, OutStoreSubEntity.class);

            tmp.setInStoreMoney(instoreMaterialVO.getStoreAmount());
            tmp.setInStorePrice(instoreMaterialVO.getUnitPrice());
            tmp.setSourceSubId(materialEntity.getId());
            tmp.setSourceMainId(e.getId());

            switch (compareResult) {
                case -1: {
                    tmp.setOutStoreNumber(instoreMaterialVO.getInstoreNumber());
                    tmp.setAmount(null != instoreMaterialVO.getUnitPrice() ? instoreMaterialVO.getInstoreNumber().multiply(instoreMaterialVO.getUnitPrice()) : BigDecimal.ZERO.setScale(8));
                    outNum = outNum.subtract(instoreMaterialVO.getInstoreNumber());
                    break;
                }
                case 0:
                case 1: {
                    //出库数量
                    tmp.setOutStoreNumber(outNum);
                    tmp.setAmount(null != instoreMaterialVO.getUnitPrice() ? outNum.multiply(instoreMaterialVO.getUnitPrice()) : BigDecimal.ZERO.setScale(8));
                    outNum = BigDecimal.ZERO.setScale(8);
                    break;
                }
            }

            outStoreList.add(tmp);

            if(outNum.compareTo(BigDecimal.ZERO.setScale(8)) == 0) {
                break;
            }
        }
        if(outNum.compareTo(BigDecimal.ZERO.setScale(8)) < 0) {
            logger.error("盘点物资-{}出库数量大于库存数量", JSONObject.toJSONString(materialEntity));
            throw new BusinessException("出库数量大于物资库存总数量！");
        }
    }

    /**
     * 弃审前校验
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return bpmCheck(billId, state, billTypeCode);
    }

    /**
     * 弃审
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return delCheckRecords(billId, state, billTypeCode);
    }

    /**
     * 撤回前校验
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return bpmCheck(billId, state, billTypeCode);
    }

    private CommonResponse<String> bpmCheck(Long billId, Integer state, String billTypeCode) {
        StoreCheckEntity e = storeCheckService.selectById(billId);
        logger.info("盘点单据-【{}】撤回前校验", JSONObject.toJSONString(e));
        QueryWrapper<StoreCheckEntity> query = new QueryWrapper<>();
        query.eq("store_id", e.getStoreId());
        query.gt("check_date", e.getCheckDate());
        List<StoreCheckEntity> entityList = storeCheckService.list(query);
        if(CollectionUtils.isNotEmpty(entityList)) {
            return CommonResponse.error("弃审失败，在该盘点单据之后存在新的盘点单据!");
        }

        if(null != e.getRelationInStoreId()) {
            //若盘点存在盘盈入库，则校验盘盈入库是否被下游引用
            CommonResponse<String> resp = billTypeApi.checkQuote(IN_STORE_BILL_TYPE_CODE, e.getRelationInStoreId());
            if(!resp.isSuccess()) {
                logger.info("盘点单据【billId-{}】执行撤回操作校验结束，未通过校验，原因：{}", billId, resp.getMsg());
                return CommonResponse.error(resp.getMsg());
            }
        }

        return CommonResponse.success("弃审前校验通过！");
    }

    /**
     * 撤回
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return delCheckRecords(billId, state, billTypeCode);
    }

    @Transactional(rollbackFor = Exception.class)
    private CommonResponse<String> delCheckRecords(Long billId, Integer state, String billTypeCode) {
        StoreCheckEntity e = storeCheckService.selectById(billId);
        logger.info("盘点单据-【{}】执行撤回逻辑", JSONObject.toJSONString(e));

        //1、删除对应入库单据
        if(null != e.getRelationInStoreId()) {
            logger.info("盘点单据-【ID: {}】撤回，删除对应入库单据-【ID：{}】", e.getId(), e.getRelationInStoreId());
            instoreService.removeById(e.getRelationInStoreId(), false);
        }

        if(null != e.getRelationOutStoreId()) {
            logger.info("盘点单据-【ID: {}】撤回，删除对应出库单据-【ID：{}】", e.getId(), e.getRelationOutStoreId());
            outStoreService.removeById(e.getRelationOutStoreId(), false);
        }

        return CommonResponse.success("执行盘点单据撤回逻辑成功！");
    }
}
