package com.ejianc.business.material.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.budget.vo.cons.BillCons;
import com.ejianc.business.material.bean.*;
import com.ejianc.business.material.service.*;
import com.ejianc.business.material.vo.ContractDetailChangeVO;
import com.ejianc.business.material.vo.MaterialContractVO;
import com.ejianc.business.utils.ComputeUtil;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2020/6/12 15:12
 */
@Service("contractChange")
public class ContractChangeBpmCallbackServiceImpl implements ICommonBusinessService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IContractChangeService contractChangeService;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IMaterialContractService materialContractService;

    @Autowired
    private IContractRecordService contractRecordService;

    @Autowired
    private IMaterialContractOtherCostSubService contractOtherCostSubService;

    @Autowired
    private IMaterialContractItemService materialContractItemService;

    @Autowired
    private IMaterialContractDetailSubService materialContractDetailSubService;

    @Autowired
    private IBillTypeApi billTypeApi;
    @Autowired
    private IAttachmentApi attachmentApi;

    @Override
    public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        logger.info("合同变更提交审批回调--billId:{}, state: {}", billId, state);

        //此处billId即为单据Id
        ContractChangeEntity entity = contractChangeService.selectById(billId);
        UserContext userContext = sessionManager.getUserContext();
        //设置提交人信息
        entity.setCommitDate(new Date());
        entity.setCommitUserCode(userContext.getUserCode());
        entity.setCommitUserName(userContext.getUserName());

        //修改合同中的变更审批状态
        MaterialContractEntity contract = materialContractService.selectById(entity.getMaterialContractId());
        contract.setChangeBillState(state);

        contractChangeService.saveOrUpdate(entity, false);
        materialContractService.saveOrUpdate(contract, false);

        return CommonResponse.success("合同提交审批回调成功！");
    }

    /**
     * 审批完成回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        logger.info("合同变更审批完成回调处理：billId: {}, state: {}", billId, state);
        ContractChangeEntity changeEntity = contractChangeService.selectById(billId);

        if(BillStateEnum.COMMITED_STATE.getBillStateCode().equals(state)){
            UserContext userContext = sessionManager.getUserContext();
            //设置提交人信息
            changeEntity.setCommitDate(new Date());
            changeEntity.setCommitUserCode(userContext.getUserCode());
            changeEntity.setCommitUserName(userContext.getUserName());
        }

        MaterialContractEntity newContractEntity = null;
        MaterialContractEntity contractEntity = materialContractService.selectById(changeEntity.getMaterialContractId());

        //1、复制合同信息到合同记录
        ContractRecordEntity recordEntity = BeanMapper.map(contractEntity, ContractRecordEntity.class);
        logger.info("合同变更审批完成回调, 合同变更信息：【{}】, 变更前合同信息: 【{}】", JSON.toJSONString(changeEntity), JSON.toJSONString(contractEntity));

        //设置合同Id
        recordEntity.setMaterialContractId(contractEntity.getId());
        //Id置为空
        recordEntity.setId(null);
        if (CollectionUtils.isNotEmpty(recordEntity.getMaterialDetailList())) {
            recordEntity.getMaterialDetailList().stream().forEach(detail -> {
                detail.setContractDetailId(detail.getId());
                detail.setId(null);
            });
        }

        if (CollectionUtils.isNotEmpty(recordEntity.getContractItemList())) {
            recordEntity.getContractItemList().stream().forEach(item -> {
                item.setContractItemId(item.getId());
                item.setId(null);
            });
        }

        if (CollectionUtils.isNotEmpty(recordEntity.getOtherCostList())) {
            recordEntity.getOtherCostList().stream().forEach(cost -> {
                cost.setContractOtherCostId(cost.getId());
                cost.setId(null);
            });
        }
        //设置合同变更单据生效时间
        recordEntity.setEffectiveDate(new Date());
        //设置编制人等信息
        recordEntity.setCreateUserCode(changeEntity.getCreateUserCode());
        recordEntity.setCreateTime(changeEntity.getCreateTime());
        recordEntity.setCreateUserName(changeEntity.getCreateUserName());
        recordEntity.setModifyUserName(changeEntity.getModifyUserName());
        recordEntity.setUpdateTime(changeEntity.getUpdateTime());
        recordEntity.setUpdateUserCode(changeEntity.getUpdateUserCode());
        recordEntity.setCommitDate(changeEntity.getCommitDate());
        recordEntity.setCommitUserCode(changeEntity.getCommitUserCode());
        recordEntity.setCommitUserName(changeEntity.getCommitUserName());

        recordEntity.setChangeAmount(ComputeUtil.safeSub(recordEntity.getChangingTaxMny(), recordEntity.getAmountWithTax()));
        recordEntity.setChangeRatio(ComputeUtil.bigDecimalPercent(recordEntity.getChangeAmount(), recordEntity.getAmountWithTax(), 6).doubleValue());
        recordEntity.setChangeReason(changeEntity.getChangeReason());
        recordEntity.setDiscountMny(contractEntity.getDiscountMny());
        recordEntity.setContractUndiscountMny(contractEntity.getContractUndiscountMny());
        //合同变更记录保存
        if (contractRecordService.saveOrUpdate(recordEntity, false)) {
            Long recordEntityId = recordEntity.getId();
            logger.info("拷贝原合同附件到物资合同变更记录单据：billId:"+billId+", billTypeCode:"+billTypeCode+", recordEntity.getId:"+ recordEntityId);
            //将原合同附件复制到历史记录
            attachmentApi.copyFilesFromSourceBillToTargetBill(String.valueOf(contractEntity.getId()),"BT200608000000001","materialContractBill", String.valueOf(recordEntityId), "BT202205000004","materialContractRecord");
        }

        //设置合同变更单据生效时间
        changeEntity.setEffectiveDate(new Date());
        contractChangeService.saveOrUpdate(changeEntity, false);

        //修改合同字段
        newContractEntity = BeanMapper.map(changeEntity, MaterialContractEntity.class);
        //变更状态-已变更
        newContractEntity.setChangeState(MaterialContractVO.CONTRACT_CHANGE_STATE_CHANGED);
        //清空变更中、前字段
        newContractEntity.setBeforeChangeMny(newContractEntity.getAmountWithoutTax());
        newContractEntity.setBeforeChangeTaxMny(newContractEntity.getAmountWithTax());
        newContractEntity.setChangingTaxMny(null);
        newContractEntity.setChangingMny(null);
        //冗余变更单审批状态清空
        newContractEntity.setChangeBillState(null);
        newContractEntity.setId(contractEntity.getId());
        newContractEntity.setVersion(contractEntity.getVersion());
        //合同初始创建信息
        newContractEntity.setCreateUserName(contractEntity.getCreateUserName());
        newContractEntity.setCreateUserCode(contractEntity.getCreateUserCode());
        newContractEntity.setCreateTime(contractEntity.getCreateTime());
        newContractEntity.setContractUndiscountMny(changeEntity.getContractUndiscountMny());
        newContractEntity.setDiscountMny(changeEntity.getDiscountMny());

        //子表信息设置
        List<Long> delDetailIds = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(changeEntity.getMaterialDetailList())) {
            String materialTypeNames = "";
            Set materialTypeNamesSet = new HashSet();
            String sectionNames = "";
            Set sectionNameSet = new HashSet();
            List<MaterialContractDetailSubEntity> list = new ArrayList<>();
            MaterialContractDetailSubEntity tmp = null;
            Map<Long, Integer> map = contractEntity.getMaterialDetailList().stream().collect(Collectors.toMap(MaterialContractDetailSubEntity::getId, MaterialContractDetailSubEntity::getVersion));
            for (ContractDetailChangeEntity detail : changeEntity.getMaterialDetailList()) {
                tmp = BeanMapper.map(detail, MaterialContractDetailSubEntity.class);
                if (null != detail.getContractDetailId()) {
                    tmp.setId(detail.getContractDetailId());
                    tmp.setVersion(map.get(detail.getContractDetailId()));
                    if (detail.getChangeType().equals(ContractDetailChangeVO.CHANGE_TYPE_DEL)) {
                        //终止项删除
                        delDetailIds.add(detail.getContractDetailId());
                    } else {
                        list.add(tmp);
                    }
                } else {
                    list.add(tmp);
                }
            }
            if (ListUtil.isNotEmpty(list)) {
                for (MaterialContractDetailSubEntity entity : list) {
                    if (StringUtils.isNotBlank(materialTypeNames)) {
                        if (!materialTypeNamesSet.contains(entity.getMaterialTypeName())) {
                            materialTypeNames += "、" + entity.getMaterialTypeName();
                            materialTypeNamesSet.add(entity.getMaterialTypeName());
                        }
                    } else {
                        materialTypeNames += entity.getMaterialTypeName();
                        materialTypeNamesSet.add(entity.getMaterialTypeName());
                    }
                    if (StringUtils.isNotBlank(sectionNames)) {
                        if (!sectionNameSet.contains(entity.getSectionName())) {
                            sectionNames += "," + entity.getSectionName();
                            sectionNameSet.add(entity.getSectionName());
                        }
                    } else {
                        sectionNames += entity.getSectionName();
                        sectionNameSet.add(entity.getSectionName());
                    }
                }
            }
            newContractEntity.setMaterialTypeNames(materialTypeNames);
            newContractEntity.setSectionNames(sectionNames);
            newContractEntity.setMaterialDetailList(list);
        }

        if (CollectionUtils.isNotEmpty(changeEntity.getOtherCostList())) {
            changeEntity.getOtherCostList().stream().forEach(cost -> {
                cost.setId(null);
            });
            newContractEntity.setOtherCostList(BeanMapper.mapList(changeEntity.getOtherCostList(), MaterialContractOtherCostSubEntity.class));
        }

        if (CollectionUtils.isNotEmpty(changeEntity.getContractItemList())) {
            Map<Long, Integer> map = contractEntity.getContractItemList().stream().collect(Collectors.toMap(MaterialContractItemEntity::getId, MaterialContractItemEntity::getVersion));
            changeEntity.getContractItemList().stream().filter(item -> null != item.getContractItemId()).forEach(item -> {
                item.setId(null);
            });
            newContractEntity.setContractItemList(BeanMapper.mapList(changeEntity.getContractItemList(), MaterialContractItemEntity.class));
        }

        //删除已有子表
        contractOtherCostSubService.deleteByContractId(changeEntity.getMaterialContractId());
        materialContractItemService.deleteByContractId(changeEntity.getMaterialContractId());
        if (CollectionUtils.isNotEmpty(delDetailIds)) {
            materialContractDetailSubService.removeByIds(delDetailIds);
        }


        if (materialContractService.saveOrUpdate(newContractEntity, false)) {
            logger.info("拷贝附件到物资合同单据：billId:"+billId+", billTypeCode:"+billTypeCode+", newContractEntity.getId:"+newContractEntity.getId());
            //变更合同附件复制到主合同
            attachmentApi.copyFiles(String.valueOf(billId), "BT200613000000001","materialContractChange",String.valueOf(newContractEntity.getId()),"BT200608000000001","materialContractBill",true);
            materialContractService.countContractNumAndMnyByEnquiry(newContractEntity.getEnquiryPriceBillId());
            return CommonResponse.success("回调业务处理成功！");
        }
        return CommonResponse.success("回调业务处理失败！");
    }

    /**
     * 撤回前校验
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        logger.info("物资合同变更单据【billId-{}】,【billTypeCode-{}】执行撤回前校验！", billId, billTypeCode);

        //此处billId即为单据Id
        ContractChangeEntity entity = contractChangeService.selectById(billId);

        if (!BillStateEnum.APPROVING_UNEXAM_STATE.getBillStateCode().equals(entity.getBillState())) {
            logger.info("物资合同变更单据【billId-{}】,【billTypeCode-{}】执行撤回操作校验结束，未通过校验，原因：合同变更单据只能在提交未审批状态下才能撤回或弃审！", billId, billTypeCode);
            return CommonResponse.error("合同变更单据只能在提交未审批状态下才能执行此操作！");
        }

        return CommonResponse.success("回调处理成功！");
    }

    /**
     * 撤回
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        logger.info("物资合同变更单据【billId-{}】,【billTypeCode-{}】执行撤回操作！", billId, billTypeCode);
        //此处billId即为单据Id
        ContractChangeEntity entity = contractChangeService.selectById(billId);

        //修改合同中的变更审批状态
        MaterialContractEntity contract = materialContractService.selectById(entity.getMaterialContractId());
        contract.setChangeBillState(state);

        materialContractService.saveOrUpdate(contract, false);

        return CommonResponse.success("合同提交审批回调成功！");
    }
}
