package com.ejianc.business.plan.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.material.bean.UseApplySubEntity;
import com.ejianc.business.material.utlis.ArchivesUtil;
import com.ejianc.business.material.vo.ParamsCheckVO;
import com.ejianc.business.material.vo.UseApplySubVO;
import com.ejianc.business.plan.bean.MaterialMasterPlanEntity;
import com.ejianc.business.plan.bean.MaterialMasterPlanSubEntity;
import com.ejianc.business.plan.bean.PlanChangeHisEntity;
import com.ejianc.business.plan.service.IMaterialMasterPlanService;
import com.ejianc.business.plan.service.IMaterialMasterPlanSubService;
import com.ejianc.business.plan.service.IPlanChangeHisService;
import com.ejianc.business.plan.vo.*;
import com.ejianc.business.utils.ComputeUtil;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.ejianc.framework.skeleton.template.BaseServiceImpl.changeToQueryWrapper;

/**
 * 物资总计划控制器
 *
 * @author CJ
 * @Description:
 * @date 2021/3/2 17:33
 */
@RestController
@RequestMapping(value = "/masterPlan/")
public class MasterPlanController {

    @Autowired
    private IMaterialMasterPlanService materialMasterPlanService;

    @Autowired
    private IBillCodeApi billCodeApi;

    private final String BILL_CODE = "MATERIAL_MASTER_PLAN";

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IPlanChangeHisService planChangeHisService;

    @Autowired
    private IMaterialMasterPlanSubService materialMasterPlanSubService;
    @Autowired
    private ArchivesUtil archivesUtil;

    /**
     * 物资总计划-保存、更新
     *
     * @param masterPlanVo
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<MaterialMasterPlanVO> saveOrUpdate(@RequestBody MaterialMasterPlanVO masterPlanVo) {
        if(StringUtils.isNotBlank(masterPlanVo.getPlanCode())) {
            //编码重复校验
            MaterialMasterPlanVO voByCode = materialMasterPlanService.queryByCode(masterPlanVo.getPlanCode());
            if(null != voByCode && (null == masterPlanVo.getId() || !voByCode.getId().equals(masterPlanVo.getId()))) {
                return CommonResponse.error("保存失败，编码重复！");
            }
        } else {
            CommonResponse<String> codeResp = billCodeApi.getCodeBatchByRuleCode(BILL_CODE, InvocationInfoProxy.getTenantid());
            if(!codeResp.isSuccess()) {
                return CommonResponse.error("保存失败，获取自动编码失败！");
            }
            masterPlanVo.setPlanCode(codeResp.getData());
        }

        //校验项目是否已有总计划
        MaterialMasterPlanVO voByProject = materialMasterPlanService.queryByProjectId(masterPlanVo.getProjectId());
        if(null != voByProject && (null == masterPlanVo.getId() || !voByProject.getId().equals(masterPlanVo.getId()))) {
            return CommonResponse.error("保存失败，该项目下已存在物资总计划，请勿重复添加！");
        }

        UserContext userContext = sessionManager.getUserContext();
        MaterialMasterPlanEntity saveEntity = null;
        if(null == masterPlanVo.getId()) {
            saveEntity = BeanMapper.map(masterPlanVo, MaterialMasterPlanEntity.class);
            //设置为未变更
            saveEntity.setChangeState(0);
            //设置为自由态
            saveEntity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
            //变更版本号
            saveEntity.setChangeVersion(1);
            //设置编制人
            saveEntity.setCreateUserName(userContext.getUserName());
        } else {
            saveEntity = materialMasterPlanService.getById(masterPlanVo.getId());
            saveEntity.setProjectId(masterPlanVo.getProjectId());
            saveEntity.setProjectName(masterPlanVo.getProjectName());
            saveEntity.setOrgId(masterPlanVo.getOrgId());
            saveEntity.setOrgName(masterPlanVo.getOrgName());
            saveEntity.setRemark(masterPlanVo.getRemark());
            saveEntity.setPlanCode(masterPlanVo.getPlanCode());
            saveEntity.setTotalPlanAmt(masterPlanVo.getTotalPlanAmt());
            saveEntity.setModifyUserName(userContext.getUserName());

            saveEntity.setMaterialSubList(BeanMapper.mapList(masterPlanVo.getMaterialSubList(), MaterialMasterPlanSubEntity.class));
        }

        if(null == saveEntity.getTotalPlanAmt()) {
            saveEntity.setTotalPlanAmt(BigDecimal.ZERO.setScale(8));
        }
        List<MaterialMasterPlanSubVO> materialSubList = masterPlanVo.getMaterialSubList();
        if (CollectionUtils.isNotEmpty(materialSubList)){
            for (MaterialMasterPlanSubVO materialMasterPlanSubVO : materialSubList) {
                String onlykey= materialMasterPlanSubVO.getMaterialCategoryName()+materialMasterPlanSubVO.getMaterialName()+materialMasterPlanSubVO.getUnit()+materialMasterPlanSubVO.getSpec()+materialMasterPlanSubVO.getMaterialCode();
                materialMasterPlanSubVO.setOnlyKey(onlykey);
            }
            JSONObject jsonObject = archivesUtil.batchSaveArchive(materialSubList, "materialCategoryId", "materialId",
                    "materialCategoryName", "materialName","materialCode", "unit", "spec", "errorMessage");
            String sourceList = JSON.toJSONString(jsonObject.get("sourceList"), SerializerFeature.WriteMapNullValue, SerializerFeature.PrettyFormat);
            List<MaterialMasterPlanSubVO> list = JSONObject.parseArray(sourceList, MaterialMasterPlanSubVO.class);
            List<MaterialMasterPlanSubEntity> materialMasterPlanSubEntities = BeanMapper.mapList(list, MaterialMasterPlanSubEntity.class);
            saveEntity.setMaterialSubList(materialMasterPlanSubEntities);
        }

        saveEntity = materialMasterPlanService.saveOrUpdateMasterPlan(saveEntity);

        return CommonResponse.success("保存成功！", BeanMapper.map(saveEntity, MaterialMasterPlanVO.class));
    }

    /**
     * 分页查询物资总计划列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "pageList")
    public CommonResponse<IPage<MaterialMasterPlanVO>> pageList(@RequestBody QueryParam queryParam) {
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("planCode");
        queryParam.getFuzzyFields().add("createUserName");

        //查询当前上下文本下的物资总计划
        queryParam.getComplexParams().add(getPageQueryParam(null));
        IPage<MaterialMasterPlanEntity> pageData = materialMasterPlanService.queryPage(queryParam);

        IPage<MaterialMasterPlanVO> resp = new Page<>();
        resp.setSize(pageData.getSize());
        resp.setRecords(BeanMapper.mapList(pageData.getRecords(), MaterialMasterPlanVO.class));
        resp.setCurrent(pageData.getCurrent());
        resp.setTotal(pageData.getTotal());
        resp.setPages(pageData.getPages());

        return CommonResponse.success("分页查询物资总计划列表成功", resp);
    }

    public ComplexParam getPageQueryParam(Long orgId) {
        ComplexParam c1 = new ComplexParam();
        c1.setLogic(ComplexParam.AND);

        if(null == orgId) {
            orgId = InvocationInfoProxy.getOrgId();
        }
        CommonResponse<List<OrgVO>> childOrgResp = null;
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)){
            childOrgResp = orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong)
                            .collect(Collectors.toList()));
        }else{
            childOrgResp=orgApi.findChildrenByParentId(orgId);
        }
        List<Long> childIds = new ArrayList<>();
        //总计划直接跟项目关联，只查询本心的所有项目部即可
        childIds.addAll(childOrgResp.getData().stream().filter(org -> 5 == org.getOrgType()).map(OrgVO::getId).collect(Collectors.toList()));

        if(CollectionUtils.isEmpty(childIds)) {
            childIds.add(1L); //添加1项，以防报错
        }

        ComplexParam c2 = new ComplexParam();
        c2.setLogic(ComplexParam.OR);
        c2.getParams().put("org_id", new Parameter(QueryParam.IN, childIds));
        c1.getComplexParams().add(c2);

        return c1;
    }

    @GetMapping("queryDetail")
    public CommonResponse<MaterialMasterPlanVO> queryDetail(@RequestParam(value = "id") Long id) {
        MaterialMasterPlanVO vo = null;
        MaterialMasterPlanEntity dbEntity = materialMasterPlanService.selectById(id);
        if(null != dbEntity) {
            vo = BeanMapper.map(dbEntity, MaterialMasterPlanVO.class);

            if(MaterialMasterPlanVO.PLAN_CHANGE_STATE_CHANGED == vo.getChangeState()) {
                List<PlanChangeHisVO> changeHisList = loadChangeHisList(vo.getId());
                vo.setPlanChangeHisList(changeHisList);
            }
        }

        return CommonResponse.success("查询物资总计划详情成功！", vo);
    }

    private List<PlanChangeHisVO> loadChangeHisList(Long planId) {
        List<PlanChangeHisVO> resp = new ArrayList<>();
        QueryWrapper<PlanChangeHisEntity> hisQuery = new QueryWrapper<>();
        hisQuery.eq("source_plan_id", planId);
        hisQuery.orderByDesc("create_time");

        List<PlanChangeHisEntity> list = planChangeHisService.list(hisQuery);
        if(CollectionUtils.isNotEmpty(list)) {
            resp = BeanMapper.mapList(list, PlanChangeHisVO.class);
        }
        return  resp;
    }

    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<MaterialMasterPlanVO> vos) {
        if(CollectionUtils.isNotEmpty(vos)) {
            List<Long> ids = vos.stream().map(MaterialMasterPlanVO::getId).collect(Collectors.toList());
            if(CollectionUtils.isNotEmpty(ids)) {
                materialMasterPlanService.removeByIds(ids);
            }
        }
        return CommonResponse.success("删除成功！");
    }

    @GetMapping(value = "projectPlanCheck")
    public CommonResponse<String> projectPlanCheck(@RequestParam(value = "projectId") Long projectId,
                                                   @RequestParam(value = "billId", required = false) Long billId) {
        QueryWrapper<MaterialMasterPlanEntity> query = new QueryWrapper<>();
        query.eq("project_id", projectId);

        MaterialMasterPlanEntity entityByProject = materialMasterPlanService.getOne(query);
        if(null != entityByProject && (null == billId || !entityByProject.getId().equals(billId))) {
            return CommonResponse.error("该项目下已存在物资总计划！");
        }
        return CommonResponse.success("该项目适用当前总计划！");
    }

    @GetMapping(value = "planMaterialListRefer")
    public CommonResponse<IPage<MaterialMasterPlanSubVO>> planMaterialListRefer(@RequestParam(required = false) String condition,
                                                                                @RequestParam(required = false) String searchText,
                                                                                @RequestParam int pageSize,
                                                                                @RequestParam int pageNumber) {
        IPage<MaterialMasterPlanSubVO> resp = new Page<>();
        QueryParam queryParam = new QueryParam();
        queryParam.setSearchText(searchText);
        queryParam.getFuzzyFields().add("material_category_name");
        queryParam.getFuzzyFields().add("material_name");
        queryParam.getFuzzyFields().add("spec");
        queryParam.getFuzzyFields().add("material_code");

        //按照物资Id正序排列
        queryParam.getOrderMap().put("id", QueryParam.ASC);
        queryParam.setPageSize(pageSize);
        queryParam.setPageIndex(pageNumber);

        if(StringUtils.isBlank(condition)) {
            return CommonResponse.error("查询失败，参数condition中缺少项projectId！");
        }

        Map<String, Object> conditionMap = JSONObject.parseObject(condition, Map.class);
        if(null == conditionMap.get("projectId")) {
            return CommonResponse.error("查询失败，参数condition中缺少项projectId！");
        }
        List<Integer> approvedBillState = new ArrayList<>();
        QueryWrapper<MaterialMasterPlanEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("project_id", conditionMap.get("projectId"));
        //查询审批通过
        approvedBillState.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
        approvedBillState.add(BillStateEnum.PASSED_STATE.getBillStateCode());
        queryWrapper.in("bill_state", approvedBillState);
        MaterialMasterPlanEntity entity = materialMasterPlanService.getOne(queryWrapper);
        if(null == entity) {
            resp.setTotal(0L);
            resp.setSize(pageSize);
            resp.setCurrent(pageNumber);
            resp.setRecords(new ArrayList<>());
            resp.setPages(0);
        } else {
            queryParam.getParams().put("materialMasterPlanId", new Parameter(QueryParam.EQ, entity.getId()));
            Page<MaterialMasterPlanSubVO> page = new Page<>(queryParam.getPageIndex(), queryParam.getPageSize());
            QueryWrapper wrapper = changeToQueryWrapper(queryParam);
            wrapper.eq("dr",0);
            wrapper.groupBy("material_id");
            List<MaterialMasterPlanSubVO> list =  materialMasterPlanSubService.queryPlanSubList(page,wrapper);
            if (CollectionUtils.isNotEmpty(list)){
                for (MaterialMasterPlanSubVO materialMasterPlanSubVO : list) {
                    BigDecimal num = new BigDecimal(materialMasterPlanSubVO.getNum());
                    BigDecimal amount = ComputeUtil.safeMultiply(materialMasterPlanSubVO.getPrice(),num );
                    materialMasterPlanSubVO.setAmount(amount);
                    materialMasterPlanSubVO.setBillCode(entity.getPlanCode());
                }
            }
            IPage<MaterialMasterPlanSubVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
            pageData.setRecords(list);
            /*IPage<MaterialMasterPlanSubEntity> pageData = materialMasterPlanSubService.queryPage(queryParam, false);*/
            resp.setPages(pageData.getPages());
            resp.setCurrent(pageData.getCurrent());
            resp.setTotal(pageData.getTotal());
            resp.setSize(pageData.getSize());

            List<MaterialMasterPlanSubVO> dataList = new ArrayList<>();
            if (CollectionUtils.isNotEmpty(pageData.getRecords())) {
                dataList = BeanMapper.mapList(pageData.getRecords(), MaterialMasterPlanSubVO.class);
            }

            resp.setRecords(dataList);
        }


        return CommonResponse.success(resp);
    }


    /**
     * 参数控制
     *
     * @param vo 物资总计划vo
     *
     * @return {@link CommonResponse}<{@link ParamsCheckVO}>
     */
    @PostMapping("checkParams")
    public CommonResponse<ParamsCheckVO> checkParams(@RequestBody PlanControlVO vo) {
        return CommonResponse.success("参数校验成功！", materialMasterPlanService.checkParams(vo));
    }

}
