package com.ejianc.foundation.mdm.utils;

import java.sql.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import javax.sql.DataSource;
import com.ejianc.foundation.mdm.config.AggConfig;
import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.druid.pool.DruidDataSourceFactory;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.mdm.dataprovider.Aggregatable;
import com.ejianc.foundation.mdm.result.AggregateResult;
import com.ejianc.foundation.mdm.exception.DBException;
import com.google.common.base.Charsets;
import com.google.common.base.Stopwatch;
import com.google.common.hash.Hashing;

public class JdbcDataProvider extends DataProvider implements Aggregatable, Initializing {

    private static final Logger LOG = LoggerFactory.getLogger(JdbcDataProvider.class);
    private final static int resultLimit = 1000;

    private String DRIVER = "driver";
    private String JDBC_URL = "jdbcurl";
    private String USERNAME = "username";
    private String PASSWORD = "password";
    private String POOLED = "pooled";
    private String aggregateProvider = "aggregateProvider";
    private String SQL = "sql";

    private static final ConcurrentMap<String, DataSource> datasourceMap = new ConcurrentHashMap<>();
    private SqlHelper sqlHelper;

    private JdbcDataProvider() {}

    private static class JdbcDataProviderInstance {
        private static final JdbcDataProvider instance = new JdbcDataProvider();
    }

    public static JdbcDataProvider getInstance() {
        return JdbcDataProviderInstance.instance;
    }

    public void assignVal(Map<String, String> dataSource, Map<String, String> query) {
        this.dataSource = dataSource;
        this.query = query;
    }

    @Override
    public boolean doAggregationInDataSource() {
        String v = dataSource.get(aggregateProvider);
        return v != null && "true".equals(v);
    }

    public List<JSONObject> getDataList() {
        List<JSONObject> dataList = new ArrayList<>();
        try {
            String[][] datas = getData();
            if(datas != null && datas.length > 1) {
                String[] headers = datas[0];
                for(int i = 1; i < datas.length; i++) {
                    JSONObject jsonObject = new JSONObject();
                    for (int j = 0; j < datas[i].length; j++){
                        jsonObject.put(headers[j], datas[i][j]);
                    }
                    dataList.add(jsonObject);
                }
            }
        } catch (Exception e) {}
        return dataList;
    }

    @Override
    public String[][] getData() throws Exception {
        final int batchSize = 100000;
        Stopwatch stopwatch = Stopwatch.createStarted();
        LOG.debug("Execute JdbcDataProvider.getData() Start!");
        String sql = getAsSubQuery(query.get(SQL));
        List<String[]> list = null;
        LOG.info("SQL String: " + sql);

        Connection con = null;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            con = getConnection();
            pstmt = con.prepareStatement(sql);
            rs = pstmt.executeQuery();

            ResultSetMetaData metaData = rs.getMetaData();
            int columnCount = metaData.getColumnCount();
            list = new LinkedList<>();
            String[] row = new String[columnCount];
            for (int i = 0; i < columnCount; i++) {
                row[i] = metaData.getColumnLabel(i + 1);
            }

            String[] header = row;
            list.add(row);

            int resultCount = 0;
            int threadId = 0;
            ExecutorService executor = Executors.newFixedThreadPool(5);
            while (rs.next()) {
                resultCount++;
                row = new String[columnCount];
                for (int j = 0; j < columnCount; j++) {
                    row[j] = rs.getString(j + 1);
                }
                list.add(row);

                if (resultCount % batchSize == 0) {
                    LOG.info("JDBC load batch {}", resultCount);
                    final String[][] batchData = list.toArray(new String[][]{});
                    Thread loadThread = new Thread(() -> {
                        getInnerAggregator().loadBatch(header, batchData);
                    }, threadId++ + "");
                    executor.execute(loadThread);
                    list.clear();
                }
                if (resultCount > resultLimit) {
                    throw new DBException("Cube result count " + resultCount + ", is greater than limit " + resultLimit);
                }
            }
            executor.shutdown();
            while (!executor.awaitTermination(10, TimeUnit.SECONDS));
            final String[][] batchData = list.toArray(new String[][]{});
            LOG.info("getData() using time: {} ms", stopwatch.elapsed(TimeUnit.MILLISECONDS));
            return batchData;
        } catch (Exception e) {
            LOG.error("ERROR:" + e.getMessage());
            throw new Exception("ERROR:" + e.getMessage(), e);
        }finally {
            rs.close();
            pstmt.close();
            con.close();
        }
    }

    public boolean testConnection() throws Exception {
        String queryStr = query.get(SQL);
        LOG.info("Execute query: {}", queryStr);
        try (Connection con = getConnection();
            Statement ps = con.createStatement()) {
            ps.executeQuery(queryStr);
            return true;
        } catch (Exception e) {
            LOG.error("Error when execute: {}",  queryStr);
        }
        return false;
    }

    private String getAsSubQuery(String rawQueryText) {
        String deletedBlankLine = rawQueryText.replaceAll("(?m)^[\\s\t]*\r?\n", "").trim();
        return deletedBlankLine.endsWith(";") ? deletedBlankLine.substring(0, deletedBlankLine.length() - 1) : deletedBlankLine;
    }

	@SuppressWarnings("unchecked")
	private Connection getConnection() throws Exception {
        String usePool = dataSource.get(POOLED);
        String username = dataSource.get(USERNAME);
        String password = dataSource.get(PASSWORD);
        if(StringUtils.isBlank(usePool)) {
            usePool = "true";
        }
        if ("true".equals(usePool)) {
            String key = Hashing.md5().newHasher().putString(JSONObject.toJSON(dataSource).toString(), Charsets.UTF_8).hash().toString();
            DataSource ds = datasourceMap.get(key);
            if (ds == null) {
                synchronized (key.intern()) {
                    ds = datasourceMap.get(key);
                    if (ds == null) {
						Map<String, String> conf = new HashedMap();
                        conf.put(DruidDataSourceFactory.PROP_DRIVERCLASSNAME, dataSource.get(DRIVER));
                        conf.put(DruidDataSourceFactory.PROP_URL, dataSource.get(JDBC_URL));
                        conf.put(DruidDataSourceFactory.PROP_USERNAME, dataSource.get(USERNAME));
                        if (StringUtils.isNotBlank(password)) {
                            conf.put(DruidDataSourceFactory.PROP_PASSWORD, dataSource.get(PASSWORD));
                        }
                        conf.put(DruidDataSourceFactory.PROP_INITIALSIZE, "100");
                        conf.put(DruidDataSourceFactory.PROP_MINIDLE, "50");
                        conf.put(DruidDataSourceFactory.PROP_MAXACTIVE, "200");
                        conf.put(DruidDataSourceFactory.PROP_MAXWAIT, "10000");
                        conf.put(DruidDataSourceFactory.PROP_TIMEBETWEENEVICTIONRUNSMILLIS, "300000");
                        conf.put(DruidDataSourceFactory.PROP_MINEVICTABLEIDLETIMEMILLIS, "300000");
                        conf.put(DruidDataSourceFactory.PROP_TESTWHILEIDLE, "true");
                        conf.put(DruidDataSourceFactory.PROP_TESTONBORROW, "false");
                        conf.put(DruidDataSourceFactory.PROP_TESTONRETURN, "false");
                        conf.put(DruidDataSourceFactory.PROP_POOLPREPAREDSTATEMENTS, "true");
                        conf.put(DruidDataSourceFactory.PROP_REMOVEABANDONED, "true");
                        conf.put(DruidDataSourceFactory.PROP_REMOVEABANDONEDTIMEOUT, "600");
                        DruidDataSource druidDS = (DruidDataSource) DruidDataSourceFactory.createDataSource(conf);
                        druidDS.setBreakAfterAcquireFailure(true);
                        druidDS.setConnectionErrorRetryAttempts(5);
                        datasourceMap.put(key, druidDS);
                        ds = datasourceMap.get(key);
                    }
                }
            }

            Connection conn = null;
            try {
                conn = ds.getConnection();
            } catch (SQLException e) {
                datasourceMap.remove(key);
                throw e;
            }
            return conn;
        } else {
            String driver = dataSource.get(DRIVER);
            String jdbcurl = dataSource.get(JDBC_URL);

            Class.forName(driver);
            Properties props = new Properties();
            props.setProperty("user", username);
            if (StringUtils.isNotBlank(password)) {
                props.setProperty("password", password);
            }
            return DriverManager.getConnection(jdbcurl, props);
        }
    }

    @Override
    public String[] queryDimVals(String columnName, AggConfig config) throws Exception {
        String fsql = null;
        String exec = null;
        String sql = getAsSubQuery(query.get(SQL));
        List<String> filtered = new ArrayList<>();
        String whereStr = "";
        if (config != null) {
            whereStr = sqlHelper.assembleFilterSql(config);
        }
        fsql = "SELECT cb_view.%s FROM (\n%s\n) cb_view %s GROUP BY cb_view.%s";
        exec = String.format(fsql, columnName, sql, whereStr, columnName);
        LOG.info(exec);
        try (Connection connection = getConnection();
             Statement stat = connection.createStatement();
             ResultSet rs = stat.executeQuery(exec)) {
            while (rs.next()) {
                filtered.add(rs.getString(1));
            }
        } catch (Exception e) {
            LOG.error("ERROR:" + e.getMessage());
            throw new Exception("ERROR:" + e.getMessage(), e);
        }
        return filtered.toArray(new String[]{});
    }


    private ResultSetMetaData getMetaData(String subQuerySql, Statement stat) throws Exception {
        ResultSetMetaData metaData;
        try {
            stat.setMaxRows(100);
            String fsql = "\nSELECT * FROM (\n%s\n) cb_view WHERE 1=0";
            String sql = String.format(fsql, subQuerySql);
            LOG.info(sql);
            ResultSet rs = stat.executeQuery(sql);
            metaData = rs.getMetaData();
        } catch (Exception e) {
            LOG.error("ERROR:" + e.getMessage());
            throw new Exception("ERROR:" + e.getMessage(), e);
        }
        return metaData;
    }

	@SuppressWarnings("unchecked")
	private Map<String, Integer> getColumnType() throws Exception {
        Map<String, Integer> result = null;
        try (
                Connection connection = getConnection();
                Statement stat = connection.createStatement()
        ) {
            String subQuerySql = getAsSubQuery(query.get(SQL));
            ResultSetMetaData metaData = getMetaData(subQuerySql, stat);
            int columnCount = metaData.getColumnCount();
            result = new HashedMap();
            for (int i = 0; i < columnCount; i++) {
                result.put(metaData.getColumnLabel(i + 1).toUpperCase(), metaData.getColumnType(i + 1));
            }
            return result;
        }
    }

    @Override
    public String[] getColumn(boolean reload) throws Exception {
        String subQuerySql = getAsSubQuery(query.get(SQL));
        try (
                Connection connection = getConnection();
                Statement stat = connection.createStatement()
        ) {
            ResultSetMetaData metaData = getMetaData(subQuerySql, stat);
            int columnCount = metaData.getColumnCount();
            String[] row = new String[columnCount];
            for (int i = 0; i < columnCount; i++) {
                row[i] = metaData.getColumnLabel(i + 1);
            }
            return row;
        }
    }

	@Override
	public Map<String, String> getSelectColumn(boolean reload) throws Exception {
		Map<String, String> map = new HashMap<String, String>();
		String[] columns = getColumn(reload);
		for (String column : columns) {
			map.put(column, column);
		}
		return map;
	}

    @Override
    public AggregateResult queryAggData(AggConfig config) throws Exception {
        String exec = sqlHelper.assembleAggDataSql(config);
        List<String[]> list = new LinkedList<>();
        LOG.info(exec);
        try (
                Connection connection = getConnection();
                Statement stat = connection.createStatement();
                ResultSet rs = stat.executeQuery(exec)
        ) {
            ResultSetMetaData metaData = rs.getMetaData();
            int columnCount = metaData.getColumnCount();
            while (rs.next()) {
                String[] row = new String[columnCount];
                for (int j = 0; j < columnCount; j++) {
                    row[j] = rs.getString(j + 1);
                }
                list.add(row);
            }
        } catch (Exception e) {
            LOG.error("ERROR:" + e.getMessage());
            throw new Exception("ERROR:" + e.getMessage(), e);
        }
        return DPCommonUtils.transform2AggResult(config, list);
    }


    @Override
    public String viewAggDataQuery(AggConfig config) throws Exception {
        return sqlHelper.assembleAggDataSql(config);
    }


    @Override
    public void afterPropertiesSet() throws Exception {
        String subQuery = null;
        if (query != null) {
            subQuery = getAsSubQuery(query.get(SQL));
        }
        SqlHelper sqlHelper = new SqlHelper(subQuery, true);
        if (!isUsedForTest()) {
            Map<String, Integer> columnTypes = null;
            try {
                columnTypes = getColumnType();
            } catch (Exception e) {
                //  getColumns() and test() methods do not need columnTypes properties,
                // it doesn't matter for these methods even getMeta failed
                LOG.warn("getColumnType failed: {}", e.getMessage());
            }
            sqlHelper.getSqlSyntaxHelper().setColumnTypes(columnTypes);
        }
        this.sqlHelper = sqlHelper;
    }

}
