package com.ejianc.framework.mongodb.template;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.mongodb.BasicDBObject;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.UpdateOptions;

@Component
public class MongodbCrudTemplate {

	@Autowired
	private MongoTemplate mongoTemplate;

	/**
	 * 插入
	 * 
	 * @param jsonObject
	 * @param tableName
	 */
	public void insertOne(JSONObject jsonObject, String tableName) {
		Document document = Document.parse(jsonObject.toJSONString());
		MongoCollection<Document> collection = mongoTemplate.getCollection(tableName);

		collection.insertOne(document);
	}
	
	/**
	 * 插入or修改
	 * 
	 * @param jsonObject
	 * @param tableName
	 */
	public void insertOrUpdate(JSONObject jsonObject, String tableName) {
		Document document = Document.parse(jsonObject.toJSONString());
		MongoCollection<Document> collection = mongoTemplate.getCollection(tableName);
		
		Bson bson = Filters.eq("id", jsonObject.getString("id"));
		FindIterable<Document> find = collection.find(bson);
		MongoCursor<Document> iterator = find.iterator();
		JSONObject uniqueObj = null;
		try {
			while (iterator.hasNext()) {
				Document uniqueDoc = iterator.next();
				uniqueObj = JSON.parseObject(uniqueDoc.toJson());

				break;
			}
		} catch (Exception e) {
		}
		
		if(uniqueObj == null) {
			collection.insertOne(document);
		}else{
			Bson updateBson = Filters.eq("id", jsonObject.getString("id"));
			collection.updateOne(updateBson, new Document("$set", document), new UpdateOptions().upsert(true));
		}
	}

	/**
	 * 根据主键Id修改
	 * 
	 * @param jsonObject
	 * @param tableName
	 */
	public void updateById(JSONObject jsonObject, String tableName) {
		Document document = Document.parse(jsonObject.toJSONString());
		MongoCollection<Document> collection = mongoTemplate.getCollection(tableName);

		Bson bson = Filters.eq("id", jsonObject.getString("id"));
		collection.updateOne(bson, new Document("$set", document), new UpdateOptions().upsert(true));
	}

	/**
	 * 根据主键Id删除
	 * 
	 * @param pkId
	 * @param tableName
	 */
	public void deleteById(String pkId, String tableName) {
		MongoCollection<Document> collection = mongoTemplate.getCollection(tableName);

		Bson bson = Filters.eq("id", pkId);
		collection.deleteOne(bson);
	}
	
	
	/**
	 * 根据条件删除
	 * 
	 * @param pkId
	 * @param tableName
	 */
	public void deleteByQuery(QueryParam queryParam, String tableName) {
		MongoCollection<Document> collection = mongoTemplate.getCollection(tableName);
		Bson queryBson = getSearchParam(queryParam);
		
		collection.deleteOne(queryBson);
	}
	
	

	/**
	 * 根据主键查询单个bean
	 * 
	 * @param pkId
	 * @param tableName
	 * @return
	 */
	public JSONObject selectById(String pkId, String tableName) {
		MongoCollection<Document> collection = mongoTemplate.getCollection(tableName);

		Bson bson = Filters.eq("id", pkId);
		FindIterable<Document> find = collection.find(bson);
		MongoCursor<Document> iterator = find.iterator();

		JSONObject jsonObject = null;

		try {
			while (iterator.hasNext()) {
				Document document = iterator.next();
				jsonObject = JSON.parseObject(document.toJson());
				jsonObject.remove("_id");
				String billState = jsonObject.getString("billState");
				if(StringUtils.isNotBlank(billState)) {
					jsonObject.put("billState", Integer.parseInt(billState));
				}
				break;
			}
		} catch (Exception e) {
		}

		return jsonObject;
	}

	private static List<String> getMonthRange(String startDateStr, String endDateStr) {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
		Date startDate = null;
		Date endDate = null;
		try {
			startDate = sdf.parse(startDateStr);
			endDate = sdf.parse(endDateStr);
		} catch (ParseException e) {
			e.printStackTrace();
		}
		List<String> monthList = new ArrayList<>();
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(startDate);
		while (!calendar.getTime().after(endDate)) {
			int year = calendar.get(Calendar.YEAR);
			int month = calendar.get(Calendar.MONTH) + 1;
			String monthStr = String.format("%04d-%02d", year, month);
			monthList.add(monthStr);
			calendar.add(Calendar.MONTH, 1);
		}
		return monthList;
	}
	
	@SuppressWarnings("unchecked")
	private Bson getSearchParam(QueryParam queryParam) {
		List<Bson> searchParamList = new ArrayList<>();
		/** 模糊查询配置的字段 */
		Bson fuzzyBson = null;
		if (StringUtils.isNotBlank(queryParam.getSearchText()) && !ListUtil.isEmpty(queryParam.getFuzzyFields())) {
			String searchText = queryParam.getSearchText();
			List<String> fuzzyFields = queryParam.getFuzzyFields();
			if (queryParam.getFuzzyFields().size() == 1) {
				fuzzyBson = Filters.regex(queryParam.getFuzzyFields().get(0), searchText);
			} else {
				fuzzyBson = Filters.or(new Iterable<Bson>() {
					@Override
					public Iterator<Bson> iterator() {
						List<Bson> bsonList = new ArrayList<>();
						for (int i = 0; i < fuzzyFields.size(); i++) {
							bsonList.add(Filters.regex(queryParam.getFuzzyFields().get(i), searchText));
						}
						return bsonList.iterator();
					}

				});
			}
		}
		if(fuzzyBson != null) {
			searchParamList.add(fuzzyBson);
		}
		/** 基础字段查询 */
		List<Bson> paramBsonList = new ArrayList<>();
		Map<String, Parameter> params = queryParam.getParams();
		for (Map.Entry<String, Parameter> entry : params.entrySet()) {
			Parameter param = entry.getValue();
			String type = param.getType();
			switch (type) {
			case QueryParam.EQ:
				paramBsonList.add(Filters.eq(entry.getKey(), null == param.getValue() ? null : param.getValue()));
				break;
			case QueryParam.NE:
				paramBsonList.add(Filters.ne(entry.getKey(), null == param.getValue() ? null : param.getValue()));
				break;
			case QueryParam.IN:
				if (param.getValue() instanceof List) {
					paramBsonList.add(Filters.in(entry.getKey(), ((List<?>) param.getValue()).toArray()));
				} else if (param.getValue() instanceof String) {
					String[] paramArr = param.getValue().toString().split(",");
					paramBsonList.add(Filters.in(entry.getKey(), paramArr));

				}
				break;
			case QueryParam.NOT_IN:
				if (param.getValue() instanceof List) {
					paramBsonList.add(Filters.nin(entry.getKey(), ((List<?>) param.getValue()).toArray()));
				} else if (param.getValue() instanceof String) {
					String[] paramArr = param.getValue().toString().split(",");
					paramBsonList.add(Filters.nin(entry.getKey(), paramArr));
				}
				break;
			case QueryParam.LIKE:
				if (param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
					paramBsonList.add(Filters.regex(entry.getKey(), param.getValue().toString()));
				}
				break;
			case QueryParam.BETWEEN:
				if (param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
					String[] paramArr = param.getValue().toString().split(",");
					if ("dateInSelect".equals(param.getInputType()) && paramArr[0].length() == 7) {
						/** 日期区间 月份区间 */
						if (paramArr[0].equals(paramArr[1])) {
							/** 月份相同，就是一个月 */
							paramBsonList.add(Filters.regex(entry.getKey(), paramArr[0]));
						} else {
							paramBsonList.add(Filters.in(entry.getKey(), getMonthRange(paramArr[0], paramArr[1]).iterator()));
						}
					} else {
						paramBsonList.add(Filters.and(Filters.lte(entry.getKey(), paramArr[0]), Filters.gte(entry.getKey(), paramArr[1])));
					}
				}
				break;
			case QueryParam.LT:
				if (param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
					paramBsonList.add(Filters.lt(entry.getKey(), param.getValue()));
				}
				break;
			case QueryParam.LE:
				if (param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
					paramBsonList.add(Filters.lte(entry.getKey(), param.getValue()));
				}
				break;
			case QueryParam.GT:
				if (param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
					paramBsonList.add(Filters.gt(entry.getKey(), param.getValue()));
				}
				break;
			case QueryParam.GE:
				if (param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
					paramBsonList.add(Filters.gte(entry.getKey(), param.getValue()));
				}
				break;
			}
		}
		Bson paramBson = null;
		if(paramBsonList.size() > 0) {
			paramBson = Filters.and(new Iterable<Bson>() {
				@Override
				public Iterator<Bson> iterator() {
					return paramBsonList.iterator();
				}
			});
		}
		if(paramBson != null) {
			searchParamList.add(paramBson);
		}
		/** 具体模糊查询字段 */
		List<Bson> searchObjectList = new ArrayList<>();
		Bson searchObjectBson = null;
		if(StringUtils.isNotBlank(queryParam.getSearchObject())) {
			JSONObject searchObject = JSONObject.parseObject(queryParam.getSearchObject());
			for(String key: searchObject.keySet()){
				searchObjectList.add(Filters.regex(key, searchObject.getString(key)));
			}
			
			searchObjectBson = Filters.and(new Iterable<Bson>() {
				@Override
				public Iterator<Bson> iterator() {
					return searchObjectList.iterator();
				}
			});
		}
		if(searchObjectBson != null) {
			searchParamList.add(searchObjectBson);
		}
		
		if(searchParamList.size() > 0) {
			return Filters.and(new Iterable<Bson>() {
				@Override
				public Iterator<Bson> iterator() {
					return searchParamList.iterator();
				}
			});
		}
		return null;
	}

	/**
	 * 查询列表
	 * 
	 * @param queryParam
	 * @return
	 */
	public List<JSONObject> selectList(QueryParam queryParam, String tableName) {
		MongoCollection<Document> collection = mongoTemplate.getCollection(tableName);

		Bson queryBson = getSearchParam(queryParam);
		
		//排序
		BasicDBObject sortField = new BasicDBObject();	
		Map<String, String> orderMap = queryParam.getOrderMap();
		for(String key : orderMap.keySet()){
			String value = orderMap.get(key);
			if(QueryParam.ASC.equals(value.toLowerCase())){
				sortField.put(key, 1);
			}else if(QueryParam.DESC.equals(value.toLowerCase())){
				sortField.put(key, -1);
			}
		}
		
		FindIterable<Document> find = null;
		if(queryBson != null) {
			find = collection.find(queryBson);
		}else{
			find = collection.find();
		}
		if(!sortField.isEmpty()) {
			find.sort(sortField);
		}
		MongoCursor<Document> iterator = find.iterator();

		List<JSONObject> resultJson = new ArrayList<>();
		try {
			while (iterator.hasNext()) {
				Document document = iterator.next();
				JSONObject jsonObject = JSON.parseObject(document.toJson());
				jsonObject.remove("_id");
				String billState = jsonObject.getString("billState");
				if(StringUtils.isNotBlank(billState)) {
					jsonObject.put("billState", Integer.parseInt(billState));
				}
				resultJson.add(jsonObject);
			}
		} catch (Exception e) {
		}

		return resultJson;
	}
	
	/**
	 * 查询总数
	 * 
	 * @param queryParam
	 * @param tableName
	 * @return
	 */
	public Long selectCount(QueryParam queryParam, String tableName) {
		MongoCollection<Document> collection = mongoTemplate.getCollection(tableName);
		Bson queryBson = getSearchParam(queryParam);
		long totalCount = 0l;
		if(queryBson != null) {
			totalCount = collection.countDocuments(queryBson);
		}else{
			totalCount = collection.countDocuments();
		}
		return totalCount;
	}

	/**
	 * 查询分页对象
	 *
	 * @param queryParam 查询参数
	 * @param isEs 是否查询es
	 * @return
	 */
	public IPage<JSONObject> queryPage(QueryParam queryParam, String tableName) {
		IPage<JSONObject> page = new Page<>();
		if(queryParam.getPageSize() == -1) {
			page.setRecords(selectList(queryParam, tableName));
			return page;
		}else{
			MongoCollection<Document> collection = mongoTemplate.getCollection(tableName);

			Bson queryBson = getSearchParam(queryParam);
			
			//排序
			BasicDBObject sortField = new BasicDBObject();	
			Map<String, String> orderMap = queryParam.getOrderMap();
			for(String key : orderMap.keySet()){
				String value = orderMap.get(key);
				if(QueryParam.ASC.equals(value.toLowerCase())){
					sortField.put(key, 1);
				}else if(QueryParam.DESC.equals(value.toLowerCase())){
					sortField.put(key, -1);
				}
			}
			
			FindIterable<Document> find = null;
			if(queryBson != null) {
				find = collection.find(queryBson);
			}else{
				find = collection.find();
			}
			if(!sortField.isEmpty()) {
				find.sort(sortField);
			}
			if(queryParam.getPageIndex() < 1) {
				queryParam.setPageIndex(1);
			}
			find.skip((queryParam.getPageIndex() - 1) * queryParam.getPageSize()).limit(queryParam.getPageSize());
			MongoCursor<Document> iterator = find.iterator();
			List<JSONObject> resultJson = new ArrayList<>();
			try {
				while (iterator.hasNext()) {
					Document document = iterator.next();
					JSONObject jsonObject = JSON.parseObject(document.toJson());
					jsonObject.remove("_id");
					String billState = jsonObject.getString("billState");
					if(StringUtils.isNotBlank(billState)) {
						jsonObject.put("billState", Integer.parseInt(billState));
					}
					resultJson.add(jsonObject);
				}
			} catch (Exception e) {
			}
			page.setRecords(resultJson);
			
			long totalCount = 0l;
			if(queryBson != null) {
				totalCount = collection.countDocuments(queryBson);
			}else{
				totalCount = collection.countDocuments();
			}
			page.setTotal(totalCount);
			
			return page;
		}
	}
}
