package com.ejianc.business.othprice.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.othprice.bean.OtherQuoteEntity;
import com.ejianc.business.othprice.service.IOtherQuoteService;
import com.ejianc.business.othprice.vo.OtherQuoteVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.io.Serializable;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 报价记录
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("otherQuote")
@Api(tags = {"报价记录"})
public class OtherQuoteController implements Serializable {
    private static final long serialVersionUID = -6476701269896286249L;
    private static final String RULE_CODE = "OTHER_QUOTE";
    private static final String BILL_CODE = "BT202311000002";
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    @Resource
    private IBillTypeApi billTypeApi;
    @Resource
    private IBillCodeApi billCodeApi;
    @Resource
    private IOrgApi iOrgApi;
    @Resource
    private IOtherQuoteService service;
    @Autowired
    private SessionManager sessionManager;
    @Resource
    private IAttachmentApi attachmentApi;


    /**
     * 保存或更新
     *
     * @param materialQuoteVO 报价记录
     *
     * @return {@link CommonResponse}<{@link OtherQuoteVO}>
     */
    @ApiOperation(value = "保存或更新")
    @PostMapping(value = "/saveOrUpdate")
    public CommonResponse<OtherQuoteVO> saveOrUpdate(@RequestBody OtherQuoteVO materialQuoteVO) {
        OtherQuoteEntity entity = BeanMapper.map(materialQuoteVO, OtherQuoteEntity.class);
        // 校验当前时间是否超过询价单报价截止时间
        service.checkQuoteEndDate(entity.getInquiryId());
        if (entity.getId() == null || entity.getId() == 0) {
            BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), materialQuoteVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        service.saveOrUpdate(entity, false);
        OtherQuoteVO vo = BeanMapper.map(entity, OtherQuoteVO.class);
        return CommonResponse.success("保存或修改单据成功！", vo);
    }


    /**
     * 查询详情
     *
     * @param id id
     *
     * @return {@link CommonResponse}<{@link OtherQuoteVO}>
     */
    @ApiOperation(value = "查询详情")
    @GetMapping(value = "/queryDetail")
    public CommonResponse<OtherQuoteVO> queryDetail(Long id) {
        OtherQuoteEntity entity = service.selectById(id);
        if (entity == null) {
            return CommonResponse.error("未查询到报价记录数据！");
        }
        OtherQuoteVO vo = BeanMapper.map(entity, OtherQuoteVO.class);
        return CommonResponse.success("查询详情数据成功！", vo);
    }


    /**
     * 批量删除单据
     *
     * @param vos vos
     *
     * @return {@link CommonResponse}<{@link String}>
     */
    @ApiOperation(value = "批量删除单据")
    @PostMapping(value = "/delete")
    public CommonResponse<String> delete(@RequestBody List<OtherQuoteVO> vos) {
        if (ListUtil.isNotEmpty(vos)) {
            for (OtherQuoteVO vo : vos) {
                OtherQuoteEntity otherQuoteEntity = service.selectById(vo.getId());
                if (otherQuoteEntity.getBillState() == 1){
                    return CommonResponse.error("删除失败,已提交的报价单无法删除" );
                }
                CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
                if (!resp.isSuccess()) {
                    return CommonResponse.error("删除失败！" + resp.getMsg());
                }
            }
        }
        service.removeByIds(vos.stream().map(OtherQuoteVO::getId).collect(Collectors.toList()), true);
        return CommonResponse.success("删除成功！");
    }


    /**
     * 查询列表
     *
     * @param param 参数
     *
     * @return {@link CommonResponse}<{@link IPage}<{@link OtherQuoteVO}>>
     */
    @ApiOperation(value = "查询列表")
    @PostMapping(value = "/queryList")
    public CommonResponse<IPage<OtherQuoteVO>> queryList(@RequestBody QueryParam param) {
        LinkedHashMap<String, String> orderMap = new LinkedHashMap<>();
        orderMap.put("create_time", "desc");
        param.getOrderMap().putAll(orderMap);
        IPage<OtherQuoteEntity> page = service.queryPage(param, false);
        IPage<OtherQuoteVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        List<OtherQuoteVO> materialQuoteVOS = BeanMapper.mapList(page.getRecords(), OtherQuoteVO.class);
        if (CollectionUtils.isNotEmpty(materialQuoteVOS)) {
            List<Long> quoteIds = materialQuoteVOS.stream().map(OtherQuoteVO::getId).collect(Collectors.toList());
            JSONObject fileParam = new JSONObject();
            fileParam.put("sourceIds", quoteIds);
            JSONArray sourceTypeArr = new JSONArray();
            sourceTypeArr.add("quote");
            fileParam.put("sourceTypeList", JSON.toJSONString(sourceTypeArr));
            CommonResponse<List<AttachmentVO>> listCommonResponse = attachmentApi.queryAllBySourceIdList(fileParam);
            List<AttachmentVO> fileData = listCommonResponse.getData();
            Map<Long, AttachmentVO> fileMap = new HashMap<>();
            if (listCommonResponse.isSuccess() && CollectionUtils.isNotEmpty(fileData)) {
                fileMap = fileData.stream().collect(Collectors.toMap(AttachmentVO::getSourceId, Function.identity(), (e1, e2) -> e1));
            }

            for (OtherQuoteVO materialQuoteVO : materialQuoteVOS) {
                if (fileMap.containsKey(materialQuoteVO.getId())) {
                    AttachmentVO attachmentVO = fileMap.get(materialQuoteVO.getId());
                    materialQuoteVO.setQuoteFileId(attachmentVO.getId());
                    materialQuoteVO.setQuoteFileName(attachmentVO.getFileName());
                }
            }
        }
        pageData.setRecords(materialQuoteVOS);
        return CommonResponse.success("查询列表数据成功！", pageData);
    }

    /**
     * 指定供应商登录、所有供应商登录校验查看权限
     *
     * @param inquiryId   询价单id
     * @param phoneNumber 电话号码
     *
     * @return {@link CommonResponse}<{@link Void}>
     */
    @GetMapping(value = "/checkViewAuthority")
    public CommonResponse<Void> checkViewAuthority(@RequestParam(value = "inquiryId") Long inquiryId,
                                                   @RequestParam(value = "phoneNumber") String phoneNumber) {
        service.checkViewAuthority(inquiryId, phoneNumber);
        return CommonResponse.success();
    }

    /**
     * 指定供应商报价校验报价权限
     *
     * @param inquiryId   询价单id
     * @param phoneNumber 电话号码
     *
     * @return {@link CommonResponse}<{@link Void}>
     */
    @GetMapping(value = "/checkQuotationAuthority")
    public CommonResponse<Void> checkQuotationAuthority(@RequestParam(value = "inquiryId") Long inquiryId,
                                                        @RequestParam(value = "phoneNumber") String phoneNumber) {
        service.checkQuotationAuthority(inquiryId, phoneNumber);
        return CommonResponse.success();
    }


    /**
     * 获取验证码
     *
     * @param inquiryId   询价单id
     * @param phoneNumber 电话号码
     *
     * @return {@link CommonResponse}<{@link Void}>
     *
     * @throws Exception 异常
     */
    @GetMapping(value = "/fetchVerifyCode")
    public CommonResponse<Void> fetchVerifyCode(@RequestParam(value = "inquiryId") Long inquiryId,
                                                @RequestParam(value = "phoneNumber") String phoneNumber) throws Exception {
        service.fetchVerifyCode(inquiryId, phoneNumber);
        return CommonResponse.success("验证码发送成功！");
    }


    /**
     * 登录
     *
     * @param phoneNumber 电话号码
     * @param validate    验证码
     *
     * @return {@link CommonResponse}<{@link String}>
     */
    @ApiOperation(value = "登录")
    @GetMapping(value = "/login")
    public CommonResponse<String> login(@RequestParam(value = "phoneNumber") String phoneNumber,
                                        @RequestParam(value = "validate") String validate) {
        service.login(phoneNumber, validate);
        return CommonResponse.success("登录成功！", phoneNumber);
    }

    /**
     * 收回
     *
     * @param id 报价单id
     *
     * @return {@link CommonResponse}<{@link String}>
     */
    @ApiOperation(value = "收回")
    @GetMapping(value = "/withdraw")
    public CommonResponse<String> withdraw(@RequestParam(value = "id") Long id) {
        service.withdraw(id);
        return CommonResponse.success("收回成功！");
    }

    /**
     * 修改单据状态
     *
     * @param id        id
     * @param billState 单据状态
     *
     * @return {@link CommonResponse}<{@link String}>
     */
    @ApiOperation(value = "修改单据状态")
    @GetMapping(value = "/changeStatus")
    public CommonResponse<String> changeStatus(@RequestParam(value = "id") Long id,
                                               @RequestParam(value = "billState") Integer billState) {
        service.changeStatus(id, billState);
        return CommonResponse.success("单据状态修改成功！");
    }

    /**
     * 最近一次的报价记录
     *
     * @param phoneNumber 报价手机号码（当前登录的手机号码）
     *
     * @return {@link CommonResponse}<{@link OtherQuoteVO}>
     */
    @ApiOperation(value = "最近一次的报价记录")
    @GetMapping(value = "/recentQuote")
    public CommonResponse<OtherQuoteVO> recentQuote(@RequestParam(value = "phoneNumber") String phoneNumber) {
        return CommonResponse.success("单据状态修改成功！", service.recentQuote(phoneNumber));
    }
}
