package com.ejianc.foundation.print.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.foundation.metadata.api.IMdClassApi;
import com.ejianc.foundation.metadata.vo.MdClassVO;
import com.ejianc.foundation.print.bean.BusinessObjectEntity;
import com.ejianc.foundation.print.bean.TemplateEntity;
import com.ejianc.foundation.print.model.TemplatePageModel;
import com.ejianc.foundation.print.service.IBusinessObjectService;
import com.ejianc.foundation.print.service.ITemplateService;
import com.ejianc.foundation.print.vo.TemplateVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillTypeVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

@RestController
@RequestMapping("/template/")
public class TemplateController {

    @Value("${oms.tenantid}")
    private Long BASE_TANANT_ID;

    @Autowired
    private ITemplateService templateService;

    @Autowired
    private IBillTypeApi billTypeApi;

    @Autowired
    private IMdClassApi mdClassApi;

    @Autowired
    private IBusinessObjectService businessObjectService;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    /**
     * 分页查询模板列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping("page")
    public CommonResponse<JSONObject> page(@RequestBody QueryParam queryParam) {
        JSONObject resp = new JSONObject();
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        IPage<TemplateEntity> pageData = templateService.queryPage(queryParam, false);
        resp.put("data", pageData);
        return CommonResponse.success(resp);
    }

    /**
     * 模板新增、更新
     *
     * @param templateVO
     * @return
     */
    @PostMapping("save")
    public CommonResponse<TemplateVO> save(@RequestBody TemplateVO templateVO) {
        TemplateVO resp = null;
        TemplateEntity dbEntity = null;
        if(TemplateVO.TEMPLATE_TYPE_CUSTOM_FORM.equals(templateVO.getTemplateType())) {
            //自定义模板
            if(null == templateVO.getId()) {
                TemplateEntity entityByCode = templateService.detailByCode(templateVO.getTemplateCode(), InvocationInfoProxy.getTenantid());
                if(null != entityByCode) {
                    return CommonResponse.error("保存失败，编码重复！");
                }
                //根据单据类型找到单据类型对应的元数据Id
                CommonResponse<BillTypeVO> billType = billTypeApi.getById(templateVO.getBillTypeId());
                CommonResponse<MdClassVO> clsResp = mdClassApi.queryDetail(billType.getData().getMetadataId());
                if(1 == clsResp.getCode()) {
                    logger.error("保存模板失败，查询单据对应元数据源失败: " + clsResp.getMsg());
                    return CommonResponse.error("保存失败，查询单据对应元数据源失败！");
                }
                BusinessObjectEntity bo = businessObjectService.getByEntityIdAndProjectId(clsResp.getData().getId(), clsResp.getData().getProjectId());
                if(null == bo) {
                    return CommonResponse.error("保存失败，该单据类型下为创建业务对象数据源！");
                }
                dbEntity = BeanMapper.map(templateVO, TemplateEntity.class);
                dbEntity.setBoCode(bo.getCode());
            } else {
                dbEntity = templateService.detail(templateVO.getId());
                dbEntity.setTemplateName(templateVO.getTemplateName());
                dbEntity.setDetailUrl(templateVO.getDetailUrl());
            }
        }

        templateService.saveOrUpdate(dbEntity, false);
        resp = BeanMapper.map(dbEntity, TemplateVO.class);

        return CommonResponse.success("保存成功！", resp);
    }


    @PostMapping("saveDesign")
    public CommonResponse<String> saveDesign(@RequestBody TemplateVO templateVO) {
        TemplateEntity dbEntity = templateService.detail(templateVO.getId());
        dbEntity.setContent(templateVO.getContent());
        TemplatePageModel model = new TemplatePageModel(templateVO.getContent());
        dbEntity.setFieldRelationJson(model.getRelationJson());

        templateService.saveOrUpdate(dbEntity, false);
        return CommonResponse.success("保存成功！");
    }

    /**
     * 模板复制
     *
     * @param templateVO
     * @return
     */
    @PostMapping("copyTemplate")
    public CommonResponse<String> copyTemplate(@RequestBody TemplateVO templateVO) {
        TemplateEntity dbEntity = templateService.detail(templateVO.getId());
        TemplateEntity saveEntity = new TemplateEntity();

        saveEntity.setTemplateCode(templateVO.getTemplateCode());
        saveEntity.setTemplateName(templateVO.getTemplateName());
        saveEntity.setContent(dbEntity.getContent());
        saveEntity.setFieldRelationJson(dbEntity.getFieldRelationJson());
        saveEntity.setBoCode(dbEntity.getBoCode());
        saveEntity.setBillTypeId(dbEntity.getBillTypeId());
        saveEntity.setBillTypeCode(dbEntity.getBillTypeCode());
        saveEntity.setTemplateType(dbEntity.getTemplateType());
        templateService.saveOrUpdate(saveEntity, false);

        return CommonResponse.success("模板复制成功！");
    }

    /**
     * 根据主键Id查询模板详情
     *
     * @param id
     * @return
     */
    @GetMapping("detail")
    public CommonResponse<TemplateVO> detail(@RequestParam Long id) {
        TemplateEntity dbEntity = templateService.detail(id);
        return CommonResponse.success(BeanMapper.map(dbEntity, TemplateVO.class));
    }

    @PostMapping("checkCode")
    public CommonResponse<String> checkCode(@RequestBody QueryParam queryParam) {
        List<TemplateEntity> list = templateService.queryList(queryParam, false);
        if(CollectionUtils.isNotEmpty(list)) {
            return CommonResponse.error("编码重复！");
        }
        return CommonResponse.success("编码可以使用！");
    }

    /**
     * 根据code查询模板详情
     *
     * @param code
     * @return
     */
    @GetMapping("detailByCode")
    public CommonResponse<TemplateVO> detailByCode(@RequestParam String code) {
        TemplateEntity dbEntity = templateService.detailByCode(code, InvocationInfoProxy.getTenantid());
        return CommonResponse.success(BeanMapper.map(dbEntity, TemplateVO.class));
    }

    /**
     * 根据主键Id删除对应模板
     *
     * @param id
     * @return
     */
    @PostMapping("delete")
    public CommonResponse<String> delete(@RequestBody Long id) {
        templateService.removeById(id, false);

        return CommonResponse.success("删除成功！");
    }

    @PostMapping(value = "queryList")
    public CommonResponse<List<TemplateVO>> queryList(@RequestBody TemplateVO templateVO) {
        List<TemplateVO> voList = new ArrayList<>();
        QueryParam queryParam = new QueryParam();
        if(null != templateVO.getBillTypeId()) {
            queryParam.getParams().put("billTypeId", new Parameter(QueryParam.EQ, templateVO.getBillTypeId()));
        }
        if(StringUtils.isNotBlank(templateVO.getBillTypeCode())) {
            queryParam.getParams().put("billTypeCode", new Parameter(QueryParam.EQ, templateVO.getBillTypeCode()));
        }
        Long tenantId = InvocationInfoProxy.getTenantid();
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, tenantId));
        List<TemplateEntity> list = templateService.queryList(queryParam, false);
        if(CollectionUtils.isEmpty(list) && !tenantId.equals(BASE_TANANT_ID)) {
            //查询公共模板
            queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, BASE_TANANT_ID));
            list = templateService.queryList(queryParam, false);
        }
        if(CollectionUtils.isNotEmpty(list)) {
            voList = BeanMapper.mapList(list, TemplateVO.class);
        }

        return CommonResponse.success(voList);
    }
}
