package com.ejianc.business.win.service.impl;

import com.ejianc.business.bid.enums.BidStatusEnum;
import com.ejianc.business.bid.service.IProjectService;
import com.ejianc.business.win.bean.WinEntity;
import com.ejianc.business.win.service.IWinService;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.Date;

@Service("win")
public class WinBpmServiceImpl implements ICommonBusinessService {

	private final IWinService service;
	private final IProjectService projectService;
	private final SessionManager sessionManager;
	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	public WinBpmServiceImpl(IWinService service, IProjectService projectService, SessionManager sessionManager) {
		this.service = service;
		this.projectService = projectService;
		this.sessionManager = sessionManager;
	}


	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		WinEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		UserContext userContext = sessionManager.getUserContext();

		// 区分提交和审批
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		// 回写投标立项生效状态
		projectService.updateProjectStatus(entity.getProjectId(), "1".equals(entity.getBidWinFlag()) ? BidStatusEnum.未中标.getTypeName() : BidStatusEnum.中标.getTypeName());

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		/*
			单据撤回逻辑：
			1、单据操作包含：保存、返回列表、编辑、提交、删除，规则同工程云一致。
			2、单据撤回：当存在审批流程时且第一个人没有审批时，支持撤回；当不存在审批流程且没有下游业务时支持撤回。
			3、单据弃审：审批过程中，支持后台弃审；审批通过后，没有下游业务时支持弃审。
		*/

		WinEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 回写投标立项生效状态
		projectService.updateProjectStatus(entity.getProjectId(), BidStatusEnum.投标中.getTypeName());

		return CommonResponse.success("单据撤回成功");
	}

}
