package com.ejianc.business.review.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.review.bean.ReviewEntity;
import com.ejianc.business.review.service.IReviewService;
import com.ejianc.business.review.vo.ReviewVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 投标文件评审实体
 *
 * @author generator
 */
@RestController
@RequestMapping("review")
public class ReviewController implements Serializable {
	private static final long serialVersionUID = 3536444014220643625L;

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private static final String RULE_CODE = "PROBID_REVIEW";
	private static final String BILL_CODE = "EJCBT202207000021";
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final IReviewService service;
	private final SessionManager sessionManager;

	public ReviewController(IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, IReviewService service, SessionManager sessionManager) {
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
		this.sessionManager = sessionManager;
	}

	/**
	 * @Description saveOrUpdate 新增或者修改
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<ReviewVO> saveOrUpdate(@RequestBody ReviewVO saveOrUpdateVO) {
		ReviewEntity entity = BeanMapper.map(saveOrUpdateVO, ReviewEntity.class);
		if (entity.getId() == null || entity.getId() == 0) {
			// 校验只能有一个未生效的单据
			service.checkOnlyOne(entity.getProjectId());
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}

		// 单据名称、单据类型
		entity.setBillName("投标文件评审");
		entity.setBillType(BILL_CODE);
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());

		service.saveOrUpdate(entity, false);
		ReviewVO vo = BeanMapper.map(entity, ReviewVO.class);
		return CommonResponse.success("保存或修改单据成功！", vo);
	}

	/**
	 * @param id
	 *
	 * @Description queryDetail 查询详情
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<ReviewVO> queryDetail(Long id) {
		ReviewEntity entity = service.selectById(id);
		ReviewVO vo = BeanMapper.map(entity, ReviewVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * @Description delete 批量删除单据
	 * @Param [ids]
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<ReviewVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (ReviewVO vo : vos) {
				// 参数是单据类型编码字符串 根据需求是否打开下面代码
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(ReviewVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * @param param
	 *
	 * @Description queryList 查询列表
	 * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<IPage<ReviewVO>> queryList(@RequestBody QueryParam param) {
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("projectName");
		fuzzyFields.add("bidUnitName");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.getParams().put("bid_unit_id", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

		IPage<ReviewEntity> page = service.queryPage(param, false);
		IPage<ReviewVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), ReviewVO.class));

		return CommonResponse.success("查询列表数据成功！", pageData);
	}


	/**
	 * @param param
	 *
	 * @Description 导出
	 * @Return void
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("billCode");
		fuzzyFields.add("projectName");
		fuzzyFields.add("bidUnitName");
		fuzzyFields.add("employeeName");

		param.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		param.setPageIndex(1);
		param.setPageSize(-1);

		param.getParams().put("bid_unit_id", new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

		IPage<ReviewEntity> page = service.queryPage(param, false);
		List<ReviewVO> reviews = new ArrayList<>();

		page.getRecords().forEach(e -> {
			ReviewVO review = BeanMapper.map(e, ReviewVO.class);
			review.setBillStateName(BillStateEnum.getEnumByStateCode(review.getBillState()).getDescription());
			reviews.add(review);
		});

		Map<String, Object> beans = new HashMap<>();
		beans.put("records", reviews);
		ExcelExport.getInstance().export("review-export.xlsx", beans, response);
	}



	/**
	 * 招标文件评审打印详情处理
	 *
	 * @param detailJsonStr
	 * @return
	 */
	@PostMapping("/handlePrintDetail")
	@ResponseBody
	public CommonResponse<JSONObject> handlePrintDetail(@RequestBody String detailJsonStr) {
		logger.info("进入招标文件评审打印详情处理！");
//		{
//			"detailJson": {
//				"Review": {
//
//				}
//			}
//		}

		if(StringUtils.isBlank(detailJsonStr)) {
			return CommonResponse.success(null);
		}
		JSONObject data = JSONObject.parseObject(detailJsonStr);
		if(null == data) {
			return CommonResponse.success(null);
		}
		logger.info("获得打印文件信息：{}", JSONObject.toJSONString(data));

		JSONObject detailJson = data.getJSONObject("detailJson");
		if(null == detailJson) {
			return CommonResponse.success(null);
		}
		JSONObject mainData = detailJson.getJSONObject("Review");

		logger.info("获得招标文件信息：{}", JSONObject.toJSONString(mainData));


		String content = "";
		String result = "";

		//放置经营负责人相关信息
		List<JSONObject> reviewContentManageList = JSONObject.parseArray(mainData.getString("ReviewContentManage"), JSONObject.class);
		logger.info("获得经营负责人信息：{}", JSONObject.toJSONString(reviewContentManageList));
		for (JSONObject manage : reviewContentManageList) {
			content = manage.getString("manageReviewContent");
			result = manage.getString("manageReviewResult");

			if (StringUtils.isNotBlank(content)){
				switch (content) {
					case "标书条款是否合理、清楚明了？":
						mainData.put("manageReviewContentOne", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "投标书内容是否满足招标文件要求？":
						mainData.put("manageReviewContentTwo", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "投标书条款是否与招标文件及合同法等相关法律法规及标准要求相吻合？":
						mainData.put("manageReviewContentThree", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否投标？":
						mainData.put("manageReviewContentFour", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
				}
			}
		}
		//放置 技术负责人 相关信息
		List<JSONObject> reviewContentTechnologyList = JSONObject.parseArray(mainData.getString("ReviewContentTechnology"), JSONObject.class);
		logger.info("获得技术负责人信息：{}", JSONObject.toJSONString(reviewContentTechnologyList));
		for (JSONObject technology : reviewContentTechnologyList) {
			content = technology.getString("technologyReviewContent");
			result = technology.getString("technologyReviewResult");
			if (StringUtils.isNotBlank(content)){
				switch (content) {
					case "标书中的技术标准要求是否能够满足？":
						mainData.put("technologyReviewContentOne", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "保障质量、安全、进度方面的技术措施是否能够满足？":
						mainData.put("technologyReviewContentTwo", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否具备满足工程质量、工期、进度等要求的资源？":
						mainData.put("technologyReviewContentThree", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "标书中对安全、环境方面的条款是否能够满足？":
						mainData.put("technologyReviewContentFour", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否投标？":
						mainData.put("technologyReviewContentFive", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
				}
			}
		}
		//放置 预算负责人 相关信息
		List<JSONObject> ReviewContentBudgetList = JSONObject.parseArray(mainData.getString("ReviewContentBudget"), JSONObject.class);
		logger.info("获得预算负责人信息：{}", JSONObject.toJSONString(ReviewContentBudgetList));
		for (JSONObject budget : ReviewContentBudgetList) {
			content = budget.getString("budgetReviewContent");
			result = budget.getString("budgetReviewResult");
			if (StringUtils.isNotBlank(content)){
				switch (content) {
					case "编制投标预算/决算等计价依据是否合理、清楚明了？":
						mainData.put("budgetReviewContentOne", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "投标报价是否能够接受？":
						mainData.put("budgetReviewContentTwo", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "投标报价范围是否清楚？":
						mainData.put("budgetReviewContentThree", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否投标？":
						mainData.put("budgetReviewContentFour", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
				}
			}
		}
		//放置 财务负责人 相关信息
		List<JSONObject> ReviewContentFinanceList = JSONObject.parseArray(mainData.getString("ReviewContentFinance"), JSONObject.class);
		logger.info("获得财务负责人信息：{}", JSONObject.toJSONString(ReviewContentFinanceList));
		for (JSONObject finance : ReviewContentFinanceList) {
			content = finance.getString("financeReviewContent");
			result = finance.getString("financeReviewResult");
			if (StringUtils.isNotBlank(content)){
				switch (content) {
					case "公司是否具备投标保函/保证金支付的能力？":
						mainData.put("financeReviewContentOne", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否能够接受款项支付或垫款条件？":
						mainData.put("financeReviewContentTwo", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否具备工程款支付方面的能力？":
						mainData.put("financeReviewContentThree", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否投标？":
						mainData.put("financeReviewContentFour", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
				}
			}
		}
		//放置 直管部 （分公司）负责人 相关信息
		List<JSONObject> ReviewContentDirectList = JSONObject.parseArray(mainData.getString("ReviewContentDirect"), JSONObject.class);
		logger.info("获得直管部 （分公司）负责人信息：{}", JSONObject.toJSONString(ReviewContentDirectList));
		for (JSONObject direct : ReviewContentDirectList) {
			content = direct.getString("directReviewContent");
			result = direct.getString("directReviewResult");
			if (StringUtils.isNotBlank(content)){
				switch (content) {
					case "是否具备工程施工的能力？":
						mainData.put("directReviewContentOne", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否同意上述条款内容？":
						mainData.put("directReviewContentTwo", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否同意其他承诺条款？":
						mainData.put("directReviewContentThree", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否投标？":
						mainData.put("directReviewContentFour", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
				}
			}
		}
		//放置 经营部负责人 相关信息
		List<JSONObject> ReviewContentBusinessList = JSONObject.parseArray(mainData.getString("ReviewContentBusiness"), JSONObject.class);
		logger.info("获得经营部负责人信息：{}", JSONObject.toJSONString(ReviewContentBusinessList));
		for (JSONObject business : ReviewContentBusinessList) {
			content = business.getString("businessReviewContent");
			result = business.getString("businessReviewResult");
			if (StringUtils.isNotBlank(content)){
				switch (content) {
					case "是否同意上述条款内容？":
						mainData.put("businessReviewContentOne", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否同意其他承诺条款？":
						mainData.put("businessReviewContentTwo", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
					case "是否投标？":
						mainData.put("businessReviewContentThree", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
						break;
				}
			}
		}
		//放置 经营副总经理审核意见 相关信息
		List<JSONObject> ReviewContentDeputyList = JSONObject.parseArray(mainData.getString("ReviewContentDeputy"), JSONObject.class);
		logger.info("获得经营副总经理审核意见信息：{}", JSONObject.toJSONString(ReviewContentDeputyList));
		for (JSONObject deputy : ReviewContentDeputyList) {
			content = deputy.getString("deputyReviewContent");
			result = deputy.getString("deputyReviewResult");
			if (StringUtils.isNotBlank(content) && content.equals("是否投标？")){
				mainData.put("deputyReviewContentOne", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
			}
		}
		//放置 总经理审批意见 相关信息
		List<JSONObject> ReviewContentGmList = JSONObject.parseArray(mainData.getString("ReviewContentGm"), JSONObject.class);
		logger.info("获得总经理审批意见信息：{}", JSONObject.toJSONString(ReviewContentGmList));
		for (JSONObject gm : ReviewContentGmList) {
			content = gm.getString("gmReviewContent");
			result = gm.getString("gmReviewResult");
			if (StringUtils.isNotBlank(content) && content.equals("是否投标？")){
				mainData.put("gmReviewContentOne", StringUtils.isNotBlank(result) ? ("0".equals(result) ? "否" : "是") : "");
			}
		}

		logger.info("内容赋值后的mainData：{}", JSONObject.toJSONString(mainData));
		detailJson.put("Review", mainData);
		logger.info("detailJson：{}", JSONObject.toJSONString(detailJson));
		return CommonResponse.success(detailJson);
	}


}
