package com.ejianc.business.budget.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.budget.bean.BudgetDetailEntity;
import com.ejianc.business.budget.bean.BudgetEntity;
import com.ejianc.business.budget.service.IBudgetDetailService;
import com.ejianc.business.budget.service.IBudgetRecordService;
import com.ejianc.business.budget.service.IBudgetService;
import com.ejianc.business.budget.utils.TreeNodeBUtil;
import com.ejianc.business.budget.vo.BudgetSubRefVO;
import com.ejianc.business.budget.vo.BudgetVO;
import com.ejianc.business.enums.ChangeTypeEnum;
import com.ejianc.business.utils.BigDecimalUtil;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IMaterialApi;
import com.ejianc.foundation.share.vo.MaterialVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ImportTemplate;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 预算管理
 *
 * @author generator
 *
 */
@Controller
@RequestMapping("budget")
public class BudgetController implements Serializable {
	private static final long serialVersionUID = 1L;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi iOrgApi;

    @Autowired
    private IBudgetService budgetService;

    @Autowired
    private IBudgetRecordService budgetRecordService;

    @Autowired
    private IBudgetDetailService budgetDetailService;

    @Autowired
    private IMaterialApi materialApi;

    /**
     * @Description saveOrUpdate 新增或者修改
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<BudgetVO> saveOrUpdate(@RequestBody BudgetVO saveorUpdateVO) {
    	BudgetVO vo = budgetService.insertOrUpdate(saveorUpdateVO);
    	return CommonResponse.success("保存或修改单据成功！",vo);
    }

    /**
     * @Description queryDetail 查询详情
     * @param id
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<BudgetVO> queryDetail(Long id) {
    	BudgetVO vo = budgetService.queryDetail(id, true);
        return CommonResponse.success("查询详情数据成功！",vo);
    }

    /**
     * @Description saveOrUpdate 查询详情
     * @param id
     * @Return com.ejianc.framework.core.response.CommonResponse<BudgetVO>
     */
    @RequestMapping(value = "/queryRecordDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<BudgetVO> queryRecordDetail(Long id) {
        BudgetVO vo = budgetRecordService.queryDetail(id, true);
        return CommonResponse.success("查询详情数据成功！",vo);
    }

    /**
     * @Description delete 批量删除单据
     * @Param [ids]
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<BudgetVO> vos) {
        if(CollectionUtils.isNotEmpty(vos)) {
            budgetService.removeByIds(vos.stream().map(BudgetVO::getId).collect(Collectors.toList()),true);
        }
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<BudgetVO>> queryList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId",new Parameter(QueryParam.IN, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

        IPage<BudgetEntity> page = budgetService.queryPage(param,false);
        IPage<BudgetVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
  		pageData.setRecords(BeanMapper.mapList(page.getRecords(), BudgetVO.class));

        return CommonResponse.success("查询列表数据成功！",pageData);
    }


    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        param.getParams().put("tenant_id",new Parameter(QueryParam.EQ,InvocationInfoProxy.getTenantid()));
        param.setPageIndex(1);
        param.setPageSize(-1);
        /** 数据隔离 本下 没有组织orgId的删除下面代码 */
        param.getParams().put("orgId",new Parameter(QueryParam.IN,iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        List<BudgetEntity> list = budgetService.queryList(param);
        //todo:字段翻译等等
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        ExcelExport.getInstance().export("Budget-export.xlsx", beans, response);
    }

    /**
     * 导入模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping(value = "/download")
    @ResponseBody
    public void download(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "budget-detail-import.xlsx", "预算清单导入模板");
    }

    /**
     * excel导入
     *
     * @param request
     * @return
     */
    @RequestMapping(value = "/excelImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelImport(HttpServletRequest request, HttpServletResponse response) {
        return budgetService.excelImport(request, response);
    }

    /**
     * @Description 预算参照
     * @Return void
     */
    @RequestMapping(value = "/budgetRefData", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<BudgetDetailEntity>> budgetRefData(@RequestParam Integer
                                                                          pageNumber, @RequestParam Integer pageSize,
                                                                  String condition,
                                                                  String searchObject,
                                                                  String searchText) {
        QueryParam queryParam = new QueryParam();

        //只查询审批通过的预算
        List<Integer> billStateList = new ArrayList<>();
        billStateList.add(BillStateEnum.PASSED_STATE.getBillStateCode());
        billStateList.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.IN, billStateList));
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));



        if (StringUtils.isNotBlank(condition)) {
            Map<String, Object> conditionMap = JSONObject.parseObject(condition, Map.class);
            if (null != conditionMap.get("projectId")) {
                Long projectId = Long.valueOf(conditionMap.get("projectId").toString());
                queryParam.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
            }
        }
        /** 数据隔离 本下 */
        queryParam.getParams().put("orgId",new Parameter(QueryParam.IN,iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        //查询所选项目下  所有预算参照主表数据
        IPage<BudgetEntity> pageData = budgetService.queryPage(queryParam,false);

        List<BudgetEntity> records = pageData.getRecords();
        IPage<BudgetDetailEntity> budgetDetailEntityIPage = new Page<>(); ;
        for (BudgetEntity budget :records) {
            queryParam = new QueryParam();
            queryParam.getParams().put("budget_id",new Parameter(QueryParam.EQ , budget.getId()));
            queryParam.getParams().put("level_no",new Parameter(QueryParam.EQ, 4));
            queryParam.getFuzzyFields().add("materialName");
            if (StringUtils.isNotBlank(searchText)) {
                queryParam.setSearchText(searchText);
            }
            queryParam.setPageSize(pageSize);
            queryParam.setPageIndex(pageNumber);
            Page<BudgetDetailEntity> page = new Page<>(pageNumber, pageSize);
            budgetDetailEntityIPage = budgetDetailService.queryPageList(page,queryParam);
            //budgetDetailEntityIPage = budgetDetailService.queryPage(queryParam, false);
            List<Long> ids = new ArrayList<>();
            budgetDetailEntityIPage.getRecords().forEach(budgetDetail->{
                ids.add(budgetDetail.getMaterialId());
            });
            //根据物资id获取物资分类id
            CommonResponse<List<MaterialVO>> result = materialApi.queryMaterialByIds(ids);
            if(result.getCode()==0){
                //请求成功
                List<MaterialVO> data = result.getData();
                for (BudgetDetailEntity budgetDetail :budgetDetailEntityIPage.getRecords()) {
                    for (MaterialVO material: data) {
                        if(budgetDetail.getMaterialId().equals(material.getId())){
                            budgetDetail.setMaterialCategoryId(material.getCategoryId());
                            budgetDetail.setMaterialCategoryName(material.getCategoryName());
                            budgetDetail.setSpec(material.getSpec());
                            budgetDetail.setUnitName(material.getUnitName());
                            budgetDetail.setMaterialPrice(BigDecimalUtil.safeDiv(budgetDetail.getSumMoney(),budgetDetail.getSumNum()));
                        }
                    }
                }
            }
        }

        return CommonResponse.success("查询参照数据成功！",budgetDetailEntityIPage);
    }

    /**
     * @Description 预算参照（分包合同专用）
     * @Return void
     */
    @RequestMapping(value = "/budgetRefForSub", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<BudgetSubRefVO>> budgetRefForSub(@RequestParam(value = "projectId")Long projectId) {
        List<BudgetSubRefVO> result = new ArrayList<>();
        //只查询审批通过的预算
        List<Integer> billStateList = new ArrayList<>();
        billStateList.add(BillStateEnum.PASSED_STATE.getBillStateCode());
        billStateList.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
        LambdaQueryWrapper<BudgetEntity> budgetQuery = new LambdaQueryWrapper<>();
        budgetQuery.in(BudgetEntity::getBillState, billStateList);
        budgetQuery.eq(BudgetEntity::getTenantId, InvocationInfoProxy.getTenantid());
        budgetQuery.eq(BudgetEntity::getProjectId, projectId);
        /** 数据隔离 本下 */
        budgetQuery.in(BudgetEntity::getOrgId, iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        //查询所选项目下  所有预算参照主表数据
        List<BudgetEntity> budgetEntities = budgetService.list(budgetQuery);
        if(CollectionUtils.isEmpty(budgetEntities)){
            return CommonResponse.error("该项目没有做预算！");
        }

        List<Integer> levelList = new ArrayList<>();
        levelList.add(1);
        levelList.add(2);
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("budget_id", new Parameter(QueryParam.EQ , budgetEntities.get(0).getId()));
        queryParam.getParams().put("level_no", new Parameter(QueryParam.IN, levelList));
        queryParam.getParams().put("changeType", new Parameter(QueryParam.NE, ChangeTypeEnum.中止.getCode()));
        queryParam.getOrderMap().put("treeIndex", "asc");
        List<BudgetDetailEntity> detailEntityList = budgetDetailService.queryList(queryParam, false);
        if(CollectionUtils.isNotEmpty(detailEntityList)) {
            for (int i = 0; i < detailEntityList.size(); i++) {
                BudgetDetailEntity cdEntity = detailEntityList.get(i);
                cdEntity.setTid(cdEntity.getId().toString());
                cdEntity.setTpid(cdEntity.getParentId() != null && cdEntity.getParentId() > 0 ? cdEntity.getParentId().toString() : "");

                BudgetSubRefVO subRefVO = BeanMapper.map(cdEntity, BudgetSubRefVO.class);
                int num = i + 1;
                if(i < 9){
                    subRefVO.setCode("0" + num);
                }else{
                    subRefVO.setCode(String.valueOf(num));
                }
                subRefVO.setName(StringUtils.isNotBlank(cdEntity.getBuildingNo()) ? cdEntity.getBuildingNo() : cdEntity.getBuildContent());
                subRefVO.setNum(cdEntity.getCostNum());
                subRefVO.setBudgetDetailId(cdEntity.getId());
                result.add(subRefVO);
            }
            result = TreeNodeBUtil.buildTree(result);
        }

        return CommonResponse.success("查询参照数据成功！", result);
    }
}
