package com.ejianc.business.procost.service.impl;

import com.ejianc.business.procost.bean.UnitDetailEntity;
import com.ejianc.business.procost.service.IUnitDetailService;
import com.ejianc.business.procost.utils.TreeNodeBUtil;
import com.ejianc.business.procost.vo.UnitDetailVO;
import com.ejianc.business.procost.vo.UnitVO;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;

import com.ejianc.business.procost.mapper.UnitMapper;
import com.ejianc.business.procost.bean.UnitEntity;
import com.ejianc.business.procost.service.IUnitService;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 主材量价对比主表
 *
 * @author generator
 */
@Service("unitService")
public class UnitServiceImpl extends BaseServiceImpl<UnitMapper, UnitEntity> implements IUnitService {

    @Autowired
    private IUnitDetailService unitDetailService;

    /**
     * 批量进行保存
     *
     * @param voList 数据
     * @return 保存结果
     */
    @Override
    public List<UnitVO> saveUnitBatch(List<UnitVO> voList) {
        if (CollectionUtils.isEmpty(voList)) return voList;
        // 删除原来的数据
        baseMapper.deleteUnitAll();
        List<UnitEntity> entityList = BeanMapper.mapList(voList, UnitEntity.class);
        super.saveOrUpdateBatch(entityList);
        return this.buildUnitTree(entityList);
    }

    /**
     * 批量保存子表数据
     *
     * @param detailList 子表数据
     * @return 保存结果
     */
    @Override
    public List<UnitDetailVO> saveUnitDetailBatch(List<UnitDetailVO> detailList) {
        if (CollectionUtils.isEmpty(detailList)) return detailList;
        // 子表父id相同，取第一个
        Long pid = detailList.get(0).getPid();
        baseMapper.deleteUnitDetailByPid(pid);
        List<UnitDetailEntity> detailEntityList = BeanMapper.mapList(detailList, UnitDetailEntity.class);
        unitDetailService.saveOrUpdateBatch(detailEntityList);
        return BeanMapper.mapList(detailEntityList, UnitDetailVO.class);
    }

    /**
     * 查询列表
     *
     * @param param 查询条件
     * @return 查询结果
     */
    @Override
    public List<UnitVO> queryUnitList(QueryParam param) {
        List<UnitEntity> unitEntityList = super.queryList(param, false);
        List<UnitVO> voList = this.buildUnitTree(unitEntityList);
        return voList;
    }

    /**
     * 构建树
     *
     * @param unitEntityList 实体
     * @return 结果
     */
    private List<UnitVO> buildUnitTree(List<UnitEntity> unitEntityList) {
        if (CollectionUtils.isNotEmpty(unitEntityList)) {
            return TreeNodeBUtil.buildTree(BeanMapper.mapList(unitEntityList, UnitVO.class));
        }
        return new ArrayList<>();
    }

    /**
     * 根据父id查询字表列表
     *
     * @param pid 父ID
     * @return 查询结果
     */
    @Override
    public List<UnitDetailVO> queryUnitDetailByPid(Long pid) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("pid", new Parameter(QueryParam.EQ, pid));
        List<UnitDetailEntity> list = unitDetailService.queryList(queryParam, false);
        return BeanMapper.mapList(list, UnitDetailVO.class);
    }

    /**
     * 查询末级及末级对应的子集列表
     *
     * @return 查询结果
     */
    @Override
    public Map<Long, List<UnitDetailVO>> queryUnitDetailMap() {
        Map<Long, List<UnitDetailVO>> rtnMap = new HashMap<>();
        // 查询所有父id
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("parentId", new Parameter(QueryParam.NE, null));
        List<UnitEntity> unitList = super.queryList(queryParam, false);
        if (CollectionUtils.isEmpty(unitList)) return rtnMap;
        List<Long> parentIdList = unitList.stream().map(UnitEntity::getParentId).collect(Collectors.toList());
        // 查询末级
        QueryParam cParam = new QueryParam();
        cParam.getParams().put("id", new Parameter(QueryParam.NOT_IN, parentIdList));
        List<UnitEntity> cList = super.queryList(cParam, false);
        if (CollectionUtils.isEmpty(cList)) return rtnMap;
        List<Long> idList = cList.stream().map(UnitEntity::getId).collect(Collectors.toList());
        // 查询子集
        QueryParam detailParam = new QueryParam();
        detailParam.getParams().put("pid", new Parameter(QueryParam.IN, idList));
        List<UnitDetailEntity> detailList = unitDetailService.queryList(detailParam, false);
        if (CollectionUtils.isEmpty(detailList)) return rtnMap;
        List<UnitDetailVO> detailVoList = BeanMapper.mapList(detailList, UnitDetailVO.class);
        rtnMap = detailVoList.stream().collect(Collectors.groupingBy(UnitDetailVO::getPid));
        return rtnMap;
    }
}
