package com.ejianc.business.pricelib.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.pricelib.bean.RentPriceGuideDetailEntity;
import com.ejianc.business.pricelib.service.IRentPriceGuideDetailService;
import com.ejianc.business.pricelib.vo.PriceLibVO;
import com.ejianc.business.pricelib.vo.PriceTrendVo;
import com.ejianc.business.pricelib.vo.RentPriceContractVO;
import com.ejianc.business.rent.bean.*;
import com.ejianc.business.rent.enums.OutFlagEnum;
import com.ejianc.business.rent.service.IRentContractService;
import com.ejianc.business.rent.service.IRentSettlementService;
import com.ejianc.business.settle.enums.SettleTypeEnum;
import com.ejianc.business.utils.DateUtils;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IPriceDepotPowerSetApi;
import com.ejianc.foundation.share.api.IProjectSetApi;
import com.ejianc.foundation.share.api.IShareEquipmentApi;
import com.ejianc.foundation.share.vo.EquipmentCategoryVO;
import com.ejianc.foundation.share.vo.EquipmentVO;
import com.ejianc.foundation.share.vo.dto.ProjectPoolDTO;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;

import com.ejianc.business.pricelib.mapper.RentPriceContractMapper;
import com.ejianc.business.pricelib.bean.RentPriceContractEntity;
import com.ejianc.business.pricelib.service.IRentPriceContractService;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 设备租赁价格库-清单明细实体
 * 
 * @author generator
 * 
 */
@Service("rentPriceContractService")
public class RentPriceContractServiceImpl extends BaseServiceImpl<RentPriceContractMapper, RentPriceContractEntity> implements IRentPriceContractService{
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IPriceDepotPowerSetApi priceDepotPowerSetApi;
    @Autowired
    private IRentPriceGuideDetailService priceGuideDetailService;
    @Autowired
    private IRentContractService contractService;
    @Autowired
    private IShareEquipmentApi shareEquipmentApi;
    @Autowired
    private IProjectSetApi projectSetApi;
    @Autowired
    private IRentSettlementService settleService;
    @Autowired
    private IOrgApi iOrgApi;

    /**
     * 查询价格库汇总信息
     *
     * @param param 查询参数
     * @return 查询结果
     */
    @Override
    public IPage<PriceLibVO> queryPriceLibListByTime(QueryParam param) {
        // 初始化参数
        logger.info("查询价格库，查询参数：{}", JSONObject.toJSONString(param));
        Date startDate;
        Date endDate;
        Map<String, Parameter> paramMap = param.getParams();
        if (paramMap.containsKey("startDate") && paramMap.containsKey("endDate")) {
            startDate = DateUtils.createDate(paramMap.get("startDate").getValue().toString());
            endDate = DateUtils.createDate(paramMap.get("endDate").getValue().toString());
            // 清除日期信息，保留其他查询条件
            paramMap.remove("startDate");
            paramMap.remove("endDate");
        }else {
            throw new BusinessException("查询日期不能为空");
        }
        List<Long> orgIdList = this.queryOrgIdList();
        List<PriceLibVO> priceLibList = this.queryPriceLibListAll(startDate, endDate, orgIdList, param);
        IPage<PriceLibVO> pageDate = new Page<>(param.getPageIndex(), param.getPageSize(), priceLibList.size());
        // 进行分页处理
        pageDate.setRecords(priceLibList.stream().skip((param.getPageIndex() - 1) * param.getPageSize())
                .limit(param.getPageSize()).collect(Collectors.toList()));
        return pageDate;
    }

    /**
     * 定时处理合同价格库明细
     * 如果传入日期为空，则为初始化，处理全部合同（已提交、审批通过）明细
     *
     * @param date 日期
     */
    @Override
    public void savePriceContractByTiming(Date date) {
        logger.info("========================价格库合同明细定时任务执行开始===========================");
        QueryParam queryParam = new QueryParam();
        List<RentContractEntity> contractEntityList;
        String outLogDate = "";
        if (date != null) {
            // 日期不为空，则取大于上日日期数据
            Date qDate = DateUtils.calculationDay(date, -1);
            outLogDate = DateUtils.transformationDefaultDate(date);
            logger.info("执行定时任务，查询参数：{}", qDate);
            contractEntityList = baseMapper.getContractListByTime(qDate);
        }
        else {
            // 日期为空，默认为初始化，取提交后的数据
            queryParam.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));
            contractEntityList = contractService.queryList(queryParam, false);
        }

        logger.info("执行日期：{}，查询合同结果：{}", outLogDate, JSONObject.toJSONString(contractEntityList));
        // 合同列表为空，不处理
        if (CollectionUtils.isEmpty(contractEntityList)) return;
        // 查询合同明细
        List<Long> contractIdList = contractEntityList.stream().map(RentContractEntity::getId).collect(Collectors.toList());
        // 清单明细id对应清单名称（档案分类名称）
        List<RentPriceContractVO> detailVOList = new ArrayList<>();
        //添加日租明细
        List<RentDayDetailedEntity> dayDetailEntityList = baseMapper.getDayDetailListByContractId(contractIdList);
        if (CollectionUtils.isNotEmpty(dayDetailEntityList)){
            for (RentDayDetailedEntity dayDetail : dayDetailEntityList) {
                RentPriceContractVO contractDetailedVO = new RentPriceContractVO();

                contractDetailedVO.setContractId(dayDetail.getPid());
                contractDetailedVO.setId(dayDetail.getId());
                contractDetailedVO.setSourceId(dayDetail.getId());
                contractDetailedVO.setContractTaxRate(dayDetail.getTax());
                contractDetailedVO.setContractNum(new BigDecimal(dayDetail.getRentNum()));
                contractDetailedVO.setContractPrice(dayDetail.getDayRentNotTaxPrice());
                contractDetailedVO.setContractTaxPrice(dayDetail.getDayRentPrice());
                contractDetailedVO.setContractMny(dayDetail.getRentMny());
                contractDetailedVO.setContractTaxMny(dayDetail.getRentTaxMny());
                contractDetailedVO.setMemo(dayDetail.getMemo());
                contractDetailedVO.setSpec(dayDetail.getSpec());
                contractDetailedVO.setManufacturer(dayDetail.getProductionManufactor());
                contractDetailedVO.setDocId(dayDetail.getEquipmentId());
                contractDetailedVO.setDocCode(dayDetail.getEquipmentCode());
                contractDetailedVO.setDocName(dayDetail.getName());
                contractDetailedVO.setDocCategoryId(dayDetail.getCategoryId());
                contractDetailedVO.setDocCategoryName(dayDetail.getCategoryName());
                //计量单位
                contractDetailedVO.setUnitName(dayDetail.getUnitName());
                contractDetailedVO.setRentType(1);
                detailVOList.add(contractDetailedVO);
            }
        }
        //添加月租明细
        List<RentMonthDetailedEntity> monthDetailEntityList = baseMapper.getMonthDetailListByContractId(contractIdList);
        if (CollectionUtils.isNotEmpty(monthDetailEntityList)){
            for (RentMonthDetailedEntity monthDetail : monthDetailEntityList) {
                RentPriceContractVO contractDetailedVO = new RentPriceContractVO();

                contractDetailedVO.setContractId(monthDetail.getPid());
                contractDetailedVO.setId(monthDetail.getId());
                contractDetailedVO.setSourceId(monthDetail.getId());
                contractDetailedVO.setContractTaxRate(monthDetail.getTax());
                contractDetailedVO.setContractNum(new BigDecimal(monthDetail.getRentNum()));
                contractDetailedVO.setContractPrice(monthDetail.getMonthRentNotTaxPrice());
                contractDetailedVO.setContractTaxPrice(monthDetail.getMonthRentPrice());
                contractDetailedVO.setContractMny(monthDetail.getRentMny());
                contractDetailedVO.setContractTaxMny(monthDetail.getRentTaxMny());
                contractDetailedVO.setMemo(monthDetail.getMemo());
                contractDetailedVO.setSpec(monthDetail.getSpec());
                contractDetailedVO.setManufacturer(monthDetail.getProductionManufactor());
                contractDetailedVO.setDocId(monthDetail.getEquipmentId());
                contractDetailedVO.setDocCode(monthDetail.getEquipmentCode());
                contractDetailedVO.setDocName(monthDetail.getName());
                contractDetailedVO.setDocCategoryId(monthDetail.getCategoryId());
                contractDetailedVO.setDocCategoryName(monthDetail.getCategoryName());
                //计量单位
                contractDetailedVO.setUnitName(monthDetail.getUnitName());
                contractDetailedVO.setRentType(2);
                detailVOList.add(contractDetailedVO);
            }
        }
        //添加工程量租明细
        List<RentQuantitiesDetailedEntity> quantitiesDetailEntityList = baseMapper.getQuantitiesDetailListByContractId(contractIdList);
        if (CollectionUtils.isNotEmpty(quantitiesDetailEntityList)){
            for (RentQuantitiesDetailedEntity quantitiesDetail : quantitiesDetailEntityList) {
                RentPriceContractVO contractDetailedVO = new RentPriceContractVO();

                contractDetailedVO.setContractId(quantitiesDetail.getPid());
                contractDetailedVO.setId(quantitiesDetail.getId());
                contractDetailedVO.setSourceId(quantitiesDetail.getId());
                contractDetailedVO.setContractTaxRate(quantitiesDetail.getTax());
                contractDetailedVO.setContractNum(new BigDecimal(quantitiesDetail.getRentNum()));
                contractDetailedVO.setContractPrice(quantitiesDetail.getNotTaxQuantitiesPrice());
                contractDetailedVO.setContractTaxPrice(quantitiesDetail.getQuantitiesPrice());
                contractDetailedVO.setContractMny(quantitiesDetail.getRentMny());
                contractDetailedVO.setContractTaxMny(quantitiesDetail.getRentTaxMny());
                contractDetailedVO.setMemo(quantitiesDetail.getMemo());
                contractDetailedVO.setSpec(quantitiesDetail.getSpec());
                contractDetailedVO.setManufacturer(quantitiesDetail.getProductionManufactor());
                contractDetailedVO.setDocId(quantitiesDetail.getEquipmentId());
                contractDetailedVO.setDocCode(quantitiesDetail.getEquipmentCode());
                contractDetailedVO.setDocName(quantitiesDetail.getName());
                contractDetailedVO.setDocCategoryId(quantitiesDetail.getCategoryId());
                contractDetailedVO.setDocCategoryName(quantitiesDetail.getCategoryName());
                //计量单位
                contractDetailedVO.setUnitName(quantitiesDetail.getUnitName());
                contractDetailedVO.setRentType(3);
                detailVOList.add(contractDetailedVO);
            }
        }

        // 合同明细列表为空，不处理
        if (CollectionUtils.isEmpty(detailVOList)) return;

        //查合同最终结算日期
        QueryParam settleParam = new QueryParam();
        settleParam.getParams().put("contract_id", new Parameter(QueryParam.IN, contractIdList));
        settleParam.getParams().put("bill_state", new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));
        settleParam.getParams().put("settlement_type", new Parameter(QueryParam.EQ, SettleTypeEnum.完工.getCode()));
        List<RentSettlementEntity> settleList = settleService.queryList(settleParam);
        Map<Long, Date> settleDateMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(settleList)){
            settleDateMap = settleList.stream()
                    .collect(Collectors.toMap(RentSettlementEntity::getContractId, RentSettlementEntity::getSettlementDate,
                            (key1, key2) -> key2));
        }


        // 查询档案内码
        List<Long> docCategoryIdList = detailVOList.stream().map(RentPriceContractVO::getDocCategoryId)
                .distinct().collect(Collectors.toList());
        List<Long> docIs = detailVOList.stream().filter(item -> null != item.getContractNum()).map(RentPriceContractVO::getDocId).distinct().collect(Collectors.toList());
        CommonResponse<List<EquipmentCategoryVO>> equipmentDocResponse = shareEquipmentApi
                .queryCategoryListByIds(docCategoryIdList);


        Map<Long, String> docCategoryMap = new HashMap<>();
        logger.info("查询设备档案内码，查询结果：{}", JSONObject.toJSONString(equipmentDocResponse));
        if (equipmentDocResponse.isSuccess()) {
            List<EquipmentCategoryVO> equipmentDocCategoryVOList = equipmentDocResponse.getData();
            docCategoryMap = equipmentDocCategoryVOList.stream()
                    .collect(Collectors.toMap(EquipmentCategoryVO::getId, EquipmentCategoryVO::getInnerCode,
                            (key1, key2) -> key2));
            CommonResponse<List<EquipmentVO>> equipmentDocItemResponse = shareEquipmentApi
                    .queryEquipmentItemByIds(docIs);
            if(equipmentDocItemResponse.isSuccess()) {
                logger.info("查询设备档案清单列表：{}", JSONObject.toJSONString(equipmentDocItemResponse.getData()));
            } else {
                logger.error("查询设备档案信息失败，{}", JSONObject.toJSONString(equipmentDocItemResponse));
            }

        }else {
            logger.error("查询设备档案信分类息失败，{}", JSONObject.toJSONString(equipmentDocResponse));
        }


        // 查询项目信息
        List<Long> projectIdList = contractEntityList.stream().map(RentContractEntity::getProjectId)
                .distinct().collect(Collectors.toList());
        CommonResponse<List<ProjectPoolDTO>> projectResponse = projectSetApi.getProjectListIds(projectIdList);
        logger.info("查询项目地址，查询结果：{}", JSONObject.toJSONString(projectResponse));
        Map<Long, String> projectAreaMap = new HashMap<>();
        Map<Long, String> projectAreaNameMap = new HashMap<>();
        if (projectResponse.isSuccess()) {
            List<ProjectPoolDTO> projectPoolList = projectResponse.getData();
            projectAreaMap = projectPoolList.stream().filter(t -> StringUtils.isNotEmpty(t.getArea()))
                    .collect(Collectors.toMap(ProjectPoolDTO::getId, ProjectPoolDTO::getArea, (key1, key2) -> key2));
            projectAreaNameMap = projectPoolList.stream().filter(t -> StringUtils.isNotEmpty(t.getAreaName()))
                    .collect(Collectors
                            .toMap(ProjectPoolDTO::getId, ProjectPoolDTO::getAreaName, (key1, key2) -> key2));
        }

        // 清单明细按合同id分组
        Map<Long, List<RentPriceContractVO>> detailMap = detailVOList.stream()
                .collect(Collectors.groupingBy(RentPriceContractVO::getContractId));

        // 处理合同及合同明细
        // 进行插入数据组装
        List<RentPriceContractEntity> saveEntityList = new ArrayList<>();
        for (RentContractEntity entity : contractEntityList) {
            List<RentPriceContractVO> detailList = detailMap.get(entity.getId());
            if (CollectionUtils.isEmpty(detailList)) continue;

            for (RentPriceContractVO detailVO : detailList) {
                // 档案id为空不做价格库处理（大类不处理）
                if (detailVO.getDocId() == null) continue;
                RentPriceContractEntity rentPriceContractEntity = BeanMapper.map(entity, RentPriceContractEntity.class);
                //设置最终结算日期
                rentPriceContractEntity.setSettlementDate(settleDateMap.get(entity.getId()));

                //主合同信息
                rentPriceContractEntity.setContractId(entity.getId());
                rentPriceContractEntity.setContractName(entity.getContractName());
                rentPriceContractEntity.setContractCode(entity.getContractCode());
                rentPriceContractEntity.setSignDate(entity.getSignedDate());
                rentPriceContractEntity.setPerformanceStatus(entity.getContractPerformanceState());

                rentPriceContractEntity.setPcCardUrl("/ejc-proequipment-frontend/#/leaseContract/cardList?id=" + entity.getId() + "&showTitle=true&supplementFlag=" + entity.getSupplementFlag());

                rentPriceContractEntity.setArea(projectAreaMap.get(entity.getProjectId())); // 项目地区  查询项目获取
                rentPriceContractEntity.setAreaName(projectAreaNameMap.get(entity.getProjectId())); // 项目地区  查询项目获取

                rentPriceContractEntity.setTargetResultId(entity.getTargetResultId());
                rentPriceContractEntity.setSourceId(detailVO.getId());

                // 子表信息
                rentPriceContractEntity.setContractTaxRate(detailVO.getContractTaxRate());
                rentPriceContractEntity.setContractNum(detailVO.getContractNum());
                rentPriceContractEntity.setContractPrice(null != detailVO.getContractPrice() ? detailVO.getContractPrice() : BigDecimal.ZERO);
                rentPriceContractEntity.setContractTaxPrice(null != detailVO.getContractTaxPrice() ? detailVO.getContractTaxPrice() : BigDecimal.ZERO);
                rentPriceContractEntity.setContractMny(null != detailVO.getContractMny() ? detailVO.getContractMny() : BigDecimal.ZERO);
                rentPriceContractEntity.setContractTaxMny(null != detailVO.getContractTaxMny() ? detailVO.getContractTaxMny() : BigDecimal.ZERO);
                rentPriceContractEntity.setMemo(detailVO.getMemo());
                rentPriceContractEntity.setSpec(detailVO.getSpec());
                rentPriceContractEntity.setManufacturer(detailVO.getManufacturer());

                rentPriceContractEntity.setDocId(detailVO.getDocId());
                rentPriceContractEntity.setDocCode(detailVO.getDocCode());
                rentPriceContractEntity.setDocName(detailVO.getDocName());
                rentPriceContractEntity.setDocCategoryId(detailVO.getDocCategoryId());
                rentPriceContractEntity.setDocCategoryName(detailVO.getDocCategoryName());
                rentPriceContractEntity.setRentType(detailVO.getRentType());

                rentPriceContractEntity.setDocInnerCode(docCategoryMap.get(detailVO.getDocCategoryId())); // 分类内码，查询获取

                //计量单位
                rentPriceContractEntity.setUnitName(detailVO.getUnitName());

                rentPriceContractEntity.setOutFlag(OutFlagEnum.未移除.getCode());
                rentPriceContractEntity.setId(null);
                rentPriceContractEntity.setCreateTime(null);
                rentPriceContractEntity.setCreateUserCode(null);
                rentPriceContractEntity.setUpdateTime(null);
                rentPriceContractEntity.setUpdateUserCode(null);
                saveEntityList.add(rentPriceContractEntity);
            }
        }
        // 查询是否为修改
        if (CollectionUtils.isEmpty(saveEntityList)) return;
        List<Long> sourceDetailIdList = saveEntityList.stream().map(RentPriceContractEntity::getSourceId)
                .collect(Collectors.toList());
        // 查询数据库
        QueryParam priceParam = new QueryParam();
        priceParam.getParams().put("sourceId", new Parameter(QueryParam.IN, sourceDetailIdList));
        List<RentPriceContractEntity> queryPriceList = super.queryList(priceParam, false);
        if (CollectionUtils.isNotEmpty(queryPriceList)) {
            // sourceId,id
            Map<Long, RentPriceContractEntity> queryPriceMap = queryPriceList.stream()
                    .collect(Collectors.toMap(RentPriceContractEntity::getSourceId, Function.identity(),
                            (key1, key2) -> key2));
            for (RentPriceContractEntity entity : saveEntityList) {
                RentPriceContractEntity queryEntity = queryPriceMap.get(entity.getSourceId());
                if (queryEntity != null) {
                    entity.setId(queryEntity.getId());
                    entity.setOutFlag(queryEntity.getOutFlag());
                }
            }
        }
        // 批量变更保存数据
//        logger.info("入库数据：{}", JSONObject.toJSONString(saveEntityList));
        super.saveOrUpdateBatch(saveEntityList);
        // 删除bill_state非1和3的数据
        // 不使用物理删除，使用逻辑删除
        QueryWrapper<RentPriceContractEntity> wrapper = new QueryWrapper<>();
        wrapper.notIn("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(),
                BillStateEnum.PASSED_STATE.getBillStateCode()));
        super.remove(wrapper, false);
        // 删除dr!=0的数据
        // 删除的id
        List<Long> delIdList = saveEntityList.stream().filter(t -> t.getDr() != 0).map(RentPriceContractEntity::getId)
                .collect(Collectors.toList());
//        logger.info("删除数据id：{}",JSONObject.toJSONString(delIdList));
        if (CollectionUtils.isNotEmpty(delIdList)) {
            super.removeByIds(delIdList, false);
        }
        logger.info("========================价格库合同明细定时任务执行结束===========================");
    }

    /**
     * 根据档案id查询价格库信息
     *
     * @param param 查询参数
     * @return 查询结果
     */
    @Override
    public PriceLibVO queryPriceLibDetail(QueryParam param) {
        Map<String, Date> checkMap = this.checkQueryParam(param);
        Date startDate = checkMap.get("startDate");
        Date endDate = checkMap.get("endDate");
        List<Long> orgIdList = this.queryOrgIdList();
        List<PriceLibVO> priceLibList = this.queryPriceLibListAll(startDate, endDate, orgIdList, param);
//        if (CollectionUtils.isEmpty(priceLibList)) throw new BusinessException("查询详情失败！");
        return priceLibList.get(0);
    }

    @Override
    public List<PriceTrendVo> queryPriceTrendContractList(QueryParam param) {
        Map<String, Date> checkMap = this.checkQueryParam(param);
        Date startDate = checkMap.get("startDate");
        Date endDate = checkMap.get("endDate");
        List<Long> orgIdList = this.queryOrgIdList();
        List<PriceTrendVo> rtnList;
        // list转Map，同一天只会有一个价格
        Map<String, PriceTrendVo> trendMap = new HashMap<>();
        // 查询价格库合同明细
        List<RentPriceContractEntity> priceContractList = this.queryPriceContractList(startDate, orgIdList, param);
        if (CollectionUtils.isNotEmpty(priceContractList)) {
            Map<Date, List<RentPriceContractEntity>> contractMap = priceContractList.stream()
                    .collect(Collectors.groupingBy(RentPriceContractEntity::getSignDate));
            for (Map.Entry<Date, List<RentPriceContractEntity>> entry : contractMap.entrySet()) {
                List<RentPriceContractEntity> list = entry.getValue();
                if (CollectionUtils.isEmpty(list)) continue;
                RentPriceContractEntity rentPriceContractEntity = list.get(0);
                List<BigDecimal> priceList = list.stream().map(RentPriceContractEntity::getContractPrice).collect(Collectors.toList());
                List<BigDecimal> taxPriceList = list.stream().map(RentPriceContractEntity::getContractTaxPrice).collect(Collectors.toList());
                BigDecimal num = BigDecimal.ZERO;
                BigDecimal mny = BigDecimal.ZERO;
                BigDecimal taxMny = BigDecimal.ZERO;
                BigDecimal avgPrice = BigDecimal.ZERO;
                BigDecimal taxAvgPrice = BigDecimal.ZERO;
                for (RentPriceContractEntity entity : list) {
                    num = num.add(entity.getContractNum());
                    mny = mny.add(entity.getContractMny());
                    taxMny = taxMny.add(entity.getContractTaxMny());
                }
                if (num.compareTo(BigDecimal.ZERO) != 0) {
                    avgPrice = mny.divide(num, 4, BigDecimal.ROUND_HALF_UP);
                    taxAvgPrice = taxMny.divide(num, 4, BigDecimal.ROUND_HALF_UP);
                }
                PriceTrendVo vo;
                String dateStr = DateUtils.transformationDefaultDate(entry.getKey());
                if (trendMap.containsKey(dateStr)) {
                    vo = trendMap.get(dateStr);
                }
                else {
                    vo = new PriceTrendVo();
                    trendMap.put(dateStr, vo);
                }
                vo.setContractPrice(rentPriceContractEntity.getContractPrice());
                vo.setContractTaxPrice(rentPriceContractEntity.getContractTaxPrice());
                vo.setContractNum(num);
                vo.setContractAvgPrice(avgPrice);
                vo.setContractMaxPrice(Collections.max(priceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP));
                vo.setContractMinPrice(Collections.min(priceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP));
                vo.setContractTaxAvgPrice(taxAvgPrice);
                vo.setContractTaxMaxPrice(Collections.max(taxPriceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP));
                vo.setContractTaxMinPrice(Collections.min(taxPriceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP));
                vo.setPriceDate(DateUtils.transformationDefaultDate(entry.getKey()));
            }
        }
        // map转list
        rtnList = new ArrayList<>(trendMap.values());
        // 按照日期排序
        if (CollectionUtils.isNotEmpty(rtnList)) {
            rtnList.sort(Comparator.comparing(PriceTrendVo::getPriceDate));
        }
        return rtnList;
    }

    /**
     * 移除合同明细
     *
     * @param id 明细id
     */
    @Override
    public void removePriceContractDetail(Long id) {
        RentPriceContractEntity entity = super.selectById(id);
        if (entity == null) throw new BusinessException("明细不存在");
        entity.setOutFlag(OutFlagEnum.已移除.getCode());
        super.saveOrUpdate(entity, false);
    }

    /**
     * 根据档案id查询对应的价格库合同明细
     *
     * @param param 查询参数
     * @return 查询结果
     */
    @Override
    public IPage<RentPriceContractVO> queryPriceContractListDetail(QueryParam param) {
        Map<String, Date> checkMap = this.checkQueryParam(param);
        Date startDate = checkMap.get("startDate");
        Date endDate = checkMap.get("endDate");
        if (param.getOrderMap().isEmpty()) {
            param.getOrderMap().put("signDate",QueryParam.DESC);
            param.getOrderMap().put("contractMny",QueryParam.DESC);
        }
        List<Long> orgIdList = this.queryOrgIdList();
        List<RentPriceContractEntity> priceContractList = this.queryPriceContractList(startDate, orgIdList, param);
        List<RentPriceContractVO> voList;
        IPage<RentPriceContractVO> pageDate = new Page<>(param.getPageIndex(), param.getPageSize(), priceContractList.size());
        if (CollectionUtils.isEmpty(priceContractList)) return pageDate;
        voList = BeanMapper.mapList(priceContractList, RentPriceContractVO.class);
        // 处理合同URL
        List<Long> myOrgList = this.getMyOrgList();
        for (RentPriceContractVO vo : voList) {
            if (!myOrgList.contains(vo.getOrgId())) {
                vo.setPcCardUrl(null);
            }
        }
        // 进行分页处理
        pageDate.setRecords(voList.stream().skip((param.getPageIndex() - 1) * param.getPageSize())
                .limit(param.getPageSize()).collect(Collectors.toList()));
        return pageDate;
    }
    /**
     * 查询当前组织本下
     * @return 组织本下list
     */
    private List<Long> getMyOrgList() {
        CommonResponse<List<OrgVO>> orgVOResp = iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if (!orgVOResp.isSuccess()) {
            throw new BusinessException("查询组织信息失败！");
        }
        List<OrgVO> orgVOList = orgVOResp.getData();
        if (CollectionUtils.isNotEmpty(orgVOList)) {
            return orgVOList.stream().map(OrgVO::getId).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }


    /**
     * 校验查询条件
     *
     * @param param 查询条件
     * @return 日期
     */
    private Map<String, Date> checkQueryParam(QueryParam param) {
        Date startDate;
        Date endDate;
        Map<String, Parameter> paramMap = param.getParams();
        if (paramMap.containsKey("startDate") && paramMap.containsKey("endDate")) {
            startDate = DateUtils.createDate(paramMap.get("startDate").getValue().toString());
            endDate = DateUtils.createDate(paramMap.get("endDate").getValue().toString());
            // 清除日期信息，保留其他查询条件
            paramMap.remove("startDate");
            paramMap.remove("endDate");
        }
        else {
            throw new BusinessException("查询日期不能为空");
        }
        if (!paramMap.containsKey("docId")) {
            throw new BusinessException("查询档案id不能为空！");
        }
        Map<String, Date> rtnMap = new HashMap<>();
        rtnMap.put("startDate", startDate);
        rtnMap.put("endDate", endDate);
        return rtnMap;
    }


    /**
     * 查询价格库信息(参照查询列表查询都走此方法)
     *
     * @param startDate 开始日期
     * @param endDate   结束日期
     * @param orgIdList 查询结果
     * @param param     查询条件
     * @return 查询结果
     */
    private List<PriceLibVO> queryPriceLibListAll(Date startDate, Date endDate, List<Long> orgIdList,
                                                  QueryParam param) {

        Map<String, PriceLibVO> map = new HashMap<>();
        this.queryPriceContractGroupByDocId(map, startDate, orgIdList, param);
        this.queryPriceGuide(map);
        List<PriceLibVO> priceLibList = new ArrayList<>(map.values());
        if (CollectionUtils.isNotEmpty(priceLibList)) {
            priceLibList.sort(Comparator.comparing(PriceLibVO::getDocCode, Comparator.nullsLast(String::compareTo)).thenComparing(PriceLibVO::getRentType, Comparator.nullsLast(Integer::compareTo)));
        }
        return priceLibList;
    }

    /**
     * 查询合同价格库信息 通过档案id查询该有该档案的合同
     *
     * @param libMap    价格库组装列表
     * @param startDate 开始日期
     * @param orgIdList 权限列表
     * @return 合同价格库列表
     *
     *
     *
     *
     * 需要改造--
     *
     *
     */
    private void queryPriceContractGroupByDocId(Map<String, PriceLibVO> libMap, Date startDate,
                                                List<Long> orgIdList, QueryParam param) {
        // 截止到当前日期状态为履约中的合同+已终止、已结束的合同取对应最终结算日期大于等于开始日期的合同
        List<RentPriceContractEntity> priceContractList = this.queryPriceContractList(startDate, orgIdList, param);
        if (CollectionUtils.isEmpty(priceContractList)) return;

        // 按照档案id + 租赁类型分组
        Map<Long, Map<Integer, List<RentPriceContractEntity>>> priceContractGroupMap = priceContractList.stream()
                .collect(Collectors.groupingBy(RentPriceContractEntity::getDocId, Collectors.groupingBy(RentPriceContractEntity::getRentType)));

//        System.out.println(JSONObject.toJSONString(priceContractGroupMap));
//        System.out.println("123");

        Map<String, List<RentPriceContractEntity>> priceContractMap = new HashMap<>();
        for (Map.Entry<Long, Map<Integer, List<RentPriceContractEntity>>> entry : priceContractGroupMap.entrySet()) {
            Map<Integer, List<RentPriceContractEntity>> priceContractOfRentTypeMap = entry.getValue();
            for (Map.Entry<Integer, List<RentPriceContractEntity>> entryTwo : priceContractOfRentTypeMap.entrySet()) {
                //如果档案id下，日租、月租、工程量租同类型下的档案数大于1，则加入集合
                if (entryTwo.getValue().size() > 1){
                    if (priceContractMap.containsKey(entry.getKey() + "" + entryTwo.getValue().get(0).getRentType())){
                        List<RentPriceContractEntity> allRentPriceContractList = priceContractMap.get(entry.getKey());
                        allRentPriceContractList.addAll(entryTwo.getValue());
                        priceContractMap.put(entry.getKey() + "" + entryTwo.getValue().get(0).getRentType(), allRentPriceContractList);
                    }else {
                        priceContractMap.put(entry.getKey() + "" + entryTwo.getValue().get(0).getRentType(), entryTwo.getValue());
                    }
                }
            }
        }


        // 汇总信息
        for (Map.Entry<String, List<RentPriceContractEntity>> entry : priceContractMap.entrySet()) {
            String key = entry.getKey();
            List<RentPriceContractEntity> value = entry.getValue();
            List<BigDecimal> priceList = value.stream().map(RentPriceContractEntity::getContractPrice).collect(Collectors.toList());
            List<BigDecimal> taxPriceList = value.stream().map(RentPriceContractEntity::getContractTaxPrice).collect(Collectors.toList());
            RentPriceContractEntity priceContract = value.get(0);
            if (priceContract == null) continue;
            BigDecimal num = BigDecimal.ZERO; // 数量
            BigDecimal mny = BigDecimal.ZERO; // 金额(无税)
            BigDecimal taxRate = BigDecimal.ZERO; // 税率
            BigDecimal taxMny = BigDecimal.ZERO; // 金额(含税)
            BigDecimal avgPrice = BigDecimal.ZERO; // 合同均价(无税)
            // 合同最高价(无税)
            BigDecimal maxPrice = Collections.max(priceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
            // 合同最低价(无税)
            BigDecimal minPrice = Collections.min(priceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
            String priceArea = minPrice + "-" + maxPrice; // 合同价格区间（无税）
            BigDecimal taxAvgPrice = BigDecimal.ZERO; // 合同均价（含税）
            // 合同最高价（含税）
            BigDecimal taxMaxPrice = Collections.max(taxPriceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
            // 合同最低价（含税）
            BigDecimal taxMinPrice = Collections.min(taxPriceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
            String taxPriceArea = taxMinPrice + "-" + taxMaxPrice; // 合同价格区间（含税）
            for (RentPriceContractEntity entity : value) {
                num = num.add(entity.getContractNum());
                mny = mny.add(entity.getContractMny());
                taxMny = taxMny.add(entity.getContractTaxMny());
            }
            if (num.compareTo(BigDecimal.ZERO) != 0) {
                avgPrice = mny.divide(num, 4, BigDecimal.ROUND_HALF_UP);
                taxAvgPrice = taxMny.divide(num, 4, BigDecimal.ROUND_HALF_UP);
            }
            taxRate = this.calculationTaxRate(taxMny, mny);
            PriceLibVO priceLibVO;
//            if (libMap.containsKey(key)) {
//                priceLibVO = libMap.get(key);
//            } else {
                priceLibVO = new PriceLibVO();
                priceLibVO.setDocId(priceContract.getDocId());
                priceLibVO.setDocCode(priceContract.getDocCode());
                priceLibVO.setDocName(priceContract.getDocName());
                priceLibVO.setDocCategoryId(priceContract.getDocCategoryId());
                priceLibVO.setDocCategoryName(priceContract.getDocCategoryName());
                priceLibVO.setUnitId(priceContract.getUnitId());
                priceLibVO.setUnitName(priceContract.getUnitName());
                priceLibVO.setDocInnerCode(priceContract.getDocInnerCode());
                priceLibVO.setUnitName(priceContract.getUnitName());
                priceLibVO.setSpec(priceContract.getSpec());
                priceLibVO.setRentType(priceContract.getRentType());
                if (null != priceLibVO.getRentType()){
                    switch (priceLibVO.getRentType()) {
                        case 1:
                            priceLibVO.setRentTypeName("日租");
                            break;
                        case 2:
                            priceLibVO.setRentTypeName("月租");
                            break;
                        default:
                            priceLibVO.setRentTypeName("工程量租");
                            break;
                    }
                }
                //key为 档案id + 租赁方式
                libMap.put(priceContract.getDocId() + "" + priceContract.getRentType(), priceLibVO);
//            }
            priceLibVO.setContractNum(num);
            priceLibVO.setContractMny(mny);
            priceLibVO.setContractTaxRate(taxRate);
            priceLibVO.setContractTaxMny(taxMny);
            priceLibVO.setContractAvgPrice(avgPrice);
            priceLibVO.setContractMaxPrice(maxPrice);
            priceLibVO.setContractMinPrice(minPrice);
            priceLibVO.setContractPriceArea(priceArea);
            priceLibVO.setContractTaxAvgPrice(taxAvgPrice);
            priceLibVO.setContractTaxMaxPrice(taxMaxPrice);
            priceLibVO.setContractTaxMinPrice(taxMinPrice);
            priceLibVO.setContractTaxPriceArea(taxPriceArea);
            priceLibVO.setId(priceLibVO.getDocId() + priceLibVO.getRentType());
            // TODO: 2022/8/29 市场价赋值
        }
    }

    /**
     * 查询符合条件的价格库合同明细
     *
     * @param startDate 开始日期
     * @param orgIdList 权限列表
     * @param param     查询参数
     * @return 查询结果
     */
    private List<RentPriceContractEntity> queryPriceContractList(Date startDate, List<Long> orgIdList,
                                                             QueryParam param) {
        // 截止到当前日期状态为履约中的合同+已终止、已结束的合同取对应最终结算日期大于等于开始日期的合同
        QueryParam contractParam = new QueryParam();
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = contractParam.getFuzzyFields();
        fuzzyFields.add("docCode");
        fuzzyFields.add("docName");
        fuzzyFields.add("unitName");
        fuzzyFields.add("area");
        fuzzyFields.add("areaName");
        contractParam.setSearchText(param.getSearchText());
        contractParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        contractParam.getParams().put("outFlag", new Parameter(QueryParam.EQ, OutFlagEnum.未移除.getCode()));

        // 分包价格库查询价格类型
//        CommonResponse<ParamRegisterSetVO> response = paramConfigApi.getByCode(PRICE_LIB_CHECK_TYPE);
//        if (!response.isSuccess() || response.getData() == null) {
//            throw new BusinessException("获取分包价格库查询价格类型系统参数请求失败，失败原因：" + response.getMsg());
//        }
//        String valueData = response.getData().getValueData();
//        if (StringUtils.isNotBlank(valueData) && !valueData.contains("全部")){
//            contractParam.getParams().put("priceTypeName", new Parameter(QueryParam.IN, Arrays.asList(valueData.split(","))));
//        }

        if (CollectionUtils.isNotEmpty(orgIdList)) {
            contractParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgIdList));
        }
        if (param != null) {
            contractParam.getParams().putAll(param.getParams());
            if (CollectionUtils.isNotEmpty(param.getComplexParams())) {
                contractParam.getComplexParams().addAll(param.getComplexParams());
            }
            if (!param.getOrderMap().isEmpty()) {
                contractParam.getOrderMap().putAll(param.getOrderMap());
            }
        }
        ComplexParam c1 = new ComplexParam();
        c1.setLogic(ComplexParam.AND);
        // 履约中
        ComplexParam c2 = new ComplexParam();
        c2.setLogic(ComplexParam.OR);
        c2.getParams().put("performance_status", new Parameter(QueryParam.IN, Arrays.asList(2, 5)));
//        c2.getParams().put("sign_date", new Parameter(QueryParam.GE, startDate));
        c1.getComplexParams().add(c2);
        // 签订日期大于等于开始日期的合同
        ComplexParam c3 = new ComplexParam();
        c3.setLogic(ComplexParam.OR);
        // 履约中+已冻结+已终止+已结束
        c3.getParams().put("performance_status", new Parameter(QueryParam.IN, Arrays.asList(3, 6)));
        c3.getParams().put("settlement_date", new Parameter(QueryParam.GE, startDate));
        c1.getComplexParams().add(c3);
        contractParam.getComplexParams().add(c1);
        return super.queryList(contractParam, false);
    }


    /**
     * 查询组织权限
     * 数据权限范围：
     * 通过当前上下文组织查询本上是否设置价格保密（从价格库权限设置中查询）。
     * 查到本上第一个设置价格保密的组织，以这个组织查询本下数据。
     * 如没有查到设置价格保密的组织，那么以根组织查询本下数据；
     *
     * @return 权限列表
     */
    private List<Long> queryOrgIdList() {
        CommonResponse<List<Long>> orgIdListResponse = priceDepotPowerSetApi
                .queryOrgList(InvocationInfoProxy.getOrgId());
        logger.info("查询组织权限，查询结果：{}", JSONObject.toJSONString(orgIdListResponse));
        if (!orgIdListResponse.isSuccess()) {
            throw new BusinessException("查询组织权限失败:" + orgIdListResponse.getMsg());
        }
        return orgIdListResponse.getData();
    }


    /**
     * 计算税率（（含税/无税 - 1）*100）
     *
     * @param taxMny 含税金额
     * @param mny    无税金额
     * @return 税率
     */
    private BigDecimal calculationTaxRate(BigDecimal taxMny, BigDecimal mny) {
        BigDecimal b = new BigDecimal("100");
        if (mny.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        }
        return taxMny.divide(mny, 4, BigDecimal.ROUND_HALF_UP).subtract(BigDecimal.ONE).multiply(b);
    }

    /**
     * 查询公司指导价
     *
     * @param libMap 价格库信息
     */
    private void queryPriceGuide(Map<String, PriceLibVO> libMap) {
        if (libMap.isEmpty()) return;
        List<RentPriceGuideDetailEntity> list = priceGuideDetailService.list();
        if (CollectionUtils.isEmpty(list)) return;
        Map<String, RentPriceGuideDetailEntity> guideMap = new HashMap<>();
        for (RentPriceGuideDetailEntity rentPriceGuideDetailEntity : list) {
            guideMap.put(rentPriceGuideDetailEntity.getDocId() + "" + rentPriceGuideDetailEntity.getRentType(), rentPriceGuideDetailEntity);
        }
        for (Map.Entry<String, PriceLibVO> entry : libMap.entrySet()) {
            String docIdRentType = entry.getKey();
            PriceLibVO vo = entry.getValue();
            if (guideMap.containsKey(docIdRentType)) {
                RentPriceGuideDetailEntity guide = guideMap.get(docIdRentType);
                BigDecimal min = guide.getMinPrice() == null ? BigDecimal.ZERO : guide.getMinPrice();
                BigDecimal max = guide.getMaxPrice() == null ? BigDecimal.ZERO : guide.getMaxPrice();
                min = min.divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
                max = max.divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
                vo.setGuidePriceArea(min + "-" + max);
            }
        }
    }
}
